<?php

namespace WDevs\CustomerOrderExport\Admin;

use WDevs\CustomerOrderExport\Ajax;
use WDevs\CustomerOrderExport\Helper;
use WDevs\CustomerOrderExport\Template;
use WDevs\CustomerOrderExport\Variables;

/**
 * The admin-specific functionality of the plugin.
 *
 * @link       https://wijnberg.dev
 * @since      1.0.0
 *
 * @package    Woocommerce_Order_Items_Attachment
 * @subpackage Woocommerce_Order_Items_Attachment/admin
 */

/**
 * The admin-specific functionality of the plugin.
 *
 * Defines the plugin name, version, and two examples hooks for how to
 * enqueue the admin-specific stylesheet and JavaScript.
 *
 * @package    Woocommerce_Order_Items_Attachment
 * @subpackage Woocommerce_Order_Items_Attachment/admin
 * @author     Wijnberg Developments <contact@wijnberg.dev>
 */
class CustomerOrderExportAdmin {

	use Helper;

	/**
	 * The ID of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string $plugin_name The ID of this plugin.
	 */
	private $plugin_name;

	/**
	 * The version of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string $version The current version of this plugin.
	 */
	private $version;


	/**
	 * Initialize the class and set its properties.
	 *
	 * @param string $plugin_name The name of this plugin.
	 * @param string $version The version of this plugin.
	 *
	 * @since    1.0.0
	 */
	public function __construct( $plugin_name, $version ) {
		$this->plugin_name = $plugin_name;
		$this->version     = $version;
	}

	/**
	 * Register the stylesheets for the admin area.
	 *
	 * @since    1.0.0
	 */
	public function enqueue_styles() {
		wp_enqueue_style( $this->plugin_name, plugin_dir_url( __FILE__ ) . 'css/wdevs-customer-order-export-admin.css', array(), $this->version, 'all' );
	}

	/**
	 * Register the JavaScript for the admin area.
	 *
	 * @since    1.0.0
	 */
	public function enqueue_scripts() {

		if ( is_admin() ) {
			$page = isset( $_GET['page'] ) ? sanitize_text_field( wp_unslash( $_GET['page'] ) ) : '';
			$tab  = isset( $_GET['tab'] ) ? sanitize_text_field( wp_unslash( $_GET['tab'] ) ) : '';

			if ( $page === 'wc-settings' && $tab === 'wdevs_coe' ) {
				$script_asset = require( plugin_dir_path( dirname( __FILE__ ) ) . 'build/wdevs-coe-admin.asset.php' );
				$admin_handle = $this->plugin_name . '-admin';
				wp_enqueue_script(
					$admin_handle,
					plugin_dir_url( dirname( __FILE__ ) ) . 'build/wdevs-coe-admin.js',
					array_merge( $script_asset['dependencies'], [
						'wp-i18n',
						'wp-util',
						'underscore',
						'backbone',
						'wc-backbone-modal',
						'jquery-ui-sortable'
					] ),
					$script_asset['version'],
					true
				);

				$localized_object = [
					'nonce' => wp_create_nonce( Ajax::AJAX_ACTION_NONCE ),
				];

				$section = isset( $_GET['section'] ) ? sanitize_text_field( wp_unslash( $_GET['section'] ) ) : '';

				//main settings
				if ( empty( $section ) ) {
					$main_settings_localized_object = [
						'variables' => Variables::get_variables(),
					];

					$localized_object = array_merge( $localized_object, $main_settings_localized_object );
				} elseif ( $section === 'templates' ) {
					$template_id               = Template::get_requested_template_id();
					$template_localized_object = [
						'ajaxUrl' => admin_url( 'admin-ajax.php' ),
					];

					if ( isset( $template_id ) ) {
						$template                  = new Template( $template_id );
						$template_localized_object += [
							'selectableColumns'     => $template->get_selectable_columns(),
							'name'                  => $template->get_name(),
							'id'                    => $template->get_id(),
							'columnsFlattened'      => $template->get_selected_columns_flattened(),
							'templateColumnsAction' => Ajax::AJAX_ACTION_TEMPLATE_COLUMNS,
							'strings'               => [
								'unload_confirmation_msg'    => __( 'Your changed data will be lost if you leave this page without saving.', 'orderbridge-for-woocommerce' ),
								'save_changes_prompt'        => __( 'Do you wish to save your changes first? Your changed data will be discarded if you choose to cancel.', 'orderbridge-for-woocommerce' ),
								'save_failed'                => __( 'Your changes were not saved. Please retry.', 'orderbridge-for-woocommerce' ),
								'add_column_failed'          => __( 'Column could not be added. Please retry.', 'orderbridge-for-woocommerce' ),
								'remove_column_failed'       => __( 'Column could not be removed. Please retry.', 'orderbridge-for-woocommerce' ),
								'yes'                        => __( 'Yes', 'orderbridge-for-woocommerce' ),
								'no'                         => __( 'No', 'orderbridge-for-woocommerce' ),
								'default_template_name'      => Template::get_default_name(),
								'delete_column_confirmation' => __( 'Are you sure you want to delete this column?', 'orderbridge-for-woocommerce' ),
							],
						];
					} else {
						$template_localized_object += [
							'templateDeleteAction' => Ajax::AJAX_ACTION_TEMPLATE_DELETE,
							'strings'              => [
								'delete_template_confirmation' => __( 'Are you sure you want to delete this template?', 'orderbridge-for-woocommerce' ),
							],
						];
					}

					$localized_object = array_merge( $localized_object, $template_localized_object );
				} elseif ( $section === 'preview' ) {
					$preview_localized_object = [
						'postUrl'       => admin_url( 'admin-post.php' ),
						'previewAction' => Ajax::POST_ACTION_PREVIEW_FILE,
					];

					$localized_object = array_merge( $localized_object, $preview_localized_object );
				}

				wp_localize_script(
					$admin_handle,
					'wdceoData',
					$localized_object
				);
			}
		}
	}

	/**
	 * @since 1.0.0
	 */
	public function add_action_links( $actions ) {
		$links = array(
			'<a href="' . admin_url( 'admin.php?page=wc-settings&tab=wdevs_coe' ) . '">' . __( 'Settings', 'orderbridge-for-woocommerce' ) . '</a>',
		);

		$actions = array_merge( $actions, $links );

		return $actions;
	}

}
