<?php
/**
 * Uninstall script for Order Status & Tracking Emails for WooCommerce
 *
 * Cleans up plugin options, meta fields, and reverts custom order statuses
 * to prevent order invisibility after uninstall.
 *
 * @package OSTE
 */

// Prevent direct access
if ( ! defined( 'WP_UNINSTALL_PLUGIN' ) ) {
	exit;
}

global $wpdb;

/**
 * STEP 1: Ripristina ordini con post_status personalizzati a uno stato valido WooCommerce
 * Questo evita che ordini con stati tipo 'wc-oste_mio_stato' diventino invisibili dopo la disinstallazione.
 */

$oste_core_statuses = array(
	'wc-pending',
	'wc-processing',
	'wc-on-hold',
	'wc-completed',
	'wc-cancelled',
	'wc-refunded',
	'wc-failed',
);

// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
$oste_custom_statuses = $wpdb->get_col("
	SELECT DISTINCT post_status
	FROM {$wpdb->posts}
	WHERE post_type = 'shop_order'
	AND post_status LIKE 'wc-%'
");

$oste_custom_statuses_to_convert = array_diff( $oste_custom_statuses, $oste_core_statuses );
$oste_fallback_status = 'wc-on-hold';

foreach ( $oste_custom_statuses_to_convert as $oste_old_status ) {
	$wpdb->query( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
		$wpdb->prepare(
			"UPDATE {$wpdb->posts} SET post_status = %s WHERE post_status = %s",
			$oste_fallback_status,
			$oste_old_status
		)
	);
}

/**
 * STEP 2: Elimina tutte le opzioni salvate
 */
$oste_option_keys = array(
	'oste_gestione_email_stati_ordine',
	'oste_email_style_settings',
	'oste_email_styling',
	'oste_tracking_field_settings',
	'oste_order_actions_settings',
	'oste_tracking_carriers',
	'oste_tracking_custom_message',
	'oste_tracking_message_position',
	'oste_custom_statuses',
	'oste_base_status_labels',
);

foreach ( $oste_option_keys as $oste_key ) {
	delete_option( $oste_key );
	delete_site_option( $oste_key );
}

/**
 * STEP 3: Elimina tutti i metadati associati agli ordini
 */

// Chiavi di meta note
$oste_meta_keys = array(
	'_custom_tracking_number',
	'_custom_tracking_url',
	'_custom_tracking_carrier',
	'_custom_tracking_message',
	'_custom_tracking_position',
	'_oste_carrier',
	'_oste_tracking',
);

// Processa ordini in batch
$oste_batch_size = 200;
$oste_paged      = 1;

do {
	$oste_orders = get_posts( array(
		'post_type'      => 'shop_order',
		'post_status'    => 'any',
		'posts_per_page' => $oste_batch_size,
		'paged'          => $oste_paged,
		'fields'         => 'ids',
		'no_found_rows'  => true,
		'orderby'        => 'ID',
		'order'          => 'ASC',
	) );

	if ( empty( $oste_orders ) ) {
		break;
	}

	foreach ( $oste_orders as $oste_order_id ) {
		foreach ( $oste_meta_keys as $oste_key ) {
			delete_post_meta( $oste_order_id, $oste_key );
		}
	}

	$oste_paged++;
} while ( count( $oste_orders ) === $oste_batch_size );

/**
 * STEP 4: Rimozione avanzata di metadati dinamici
 */

// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
$wpdb->query( "DELETE FROM {$wpdb->postmeta} WHERE meta_key LIKE '_custom_email_sent_for_status_%'" );
// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
$wpdb->query( "DELETE FROM {$wpdb->postmeta} WHERE meta_key LIKE 'oste_%'" );
