<?php
/**
 * Plugin Name:       Order Status & Tracking Emails for WooCommerce
 * Description:       Custom order statuses, tracking fields, and transactional emails for WooCommerce.
 * Version:           1.0.4
 * Requires at least: 6.0
 * Tested up to:      6.9
 * Requires PHP:      7.4
 * Requires Plugins:  woocommerce
 * Author:            imbro
 * Author URI:        https://www.luciasoscia.com/
 * Text Domain:       order-status-tracking-emails-for-woocommerce
 * Domain Path:       /languages
 * License:           GPLv2 or later
 * License URI:       https://www.gnu.org/licenses/gpl-2.0.html
 */


if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

// --- Plugin constants -------------------------------------------------------
if ( ! defined( 'OSTE_VERSION' ) ) {
	define( 'OSTE_VERSION', '1.0.0' ); // mantienila in sync con l'header
}
if ( ! defined( 'OSTE_PLUGIN_FILE' ) ) {
	define( 'OSTE_PLUGIN_FILE', __FILE__ );
}
if ( ! defined( 'OSTE_PLUGIN_DIR' ) ) {
	define( 'OSTE_PLUGIN_DIR', plugin_dir_path( __FILE__ ) );
}
if ( ! defined( 'OSTE_PLUGIN_URL' ) ) {
	define( 'OSTE_PLUGIN_URL', plugin_dir_url( __FILE__ ) );
}


// Stop se WooCommerce non è attivo
function oste_is_wc_active() {
    return class_exists('WooCommerce');
}

add_action( 'admin_init', function () {
	// Solo in admin, evita AJAX/cron
	if ( ! is_admin() || wp_doing_ajax() ) {
		return;
	}

	if ( ! oste_is_wc_active() ) {
		// Disattiva se WooCommerce manca
		deactivate_plugins( plugin_basename( __FILE__ ) );

		// Niente lettura di $_GET: ricarica la pagina plugin senza 'activate'
		wp_safe_redirect( self_admin_url( 'plugins.php' ) );
		exit;
	}
} );


add_action('admin_notices', function () {
    if ( ! oste_is_wc_active() ) {
        echo '<div class="notice notice-error"><p>'
           . esc_html__('Order Status & Tracking Emails for WooCommerce requires WooCommerce to be installed and active.', 'order-status-tracking-emails-for-woocommerce')
           . '</p></div>';
    }
});

// ===== Ledger: segna quando una email è stata "costruita" per un ordine (solo per questa request)
add_action('woocommerce_email_before_order_table', function($order, $sent_to_admin, $plain_text, $email){
    if ( $order instanceof WC_Order && isset($email->id) ) {
        if ( ! isset($GLOBALS['oste_sent_runtime']['built'][ $email->id ]) ) {
            $GLOBALS['oste_sent_runtime']['built'][ $email->id ] = [];
        }
        $GLOBALS['oste_sent_runtime']['built'][ $email->id ][ $order->get_id() ] = true;
    }
}, 1, 4);






// INCLUDES

$oste_includes = [
    'oste-functions.php',
    'register-statuses.php',
    'admin-panel.php',
    'oste-carriers.php',
    'oste-tracking-fields.php',
    'oste-admin-email-stati-ordine.php',
    'oste-funzioni-stati-ordine.php',
    'oste-admin-stili-email.php',
    // se usi l’email custom come file separato, aggiungila:
    // 'class-oste-email-customer-custom-order.php',
];

foreach ($oste_includes as $oste_include) {
    $oste_include_path = OSTE_PLUGIN_DIR . 'includes/' . $oste_include;
    if ( file_exists($oste_include_path) ) {
        require_once $oste_include_path;
    }
}




// Registrazione globale (non enqueua ancora)
add_action( 'init', function() {
    // Front
    wp_register_style(
        'oste-front',
        OSTE_PLUGIN_URL . 'assets/css/front.css',
        [],
        OSTE_VERSION
    );
    wp_register_script(
        'oste-front',
        OSTE_PLUGIN_URL . 'assets/js/front.js',
        [ 'jquery' ],
        OSTE_VERSION,
        true
    );

    // Admin
    wp_register_style(
        'oste-admin',
        OSTE_PLUGIN_URL . 'assets/css/admin.css',
        [],
        OSTE_VERSION
    );
    wp_register_style(
        'oste-admin-ui',
        OSTE_PLUGIN_URL . 'assets/css/admin-ui.css',
        [],
        OSTE_VERSION
    );
    wp_register_script(
        'oste-admin',
        OSTE_PLUGIN_URL . 'assets/js/admin.js',
        [ 'jquery' ],
        OSTE_VERSION,
        true
    );
} );

// Enqueue in ADMIN solo nelle tue schermate
add_action('admin_enqueue_scripts', function ($hook) {
    $screen = function_exists('get_current_screen') ? get_current_screen() : null;
    $screen_id = $screen ? $screen->id : '';

    // --- 1) Le tue pagine admin del plugin ---
    $allowed = [
        'woocommerce_page_oste-statuses',
        'woocommerce_page_oste-email-styles',
        'woocommerce_page_oste-carriers',
        'woocommerce_page_oste-email-templates',
        'oste-order-manager_page_oste-email-templates',
    ];
    $allowed_hooks = [
        'woocommerce_page_oste-statuses',
        'woocommerce_page_oste-email-styles',
        'woocommerce_page_oste-carriers',
        'woocommerce_page_oste-email-templates',
        'oste-order-manager_page_oste-email-templates',
    ];
    if (
        in_array($screen_id, $allowed, true) ||
        in_array($hook, $allowed_hooks, true) ||
        ( is_string($hook) && false !== strpos($hook, 'oste-') )
    ) {
        wp_enqueue_style('oste-admin');
        wp_enqueue_style(
            'oste-admin-ui',
            OSTE_PLUGIN_URL . 'assets/css/admin-ui.css',
            [],
            OSTE_VERSION
        );
        wp_enqueue_script('oste-admin');

        // opzionale: CSS dinamico
        $dynamic_css = '';
        if ( $dynamic_css ) {
            wp_add_inline_style('oste-admin', $dynamic_css);
        }

        wp_localize_script('oste-admin', 'osteAdmin', [
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'nonce'   => wp_create_nonce('oste_admin_nonce'),
        ]);
    }

    // --- 2) Lista ordini (edit-shop_order): inline per salvataggio veloce tracking ---
    if ( $screen_id === 'edit-shop_order' ) {
        wp_enqueue_script('oste-admin');

        wp_localize_script('oste-admin', 'osteInline', [
            'ajaxUrl'   => admin_url('admin-ajax.php'),
            'nonceSave' => wp_create_nonce('oste_save_tracking_inline'),
            'nonceGet'  => wp_create_nonce('oste_get_tracking_link'),
            'txtCheck'  => __('Check Tracking', 'order-status-tracking-emails-for-woocommerce'),
            'txtError'  => __('Error saving tracking.', 'order-status-tracking-emails-for-woocommerce'),
        ]);

        wp_add_inline_script('oste-admin', "
jQuery(function($){
  const t = osteInline;

  $(document).on('click', '.oste-inline-edit-tracking .oste-save-track', function(){
    const wrap     = $(this).closest('.oste-inline-edit-tracking');
    const order    = wrap.data('order');
    const tracking = wrap.find('input[name=\"oste_tracking\"]').val();
    const carrier  = wrap.find('select[name=\"oste_carrier\"]').val();
    const \$btn    = $(this);

    \$btn.prop('disabled', true);

    $.post(t.ajaxUrl, {
      action: 'oste_save_tracking_inline',
      order_id: order,
      tracking: tracking,
      carrier: carrier,
      _wpnonce: t.nonceSave
    }).done(function(resp){
      if (resp && resp.success) {
        wrap.css('background','#e5ffe5'); setTimeout(()=>wrap.css('background',''),800);

        $.post(t.ajaxUrl, {
          action: 'oste_get_tracking_link',
          order_id: order,
          carrier: carrier,
          tracking: tracking,
          _wpnonce: t.nonceGet
        }).done(function(data){
          const link = (data && data.success && data.data && data.data.link) ? data.data.link : '';
          const html = link ? '<a href=\"'+link+'\" target=\"_blank\" rel=\"noopener\" style=\"font-size:12px;color:#2271b1;text-decoration:underline;\">'+ t.txtCheck +'</a>' : '';
          wrap.find('.oste-link-wrap').html(html);
        });
      } else {
        alert(t.txtError);
      }
    }).always(function(){ \$btn.prop('disabled', false); });
  });

  $(document).on('mousedown click focus', '.oste-inline-edit-tracking input, .oste-inline-edit-tracking select, .oste-inline-edit-tracking button', function(e){
    e.stopPropagation();
  });
});
        ");
    }
});




// Filtra la ricerca dei template WooCommerce (funziona da WC 3.x a 8.x)
add_filter('woocommerce_locate_template', function($template, $template_name, $template_path) {
    if (strpos($template_name, 'emails/') === 0) {
        $plugin_path = OSTE_PLUGIN_DIR . 'templates/' . $template_name;
        if (file_exists($plugin_path)) {
            return $plugin_path;
        }
    }
    return $template;
}, 20, 3);

// (Facoltativo ma consigliato per WooCommerce 8.x+)
add_filter('woocommerce_email_locate_template', function($located, $template_name, $args) {
    $plugin_path = OSTE_PLUGIN_DIR . 'templates/' . $template_name;
    if (file_exists($plugin_path)) {
        return $plugin_path;
    }
    return $located;
}, 10, 3);

add_filter('woocommerce_locate_template', function($template, $template_name, $template_path) {
    // Solo email cliente tipo "customer-foo-order.php"
    if (strpos($template_name, 'emails/customer-') === 0 && substr($template_name, -10) === '-order.php') {
        if (preg_match('#^emails/customer-([a-z0-9\-_]+)-order\.php$#', $template_name, $m)) {
            $slug = $m[1]; // es. 'spedito', 'ritiro', 'processing', ...

            // Limita SOLO ai tuoi stati custom (senza 'wc-')
            if (function_exists('oste_get_custom_statuses_slug')) {
                $custom_slugs = (array) oste_get_custom_statuses_slug(); // ['spedito','ritiro',...]
                if (in_array($slug, $custom_slugs, true)) {
                    $custom_template = OSTE_PLUGIN_DIR . 'templates/emails/customer-custom-order.php';
                    if (file_exists($custom_template)) {
                        return $custom_template;
                    }
                }
            }
        }
    }
    return $template;
}, 25, 3);


// === EMAIL PIPELINE (core + custom) ===


// Registra la email custom in modo tollerante a nome classe
add_filter('woocommerce_email_classes', function($emails) {
    $path = OSTE_PLUGIN_DIR . 'includes/class-oste-email-customer-custom-order.php';
    if ( file_exists($path) ) {
        require_once $path;
        if ( class_exists('OSTE_Email_Customer_Custom_Order') ) {
            $emails['oste_customer_custom_order'] = new OSTE_Email_Customer_Custom_Order();
        }
    }
    return $emails;
});



// Tutta la pipeline email dopo che Woo è sicuramente inizializzato
add_action('woocommerce_init', function() {
	if ( ! oste_is_wc_active() ) return;
    // Mappa ID email Woo -> slug stato
    $woo_emails = WC()->mailer()->get_emails() ?: [];
    $slug_to_email_hook = [];
    $id_email_to_slug   = [];
	$slug_to_email_id   = [];
	
    foreach ($woo_emails as $email_class) {
        if (!isset($email_class->id)) continue;
        if (preg_match('/customer[_\-]([a-z0-9_\-]+)[_\-]order/i', $email_class->id, $m)) {
            $slug = 'wc-' . str_replace('_','-',$m[1]);
            $id_email_to_slug[$email_class->id] = $slug;
            $slug_to_email_hook[$slug] = 'woocommerce_email_subject_' . $email_class->id;
			$slug_to_email_id[$slug]   = $email_class->id; // <— aggiunto
        }
    }

    // Stati core (ridondanza sicura)
    foreach ([
        'customer_on_hold_order'    => 'wc-on-hold',
        'customer_processing_order' => 'wc-processing',
        'customer_completed_order'  => 'wc-completed',
        'customer_refunded_order'   => 'wc-refunded',
		'customer_cancelled_order'  => 'wc-cancelled', // aggiunto
    ] as $id_email => $slug) {
        $slug_to_email_hook[$slug] = 'woocommerce_email_subject_' . $id_email;
        $id_email_to_slug[$id_email] = $slug;
		$slug_to_email_id[$slug]   = $id_email; // <— aggiunto
    }

    // SUBJECT per stato (se abilitato)
    foreach ($slug_to_email_hook as $slug => $hook) {
        add_filter($hook, function($subject, $order) use ($slug) {
            if (!$order) return $subject;
            $opts = oste_get_opzioni_stato($slug);
            if ( empty($opts['enabled']) || empty($opts['subject']) ) return $subject;
            return sanitize_text_field( oste_sostituisci_variabili_email($opts['subject'], $order) );
        }, 10, 2);
    }

	// Intro (dopo il titolo) per email CORE; le CUSTOM lo stampano già nel template
	// Intro (dopo il titolo) per email CORE; le CUSTOM lo stampano già nel template
	add_action('woocommerce_email_header', function ($email_heading, $email) {
		if (!is_object($email) || empty($email->id)) return;

		// evita doppione sulle tue custom
		if ($email->id === 'oste_customer_custom_order' || is_a($email, 'OSTE_Email_Customer_Custom_Order')) {
			return;
		}

		// prendi l'ordine dall'istanza email
		$order = (isset($email->object) && $email->object instanceof WC_Order) ? $email->object : null;
		if (!$order) return;

		// leggi le opzioni del tuo pannello
		$opts_all = get_option('oste_gestione_email_stati_ordine', []);
		$slug     = 'wc-' . $order->get_status();
		$opts     = $opts_all[$slug] ?? [];

		if (empty($opts['enabled']) || empty($opts['intro'])) return;

		// variabili + stampa sicura
		$msg = function_exists('oste_sostituisci_variabili_email')
			? oste_sostituisci_variabili_email($opts['intro'], $order)
			: $opts['intro'];

		echo wp_kses_post( wpautop($msg) );
	}, 20, 2);


    // Testo extra TOP
	add_action('woocommerce_email_before_order_table', function($order, $sent_to_admin, $plain_text, $email) {
		if (!$order || !function_exists('oste_get_opzioni_stato')) return;
		$slug = 'wc-' . $order->get_status();
		$opts = oste_get_opzioni_stato($slug);
		if (empty($opts['enabled']) || empty($opts['extra_top'])) return;
		$msg = oste_sostituisci_variabili_email($opts['extra_top'], $order);
		echo $plain_text ? esc_html(wp_strip_all_tags($msg))."\n" : wp_kses_post(wpautop($msg));
	}, 10, 4);


    // Additional content + Heading per ID email core
    foreach ($id_email_to_slug as $id_email => $slug) {
        add_filter('woocommerce_email_additional_content_'.$id_email, function($content, $order) use ($slug) {
            $opts = oste_get_opzioni_stato($slug);
            if ( empty($opts['enabled']) || empty($opts['extra']) ) return $content;
            return wp_kses_post( oste_sostituisci_variabili_email($opts['extra'], $order) );
        }, 10, 2);

        add_filter('woocommerce_email_heading_'.$id_email, function($heading, $order) use ($slug) {
            $opts = oste_get_opzioni_stato($slug);
            if ( empty($opts['enabled']) || empty($opts['thanks']) ) return $heading;
            return wp_kses_post( oste_sostituisci_variabili_email($opts['thanks'], $order) );
        }, 10, 2);
    }

    // Footer aggiuntivo
    add_action('woocommerce_email_footer', function($email) {
        if ( isset($email->object) && is_a($email->object, 'WC_Order') ) {
            $order = $email->object;
            $slug  = 'wc-' . $order->get_status();
            $opts  = oste_get_opzioni_stato($slug);
            if ( ! empty($opts['enabled']) && ! empty($opts['footer']) ) {
                echo wp_kses_post( wpautop( oste_sostituisci_variabili_email($opts['footer'], $order) ) );
            }
        }
    }, 8);


	// TRIGGER email CUSTOM SOLO per stati custom (solo OSTE)
	add_action('woocommerce_order_status_changed', function($order_id, $old_status, $new_status, $order) {
		if ( ! $order ) $order = wc_get_order($order_id);

		$custom_no_prefix = (array) oste_get_custom_statuses_slug();
		$custom_with_wc   = (array) oste_get_custom_statuses_slug(true);

		$is_custom = in_array($new_status, $custom_no_prefix, true)
				  || in_array('wc-'.$new_status, $custom_with_wc, true);

		if ( ! $is_custom ) return;

		$emails = WC()->mailer()->get_emails();
		if ( isset($emails['oste_customer_custom_order']) ) {
			$emails['oste_customer_custom_order']->trigger($order_id, $order);
		}
	}, 20, 4);

	// --- FORZA INVIO EMAIL (anche se Woo non lo farebbe) ---
	add_action('woocommerce_order_status_changed', function ( $order_id, $old_status, $new_status, $order ) use ( $slug_to_email_hook ) {

		if ( ! $order instanceof WC_Order ) { $order = wc_get_order( $order_id ); }
		if ( ! $order ) return;

		$slug = 'wc-' . sanitize_key( $new_status );

		// Rispetta il toggle del tuo stato
		if ( function_exists( 'oste_get_opzioni_stato' ) ) {
			$opts = oste_get_opzioni_stato( $slug );
			if ( empty( $opts['enabled'] ) ) return;
		}

		// Calcola l'hook "atteso" se mappato; se non mappato lasciamo null (gestiamo comunque)
		$expected_subject_hook = ! empty( $slug_to_email_hook[ $slug ] ) ? $slug_to_email_hook[ $slug ] : null;

		$mailer = WC()->mailer();
		if ( ! $mailer || ! method_exists( $mailer, 'get_emails' ) ) return;

		$woo_emails = $mailer->get_emails();
		if ( empty( $woo_emails ) || ! is_array( $woo_emails ) ) return;

		foreach ( $woo_emails as $email_class ) {
			if ( ! is_object( $email_class ) || empty( $email_class->id ) || ! method_exists( $email_class, 'trigger' ) ) {
				continue;
			}

			$email_id          = $email_class->id;
			$this_subject_hook = 'woocommerce_email_subject_' . $email_id;

			// È quella attesa (mappata)?
			$is_expected   = ( $expected_subject_hook && $this_subject_hook === $expected_subject_hook );
			// È una core di tipo "customer_*_order"?
			$is_core_order = ( strpos( $email_id, 'customer_' ) === 0 && substr( $email_id, -6 ) === '_order' );

			// Forza se:
			// - è quella attesa (mappata) OPPURE
			// - lo stato NON è mappato e l'email è una core d'ordine
			if ( ! $is_expected && ! ( ! $expected_subject_hook && $is_core_order ) ) {
				continue;
			}

			// Init strutture globali (anti‑notice)
			if ( ! isset($GLOBALS['oste_sent_runtime']['forced'][ $email_id ]) ) {
				$GLOBALS['oste_sent_runtime']['forced'][ $email_id ] = [];
			}
			if ( ! isset($GLOBALS['oste_guard_state'][ $email_id ]) ) {
				$GLOBALS['oste_guard_state'][ $email_id ] = [];
			}

			// Anti‑duplica nella stessa request per stesso email_id+ordine
			if (
				! empty( $GLOBALS['oste_sent_runtime']['built'][  $email_id ][ $order_id ] ) ||
				! empty( $GLOBALS['oste_sent_runtime']['forced'][ $email_id ][ $order_id ] )
			) {
				break;
			}

			// Guard: se Woo prova a inviarla DOPO nella stessa request sullo stesso ordine, svuotiamo recipient
			$GLOBALS['oste_guard_state'][ $email_id ][ $order_id ] = 'sending';
			$__oste_guard = function( $recipient, $order_obj ) use ( $email_id ) {
				if ( $order_obj instanceof WC_Order ) {
					$oid   = $order_obj->get_id();
					$state = isset( $GLOBALS['oste_guard_state'][ $email_id ][ $oid ] ) ? $GLOBALS['oste_guard_state'][ $email_id ][ $oid ] : '';
					if ( $state === 'sent' ) {
						return ''; // blocca il duplicato della stessa email-id su questo ordine in questa request
					}
				}
				return $recipient;
			};
			add_filter( "woocommerce_email_recipient_{$email_id}", $__oste_guard, 999, 2 );

			try {
				// Forza SEMPRE l'email core determinata sopra
				$email_class->trigger( $order_id, $order );

				// Marca come inviata/costruita per evitare ulteriori forzature/duplicati nella stessa request
				$GLOBALS['oste_guard_state'][ $email_id ][ $order_id ]   = 'sent';
				$GLOBALS['oste_sent_runtime']['forced'][ $email_id ][ $order_id ] = true;
				if ( ! isset($GLOBALS['oste_sent_runtime']['built'][ $email_id ]) ) {
					$GLOBALS['oste_sent_runtime']['built'][ $email_id ] = [];
				}
				$GLOBALS['oste_sent_runtime']['built'][  $email_id ][ $order_id ] = true;

			} catch ( \Throwable $e ) {
				// nessun log in produzione
				$GLOBALS['oste_guard_state'][ $email_id ][ $order_id ] = 'sent';
			} finally {
				// pulizia filtro per non “sporcare” altre email/ordini nella stessa request
				remove_filter( "woocommerce_email_recipient_{$email_id}", $__oste_guard, 999 );
			}

			break; // una sola email per stato
		}

	}, 20, 4);


    
});


