<?php
if ( ! defined( 'ABSPATH' ) ) { exit; }

/**
 * Registra gli stati custom (in base alle opzioni salvate)
 */
add_action( 'init', function () {
	$statuses = oste_get_custom_statuses();
	$statuses = is_array( $statuses ) ? $statuses : [];

	foreach ( $statuses as $s ) {
		if ( empty( $s['slug'] ) || empty( $s['label'] ) ) { continue; }

		$slug  = sanitize_key( $s['slug'] );
		$label = sanitize_text_field( $s['label'] );

		register_post_status( 'wc-' . $slug, [
			'label'                     => $label,
			'public'                    => true,
			'exclude_from_search'       => false,
			'show_in_admin_all_list'    => true,
			'show_in_admin_status_list' => true,
			// translators: %s is the number of orders with this status
			'label_count'               => _n_noop( '%s order', '%s orders', 'order-status-tracking-emails-for-woocommerce' ),
		] );
	}
} );

/**
 * Sostituisce/aggiunge le etichette negli elenchi WooCommerce
 */
add_filter( 'wc_order_statuses', function ( $order_statuses ) {
	$base_labels = oste_get_base_labels();
	$base_labels = is_array( $base_labels ) ? $base_labels : [];

	// Sovrascrivi le label base se impostate
	foreach ( $order_statuses as $key => $label ) {
		$short = str_replace( 'wc-', '', $key );
		if ( ! empty( $base_labels[ $short ]['label'] ) ) {
			$order_statuses[ $key ] = sanitize_text_field( $base_labels[ $short ]['label'] );
		}
	}

	// Aggiungi stati custom
	$custom = oste_get_custom_statuses();
	$custom = is_array( $custom ) ? $custom : [];
	foreach ( $custom as $s ) {
		if ( empty( $s['slug'] ) || empty( $s['label'] ) ) { continue; }
		$order_statuses[ 'wc-' . sanitize_key( $s['slug'] ) ] = sanitize_text_field( $s['label'] );
	}

	return $order_statuses;
} );

/**
 * Stili badge & pulsanti azione (inline CSS via enqueue, NON admin_head)
 */
add_action( 'admin_enqueue_scripts', function () {
	if ( ! function_exists( 'get_current_screen' ) ) { return; }
	$screen = get_current_screen();
	if ( ! $screen || 'edit-shop_order' !== $screen->id ) { return; }

	// Assicurati che un handle di stile sia enqueuato per poter iniettare CSS
	wp_enqueue_style( 'oste-admin' );

	$statuses    = oste_get_custom_statuses();
	$statuses    = is_array( $statuses ) ? $statuses : [];
	$base_labels = oste_get_base_labels();
	$base_labels = is_array( $base_labels ) ? $base_labels : [];

	$css = '';

	// Colori per stati custom
	foreach ( $statuses as $st ) {
		if ( empty( $st['slug'] ) || empty( $st['color'] ) ) { continue; }
		$slug  = sanitize_key( $st['slug'] );
		$color = sanitize_hex_color( $st['color'] );
		if ( ! $color ) { continue; }

		$css .= ".status-{$slug},.order-status.status-{$slug}{background:{$color}!important;color:#fff!important;}";
		$css .= ".oste-status-btn-{$slug}{background:{$color}!important;color:#fff!important;border-radius:6px;}";
	}

	// Colori per stati base
	foreach ( $base_labels as $slug => $arr ) {
		if ( empty( $arr['color'] ) ) { continue; }
		$slug  = sanitize_key( $slug );
		$color = sanitize_hex_color( $arr['color'] );
		if ( ! $color ) { continue; }

		$css .= ".status-{$slug},.order-status.status-{$slug}{background:{$color}!important;color:#fff!important;}";
		$css .= ".wc-action-button.{$slug}{background:{$color}!important;color:#fff!important;border-radius:6px;}";
	}

	// Stile pulsanti (base)
	$css .= "
		.widefat .column-wc_actions a.button:after{display:none!important;content:none!important;}
		.widefat .column-wc_actions a.button{
			display:inline-block!important;
			width:60px!important;
			height:30px!important;
			line-height:30px!important;
			margin:0 2px 2px 0;
			padding:0!important;
			font:600 11px/30px sans-serif;
			text-align:center;
			text-indent:0!important;
			overflow:visible!important;
			border-radius:4px;
			color:#fff!important;
		}";

	// Nascondi pulsanti core se opzione attiva
	if ( get_option( 'oste_hide_core_action', 0 ) ) {
		$css .= "
			.widefat .column-wc_actions a.button.preview,
			.widefat .column-wc_actions a.button.view { display:none!important; }";
	}

	if ( $css ) {
		wp_add_inline_style( 'oste-admin', $css );
	}
} );

/**
 * Pulsanti azione (lista ordini)
 */
add_filter( 'woocommerce_admin_order_actions', function ( $actions, $order ) {
	$order_status  = $order->get_status(); // es. 'completed', 'processing', ...
	$order_id      = $order->get_id();
	$actions_show  = oste_get_actions_settings();
	$actions_show  = is_array( $actions_show ) ? $actions_show : [];

	// Se l'opzione NON è attiva (0), rimuovi i pulsanti core di Woo
	if ( 0 == get_option( 'oste_hide_core_action', 0 ) ) {
		foreach ( array_keys( oste_get_base_statuses() ) as $slug ) {
			unset( $actions[ $slug ] );
			$mark_slug = 'mark_' . $slug;
			unset( $actions[ $mark_slug ] );
		}
		unset( $actions['complete'] ); // in alcuni Woo
	}

	// Pulsanti custom
	$custom = oste_get_custom_statuses();
	$custom = is_array( $custom ) ? $custom : [];
	foreach ( $custom as $s ) {
		if ( empty( $s['slug'] ) || empty( $s['label'] ) ) { continue; }

		$slug  = sanitize_key( $s['slug'] );
		$label = sanitize_text_field( $s['label'] );

		if ( ! empty( $actions_show[ $slug ] ) && $order_status !== $slug ) {
			$short = function_exists( 'mb_substr' ) ? mb_substr( $label, 0, 6 ) : substr( $label, 0, 6 );

			$url = add_query_arg(
				[
					'action'     => 'oste_change_order_status',
					'order_id'   => $order_id,
					'new_status' => 'wc-' . $slug,
				],
				admin_url( 'admin-post.php' )
			);
			$url = wp_nonce_url( $url, 'oste_change_order_status', '_wpnonce' );

			$actions[ $slug ] = [
				'url'    => $url,
				'name'   => $short,
				'action' => 'oste-status-btn oste-status-btn-' . $slug,
				'title'  => $label,
			];
		}
	}

	// Pulsanti base personalizzati
	$base_statuses = oste_get_base_statuses(); // es. ['completed' => 'Completed', ...]
	foreach ( $base_statuses as $slug => $default_label ) {
		$slug  = sanitize_key( $slug );
		if ( empty( $actions_show[ $slug ] ) || $order_status === $slug ) {
			continue;
		}

		$row   = oste_get_base_labels()[ $slug ] ?? [];
		$label = sanitize_text_field( $row['label'] ?? $default_label );
		$short = function_exists( 'mb_substr' ) ? mb_substr( $label, 0, 6 ) : substr( $label, 0, 6 );

		// Link standard Woo per cambiare stato base
		$url = add_query_arg(
			[
				'action'   => 'woocommerce_mark_order_status',
				'status'   => $slug,
				'order_id' => $order_id,
			],
			admin_url( 'admin-ajax.php' )
		);
		$url = wp_nonce_url( $url, 'woocommerce-mark-order-status', '_wpnonce' );

		$actions[ $slug ] = [
			'url'    => $url,
			'name'   => $short,
			'action' => 'wc-action-button ' . $slug,
			'title'  => $label,
		];
	}

	return $actions;
}, 10, 2 );

/**
 * Handler: cambia stato custom via admin-post.php
 */
add_action( 'admin_post_oste_change_order_status', function () {
	// Nonce
	$nonce = isset( $_GET['_wpnonce'] ) ? sanitize_text_field( wp_unslash( $_GET['_wpnonce'] ) ) : '';
	if ( ! $nonce || ! wp_verify_nonce( $nonce, 'oste_change_order_status' ) ) {
		wp_die( esc_html__( 'Invalid request.', 'order-status-tracking-emails-for-woocommerce' ) );
	}

	// Permessi
	if ( ! current_user_can( 'manage_woocommerce' ) && ! current_user_can( 'edit_shop_orders' ) ) {
		wp_die( esc_html__( 'You do not have permission to change order status.', 'order-status-tracking-emails-for-woocommerce' ) );
	}

	// Parametri
	$order_id   = isset( $_GET['order_id'] )   ? absint( $_GET['order_id'] ) : 0;
	$new_status = isset( $_GET['new_status'] ) ? sanitize_key( wp_unslash( $_GET['new_status'] ) ) : '';

	$referer = wp_get_referer();
	$back    = $referer ? $referer : admin_url( 'edit.php?post_type=shop_order' );

	if ( ! $order_id || ! $new_status ) {
		wp_safe_redirect( remove_query_arg( [ 'action', 'order_id', 'new_status', '_wpnonce' ], $back ) );
		exit;
	}

	// Valida lo status: core + custom
	// Valida lo status confrontando con TUTTI gli stati registrati (core + custom)
	$allowed = array_keys( wc_get_order_statuses() ); // es. ['wc-pending','wc-spedito', ...]
	if ( ! in_array( $new_status, $allowed, true ) ) {
		wp_die( esc_html__( 'Invalid status.', 'order-status-tracking-emails-for-woocommerce' ) );
	}


	// Applica lo stato
	$order = wc_get_order( $order_id );
	if ( $order ) {
		$order->update_status(
			str_replace( 'wc-', '', $new_status ),
			__( 'Status changed via admin action.', 'order-status-tracking-emails-for-woocommerce' ),
			true
		);
	}

	// Redirect sicuro pulendo la query
	wp_safe_redirect( remove_query_arg( [ 'action', 'order_id', 'new_status', '_wpnonce' ], $back ) );
	exit;
} );



