<?php
if ( ! defined( 'ABSPATH' ) ) { exit; }

/**
 * Submenu: Tracking Carriers
 */
add_action( 'admin_menu', function () {
	add_submenu_page(
		'oste-order-manager',
		__( 'Tracking Carriers', 'order-status-tracking-emails-for-woocommerce' ),
		__( 'Tracking Carriers', 'order-status-tracking-emails-for-woocommerce' ),
		'manage_options',
		'oste-tracking-carriers',
		'oste_tracking_carriers_page'
	);
}, 20 );

/**
 * Enqueue JS/CSS solo in questa pagina
 */
add_action( 'admin_enqueue_scripts', function ( $hook_suffix ) {
	$matches_hook = is_string( $hook_suffix ) && false !== strpos( $hook_suffix, 'oste-tracking-carriers' );
	if ( ! $matches_hook ) {
		return;
	}

	// Assicuriamoci che l'handle esista (registrato nel main)
	wp_enqueue_style( 'oste-admin' );
	wp_enqueue_style( 'oste-admin-ui' );
	wp_enqueue_script( 'oste-admin' );

	// JS per add/remove righe
	wp_add_inline_script( 'oste-admin', "
document.addEventListener('DOMContentLoaded', function(){
  var table = document.getElementById('oste-carriers-table');
  if (!table) return;

  var addBtn = document.getElementById('oste-add-row');
  var presetSelect = document.getElementById('oste-carrier-presets');
  var addPresetBtn = document.getElementById('oste-add-preset');
  if (addBtn) {
    addBtn.addEventListener('click', function(){
      var tbody = table.querySelector('tbody');
      var last  = tbody.rows[tbody.rows.length - 1];
      var clone = last.cloneNode(true);
      clone.querySelectorAll('input').forEach(function(inp){ inp.value=''; });
      // pulisci eventuali pulsanti/icone nella nuova riga
      var rm = clone.querySelector('.oste-remove-row');
      if (rm) rm.addEventListener('click', function(){ clone.remove(); });
      tbody.appendChild(clone);
    });
  }

  if (addPresetBtn && presetSelect) {
    addPresetBtn.addEventListener('click', function(){
      var name = presetSelect.value || '';
      var url = presetSelect.options[presetSelect.selectedIndex] ? presetSelect.options[presetSelect.selectedIndex].dataset.url : '';
      if (!name || !url) return;

      var tbody = table.querySelector('tbody');
      if (!tbody) return;

      var row = document.createElement('tr');
      row.innerHTML =
        '<td><input type=\"text\" name=\"carrier_name[]\" value=\"' + name.replace(/\"/g,'&quot;') + '\" required style=\"width:150px;\"></td>' +
        '<td><input type=\"text\" name=\"carrier_url[]\" value=\"' + url.replace(/\"/g,'&quot;') + '\" required style=\"width:350px;\"></td>' +
        '<td><button type=\"button\" class=\"button oste-remove-row\" title=\"Remove carrier\">&#10005;</button></td>';

      var rows = tbody.querySelectorAll('tr');
      if (rows.length) {
        tbody.insertBefore(row, rows[rows.length - 1]);
      } else {
        tbody.appendChild(row);
      }
    });
  }

  table.addEventListener('click', function(e){
    if (e.target && e.target.classList && e.target.classList.contains('oste-remove-row')) {
      var tr = e.target.closest('tr');
      if (tr) tr.remove();
    }
  });
});
	" );

	// CSS tabellina
	wp_add_inline_style( 'oste-admin', "
#oste-carriers-table input[type='text']{ width:95%; }
#oste-carriers-table td, #oste-carriers-table th{ text-align:left; }
	" );
} );

/**
 * Pagina impostazioni carriers
 */
function oste_tracking_carriers_page() {
	// Permessi extra (in più rispetto a add_submenu_page)
	if ( ! current_user_can( 'manage_options' ) ) {
		wp_die( esc_html__( 'You do not have permission to access this page.', 'order-status-tracking-emails-for-woocommerce' ) );
	}

	$carriers = get_option( 'oste_tracking_carriers', [] );
	$carriers = is_array( $carriers ) ? $carriers : [];

	// Valori default da opzioni (per il form)
	$custom_tracking_message  = get_option( 'oste_tracking_custom_message', '' );
	$custom_tracking_position = get_option( 'oste_tracking_message_position', 'below' );
	$carrier_presets = array(
		'Australia Post' => 'https://auspost.com.au/mypost/track/#/details/[tracking]',
		'Austria Post' => 'https://www.post.at/s/sendungsdetails?snr=[tracking]',
		'Bpost' => 'https://track.bpost.cloud/track/items?itemIdentifier=[tracking]',
		'BRT Bartolini' => 'https://services.brt.it/it/tracking?CD=[tracking]',
		'Canada Post' => 'https://www.canadapost-postescanada.ca/track-reperage/en#/search?searchFor=[tracking]',
		'Colissimo' => 'https://www.laposte.fr/outils/suivre-vos-envois?code=[tracking]',
		'Correos' => 'https://www.correos.es/es/es/herramientas/localizador/envios/detalle?tracking-number=[tracking]',
		'DHL Express' => 'https://www.dhl.com/it-it/home/tracking.html?tracking-id=[tracking]',
		'DHL Paket' => 'https://www.dhl.de/en/privatkunden/dhl-sendungsverfolgung.html?piececode=[tracking]',
		'DPD' => 'https://services.brt.it/it/tracking?CD=[tracking]',
		'FedEx' => 'https://www.office.fedex.com/default/track?gtns=[tracking]',
		'GLS' => 'https://gls-group.eu/track/[tracking]',
		'GLS Italy' => 'https://gls-group.com/IT/it/servizi-online/ricerca-spedizioni/?match=[tracking]&type=NAT',
		'InPost' => 'https://inpost.it/trova-il-tuo-pacco?number=[tracking]',
		'Poste Italiane' => 'https://www.poste.it/cerca/index.html#/risultati-spedizioni/[tracking]/',
		'PostNL' => 'https://jouw.postnl.nl/track-and-trace/[tracking]',
		'PostNord' => 'https://tracking.postnord.com/it/tracking?id=[tracking]',
		'Purolator' => 'https://www.purolator.com/en/shipping/tracker?pin=[tracking]',
		'Royal Mail' => 'https://www.royalmail.com/track-your-item#/tracking-results/[tracking]',
		'SDA' => 'https://www.poste.it/cerca/index.html#/risultati-spedizioni/[tracking]',
		'Swiss Post' => 'https://service.post.ch/ekp-web/ui/list?trackings=[tracking]',
		'TNT (FedEx)' => 'https://www.tnt.com/express/it_it/site/shipping-tools/tracking.html?searchType=con&cons=[tracking]',
		'UPS' => 'https://www.ups.com/track?tracknum=[tracking]',
		'USPS' => 'https://tools.usps.com/go/TrackConfirmAction?tLabels=[tracking]',
		'Yodel' => 'https://www.yodel.co.uk/track/[tracking]',
	);
	ksort( $carrier_presets, SORT_NATURAL | SORT_FLAG_CASE );

	// Salvataggio
	if (
		isset( $_SERVER['REQUEST_METHOD'] ) && 'POST' === $_SERVER['REQUEST_METHOD']
		&& isset( $_POST['oste_carriers_nonce'] )
		&& wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['oste_carriers_nonce'] ) ), 'save_carriers' )
	) {
		// Estrai una sola volta e rimuovi gli slash aggiunti da WP (non usare $post per evitare collisioni col globale WP_Post).
		$input = wp_unslash( $_POST );

		// ========== Raccolta input (sanificati SUBITO) ==========
		$names_raw = ( isset( $input['carrier_name'] ) && is_array( $input['carrier_name'] ) ) ? $input['carrier_name'] : array();
		$urls_raw  = ( isset( $input['carrier_url'] )  && is_array( $input['carrier_url'] ) )  ? $input['carrier_url']  : array();

		$names = array_map( 'sanitize_text_field', $names_raw );
		$urls  = array_map(
			function ( $u ) {
				$u = trim( (string) $u );
				// Consenti solo http/https; se non valida, esc_url_raw() ritorna stringa vuota.
				return esc_url_raw( $u, array( 'http', 'https' ) );
			},
			$urls_raw
		);

		// ========== Normalizzazione, validazione riga-per-riga (mantiene gli indici allineati) ==========
		$new_carriers = array();
		$seen         = array();
		$max          = max( count( $names ), count( $urls ) );

		for ( $i = 0; $i < $max; $i++ ) {
			$name = isset( $names[ $i ] ) ? trim( (string) $names[ $i ] ) : '';
			$url  = isset( $urls[ $i ] )  ? (string) $urls[ $i ]          : '';

			// Salta righe incomplete
			if ( '' === $name || '' === $url ) {
				continue;
			}

			// Limita lunghezze (difensivo, senza richiedere mbstring)
			if ( function_exists( 'mb_substr' ) ) {
				$name = mb_substr( $name, 0, 120 );
				$url  = mb_substr( $url, 0, 2048 );
			} else {
				$name = substr( $name, 0, 120 );
				$url  = substr( $url, 0, 2048 );
			}

			// Valida URL formalmente
			if ( ! filter_var( $url, FILTER_VALIDATE_URL ) ) {
				continue;
			}

			// Deduplica per coppia name|url (case-insensitive)
			$key = strtolower( $name . '|' . $url );
			if ( isset( $seen[ $key ] ) ) {
				continue;
			}
			$seen[ $key ] = true;

			$new_carriers[] = array(
				'nome' => $name,
				'url'  => $url,
			);
		}

		// ========== Messaggio personalizzato e posizione ==========
		if ( isset( $input['custom_tracking_message'] ) ) {
			update_option( 'oste_tracking_custom_message', wp_kses_post( $input['custom_tracking_message'] ) );
			$custom_tracking_message = get_option( 'oste_tracking_custom_message', '' );
		}

		if ( isset( $input['custom_tracking_position'] ) ) {
			$pos         = sanitize_text_field( $input['custom_tracking_position'] );
			$allowed_pos = array( 'above', 'below', 'after_customer_details' );
			if ( ! in_array( $pos, $allowed_pos, true ) ) {
				$pos = 'below';
			}
			update_option( 'oste_tracking_message_position', $pos );
			$custom_tracking_position = $pos;
		}

		// Salva carriers
		update_option( 'oste_tracking_carriers', $new_carriers );
		$carriers = $new_carriers;

		echo '<div class="updated"><p>' . esc_html__( 'Carriers saved successfully!', 'order-status-tracking-emails-for-woocommerce' ) . '</p></div>';

	}

	?>
	<div class="wrap oste-admin-wrap">
		<h1><?php esc_html_e( 'Shipping Tracking Carriers', 'order-status-tracking-emails-for-woocommerce' ); ?></h1>

		<form method="post">
			<?php wp_nonce_field( 'save_carriers', 'oste_carriers_nonce' ); ?>

			<p class="oste-carriers-presets">
				<label for="oste-carrier-presets"><strong><?php esc_html_e( 'Preset carriers', 'order-status-tracking-emails-for-woocommerce' ); ?></strong></label><br>
				<select id="oste-carrier-presets">
					<option value=""><?php esc_html_e( 'Select a carrier...', 'order-status-tracking-emails-for-woocommerce' ); ?></option>
					<?php foreach ( $carrier_presets as $preset_name => $preset_url ) : ?>
						<option value="<?php echo esc_attr( $preset_name ); ?>" data-url="<?php echo esc_attr( $preset_url ); ?>">
							<?php echo esc_html( $preset_name ); ?>
						</option>
					<?php endforeach; ?>
				</select>
				<button type="button" class="button" id="oste-add-preset"><?php esc_html_e( 'Add preset', 'order-status-tracking-emails-for-woocommerce' ); ?></button>
			</p>

			<table class="widefat" id="oste-carriers-table">
				<thead>
					<tr>
						<th><?php esc_html_e( 'Carrier Name', 'order-status-tracking-emails-for-woocommerce' ); ?></th>
						<th>
							<?php esc_html_e( 'Tracking URL', 'order-status-tracking-emails-for-woocommerce' ); ?><br>
							<small>
								<?php esc_html_e( 'Use', 'order-status-tracking-emails-for-woocommerce' ); ?>
								<code>[tracking]</code>
								<?php esc_html_e( 'where the code goes (e.g., https://.../[tracking] or https://...?code=[tracking])', 'order-status-tracking-emails-for-woocommerce' ); ?><br>
								<b><?php esc_html_e( 'Note:', 'order-status-tracking-emails-for-woocommerce' ); ?></b>
								<?php esc_html_e( 'curly braces {} are not supported.', 'order-status-tracking-emails-for-woocommerce' ); ?>
							</small>
						</th>
						<th><?php esc_html_e( 'Remove', 'order-status-tracking-emails-for-woocommerce' ); ?></th>
					</tr>
				</thead>
				<tbody>
				<?php if ( ! empty( $carriers ) ) : ?>
					<?php foreach ( $carriers as $i => $c ) :
						$name = isset( $c['nome'] ) ? (string) $c['nome'] : '';
						$url  = isset( $c['url'] )  ? (string) $c['url']  : '';
					?>
					<tr>
						<td>
							<input type="text" name="carrier_name[]" value="<?php echo esc_attr( $name ); ?>" required style="width:150px;">
						</td>
						<td>
							<input type="text" name="carrier_url[]" value="<?php echo esc_attr( $url ); ?>" required style="width:350px;">
						</td>
						<td>
							<button type="button" class="button oste-remove-row" title="<?php esc_attr_e( 'Remove carrier', 'order-status-tracking-emails-for-woocommerce' ); ?>">✕</button>
						</td>
					</tr>
					<?php endforeach; ?>
				<?php endif; ?>

					<tr>
						<td>
							<input type="text" name="carrier_name[]" value="" placeholder="<?php esc_attr_e( 'Carrier Name', 'order-status-tracking-emails-for-woocommerce' ); ?>" style="width:150px;">
						</td>
						<td>
							<input type="text" name="carrier_url[]" value="" placeholder="https://url/[tracking]" style="width:350px;">
						</td>
						<td></td>
					</tr>
				</tbody>
			</table>

			<p>
				<button type="button" class="button" id="oste-add-row">+ <?php esc_html_e( 'Add Carrier', 'order-status-tracking-emails-for-woocommerce' ); ?></button>
			</p>

			<h2 style="margin-top:32px;"><?php esc_html_e( 'Custom Tracking Message (frontend for customer)', 'order-status-tracking-emails-for-woocommerce' ); ?></h2>

			<textarea name="custom_tracking_message" rows="3" style="width:100%;max-width:550px;"><?php echo esc_textarea( $custom_tracking_message ); ?></textarea>

			<p style="margin:7px 0 3px 0;">
				<label>
					<b><?php esc_html_e( 'Position on the customer order page:', 'order-status-tracking-emails-for-woocommerce' ); ?></b>
					<select name="custom_tracking_position">
						<option value="above" <?php selected( $custom_tracking_position, 'above' ); ?>><?php esc_html_e( 'Above the products table', 'order-status-tracking-emails-for-woocommerce' ); ?></option>
						<option value="below" <?php selected( $custom_tracking_position, 'below' ); ?>><?php esc_html_e( 'Below the products table', 'order-status-tracking-emails-for-woocommerce' ); ?></option>
						<option value="after_customer_details" <?php selected( $custom_tracking_position, 'after_customer_details' ); ?>><?php esc_html_e( 'After the customer addresses', 'order-status-tracking-emails-for-woocommerce' ); ?></option>
					</select>
				</label>
			</p>

			<p style="color:#888;">
				<?php esc_html_e( 'You can use the following variables:', 'order-status-tracking-emails-for-woocommerce' ); ?><br>
				<code>{order_id}</code>, <code>{customer_name}</code>, <code>{carrier}</code>, <code>{tracking}</code>, <code>[link-tracking]</code>
			</p>

			<p>
				<input type="submit" class="button-primary" value="<?php esc_attr_e( 'Save Carriers', 'order-status-tracking-emails-for-woocommerce' ); ?>">
			</p>
		</form>
	</div>
	<?php
}
