<?php
if ( ! defined( 'ABSPATH' ) ) { exit; }

// === ADMIN: Pagina personalizzazione stile email ordini ===

add_action( 'admin_menu', function () {
	add_submenu_page(
		'oste-order-manager',                     // stesso parent dello slug reale
		__( 'Email Style Settings', 'order-status-tracking-emails-for-woocommerce' ),
		__( 'Email Style Settings', 'order-status-tracking-emails-for-woocommerce' ),
		'manage_options',
		'oste-email-style-settings',        //  <-- underscore, non trattini
		'oste_email_styling_admin_page'
	);

}, 40);


add_action( 'admin_init', function () {
	register_setting(
		'oste_email_styling_group',
		'oste_email_styling',
		array( 'sanitize_callback' => 'oste_sanitize_email_style_settings' )
	);
} );

function oste_sanitize_css_color_maybe_rgba( $v ) {
	$v = trim( (string) $v );
	if ( '' === $v ) {
		return '';
	}
	// HEX #rgb o #rrggbb
	if ( preg_match( '/^#([0-9a-f]{3}|[0-9a-f]{6})$/i', $v ) ) {
		return strtolower( $v );
	}
	// rgb(...) o rgba(...)
	if ( preg_match( '/^rgba?\(\s*\d{1,3}\s*,\s*\d{1,3}\s*,\s*\d{1,3}(?:\s*,\s*(?:0|1|0?\.\d+))?\s*\)$/i', $v ) ) {
		return $v;
	}
	// fallback: tenta HEX WP, altrimenti stringa vuota
	$hex = sanitize_hex_color( $v );
	return $hex ? $hex : '';
}

function oste_sanitize_css_len( $v ) {
	$v = trim( (string) $v );
	if ( $v === '' ) return '';
	$re = '/^(-?\d+(?:\.\d+)?(?:px|em|rem|vh|vw|%)?)' .
	      '(?:\s+-?\d+(?:\.\d+)?(?:px|em|rem|vh|vw|%)?){0,3}$/i';
	if ( preg_match( $re, $v ) ) return $v;
	// fallback: ripulisci grossolanamente
	return preg_replace( '/[^0-9.\-\s%a-z]/i', '', $v );
}


function oste_sanitize_box_shadow( $v ) {
	// Consenti numeri/px/em, spazi, virgole, parentesi, #, rgba
	return preg_replace( '/[^0-9a-z#(),\.\spxem\-]/i', '', (string) $v );
}

function oste_sanitize_email_style_settings( $input ) {
	if ( ! is_array( $input ) ) {
		return array();
	}
	$input  = wp_unslash( $input );
	$output = array();

	foreach ( $input as $key_raw => $value ) {
		$key = sanitize_key( $key_raw );

		// Checkbox
		if ( in_array( $key, array( 'enable_custom_styles', 'show_logo', 'show_disclaimer' ), true ) ) {
			$output[ $key ] = ( '1' === (string) $value ) ? '1' : '0';
			continue;
		}

		// URL immagini (solo http/https)
		if ( in_array( $key, array( 'logo_url', 'banner_footer_url', 'bg_image' ), true ) ) {
			$output[ $key ] = esc_url_raw( (string) $value, array( 'http', 'https' ) );
			continue;
		}

		// Colori (HEX o RGBA)
		if (
			false !== strpos( $key, 'color' ) ||
			false !== strpos( $key, '_bg' ) ||
			false !== strpos( $key, '_shadow_color' ) ||
			false !== strpos( $key, 'border_color' )
		) {
			$output[ $key ] = oste_sanitize_css_color_maybe_rgba( $value );
			continue;
		}

		// Valori numerici / dimensioni / padding / margin / radius / size / blur / *_width
		if ( preg_match( '/(_width|_size|_radius|_blur|_padding|_margin|_letter_spacing|_shadow_x|_shadow_y|_lineheight)$/', $key ) ) {
			$output[ $key ] = oste_sanitize_css_len( $value );
			continue;
		}



		// Align / font / weight / style / box_align
		if ( preg_match( '/(align|font|weight|style|box_align)$/', $key ) ) {
			$output[ $key ] = sanitize_text_field( (string) $value );
			continue;
		}

		// Box shadow generico
		if ( false !== strpos( $key, 'box_shadow' ) ) {
			$output[ $key ] = oste_sanitize_box_shadow( $value );
			continue;
		}

		// CSS custom (consenti HTML minimoâ€”verrÃ  inserito in <style>)
		if ( 'custom_css' === $key ) {
			$output[ $key ] = wp_kses_post( (string) $value );
			continue;
		}

		// Testo generico
		$output[ $key ] = sanitize_text_field( (string) $value );
	}

	return $output;
}

// 1) Enqueue color picker + media SOLO sulla pagina giusta
add_action( 'admin_enqueue_scripts', function ( $hook_suffix ) {
	$targets = array(
		'oste-order-manager_page_oste-email-style-settings',
		'woocommerce_page_oste-email-style-settings',
	);

	$ok = in_array( $hook_suffix, $targets, true );
	if ( ! $ok && isset( $_GET['page'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification.Recommended
		$ok = ( 'oste-email-style-settings' === sanitize_text_field( wp_unslash( $_GET['page'] ) ) ); // phpcs:ignore WordPress.Security.NonceVerification.Recommended
	}
	if ( ! $ok ) {
		return;
	}

	wp_enqueue_style( 'wp-color-picker' );
	wp_enqueue_media();
	wp_enqueue_script( 'wp-color-picker' ); // assicura la handle
} );

// 2) Inizializza i picker DOPO che gli script sono caricati
add_action( 'admin_print_footer_scripts', function () {
	if ( ! isset( $_GET['page'] ) || 'oste-email-style-settings' !== sanitize_text_field( wp_unslash( $_GET['page'] ) ) ) { // phpcs:ignore WordPress.Security.NonceVerification.Recommended
		return;
	}
	?>
	<script>
	jQuery(function($){
		if ($.fn.wpColorPicker) {
			$('.color-picker').wpColorPicker({
				change: function() {
					if (window.osteScheduleEmailPreview) {
						window.osteScheduleEmailPreview();
					}
				},
				clear: function() {
					if (window.osteScheduleEmailPreview) {
						window.osteScheduleEmailPreview();
					}
				}
			});
		}

		$('.oste-upload-image').on('click', function(e){
			e.preventDefault();
			var target = $(this).data('target');
			var input  = $('#'+target);
			var frame  = wp.media({
				title: '<?php echo esc_js( __( 'Choose image', 'order-status-tracking-emails-for-woocommerce' ) ); ?>',
				button:{ text: '<?php echo esc_js( __( 'Use this image', 'order-status-tracking-emails-for-woocommerce' ) ); ?>' },
				multiple: false
			});
			frame.on('select', function(){
				var att = frame.state().get('selection').first().toJSON();
				input.val(att.url);
				input.closest('td').find('img').remove();
				input.closest('td').append('<br><img src="'+att.url+'" style="max-width:200px;max-height:80px;margin-top:8px;" />');
			});
			frame.open();
		});

		$('.oste-remove-image').on('click', function(e){
			e.preventDefault();
			var input = $('#'+$(this).data('target'));
			input.val('');
			input.closest('td').find('img').remove();
		});

		var previewRoot = document.getElementById('oste-email-preview-root');
		if (previewRoot) {
			var previewConfig = {
				ajaxUrl: previewRoot.getAttribute('data-ajax-url') || '',
				nonce: previewRoot.getAttribute('data-nonce') || '',
				errorMessage: previewRoot.getAttribute('data-error') || '<?php echo esc_js( __( 'Unable to load the email preview. Please try again.', 'order-status-tracking-emails-for-woocommerce' ) ); ?>',
				noteTemplate: previewRoot.getAttribute('data-note-template') || '',
				noteDefault: previewRoot.getAttribute('data-note-default') || ''
			};

			var $form         = $('form[action="options.php"]');
			var $statusSelect = $('#oste-email-preview-status');
			var $iframe       = $('#oste-email-preview-frame');
			var $subject      = $('#oste-email-preview-subject');
			var $spinner      = $('#oste-email-preview-spinner');
			var $note         = $('#oste-email-preview-note');
			var debounceHandle;
			var lastRequest = null;

			function escapeHtml(str) {
				return String(str || '').replace(/[&<>"]/g, function(ch) {
					switch (ch) {
						case '&': return '&amp;';
						case '<': return '&lt;';
						case '>': return '&gt;';
						case '"': return '&quot;';
					}
					return ch;
				});
			}

			function setIframeContent(html) {
				if (!$iframe.length) {
					return;
				}
				var frameEl = $iframe[0];
				if ('srcdoc' in frameEl) {
					frameEl.srcdoc = html;
				} else {
					var doc = frameEl.contentDocument || frameEl.contentWindow.document;
					if (!doc) {
						return;
					}
					doc.open();
					doc.write(html);
					doc.close();
				}
			}

			function showError(message) {
				var safe = escapeHtml(message || previewConfig.errorMessage);
				var html = '<!DOCTYPE html><html><head><meta charset="utf-8"></head><body style="font-family:Arial, sans-serif;padding:24px;background:#fff5f5;color:#8a1f1f;">' + safe + '</body></html>';
				setIframeContent(html);
				if ($subject.length) {
					$subject.text('');
				}
				if ($note.length && previewConfig.noteDefault) {
					$note.text(previewConfig.noteDefault);
				}
			}

			function requestPreview() {
				if (!$form.length || !previewConfig.ajaxUrl || !previewConfig.nonce) {
					return;
				}

				if (lastRequest && lastRequest.abort) {
					lastRequest.abort();
				}

				if ($spinner.length) {
					$spinner.addClass('is-active');
				}

				var payload = {
					action: 'oste_email_style_preview',
					nonce: previewConfig.nonce,
					status: $statusSelect.val(),
					form: $form.serialize()
				};

				lastRequest = $.ajax({
					type: 'POST',
					url: previewConfig.ajaxUrl,
					dataType: 'json',
					data: payload
				})
				.done(function(resp){
					if (resp && resp.success && resp.data) {
						var html = resp.data.html || '';
						setIframeContent(html);
						if ($subject.length) {
							$subject.text(resp.data.subject || '');
						}
						if ($note.length) {
							var noteText = previewConfig.noteDefault;
							if (resp.data.preview_order && previewConfig.noteTemplate) {
								noteText = previewConfig.noteTemplate.replace('%s', resp.data.preview_order);
							}
							if (noteText) {
								$note.text(noteText);
							}
						}
					} else {
						var fallback = previewConfig.errorMessage;
						if (resp && resp.data && resp.data.message) {
							fallback = resp.data.message;
						}
						showError(fallback);
					}
				})
				.fail(function(jqXHR, textStatus){
					if (textStatus === 'abort') {
						return;
					}
					var fallback = previewConfig.errorMessage;
					if (jqXHR && jqXHR.responseJSON && jqXHR.responseJSON.data && jqXHR.responseJSON.data.message) {
						fallback = jqXHR.responseJSON.data.message;
					} else if (jqXHR && jqXHR.responseText) {
						var text = jqXHR.responseText.replace(/<[^>]+>/g, '').trim();
						if (text) {
							fallback = text;
						}
					}
					showError(fallback);
				})
				.always(function(){
					if ($spinner.length) {
						$spinner.removeClass('is-active');
					}
				});
			}

			function schedulePreview() {
				if (debounceHandle) {
					clearTimeout(debounceHandle);
				}
				debounceHandle = setTimeout(requestPreview, 400);
			}
			window.osteScheduleEmailPreview = schedulePreview;

			if ($form.length) {
				requestPreview();

				$form.on('input change', 'input, select, textarea', function(event){
					if (event.target === $statusSelect[0]) {
						return;
					}
					schedulePreview();
				});

				$statusSelect.on('change', function(){
					requestPreview();
				});
			}
		}
	});
	</script>
	<?php
} );



function oste_email_styling_default_options() {
    return [
		'enable_custom_styles' => '0', // di default disattivo

        // sfondo principale 
        'bg_color'                => '#f5f5f5',
        'bg_image'                => '',
		// inner wrapper, colonna centrale tutta
		'inner_bg_color'      => '#ffffff',
		'inner_border_width'  => '1px',
		'inner_border_style'  => 'solid',
		'inner_border_color'  => '#eeeeee',
		'inner_radius'        => '8',
		'inner_shadow_x'      => '0',
		'inner_shadow_y'      => '1px',
		'inner_shadow_blur'   => '10px',
		'inner_shadow_color'  => 'rgba(0,0,0,0.07)',
		'inner_padding'       => '24px',
		// inner wrapper, colonna centrale tutta
		
		'template_container_bg'        => '#f5f5f5',
		'template_container_radius'    => '8',
		'template_container_border_width' => '0',
		'template_container_border_style' => 'solid',
		'template_container_border_color' => '#cccccc',
		'template_container_box_shadow'   => 'none',
		'template_container_padding'      => '0px 32px 0',

		
        'container_color'         => '#ffffff',
        'container_radius'        => '12',
        'container_border_width'  => '1px',
        'container_border_style'  => 'solid',
        'container_border_color'  => '#eeeeee',
        'container_shadow_x'      => '0',
        'container_shadow_y'      => '1px',
        'container_shadow_blur'   => '10px',
        'container_shadow_color'  => 'rgba(0,0,0,0.07)',
        'container_padding'       => '32px',
		
		'container_cell_color'         => '#ffffff',
        'container_cell_radius'        => '12',
        'container_cell_border_width'  => '1px',
        'container_cell_border_style'  => 'solid',
        'container_cell_border_color'  => '#eeeeee',
        'container_cell_shadow_x'      => '0',
        'container_cell_shadow_y'      => '1px',
        'container_cell_shadow_blur'   => '10px',
        'container_cell_shadow_color'  => 'rgba(0,0,0,0.07)',
        'container_cell_padding'       => '32px',
		'element_spacing' => '16px',

		
		
        'show_logo'               => '1',
        'logo_url'                => '',
        'logo_padding'            => '32px 32px 0 32px',
        #'logo_max_width'          => '180px',

		// Introduzione Email
		'order_intro_color'           => '#d30070',
        'order_intro_font'     => 'Arial, sans-serif',
        'order_intro_font_size'      => '16px',
		'order_intro_padding'         => '0 0 24px 0',

		// Titolo riepilogo ordine
		'order_heading_color'         => '#3c3c3c',
		'order_heading_font'          => '"Helvetica Neue",Helvetica,Roboto,Arial,sans-serif',
		'order_heading_size'          => '20px',
		'order_heading_weight'        => 'bold',
		'order_heading_align'         => 'left',
		'order_heading_margin'        => '0 0 18px 0',

		// Tabella ordini principale
		'order_table_bg'              => '#fff',
		'order_table_radius'          => '8',
		'order_table_border_width'    => '0',
		'order_table_border_color'    => '#fff',
		'order_table_border_style'    => 'solid',
		'order_table_box_shadow'      => '0 1px 10px rgba(0,0,0,0.07)',
		'order_table_padding'         => '0px',

		// Celle tabella ordini
		'order_cell_color'            => '#d30070',
		'order_cell_font'             => 'Roboto, sans-serif',
		'order_cell_font_size'        => '14px',
		'order_cell_padding'          => '8px 12px',
		'order_cell_bg'               => '#fff',

		// QuantitÃ /prezzo
		'order_price_color'           => '#d30070',
		'order_price_font_weight'     => 'bold',
		'order_price_font_size'       => '14px',

		// Meta prodotti (Colore, Variante, ecc)
		'order_item_meta_color'       => '#d30070',
		'order_item_meta_font_size'   => '14px',

		// Titoli indirizzi
		'order_address_title_color'   => '#3c3c3c',
		'order_address_font'          => '"Helvetica Neue",Helvetica,Roboto,Arial,sans-serif',
		'order_address_color'         => '#3c3c3c',
		'order_address_link_color'    => '#fbba16',
		
        // DETTAGLIO ORDINE
        'detail_color'               => '#111',
        'detail_font'                => 'Arial, sans-serif',
        'detail_size'                => '15px',
        'detail_padding'             => '10px 0',
        // HEADING
        'heading_color'              => '#222222',
        'heading_size'               => '28px',
        'heading_font'               => 'Arial, sans-serif',
        'heading_weight'             => 'bold',
        'heading_align'              => 'center',
        'heading_shadow_x'           => '0',
        'heading_shadow_y'           => '2px',
        'heading_shadow_blur'        => '4px',
        'heading_shadow_color'       => '#aaa',
        'heading_letter_spacing'     => '0',

		// HEADER WRAPPER PERSONALIZZATO
		'header_wrapper_bg'            => '#ffffff',   // Sfondo
		'header_wrapper_padding'       => '20px 32px 0', // Padding default
		'header_wrapper_radius'        => '0',           // Raggio bordi
		'header_wrapper_border_width'  => '0',           // Larghezza bordo
		'header_wrapper_border_style'  => 'solid',       // Stile bordo
		'header_wrapper_border_color'  => '#eeeeee',     // Colore bordo

        // TESTO PRINCIPALE
        'text_color'                 => '#222222',
        'text_size'                  => '16px',
        'text_font'                  => 'Arial, sans-serif',
        'text_align'                 => 'left',
        'text_lineheight'            => '1.6',
        // LINK
        'link_color'                 => '#1e73be',
        'link_hover_color'           => '#156fa3',
        'link_decoration'            => 'underline',
        // PULSANTI (CTA)
        'button_bg'                  => '#1e73be',
        'button_color'               => '#ffffff',
        'button_font'                => 'Arial, sans-serif',
        'button_size'                => '18px',
        'button_radius'              => '5px',
        'button_weight'              => 'bold',
        'button_shadow_x'            => '0',
        'button_shadow_y'            => '1px',
        'button_shadow_blur'         => '4px',
        'button_shadow_color'        => 'rgba(30,115,190,0.08)',
        'button_border_width'        => '0',
        'button_border_style'        => 'none',
        'button_border_color'        => '#1e73be',
        'button_padding'             => '12px 24px',
        // FOOTER
        'footer_bg'                  => '#f7f7f7',
        'footer_color'               => '#555555',
        'footer_size'                => '14px',
        'footer_font'                => 'Arial, sans-serif',
        'footer_align'               => 'center',
        'footer_text'                => 'Ricevi questa email per aggiornamenti sul tuo ordine.<br>Per informazioni rispondi a questa email.',
        'banner_footer_url'          => '',
        // DISCLAIMER/NOTE
        'show_disclaimer'            => '0',
        'disclaimer_text'            => 'Questa email Ã¨ generata automaticamente. Non rispondere.',
        // ALTRO
		'custom_css'                 => '',

        // TOTALI ORDINE
        'order_totals_color'         => '#222222',
        'order_totals_font'          => 'Arial, sans-serif',
        'order_totals_font_size'     => '16px',
        'order_totals_font_weight'   => 'normal',
    ];
}

function oste_email_styling_admin_page() {
	if ( ! current_user_can( 'manage_options' ) ) {
		wp_die( esc_html__( 'Sorry, you are not allowed to access this page.', 'order-status-tracking-emails-for-woocommerce' ) );
	}
    $fonts = [
        'Arial, sans-serif' => 'Arial',
        'Verdana, sans-serif' => 'Verdana',
        'Tahoma, sans-serif' => 'Tahoma',
        '"Trebuchet MS", sans-serif' => 'Trebuchet MS',
        '"Segoe UI", sans-serif' => 'Segoe UI',
        '"Open Sans", sans-serif' => 'Open Sans',
        '"Roboto", sans-serif' => 'Roboto',
        '"Lato", sans-serif' => 'Lato',
        '"Montserrat", sans-serif' => 'Montserrat',
        '"Times New Roman", serif' => 'Times New Roman',
        'Georgia, serif' => 'Georgia',
    ];
    $opts = wp_parse_args( get_option( 'oste_email_styling', [] ), oste_email_styling_default_options() );
	$status_list = oste_get_tutti_gli_stati();
	if ( ! is_array( $status_list ) ) {
		$status_list = array();
	}
	$status_keys            = array_keys( $status_list );
	$default_preview_status = ! empty( $status_keys ) ? $status_keys[0] : 'wc-processing';
	$preview_nonce          = wp_create_nonce( 'oste_email_style_preview' );
	$preview_ajax_url       = admin_url( 'admin-ajax.php' );
	$preview_error_message  = __( 'Unable to load the email preview. Please try again.', 'order-status-tracking-emails-for-woocommerce' );
?>
<div class="wrap oste-admin-wrap">
<h1><?php esc_html_e( 'Customize WooCommerce Email Styles', 'order-status-tracking-emails-for-woocommerce' ); ?></h1>
<form method="post" action="options.php">
<?php settings_fields('oste_email_styling_group'); ?>
<?php if ( ! empty( $status_list ) ) : ?>
<div id="oste-email-preview-root"
     class="oste-email-preview-panel"
     data-ajax-url="<?php echo esc_url( $preview_ajax_url ); ?>"
     data-nonce="<?php echo esc_attr( $preview_nonce ); ?>"
     data-error="<?php echo esc_attr( $preview_error_message ); ?>"
     data-note-default="<?php echo esc_attr__( 'Preview uses the most recent WooCommerce order and does not send any real email.', 'order-status-tracking-emails-for-woocommerce' ); ?>"
<?php /* translators: %s: order number used in the email preview (no real email is sent). */ ?>
	 data-note-template="<?php echo esc_attr__( 'Preview uses order #%s and does not send any real email.', 'order-status-tracking-emails-for-woocommerce' ); ?>">
    <h2><?php esc_html_e( 'Live Email Preview', 'order-status-tracking-emails-for-woocommerce' ); ?></h2>
    <p><?php esc_html_e( 'Choose an order status and adjust the style options to preview the resulting email instantly.', 'order-status-tracking-emails-for-woocommerce' ); ?></p>
    <div class="oste-email-preview-panel__controls">
        <label for="oste-email-preview-status"><?php esc_html_e( 'Preview order status', 'order-status-tracking-emails-for-woocommerce' ); ?></label>
        <select id="oste-email-preview-status" name="oste_preview_status">
            <?php foreach ( $status_list as $slug => $label ) : ?>
                <option value="<?php echo esc_attr( $slug ); ?>" <?php selected( $slug, $default_preview_status ); ?>><?php echo esc_html( $label ); ?></option>
            <?php endforeach; ?>
        </select>
        <span class="spinner" id="oste-email-preview-spinner"></span>
    </div>
    <div class="oste-email-preview-panel__subject" aria-live="polite">
        <strong><?php esc_html_e( 'Subject:', 'order-status-tracking-emails-for-woocommerce' ); ?></strong>
        <span id="oste-email-preview-subject"></span>
    </div>
    <div class="oste-email-preview-panel__frame">
        <iframe id="oste-email-preview-frame" class="oste-email-preview-panel__iframe" title="<?php esc_attr_e( 'Email preview', 'order-status-tracking-emails-for-woocommerce' ); ?>" sandbox="allow-same-origin allow-scripts" loading="lazy"></iframe>
    </div>
    <p class="description oste-email-preview-panel__note" id="oste-email-preview-note"><?php esc_html_e( 'Preview uses the most recent WooCommerce order and does not send any real email.', 'order-status-tracking-emails-for-woocommerce' ); ?></p>
</div>
<?php else : ?>
<div class="oste-email-preview-panel oste-email-preview-panel--empty">
    <h2><?php esc_html_e( 'Live Email Preview', 'order-status-tracking-emails-for-woocommerce' ); ?></h2>
    <p class="description"><?php esc_html_e( 'Add at least one order to enable the live preview.', 'order-status-tracking-emails-for-woocommerce' ); ?></p>
</div>
<?php endif; ?>
<style>
.oste-email-preview-panel {
	margin: 24px 0 36px;
	padding: 24px;
	background: #ffffff;
	border: 1px solid #dcdcde;
	border-radius: 4px;
	box-shadow: 0 1px 1px rgba(0,0,0,0.04);
}
.oste-email-preview-panel h2 {
	margin-top: 0;
}
.oste-email-preview-panel__controls {
	display: flex;
	flex-wrap: wrap;
	align-items: center;
	gap: 12px;
	margin-bottom: 12px;
}
.oste-email-preview-panel__controls select {
	min-width: 220px;
}
.oste-email-preview-panel__subject {
	margin-bottom: 14px;
	font-size: 14px;
	color: #3c434a;
}
.oste-email-preview-panel__subject strong {
	margin-right: 6px;
}
.oste-email-preview-panel__frame {
	position: relative;
}
.oste-email-preview-panel__iframe {
	width: 100%;
	min-height: 560px;
	border: 1px solid #ccd0d4;
	border-radius: 4px;
	background: #f6f7f7;
}
.oste-email-preview-panel__note {
	margin-top: 14px;
}
.oste-email-preview-panel--empty {
	margin: 24px 0 36px;
	padding: 24px;
	background: #fffdf5;
	border: 1px solid #f0d8a8;
	border-radius: 4px;
}
.oste-email-preview-panel--empty h2 {
	margin-top: 0;
}
.oste-email-preview-panel--empty .description {
	margin: 0;
}
#oste-email-preview-spinner {
	margin-left: auto;
}
</style>
<table class="form-table" style="max-width:980px;">
		<tr>
    <th><?php esc_html_e( 'Enable custom email styles', 'order-status-tracking-emails-for-woocommerce' ); ?></th>
    <td>
        <label class="oste-toggle-label">
            <span class="oste-toggle">
                <input type="checkbox"
                       name="oste_email_styling[enable_custom_styles]"
                       value="1"
                       <?php checked( $opts['enable_custom_styles'] ?? '', '1' ); ?> />
                <span class="oste-toggle-slider" aria-hidden="true"></span>
            </span>
            <span><?php esc_html_e( 'If disabled, WooCommerce emails will use the default WooCommerce style (without customization from this plugin).', 'order-status-tracking-emails-for-woocommerce' ); ?></span>
        </label>
    </td>
</tr>
<tr><th colspan="2"><h2><?php esc_html_e( 'General', 'order-status-tracking-emails-for-woocommerce' ); ?></h2></th></tr>
<tr>
    <th><?php esc_html_e( 'Email background', 'order-status-tracking-emails-for-woocommerce' ); ?></th>
    <td>
        <input type="text" name="oste_email_styling[bg_color]" value="<?php echo esc_attr($opts['bg_color']); ?>" class="color-picker" data-default-color="#f5f5f5" style="width: 80px;" />
        &nbsp; <?php esc_html_e( 'Background image:', 'order-status-tracking-emails-for-woocommerce' ); ?>
        <input type="text" id="oste_bg_image" name="oste_email_styling[bg_image]" value="<?php echo esc_attr($opts['bg_image']); ?>" style="width:260px;" />
		<button class="button oste-upload-image" type="button" data-target="oste_bg_image"><?php esc_html_e( 'Choose image', 'order-status-tracking-emails-for-woocommerce' ); ?></button>
		<button class="button oste-remove-image" type="button" data-target="oste_bg_image"><?php esc_html_e( 'Remove', 'order-status-tracking-emails-for-woocommerce' ); ?></button>
		<?php if ($opts['bg_image']): ?><br><img src="<?php echo esc_url($opts['bg_image']); ?>" style="max-width:160px;max-height:50px;" /><?php endif; ?>
    </td>
</tr>
<tr><th colspan="2"><h2><?php esc_html_e( 'Main Text', 'order-status-tracking-emails-for-woocommerce' ); ?></h2></th></tr>
<tr>
    <th><?php esc_html_e( 'Text color', 'order-status-tracking-emails-for-woocommerce' ); ?></th>
    <td><input type="text" name="oste_email_styling[text_color]" value="<?php echo esc_attr($opts['text_color']); ?>" class="color-picker" /></td>
</tr>
<tr>
    <th><?php esc_html_e( 'Text size', 'order-status-tracking-emails-for-woocommerce' ); ?></th>
    <td><input type="text" name="oste_email_styling[text_size]" value="<?php echo esc_attr($opts['text_size']); ?>" style="width:80px;" /></td>
</tr>
<tr>
    <th><?php esc_html_e( 'Text font', 'order-status-tracking-emails-for-woocommerce' ); ?></th>
    <td>
        <select name="oste_email_styling[text_font]">
            <?php foreach($fonts as $val => $label): ?>
                <option value="<?php echo esc_attr($val); ?>" <?php selected($opts['text_font'],$val); ?>><?php echo esc_html($label); ?></option>
            <?php endforeach; ?>
        </select>
    </td>
</tr>
<tr>
    <th><?php esc_html_e( 'Text alignment', 'order-status-tracking-emails-for-woocommerce' ); ?></th>
    <td>
        <select name="oste_email_styling[text_align]">
            <option value="left" <?php selected($opts['text_align'],'left'); ?>><?php esc_html_e( 'Left', 'order-status-tracking-emails-for-woocommerce' ); ?></option>
            <option value="center" <?php selected($opts['text_align'],'center'); ?>><?php esc_html_e( 'Center', 'order-status-tracking-emails-for-woocommerce' ); ?></option>
            <option value="right" <?php selected($opts['text_align'],'right'); ?>><?php esc_html_e( 'Right', 'order-status-tracking-emails-for-woocommerce' ); ?></option>
        </select>
    </td>
</tr>
<tr>
    <th><?php esc_html_e( 'Line height', 'order-status-tracking-emails-for-woocommerce' ); ?></th>
    <td><input type="text" name="oste_email_styling[text_lineheight]" value="<?php echo esc_attr($opts['text_lineheight']); ?>" style="width:80px;" /></td>
</tr>

<!-- LINK -->
<tr><th colspan="2"><h2><?php esc_html_e('Links', 'order-status-tracking-emails-for-woocommerce'); ?></h2></th></tr>
<tr>
    <th><?php esc_html_e('Link color', 'order-status-tracking-emails-for-woocommerce'); ?></th>
    <td><input type="text" name="oste_email_styling[link_color]" value="<?php echo esc_attr($opts['link_color']); ?>" class="color-picker" /></td>
</tr>
<tr>
    <th><?php esc_html_e('Link hover color', 'order-status-tracking-emails-for-woocommerce'); ?></th>
    <td><input type="text" name="oste_email_styling[link_hover_color]" value="<?php echo esc_attr($opts['link_hover_color']); ?>" class="color-picker" /></td>
</tr>
<tr>
    <th><?php esc_html_e('Link decoration', 'order-status-tracking-emails-for-woocommerce'); ?></th>
    <td>
        <select name="oste_email_styling[link_decoration]">
            <option value="none" <?php selected($opts['link_decoration'],'none'); ?>><?php esc_html_e('None', 'order-status-tracking-emails-for-woocommerce'); ?></option>
            <option value="underline" <?php selected($opts['link_decoration'],'underline'); ?>><?php esc_html_e('Underline', 'order-status-tracking-emails-for-woocommerce'); ?></option>
        </select>
    </td>
</tr>

<!-- CTA BUTTONS -->
<tr><th colspan="2"><h2><?php esc_html_e('Buttons (CTA)', 'order-status-tracking-emails-for-woocommerce'); ?></h2></th></tr>
<tr>
    <th><?php esc_html_e('Button background color', 'order-status-tracking-emails-for-woocommerce'); ?></th>
    <td><input type="text" name="oste_email_styling[button_bg]" value="<?php echo esc_attr($opts['button_bg']); ?>" class="color-picker" /></td>
</tr>
<tr>
    <th><?php esc_html_e('Button text color', 'order-status-tracking-emails-for-woocommerce'); ?></th>
    <td><input type="text" name="oste_email_styling[button_color]" value="<?php echo esc_attr($opts['button_color']); ?>" class="color-picker" /></td>
</tr>
<tr>
    <th><?php esc_html_e('Button font', 'order-status-tracking-emails-for-woocommerce'); ?></th>
    <td>
        <select name="oste_email_styling[button_font]">
            <?php foreach($fonts as $val => $label): ?>
                <option value="<?php echo esc_attr($val); ?>" <?php selected($opts['button_font'],$val); ?>><?php echo esc_html($label); ?></option>
            <?php endforeach; ?>
        </select>
    </td>
</tr>
<tr>
    <th><?php esc_html_e('Button text size', 'order-status-tracking-emails-for-woocommerce'); ?></th>
    <td><input type="text" name="oste_email_styling[button_size]" value="<?php echo esc_attr($opts['button_size']); ?>" style="width:80px;" /></td>
</tr>
<tr>
    <th><?php esc_html_e('Button border radius', 'order-status-tracking-emails-for-woocommerce'); ?></th>
    <td><input type="text" name="oste_email_styling[button_radius]" value="<?php echo esc_attr($opts['button_radius']); ?>" style="width:80px;" /></td>
</tr>
<tr>
    <th><?php esc_html_e('Button text weight', 'order-status-tracking-emails-for-woocommerce'); ?></th>
    <td>
        <select name="oste_email_styling[button_weight]">
            <option value="normal" <?php selected($opts['button_weight'],'normal'); ?>><?php esc_html_e('Normal', 'order-status-tracking-emails-for-woocommerce'); ?></option>
            <option value="bold" <?php selected($opts['button_weight'],'bold'); ?>><?php esc_html_e('Bold', 'order-status-tracking-emails-for-woocommerce'); ?></option>
        </select>
    </td>
</tr>
<tr>
    <th><?php esc_html_e('Button shadow', 'order-status-tracking-emails-for-woocommerce'); ?></th>
    <td>
        X: <input type="text" name="oste_email_styling[button_shadow_x]" value="<?php echo esc_attr($opts['button_shadow_x']); ?>" style="width:40px;" />
        Y: <input type="text" name="oste_email_styling[button_shadow_y]" value="<?php echo esc_attr($opts['button_shadow_y']); ?>" style="width:40px;" />
        Blur: <input type="text" name="oste_email_styling[button_shadow_blur]" value="<?php echo esc_attr($opts['button_shadow_blur']); ?>" style="width:40px;" />
        <?php esc_html_e('Color:', 'order-status-tracking-emails-for-woocommerce'); ?> <input type="text" name="oste_email_styling[button_shadow_color]" value="<?php echo esc_attr($opts['button_shadow_color']); ?>" class="color-picker" style="width:90px;" />
    </td>
</tr>
<tr>
    <th><?php esc_html_e('Button border', 'order-status-tracking-emails-for-woocommerce'); ?></th>
    <td>
        <input type="text" name="oste_email_styling[button_border_width]" value="<?php echo esc_attr($opts['button_border_width']); ?>" style="width:60px;" placeholder="0" />
        <select name="oste_email_styling[button_border_style]" style="width:80px;">
            <option value="none" <?php selected($opts['button_border_style'],'none'); ?>><?php esc_html_e('None', 'order-status-tracking-emails-for-woocommerce'); ?></option>
            <option value="solid" <?php selected($opts['button_border_style'],'solid'); ?>><?php esc_html_e('Solid', 'order-status-tracking-emails-for-woocommerce'); ?></option>
            <option value="dashed" <?php selected($opts['button_border_style'],'dashed'); ?>><?php esc_html_e('Dashed', 'order-status-tracking-emails-for-woocommerce'); ?></option>
            <option value="dotted" <?php selected($opts['button_border_style'],'dotted'); ?>><?php esc_html_e('Dotted', 'order-status-tracking-emails-for-woocommerce'); ?></option>
            <option value="double" <?php selected($opts['button_border_style'],'double'); ?>><?php esc_html_e('Double', 'order-status-tracking-emails-for-woocommerce'); ?></option>
        </select>
        <input type="text" name="oste_email_styling[button_border_color]" value="<?php echo esc_attr($opts['button_border_color']); ?>" class="color-picker" style="width:80px;" />
    </td>
</tr>
<tr>
    <th><?php esc_html_e('Button padding', 'order-status-tracking-emails-for-woocommerce'); ?></th>
    <td><input type="text" name="oste_email_styling[button_padding]" value="<?php echo esc_attr($opts['button_padding']); ?>" style="width:120px;" /></td>
</tr>

<tr><th colspan="2"><h2><?php esc_html_e('Logo above heading', 'order-status-tracking-emails-for-woocommerce'); ?></h2></th></tr>
<tr>
    <th><?php esc_html_e('Logo', 'order-status-tracking-emails-for-woocommerce'); ?></th>
    <td>
        <input type="text" id="oste_logo_url" name="oste_email_styling[logo_url]" value="<?php echo esc_attr($opts['logo_url']); ?>" style="width:260px;" readonly />
        <button class="button oste-upload-image" type="button" data-target="oste_logo_url"><?php esc_html_e('Choose image', 'order-status-tracking-emails-for-woocommerce'); ?></button>
        <button class="button oste-remove-image" type="button" data-target="oste_logo_url"><?php esc_html_e('Remove', 'order-status-tracking-emails-for-woocommerce'); ?></button>
        <br>
        <label><?php esc_html_e('Logo width:', 'order-status-tracking-emails-for-woocommerce'); ?>
            <input type="number" name="oste_email_styling[logo_width]" value="<?php echo esc_attr($opts['logo_width'] ?? '200'); ?>" min="40" max="800" step="1" style="width:80px;" /> px
        </label>
        <br>
        <label><?php esc_html_e('Alignment:', 'order-status-tracking-emails-for-woocommerce'); ?>
            <select name="oste_email_styling[logo_align]">
                <option value="left" <?php selected($opts['logo_align'] ?? '', 'left'); ?>><?php esc_html_e('Left', 'order-status-tracking-emails-for-woocommerce'); ?></option>
                <option value="center" <?php selected($opts['logo_align'] ?? '', 'center'); ?>><?php esc_html_e('Center', 'order-status-tracking-emails-for-woocommerce'); ?></option>
                <option value="right" <?php selected($opts['logo_align'] ?? '', 'right'); ?>><?php esc_html_e('Right', 'order-status-tracking-emails-for-woocommerce'); ?></option>
            </select>
        </label>
		<br>
		<label><?php esc_html_e('Logo padding:', 'order-status-tracking-emails-for-woocommerce'); ?>
			<input type="text" name="oste_email_styling[logo_padding]" value="<?php echo esc_attr($opts['logo_padding'] ?? '32px 32px 0 32px'); ?>" style="width:140px;" />
		</label>

        <?php if ($opts['logo_url']): ?>
            <br>
            <div style="text-align:<?php echo esc_attr($opts['logo_align'] ?? 'center'); ?>">
                <img src="<?php echo esc_url($opts['logo_url']); ?>" style="width:<?php echo esc_attr($opts['logo_width'] ?? '200'); ?>px;max-height:70px;" />
            </div>
        <?php endif; ?>
    </td>
</tr>

	
<!-- HEADING -->
<tr><th colspan="2"><h2><?php esc_html_e('Heading', 'order-status-tracking-emails-for-woocommerce'); ?></h2></th></tr>
<tr>
    <th><?php esc_html_e('Heading text color', 'order-status-tracking-emails-for-woocommerce'); ?></th>
    <td><input type="text" name="oste_email_styling[heading_color]" value="<?php echo esc_attr($opts['heading_color']); ?>" class="color-picker" /></td>
</tr>
<tr>
    <th><?php esc_html_e('Heading text shadow', 'order-status-tracking-emails-for-woocommerce'); ?></th>
    <td>
        X: <input type="text" name="oste_email_styling[heading_shadow_x]" value="<?php echo esc_attr($opts['heading_shadow_x']); ?>" style="width:40px;" />
        Y: <input type="text" name="oste_email_styling[heading_shadow_y]" value="<?php echo esc_attr($opts['heading_shadow_y']); ?>" style="width:40px;" />
        Blur: <input type="text" name="oste_email_styling[heading_shadow_blur]" value="<?php echo esc_attr($opts['heading_shadow_blur']); ?>" style="width:40px;" />
        <?php esc_html_e('Color:', 'order-status-tracking-emails-for-woocommerce'); ?> <input type="text" name="oste_email_styling[heading_shadow_color]" value="<?php echo esc_attr($opts['heading_shadow_color']); ?>" class="color-picker" style="width:90px;" />
    </td>
</tr>
<tr>
    <th><?php esc_html_e('Text size', 'order-status-tracking-emails-for-woocommerce'); ?></th>
    <td><input type="text" name="oste_email_styling[heading_size]" value="<?php echo esc_attr($opts['heading_size']); ?>" style="width:80px;" /></td>
</tr>
<tr>
    <th><?php esc_html_e('Font', 'order-status-tracking-emails-for-woocommerce'); ?></th>
    <td>
        <select name="oste_email_styling[heading_font]">
            <?php foreach($fonts as $val => $label): ?>
                <option value="<?php echo esc_attr($val); ?>" <?php selected($opts['heading_font'],$val); ?>><?php echo esc_html($label); ?></option>
            <?php endforeach; ?>
        </select>
    </td>
</tr>
<tr>
    <th><?php esc_html_e('Weight', 'order-status-tracking-emails-for-woocommerce'); ?></th>
    <td>
        <select name="oste_email_styling[heading_weight]">
            <option value="normal" <?php selected($opts['heading_weight'],'normal'); ?>><?php esc_html_e('Normal', 'order-status-tracking-emails-for-woocommerce'); ?></option>
            <option value="bold" <?php selected($opts['heading_weight'],'bold'); ?>><?php esc_html_e('Bold', 'order-status-tracking-emails-for-woocommerce'); ?></option>
        </select>
    </td>
</tr>
<tr>
    <th><?php esc_html_e('Alignment', 'order-status-tracking-emails-for-woocommerce'); ?></th>
    <td>
        <select name="oste_email_styling[heading_align]">
            <option value="left" <?php selected($opts['heading_align'],'left'); ?>><?php esc_html_e('Left', 'order-status-tracking-emails-for-woocommerce'); ?></option>
            <option value="center" <?php selected($opts['heading_align'],'center'); ?>><?php esc_html_e('Center', 'order-status-tracking-emails-for-woocommerce'); ?></option>
            <option value="right" <?php selected($opts['heading_align'],'right'); ?>><?php esc_html_e('Right', 'order-status-tracking-emails-for-woocommerce'); ?></option>
        </select>
    </td>
</tr>
<tr>
    <th><?php esc_html_e('Letter spacing', 'order-status-tracking-emails-for-woocommerce'); ?></th>
    <td>
        <input type="text" name="oste_email_styling[heading_letter_spacing]" value="<?php echo esc_attr($opts['heading_letter_spacing'] ?? '0'); ?>" style="width:60px;" /> px
    </td>
</tr>
<tr>
    <th><?php esc_html_e('Heading margin', 'order-status-tracking-emails-for-woocommerce'); ?></th>
    <td>
        <input type="text" name="oste_email_styling[heading_margin]" value="<?php echo esc_attr($opts['heading_margin'] ?? '0'); ?>" style="width:80px;" />
    </td>
</tr>
<tr><th colspan="2"><h2><?php esc_html_e('Header Wrapper (inner heading container)', 'order-status-tracking-emails-for-woocommerce'); ?></h2></th></tr>
<tr>
    <th><?php esc_html_e('Header Wrapper background', 'order-status-tracking-emails-for-woocommerce'); ?></th>
    <td><input type="text" name="oste_email_styling[header_wrapper_bg]" value="<?php echo esc_attr($opts['header_wrapper_bg'] ?? '#ffffff'); ?>" class="color-picker" /></td>
</tr>
<tr>
    <th><?php esc_html_e('Header Wrapper padding', 'order-status-tracking-emails-for-woocommerce'); ?></th>
    <td><input type="text" name="oste_email_styling[header_wrapper_padding]" value="<?php echo esc_attr($opts['header_wrapper_padding'] ?? '20px 32px 0'); ?>" style="width:150px;" /></td>
</tr>
<tr>
    <th><?php esc_html_e('Header Wrapper border', 'order-status-tracking-emails-for-woocommerce'); ?></th>
    <td>
        <input type="text" name="oste_email_styling[header_wrapper_border_width]" value="<?php echo esc_attr($opts['header_wrapper_border_width'] ?? '0'); ?>" style="width:60px;" placeholder="0" /> px
        <select name="oste_email_styling[header_wrapper_border_style]" style="width:80px;">
            <option value="solid" <?php selected($opts['header_wrapper_border_style'],'solid'); ?>><?php esc_html_e('Solid', 'order-status-tracking-emails-for-woocommerce'); ?></option>
            <option value="dashed" <?php selected($opts['header_wrapper_border_style'],'dashed'); ?>><?php esc_html_e('Dashed', 'order-status-tracking-emails-for-woocommerce'); ?></option>
            <option value="dotted" <?php selected($opts['header_wrapper_border_style'],'dotted'); ?>><?php esc_html_e('Dotted', 'order-status-tracking-emails-for-woocommerce'); ?></option>
            <option value="double" <?php selected($opts['header_wrapper_border_style'],'double'); ?>><?php esc_html_e('Double', 'order-status-tracking-emails-for-woocommerce'); ?></option>
            <option value="none" <?php selected($opts['header_wrapper_border_style'],'none'); ?>><?php esc_html_e('None', 'order-status-tracking-emails-for-woocommerce'); ?></option>
        </select>
        <input type="text" name="oste_email_styling[header_wrapper_border_color]" value="<?php echo esc_attr($opts['header_wrapper_border_color'] ?? '#eeeeee'); ?>" class="color-picker" style="width:80px;" />
    </td>
</tr>
<tr>
    <th><?php esc_html_e('Header Wrapper border radius', 'order-status-tracking-emails-for-woocommerce'); ?></th>
    <td>
        <input type="number" min="0" max="40" name="oste_email_styling[header_wrapper_radius]" value="<?php echo esc_attr($opts['header_wrapper_radius'] ?? '0'); ?>" style="width:60px;" /> px
    </td>
</tr>

	
<!-- Email Inner Content -->	
<tr>
    <th colspan="2"><h2><?php esc_html_e('Email Inner Content (inner_wrapper)', 'order-status-tracking-emails-for-woocommerce'); ?></h2></th>
</tr>
<tr>
    <th><?php esc_html_e('Inner background color', 'order-status-tracking-emails-for-woocommerce'); ?></th>
    <td>
        <input type="text" name="oste_email_styling[inner_bg_color]" value="<?php echo esc_attr($opts['inner_bg_color'] ?? '#ffffff'); ?>" class="color-picker" data-default-color="#ffffff" style="width:80px;" />
    </td>
</tr>
<tr>
    <th><?php esc_html_e('Border', 'order-status-tracking-emails-for-woocommerce'); ?></th>
    <td>
        <input type="text" name="oste_email_styling[inner_border_width]" value="<?php echo esc_attr($opts['inner_border_width'] ?? '1px'); ?>" style="width:60px;" placeholder="1px" />
        <select name="oste_email_styling[inner_border_style]" style="width:80px;">
            <option value="solid" <?php selected($opts['inner_border_style'] ?? '', 'solid'); ?>><?php esc_html_e('Solid', 'order-status-tracking-emails-for-woocommerce'); ?></option>
            <option value="dashed" <?php selected($opts['inner_border_style'] ?? '', 'dashed'); ?>><?php esc_html_e('Dashed', 'order-status-tracking-emails-for-woocommerce'); ?></option>
            <option value="dotted" <?php selected($opts['inner_border_style'] ?? '', 'dotted'); ?>><?php esc_html_e('Dotted', 'order-status-tracking-emails-for-woocommerce'); ?></option>
            <option value="double" <?php selected($opts['inner_border_style'] ?? '', 'double'); ?>><?php esc_html_e('Double', 'order-status-tracking-emails-for-woocommerce'); ?></option>
            <option value="none" <?php selected($opts['inner_border_style'] ?? '', 'none'); ?>><?php esc_html_e('None', 'order-status-tracking-emails-for-woocommerce'); ?></option>
        </select>
        <input type="text" name="oste_email_styling[inner_border_color]" value="<?php echo esc_attr($opts['inner_border_color'] ?? '#eeeeee'); ?>" class="color-picker" style="width:80px;" />
    </td>
</tr>
<tr>
    <th><?php esc_html_e('Border radius', 'order-status-tracking-emails-for-woocommerce'); ?></th>
    <td>
        <input type="number" min="0" max="40" name="oste_email_styling[inner_radius]" value="<?php echo esc_attr($opts['inner_radius'] ?? '8'); ?>" style="width:60px;" /> px
    </td>
</tr>
<tr>
    <th><?php esc_html_e('Shadow', 'order-status-tracking-emails-for-woocommerce'); ?></th>
    <td>
        X: <input type="text" name="oste_email_styling[inner_shadow_x]" value="<?php echo esc_attr($opts['inner_shadow_x'] ?? '0'); ?>" style="width:40px;" />
        Y: <input type="text" name="oste_email_styling[inner_shadow_y]" value="<?php echo esc_attr($opts['inner_shadow_y'] ?? '1px'); ?>" style="width:40px;" />
        <?php esc_html_e('Blur', 'order-status-tracking-emails-for-woocommerce'); ?>: <input type="text" name="oste_email_styling[inner_shadow_blur]" value="<?php echo esc_attr($opts['inner_shadow_blur'] ?? '10px'); ?>" style="width:40px;" />
        <?php esc_html_e('Color', 'order-status-tracking-emails-for-woocommerce'); ?>: <input type="text" name="oste_email_styling[inner_shadow_color]" value="<?php echo esc_attr($opts['inner_shadow_color'] ?? 'rgba(0,0,0,0.07)'); ?>" class="color-picker" style="width:90px;" />
    </td>
</tr>
<tr>
    <th><?php esc_html_e('Inner padding', 'order-status-tracking-emails-for-woocommerce'); ?></th>
    <td>
        <input type="text" name="oste_email_styling[inner_padding]" value="<?php echo esc_attr($opts['inner_padding'] ?? '24px'); ?>" style="width:80px;" />
    </td>
</tr>
	
<!-- Template container -->	
<tr><th colspan="2"><h2><?php esc_html_e('Template Container (outer order block)', 'order-status-tracking-emails-for-woocommerce'); ?></h2></th></tr>
<tr>
    <th><?php esc_html_e('Background color', 'order-status-tracking-emails-for-woocommerce'); ?></th>
    <td><input type="text" name="oste_email_styling[template_container_bg]" value="<?php echo esc_attr($opts['template_container_bg']); ?>" class="color-picker" /></td>
</tr>
<tr>
    <th><?php esc_html_e('Border radius', 'order-status-tracking-emails-for-woocommerce'); ?></th>
    <td><input type="number" min="0" max="40" name="oste_email_styling[template_container_radius]" value="<?php echo esc_attr($opts['template_container_radius']); ?>" style="width:60px;" /> px</td>
</tr>
<tr>
    <th><?php esc_html_e('Border', 'order-status-tracking-emails-for-woocommerce'); ?></th>
    <td>
        <input type="number" min="0" max="20" name="oste_email_styling[template_container_border_width]" value="<?php echo esc_attr($opts['template_container_border_width']); ?>" style="width:60px;" /> px
        <select name="oste_email_styling[template_container_border_style]">
            <option value="solid" <?php selected($opts['template_container_border_style'], 'solid'); ?>><?php esc_html_e('Solid', 'order-status-tracking-emails-for-woocommerce'); ?></option>
            <option value="dashed" <?php selected($opts['template_container_border_style'], 'dashed'); ?>><?php esc_html_e('Dashed', 'order-status-tracking-emails-for-woocommerce'); ?></option>
            <option value="dotted" <?php selected($opts['template_container_border_style'], 'dotted'); ?>><?php esc_html_e('Dotted', 'order-status-tracking-emails-for-woocommerce'); ?></option>
            <option value="double" <?php selected($opts['template_container_border_style'], 'double'); ?>><?php esc_html_e('Double', 'order-status-tracking-emails-for-woocommerce'); ?></option>
            <option value="none" <?php selected($opts['template_container_border_style'], 'none'); ?>><?php esc_html_e('None', 'order-status-tracking-emails-for-woocommerce'); ?></option>
        </select>
        <input type="text" name="oste_email_styling[template_container_border_color]" value="<?php echo esc_attr($opts['template_container_border_color']); ?>" class="color-picker" style="width:80px;" />
    </td>
</tr>
<tr>
    <th><?php esc_html_e('Box shadow', 'order-status-tracking-emails-for-woocommerce'); ?></th>
    <td>
        <input type="text" name="oste_email_styling[template_container_box_shadow]" value="<?php echo esc_attr($opts['template_container_box_shadow']); ?>" style="width:150px;" placeholder="e.g. 0 2px 10px #bbb" />
    </td>
</tr>
<tr>
    <th><?php esc_html_e('Padding', 'order-status-tracking-emails-for-woocommerce'); ?></th>
    <td>
        <input type="text" name="oste_email_styling[template_container_padding]" value="<?php echo esc_attr($opts['template_container_padding']); ?>" style="width:150px;" placeholder="e.g. 20px 32px 0" />
    </td>
</tr>
<!-- Order Body Container 2 -->	
<tr>
    <th colspan="2"><h2><?php esc_html_e('Order Body Container (outer cell)', 'order-status-tracking-emails-for-woocommerce'); ?></h2></th>
</tr>	
<tr>
    <th><?php esc_html_e('Container color', 'order-status-tracking-emails-for-woocommerce'); ?></th>
    <td><input type="text" name="oste_email_styling[container_cell_color]" value="<?php echo esc_attr($opts['container_cell_color']); ?>" class="color-picker" data-default-color="#ffffff" /></td>
</tr>
<tr>
    <th><?php esc_html_e('Border radius', 'order-status-tracking-emails-for-woocommerce'); ?></th>
    <td><input type="number" min="0" max="60" name="oste_email_styling[container_cell_radius]" value="<?php echo esc_attr($opts['container_cell_radius']); ?>" style="width:60px;" /> px</td>
</tr>
<tr>
    <th><?php esc_html_e('Border', 'order-status-tracking-emails-for-woocommerce'); ?></th>
    <td>
        <input type="text" name="oste_email_styling[container_cell_border_width]" value="<?php echo esc_attr($opts['container_cell_border_width']); ?>" style="width:60px;" placeholder="1px" />
        <select name="oste_email_styling[container_cell_border_style]" style="width:80px;">
            <option value="solid" <?php selected($opts['container_cell_border_style'],'solid'); ?>><?php esc_html_e('Solid', 'order-status-tracking-emails-for-woocommerce'); ?></option>
            <option value="dashed" <?php selected($opts['container_cell_border_style'],'dashed'); ?>><?php esc_html_e('Dashed', 'order-status-tracking-emails-for-woocommerce'); ?></option>
            <option value="dotted" <?php selected($opts['container_cell_border_style'],'dotted'); ?>><?php esc_html_e('Dotted', 'order-status-tracking-emails-for-woocommerce'); ?></option>
            <option value="double" <?php selected($opts['container_cell_border_style'],'double'); ?>><?php esc_html_e('Double', 'order-status-tracking-emails-for-woocommerce'); ?></option>
            <option value="none" <?php selected($opts['container_cell_border_style'],'none'); ?>><?php esc_html_e('None', 'order-status-tracking-emails-for-woocommerce'); ?></option>
        </select>
        <input type="text" name="oste_email_styling[container_cell_border_color]" value="<?php echo esc_attr($opts['container_cell_border_color']); ?>" class="color-picker" style="width:80px;" />
    </td>
</tr>
<tr>
    <th><?php esc_html_e('Box shadow', 'order-status-tracking-emails-for-woocommerce'); ?></th>
    <td>
        X: <input type="text" name="oste_email_styling[container_cell_shadow_x]" value="<?php echo esc_attr($opts['container_cell_shadow_x']); ?>" style="width:40px;" />
        Y: <input type="text" name="oste_email_styling[container_cell_shadow_y]" value="<?php echo esc_attr($opts['container_cell_shadow_y']); ?>" style="width:40px;" />
        <?php esc_html_e('Blur', 'order-status-tracking-emails-for-woocommerce'); ?>: <input type="text" name="oste_email_styling[container_cell_shadow_blur]" value="<?php echo esc_attr($opts['container_cell_shadow_blur']); ?>" style="width:40px;" />
        <?php esc_html_e('Color', 'order-status-tracking-emails-for-woocommerce'); ?>: <input type="text" name="oste_email_styling[container_cell_shadow_color]" value="<?php echo esc_attr($opts['container_cell_shadow_color']); ?>" class="color-picker" style="width:90px;" />
    </td>
</tr>
<tr>
    <th><?php esc_html_e('Inner padding', 'order-status-tracking-emails-for-woocommerce'); ?></th>
    <td><input type="text" name="oste_email_styling[container_cell_padding]" value="<?php echo esc_attr($opts['container_cell_padding']); ?>" style="width:80px;" /></td>
</tr>
<tr>
    <th><?php esc_html_e('Element spacing', 'order-status-tracking-emails-for-woocommerce'); ?></th>
    <td><input type="text" name="oste_email_styling[element_spacing]" value="<?php echo esc_attr($opts['element_spacing']); ?>" style="width:80px;" /></td>
</tr>	
<!-- Order Body Container -->	
<tr>
    <th colspan="2"><h2><?php esc_html_e('Order Body Container (inner)', 'order-status-tracking-emails-for-woocommerce'); ?></h2></th>
</tr>	
<tr>
    <th><?php esc_html_e('Container color', 'order-status-tracking-emails-for-woocommerce'); ?></th>
    <td><input type="text" name="oste_email_styling[container_color]" value="<?php echo esc_attr($opts['container_color']); ?>" class="color-picker" data-default-color="#ffffff" /></td>
</tr>
<tr>
    <th><?php esc_html_e('Border radius', 'order-status-tracking-emails-for-woocommerce'); ?></th>
    <td><input type="number" min="0" max="60" name="oste_email_styling[container_radius]" value="<?php echo esc_attr($opts['container_radius']); ?>" style="width:60px;" /> px</td>
</tr>
<tr>
    <th><?php esc_html_e('Border', 'order-status-tracking-emails-for-woocommerce'); ?></th>
    <td>
        <input type="text" name="oste_email_styling[container_border_width]" value="<?php echo esc_attr($opts['container_border_width']); ?>" style="width:60px;" placeholder="1px" />
        <select name="oste_email_styling[container_border_style]" style="width:80px;">
            <option value="solid" <?php selected($opts['container_border_style'],'solid'); ?>><?php esc_html_e('Solid', 'order-status-tracking-emails-for-woocommerce'); ?></option>
            <option value="dashed" <?php selected($opts['container_border_style'],'dashed'); ?>><?php esc_html_e('Dashed', 'order-status-tracking-emails-for-woocommerce'); ?></option>
            <option value="dotted" <?php selected($opts['container_border_style'],'dotted'); ?>><?php esc_html_e('Dotted', 'order-status-tracking-emails-for-woocommerce'); ?></option>
            <option value="double" <?php selected($opts['container_border_style'],'double'); ?>><?php esc_html_e('Double', 'order-status-tracking-emails-for-woocommerce'); ?></option>
            <option value="none" <?php selected($opts['container_border_style'],'none'); ?>><?php esc_html_e('None', 'order-status-tracking-emails-for-woocommerce'); ?></option>
        </select>
        <input type="text" name="oste_email_styling[container_border_color]" value="<?php echo esc_attr($opts['container_border_color']); ?>" class="color-picker" style="width:80px;" />
    </td>
</tr>
<tr>
    <th><?php esc_html_e('Box shadow', 'order-status-tracking-emails-for-woocommerce'); ?></th>
    <td>
        X: <input type="text" name="oste_email_styling[container_shadow_x]" value="<?php echo esc_attr($opts['container_shadow_x']); ?>" style="width:40px;" />
        Y: <input type="text" name="oste_email_styling[container_shadow_y]" value="<?php echo esc_attr($opts['container_shadow_y']); ?>" style="width:40px;" />
        <?php esc_html_e('Blur', 'order-status-tracking-emails-for-woocommerce'); ?>: <input type="text" name="oste_email_styling[container_shadow_blur]" value="<?php echo esc_attr($opts['container_shadow_blur']); ?>" style="width:40px;" />
        <?php esc_html_e('Color', 'order-status-tracking-emails-for-woocommerce'); ?>: <input type="text" name="oste_email_styling[container_shadow_color]" value="<?php echo esc_attr($opts['container_shadow_color']); ?>" class="color-picker" style="width:90px;" />
    </td>
</tr>
<tr>
    <th><?php esc_html_e('Inner padding', 'order-status-tracking-emails-for-woocommerce'); ?></th>
    <td><input type="text" name="oste_email_styling[container_padding]" value="<?php echo esc_attr($opts['container_padding']); ?>" style="width:80px;" /></td>
</tr>

<!-- ORDER TABLE -->
<tr><th colspan="2"><h2><?php esc_html_e('Order Details / Table', 'order-status-tracking-emails-for-woocommerce'); ?></h2></th></tr>

<tr><th colspan="2"><b><?php esc_html_e('Introduction text (above table)', 'order-status-tracking-emails-for-woocommerce'); ?></b></th></tr>
<tr>
    <th><?php esc_html_e('Intro text color', 'order-status-tracking-emails-for-woocommerce'); ?></th>
    <td><input type="text" name="oste_email_styling[order_intro_color]" value="<?php echo esc_attr($opts['order_intro_color']); ?>" class="color-picker" /></td>
</tr>
<tr>
	<th><?php esc_html_e('Intro font', 'order-status-tracking-emails-for-woocommerce'); ?></th>
	<td>
		<select name="oste_email_styling[order_intro_font]">
			<?php foreach($fonts as $val => $label): ?>
				<option value="<?php echo esc_attr($val); ?>" <?php selected($opts['order_intro_font'],$val); ?>><?php echo esc_html($label); ?></option>
			<?php endforeach; ?>
		</select>
	</td>
</tr>
<tr>
	<th><?php esc_html_e('Intro font size', 'order-status-tracking-emails-for-woocommerce'); ?></th>
	<td><input type="text" name="oste_email_styling[order_intro_font_size]" value="<?php echo esc_attr($opts['order_intro_font_size']); ?>" style="width:80px;" /></td>
</tr>
<tr>
    <th><?php esc_html_e('Intro padding', 'order-status-tracking-emails-for-woocommerce'); ?></th>
    <td><input type="text" name="oste_email_styling[order_intro_padding]" value="<?php echo esc_attr($opts['order_intro_padding']); ?>" style="width:100px;" /></td>
</tr>

<tr><th colspan="2"><b><?php esc_html_e('Order Summary - heading', 'order-status-tracking-emails-for-woocommerce'); ?></b></th></tr>
<tr>
    <th><?php esc_html_e('Heading color', 'order-status-tracking-emails-for-woocommerce'); ?></th>
    <td><input type="text" name="oste_email_styling[order_heading_color]" value="<?php echo esc_attr($opts['order_heading_color']); ?>" class="color-picker" /></td>
</tr>
<tr>
    <th><?php esc_html_e('Heading font', 'order-status-tracking-emails-for-woocommerce'); ?></th>
    <td>
        <select name="oste_email_styling[order_heading_font]">
            <?php foreach($fonts as $val => $label): ?>
                <option value="<?php echo esc_attr($val); ?>" <?php selected($opts['order_heading_font'],$val); ?>><?php echo esc_html($label); ?></option>
            <?php endforeach; ?>
        </select>
    </td>
</tr>
<tr>
    <th><?php esc_html_e('Heading size', 'order-status-tracking-emails-for-woocommerce'); ?></th>
    <td><input type="text" name="oste_email_styling[order_heading_size]" value="<?php echo esc_attr($opts['order_heading_size']); ?>" style="width:80px;" /></td>
</tr>
<tr>
    <th><?php esc_html_e('Heading weight', 'order-status-tracking-emails-for-woocommerce'); ?></th>
    <td>
        <select name="oste_email_styling[order_heading_weight]">
            <option value="normal" <?php selected($opts['order_heading_weight'],'normal'); ?>><?php esc_html_e('Normal', 'order-status-tracking-emails-for-woocommerce'); ?></option>
            <option value="bold" <?php selected($opts['order_heading_weight'],'bold'); ?>><?php esc_html_e('Bold', 'order-status-tracking-emails-for-woocommerce'); ?></option>
        </select>
    </td>
</tr>
<tr>
    <th><?php esc_html_e('Heading alignment', 'order-status-tracking-emails-for-woocommerce'); ?></th>
    <td>
        <select name="oste_email_styling[order_heading_align]">
            <option value="left" <?php selected($opts['order_heading_align'],'left'); ?>><?php esc_html_e('Left', 'order-status-tracking-emails-for-woocommerce'); ?></option>
            <option value="center" <?php selected($opts['order_heading_align'],'center'); ?>><?php esc_html_e('Center', 'order-status-tracking-emails-for-woocommerce'); ?></option>
            <option value="right" <?php selected($opts['order_heading_align'],'right'); ?>><?php esc_html_e('Right', 'order-status-tracking-emails-for-woocommerce'); ?></option>
        </select>
    </td>
</tr>
<tr>
    <th><?php esc_html_e('Heading margin', 'order-status-tracking-emails-for-woocommerce'); ?></th>
    <td><input type="text" name="oste_email_styling[order_heading_margin]" value="<?php echo esc_attr($opts['order_heading_margin']); ?>" style="width:90px;" /></td>
</tr>

<tr><th colspan="2"><b><?php esc_html_e('Order Table', 'order-status-tracking-emails-for-woocommerce'); ?></b></th></tr>
<tr>
    <th><?php esc_html_e('Table background', 'order-status-tracking-emails-for-woocommerce'); ?></th>
    <td><input type="text" name="oste_email_styling[order_table_bg]" value="<?php echo esc_attr($opts['order_table_bg']); ?>" class="color-picker" /></td>
</tr>
<tr>
    <th><?php esc_html_e('Border radius', 'order-status-tracking-emails-for-woocommerce'); ?></th>
    <td><input type="number" name="oste_email_styling[order_table_radius]" value="<?php echo esc_attr($opts['order_table_radius']); ?>" min="0" max="32" style="width:60px;" /> px</td>
</tr>
<tr>
    <th><?php esc_html_e('Table border', 'order-status-tracking-emails-for-woocommerce'); ?></th>
    <td>
        <input type="text" name="oste_email_styling[order_table_border_width]" value="<?php echo esc_attr($opts['order_table_border_width']); ?>" style="width:40px;" /> px
        <select name="oste_email_styling[order_table_border_style]">
            <option value="solid" <?php selected($opts['order_table_border_style'],'solid'); ?>><?php esc_html_e('Solid', 'order-status-tracking-emails-for-woocommerce'); ?></option>
            <option value="dashed" <?php selected($opts['order_table_border_style'],'dashed'); ?>><?php esc_html_e('Dashed', 'order-status-tracking-emails-for-woocommerce'); ?></option>
            <option value="dotted" <?php selected($opts['order_table_border_style'],'dotted'); ?>><?php esc_html_e('Dotted', 'order-status-tracking-emails-for-woocommerce'); ?></option>
        </select>
        <input type="text" name="oste_email_styling[order_table_border_color]" value="<?php echo esc_attr($opts['order_table_border_color']); ?>" class="color-picker" style="width:70px;" />
    </td>
</tr>
<tr>
    <th><?php esc_html_e('Box shadow', 'order-status-tracking-emails-for-woocommerce'); ?></th>
    <td><input type="text" name="oste_email_styling[order_table_box_shadow]" value="<?php echo esc_attr($opts['order_table_box_shadow']); ?>" style="width:180px;" /></td>
</tr>
<tr>
    <th><?php esc_html_e('Table padding', 'order-status-tracking-emails-for-woocommerce'); ?></th>
    <td><input type="text" name="oste_email_styling[order_table_padding]" value="<?php echo esc_attr($opts['order_table_padding']); ?>" style="width:100px;" /></td>
</tr>

<tr><th colspan="2"><b><?php esc_html_e('Table Cells', 'order-status-tracking-emails-for-woocommerce'); ?></b></th></tr>
<tr>
    <th><?php esc_html_e('Text color', 'order-status-tracking-emails-for-woocommerce'); ?></th>
    <td><input type="text" name="oste_email_styling[order_cell_color]" value="<?php echo esc_attr($opts['order_cell_color']); ?>" class="color-picker" /></td>
</tr>
<tr>
    <th><?php esc_html_e('Cell font', 'order-status-tracking-emails-for-woocommerce'); ?></th>
    <td>
        <select name="oste_email_styling[order_cell_font]">
            <?php foreach($fonts as $val => $label): ?>
                <option value="<?php echo esc_attr($val); ?>" <?php selected($opts['order_cell_font'],$val); ?>><?php echo esc_html($label); ?></option>
            <?php endforeach; ?>
        </select>
    </td>
</tr>
<tr>
    <th><?php esc_html_e('Font size', 'order-status-tracking-emails-for-woocommerce'); ?></th>
    <td><input type="text" name="oste_email_styling[order_cell_font_size]" value="<?php echo esc_attr($opts['order_cell_font_size']); ?>" style="width:80px;" /></td>
</tr>
<tr>
    <th><?php esc_html_e('Cell padding', 'order-status-tracking-emails-for-woocommerce'); ?></th>
    <td><input type="text" name="oste_email_styling[order_cell_padding]" value="<?php echo esc_attr($opts['order_cell_padding']); ?>" style="width:100px;" /></td>
</tr>
<tr>
    <th><?php esc_html_e('Cell background', 'order-status-tracking-emails-for-woocommerce'); ?></th>
    <td><input type="text" name="oste_email_styling[order_cell_bg]" value="<?php echo esc_attr($opts['order_cell_bg']); ?>" class="color-picker" /></td>
</tr>



<tr><th colspan="2"><b><?php esc_html_e('Price and Quantity', 'order-status-tracking-emails-for-woocommerce'); ?></b></th></tr>

	<tr>
	    <th><?php esc_html_e('Order totals color', 'order-status-tracking-emails-for-woocommerce'); ?></th>
	    <td><input type="text" name="oste_email_styling[order_totals_color]" value="<?php echo esc_attr($opts['order_totals_color'] ?? '#222222'); ?>" class="color-picker" /></td>
	</tr>
	<tr>
	    <th><?php esc_html_e('Order totals font', 'order-status-tracking-emails-for-woocommerce'); ?></th>
	    <td>
	        <select name="oste_email_styling[order_totals_font]">
	            <?php foreach ( $fonts as $val => $label ) : ?>
	                <option value="<?php echo esc_attr( $val ); ?>" <?php selected( $opts['order_totals_font'] ?? '', $val ); ?>><?php echo esc_html( $label ); ?></option>
	            <?php endforeach; ?>
	        </select>
	    </td>
	</tr>
	<tr>
	    <th><?php esc_html_e('Order totals size', 'order-status-tracking-emails-for-woocommerce'); ?></th>
	    <td><input type="text" name="oste_email_styling[order_totals_font_size]" value="<?php echo esc_attr($opts['order_totals_font_size'] ?? '16px'); ?>" style="width:80px;" /></td>
	</tr>
	<tr>
	    <th><?php esc_html_e('Order totals weight', 'order-status-tracking-emails-for-woocommerce'); ?></th>
	    <td>
	        <select name="oste_email_styling[order_totals_font_weight]">
	            <option value="normal" <?php selected( $opts['order_totals_font_weight'] ?? 'normal', 'normal' ); ?>><?php esc_html_e( 'Normal', 'order-status-tracking-emails-for-woocommerce' ); ?></option>
	            <option value="bold" <?php selected( $opts['order_totals_font_weight'] ?? 'normal', 'bold' ); ?>><?php esc_html_e( 'Bold', 'order-status-tracking-emails-for-woocommerce' ); ?></option>
	        </select>
	    </td>
	</tr>

<tr>
    <th><?php esc_html_e('Price color', 'order-status-tracking-emails-for-woocommerce'); ?></th>
    <td><input type="text" name="oste_email_styling[order_price_color]" value="<?php echo esc_attr($opts['order_price_color']); ?>" class="color-picker" /></td>
</tr>
<tr>
    <th><?php esc_html_e('Price weight', 'order-status-tracking-emails-for-woocommerce'); ?></th>
    <td>
        <select name="oste_email_styling[order_price_font_weight]">
            <option value="normal" <?php selected($opts['order_price_font_weight'],'normal'); ?>><?php esc_html_e('Normal', 'order-status-tracking-emails-for-woocommerce'); ?></option>
            <option value="bold" <?php selected($opts['order_price_font_weight'],'bold'); ?>><?php esc_html_e('Bold', 'order-status-tracking-emails-for-woocommerce'); ?></option>
        </select>
    </td>
</tr>
<tr>
    <th><?php esc_html_e('Price size', 'order-status-tracking-emails-for-woocommerce'); ?></th>
    <td><input type="text" name="oste_email_styling[order_price_font_size]" value="<?php echo esc_attr($opts['order_price_font_size']); ?>" style="width:80px;" /></td>
</tr>

<tr><th colspan="2"><b><?php esc_html_e('Product Meta', 'order-status-tracking-emails-for-woocommerce'); ?></b></th></tr>
<tr>
    <th><?php esc_html_e('Meta color', 'order-status-tracking-emails-for-woocommerce'); ?></th>
    <td><input type="text" name="oste_email_styling[order_item_meta_color]" value="<?php echo esc_attr($opts['order_item_meta_color']); ?>" class="color-picker" /></td>
</tr>
<tr>
    <th><?php esc_html_e('Meta font size', 'order-status-tracking-emails-for-woocommerce'); ?></th>
    <td><input type="text" name="oste_email_styling[order_item_meta_font_size]" value="<?php echo esc_attr($opts['order_item_meta_font_size']); ?>" style="width:80px;" /></td>
</tr>

<tr><th colspan="2"><b><?php esc_html_e('Addresses', 'order-status-tracking-emails-for-woocommerce'); ?></b></th></tr>
<tr>
    <th><?php esc_html_e('Address title color', 'order-status-tracking-emails-for-woocommerce'); ?></th>
    <td><input type="text" name="oste_email_styling[order_address_title_color]" value="<?php echo esc_attr($opts['order_address_title_color']); ?>" class="color-picker" /></td>
</tr>
<tr>
    <th><?php esc_html_e('Address font', 'order-status-tracking-emails-for-woocommerce'); ?></th>
    <td>
        <select name="oste_email_styling[order_address_font]">
            <?php foreach($fonts as $val => $label): ?>
                <option value="<?php echo esc_attr($val); ?>" <?php selected($opts['order_address_font'],$val); ?>><?php echo esc_html($label); ?></option>
            <?php endforeach; ?>
        </select>
    </td>
</tr>
<tr>
    <th><?php esc_html_e('Address text color', 'order-status-tracking-emails-for-woocommerce'); ?></th>
    <td><input type="text" name="oste_email_styling[order_address_color]" value="<?php echo esc_attr($opts['order_address_color']); ?>" class="color-picker" /></td>
</tr>
<tr>
    <th><?php esc_html_e('Address link color', 'order-status-tracking-emails-for-woocommerce'); ?></th>
    <td><input type="text" name="oste_email_styling[order_address_link_color]" value="<?php echo esc_attr($opts['order_address_link_color']); ?>" class="color-picker" /></td>
</tr>




<!-- ORDER DETAIL -->
<tr><th colspan="2"><h2><?php esc_html_e('Order Detail', 'order-status-tracking-emails-for-woocommerce'); ?></h2></th></tr>
<tr>
    <th><?php esc_html_e('Order detail text color', 'order-status-tracking-emails-for-woocommerce'); ?></th>
    <td><input type="text" name="oste_email_styling[detail_color]" value="<?php echo esc_attr($opts['detail_color']); ?>" class="color-picker" data-default-color="#333333" /></td>

</tr>
<tr>
    <th><?php esc_html_e('Order detail font', 'order-status-tracking-emails-for-woocommerce'); ?></th>
    <td>
        <select name="oste_email_styling[detail_font]">
            <?php foreach($fonts as $val => $label): ?>
                <option value="<?php echo esc_attr($val); ?>" <?php selected($opts['detail_font'],$val); ?>><?php echo esc_html($label); ?></option>
            <?php endforeach; ?>
        </select>
    </td>
</tr>
<tr>
    <th><?php esc_html_e('Order detail font size', 'order-status-tracking-emails-for-woocommerce'); ?></th>
    <td><input type="text" name="oste_email_styling[detail_size]" value="<?php echo esc_attr($opts['detail_size']); ?>" style="width:80px;" /></td>
</tr>
<tr>
    <th><?php esc_html_e('Order detail padding', 'order-status-tracking-emails-for-woocommerce'); ?></th>
    <td><input type="text" name="oste_email_styling[detail_padding]" value="<?php echo esc_attr($opts['detail_padding']); ?>" style="width:80px;" /></td>
</tr>

<!-- FOOTER -->
<tr><th colspan="2"><h2><?php esc_html_e('Footer', 'order-status-tracking-emails-for-woocommerce'); ?></h2></th></tr>
<tr>
    <th><?php esc_html_e('Footer background', 'order-status-tracking-emails-for-woocommerce'); ?></th>
    <td><input type="text" name="oste_email_styling[footer_bg]" value="<?php echo esc_attr($opts['footer_bg']); ?>" class="color-picker" /></td>
</tr>
<tr>
    <th><?php esc_html_e('Footer text color', 'order-status-tracking-emails-for-woocommerce'); ?></th>
    <td><input type="text" name="oste_email_styling[footer_color]" value="<?php echo esc_attr($opts['footer_color']); ?>" class="color-picker" /></td>
</tr>
<tr>
    <th><?php esc_html_e('Footer font size', 'order-status-tracking-emails-for-woocommerce'); ?></th>
    <td><input type="text" name="oste_email_styling[footer_size]" value="<?php echo esc_attr($opts['footer_size']); ?>" style="width:80px;" /></td>
</tr>
<tr>
    <th><?php esc_html_e('Footer font', 'order-status-tracking-emails-for-woocommerce'); ?></th>
    <td>
        <select name="oste_email_styling[footer_font]">
            <?php foreach($fonts as $val => $label): ?>
                <option value="<?php echo esc_attr($val); ?>" <?php selected($opts['footer_font'],$val); ?>><?php echo esc_html($label); ?></option>
            <?php endforeach; ?>
        </select>
    </td>
</tr>
<tr>
    <th><?php esc_html_e('Footer alignment', 'order-status-tracking-emails-for-woocommerce'); ?></th>
    <td>
        <select name="oste_email_styling[footer_align]">
            <option value="left" <?php selected($opts['footer_align'],'left'); ?>><?php esc_html_e('Left', 'order-status-tracking-emails-for-woocommerce'); ?></option>
            <option value="center" <?php selected($opts['footer_align'],'center'); ?>><?php esc_html_e('Center', 'order-status-tracking-emails-for-woocommerce'); ?></option>
            <option value="right" <?php selected($opts['footer_align'],'right'); ?>><?php esc_html_e('Right', 'order-status-tracking-emails-for-woocommerce'); ?></option>
        </select>
    </td>
</tr>

<!-- CUSTOM CSS -->
<tr><th colspan="2"><h2><?php esc_html_e('Custom CSS', 'order-status-tracking-emails-for-woocommerce'); ?></h2></th></tr>
<tr>
    <th><?php esc_html_e('Extra CSS', 'order-status-tracking-emails-for-woocommerce'); ?></th>
    <td>
        <textarea name="oste_email_styling[custom_css]" rows="4" style="width:100%;"><?php echo esc_textarea($opts['custom_css']); ?></textarea>
        <br><small><?php esc_html_e('It will be added directly in the head of the email. Only use email-safe CSS!', 'order-status-tracking-emails-for-woocommerce'); ?></small>
    </td>
</tr>

</table>
<?php submit_button(__('Save Settings', 'order-status-tracking-emails-for-woocommerce')); ?>
</form>
</div>

<?php
}

add_filter( 'pre_update_option_oste_email_styling', function ( $new, $old ) {
	$new = is_array( $new ) ? $new : array();

	// Normalizza checkbox root
	foreach ( array( 'enable_custom_styles', 'show_logo', 'show_disclaimer' ) as $cb ) {
		if ( ! isset( $new[ $cb ] ) ) {
			$new[ $cb ] = '0';
		}
	}


	// Aggiorna opzioni WooCommerce per il logo header
	if ( ! empty( $new['logo_url'] ) ) {
		update_option( 'woocommerce_email_header_image', esc_url_raw( $new['logo_url'], array( 'http', 'https' ) ) );
		$logo_width = isset( $new['logo_width'] ) ? absint( $new['logo_width'] ) : 200;
		update_option( 'woocommerce_email_header_image_width', $logo_width );
	} else {
		delete_option( 'woocommerce_email_header_image' );
		delete_option( 'woocommerce_email_header_image_width' );
	}

	if ( empty( $new['logo_width'] ) ) {
		$new['logo_width'] = 200;
	}
	if ( empty( $new['logo_align'] ) ) {
		$new['logo_align'] = 'center';
	}

return $new;
}, 10, 2 );


add_action( 'wp_ajax_oste_email_style_preview', 'oste_email_style_preview' );

function oste_email_style_preview() {
	if ( ! current_user_can( 'manage_options' ) ) {
		wp_send_json_error(
			array( 'message' => __( 'You do not have permission to preview emails.', 'order-status-tracking-emails-for-woocommerce' ) ),
			403
		);
	}

	if ( ! check_ajax_referer( 'oste_email_style_preview', 'nonce', false ) ) {
		wp_send_json_error(
			array( 'message' => __( 'Security check failed. Please reload the page and try again.', 'order-status-tracking-emails-for-woocommerce' ) ),
			403
		);
	}
    // Sanitize status from POST input.
    // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- input is sanitized immediately via sanitize_text_field + wp_unslash.
    $status = isset( $_POST['status'] ) ? sanitize_text_field( wp_unslash( $_POST['status'] ) ) : '';
    if ( '' === $status ) {
        // Default interno, non necessita sanitizzazione.
        $status = 'wc-processing';
    }

    // Leggi la form serializzata: è raw per natura, sarà sanificata per-campo dopo il parse.
    // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- 'form' is unslashed here, validated for UTF-8/length, then parsed and sanitized field-by-field.
    $form_serialized = isset( $_POST['form'] ) ? (string) wp_unslash( $_POST['form'] ) : '';
    $parsed_form     = array();
    if ( is_string( $form_serialized ) && '' !== $form_serialized ) {
        // Validazione basica e limite lunghezza difensivo per evitare abusi.
        $form_serialized = wp_check_invalid_utf8( $form_serialized, true );
        if ( strlen( $form_serialized ) > 10000 ) {
            $form_serialized = substr( $form_serialized, 0, 10000 );
        }
        parse_str( $form_serialized, $parsed_form );
    }
	// Sanificazione per-campo tramite funzione dedicata del plugin + fallback ai default.
	$raw_styles = array();
	if ( isset( $parsed_form['oste_email_styling'] ) && is_array( $parsed_form['oste_email_styling'] ) ) {
		$raw_styles = $parsed_form['oste_email_styling'];
	}

	$styles = oste_sanitize_email_style_settings( $raw_styles );
	$styles = wp_parse_args( $styles, oste_email_styling_default_options() );
	// Forza gli stili in anteprima anche se disattivati nelle opzioni.
	$styles['enable_custom_styles'] = '1';

	$email_opts      = get_option( 'oste_gestione_email_stati_ordine', array() );
	$show_site_name  = isset( $email_opts['show_site_name'] ) ? $email_opts['show_site_name'] : '1';
	$status_options  = oste_get_opzioni_stato( $status );
	$statuses_labels = oste_get_tutti_gli_stati();
	$status_label    = isset( $statuses_labels[ $status ] ) ? $statuses_labels[ $status ] : $status;

	$error_response = null;
	$result          = null;

	$forced_status_filter       = null;
	$preview_filter_added       = false;
	$preview_logo_transient_set = false;

	try {
		$orders_collection = array();

		if ( function_exists( 'wc_get_orders' ) ) {
			$orders_collection = wc_get_orders(
				array(
					'type'    => array( 'shop_order' ),
					'limit'   => 5,
					'orderby' => 'date',
					'order'   => 'DESC',
				)
			);
		} else {
			$status_slugs = function_exists( 'wc_get_order_statuses' )
				? array_keys( wc_get_order_statuses() )
				: array( 'wc-pending', 'wc-processing', 'wc-on-hold', 'wc-completed', 'wc-cancelled', 'wc-refunded', 'wc-failed' );

			$query = new WP_Query(
				array(
					'post_type'      => 'shop_order',
					'post_status'    => $status_slugs,
					'posts_per_page' => 5,
					'orderby'        => 'date',
					'order'          => 'DESC',
					'no_found_rows'  => true,
				)
			);

			if ( ! empty( $query->posts ) ) {
				foreach ( $query->posts as $order_post ) {
					$order_obj = function_exists( 'wc_get_order' ) ? wc_get_order( $order_post->ID ) : new WC_Order( $order_post->ID );
					if ( $order_obj instanceof WC_Order ) {
						$orders_collection[] = $order_obj;
					}
				}
			}
		}

		$base_order = null;
		foreach ( $orders_collection as $maybe_order ) {
			if ( $maybe_order instanceof WC_Order ) {
				$base_order = $maybe_order;
				break;
			}
		}

		if ( ! $base_order instanceof WC_Order ) {
			throw new Exception( __( 'Create at least one WooCommerce order to preview the email.', 'order-status-tracking-emails-for-woocommerce' ) );
		}

		$order = clone $base_order;

		$preview_status = sanitize_key( preg_replace( '/^wc-/', '', $status ) );
		if ( $preview_status ) {
			$forced_status_filter = function ( $current_status, $order_obj ) use ( $order, $preview_status ) {
				if ( $order_obj === $order ) {
					return $preview_status;
				}
				return $current_status;
			};
			add_filter( 'woocommerce_order_get_status', $forced_status_filter, 10, 2 );
		}

		$mailer          = WC()->mailer();
		$email_instance  = null;
		$emails          = array();
		if ( $mailer && method_exists( $mailer, 'get_emails' ) ) {
			$emails = (array) $mailer->get_emails();
		}

		$core_email_map = array(
			'pending'    => 'customer_pending_order',
			'on-hold'    => 'customer_on_hold_order',
			'processing' => 'customer_processing_order',
			'completed'  => 'customer_completed_order',
			'cancelled'  => 'customer_cancelled_order',
			'refunded'   => 'customer_refunded_order',
			'failed'     => 'customer_failed_order',
		);

		$core_email_id = $core_email_map[ $preview_status ] ?? '';
		$force_custom  = ! empty( $status_options['enabled'] ) && '1' === (string) $status_options['enabled'];

		if ( ! $force_custom && $core_email_id ) {
			if ( isset( $emails[ $core_email_id ] ) && $emails[ $core_email_id ] instanceof WC_Email ) {
				$email_instance = clone $emails[ $core_email_id ];
			} else {
				foreach ( $emails as $email_obj ) {
					if ( $email_obj instanceof WC_Email && isset( $email_obj->id ) && $email_obj->id === $core_email_id ) {
						$email_instance = clone $email_obj;
						break;
					}
				}
			}
		}

		if ( ! $email_instance instanceof WC_Email ) {
			if ( isset( $emails['oste_customer_custom_order'] ) && $emails['oste_customer_custom_order'] instanceof WC_Email ) {
				$email_instance = clone $emails['oste_customer_custom_order'];
			} elseif ( isset( $emails['WC_Email_Customer_Custom_Order'] ) && $emails['WC_Email_Customer_Custom_Order'] instanceof WC_Email ) {
				$email_instance = clone $emails['WC_Email_Customer_Custom_Order'];
			} elseif ( $emails ) {
				$email_instance = clone reset( $emails );
			} else {
				$email_instance = new OSTE_Email_Customer_Custom_Order();
			}
		}

		$preview_logo = '';
		if ( isset( $styles['logo_url'] ) && $styles['logo_url'] && ( ! isset( $styles['show_logo'] ) || '1' === (string) $styles['show_logo'] ) ) {
			$preview_logo = (string) $styles['logo_url'];
		}

		add_filter( 'woocommerce_is_email_preview', '__return_true' );
		$preview_filter_added = true;
		set_transient( 'woocommerce_email_header_image', $preview_logo, MINUTE_IN_SECONDS );
		$preview_logo_transient_set = true;

		$email_instance->object = $order;

		$recipient_email = '';
		if ( method_exists( $order, 'get_billing_email' ) ) {
			$recipient_email = (string) $order->get_billing_email();
		}
		if ( '' === $recipient_email ) {
			$recipient_email = get_option( 'admin_email' );
		}
		$email_instance->recipient = $recipient_email;

		if ( method_exists( $order, 'get_date_created' ) && $order->get_date_created() ) {
			$email_instance->placeholders['{order_date}'] = wc_format_datetime( $order->get_date_created() );
		}
		$email_instance->placeholders['{order_number}'] = method_exists( $order, 'get_order_number' ) ? $order->get_order_number() : (string) $order->get_id();

		if ( $email_instance instanceof WC_Email && ! empty( $email_instance->id ) ) {
			$oid = $order->get_id();
			if ( isset( $GLOBALS['oste_sent_runtime']['built'][ $email_instance->id ][ $oid ] ) ) {
				unset( $GLOBALS['oste_sent_runtime']['built'][ $email_instance->id ][ $oid ] );
			}
			if ( isset( $GLOBALS['oste_sent_runtime']['forced'][ $email_instance->id ][ $oid ] ) ) {
				unset( $GLOBALS['oste_sent_runtime']['forced'][ $email_instance->id ][ $oid ] );
			}
			if ( isset( $GLOBALS['oste_guard_state'][ $email_instance->id ][ $oid ] ) ) {
				unset( $GLOBALS['oste_guard_state'][ $email_instance->id ][ $oid ] );
			}
		}

		$GLOBALS['oste_email_preview_styles']         = $styles;
		$GLOBALS['oste_email_preview_show_site_name'] = $show_site_name;

		if ( method_exists( $email_instance, 'setup_locale' ) ) {
			$email_instance->setup_locale();
		}
		$raw_html = $email_instance->get_content_html();
		if ( method_exists( $email_instance, 'restore_locale' ) ) {
			$email_instance->restore_locale();
		}

		$captured = null;
		$mailer_callback = function( $callback ) use ( &$captured ) {
			return function ( $to, $subject, $message, $headers = '', $attachments = array(), $options = array() ) use ( &$captured ) {
				$captured = array(
					'to'        => $to,
					'subject'   => $subject,
					'message'   => $message,
					'headers'   => $headers,
					'attachments' => $attachments,
				);
				return true;
			};
		};

		add_filter( 'woocommerce_mail_callback', $mailer_callback, 10, 1 );

		if ( ! empty( $email_instance->id ) ) {
			add_filter( 'woocommerce_email_enabled_' . $email_instance->id, '__return_true', 10, 1 );
		}

		if ( method_exists( $email_instance, 'setup_locale' ) ) {
			$email_instance->setup_locale();
		}

		$email_instance->trigger( $order->get_id(), $order );

		if ( method_exists( $email_instance, 'restore_locale' ) ) {
			$email_instance->restore_locale();
		}

		if ( ! empty( $email_instance->id ) ) {
			remove_filter( 'woocommerce_email_enabled_' . $email_instance->id, '__return_true', 10 );
		}

		remove_filter( 'woocommerce_mail_callback', $mailer_callback, 10 );

		if ( empty( $captured ) || empty( $captured['message'] ) ) {
			throw new Exception( __( 'Unable to generate the email preview content.', 'order-status-tracking-emails-for-woocommerce' ) );
		}

		$result = array(
			'html'          => (string) $captured['message'],
			'subject'       => (string) ( $captured['subject'] ?? $email_instance->get_subject() ),
			'status_label'  => $status_label,
			'preview_order' => method_exists( $order, 'get_order_number' ) ? $order->get_order_number() : $order->get_id(),
		);
	} catch ( Throwable $e ) {
		$error_message = wp_strip_all_tags( $e->getMessage() );
		if ( '' === $error_message ) {
			$error_message = __( 'Unable to load the email preview. Please try again.', 'order-status-tracking-emails-for-woocommerce' );
		}
		$error_response = array( 'message' => $error_message );
	}

	if ( $forced_status_filter ) {
		remove_filter( 'woocommerce_order_get_status', $forced_status_filter, 10 );
	}

	if ( $preview_filter_added ) {
		remove_filter( 'woocommerce_is_email_preview', '__return_true' );
	}

	if ( $preview_logo_transient_set ) {
		delete_transient( 'woocommerce_email_header_image' );
	}

	unset( $GLOBALS['oste_email_preview_styles'], $GLOBALS['oste_email_preview_show_site_name'] );

	if ( $error_response ) {
		wp_send_json_error( $error_response, 500 );
	}

	wp_send_json_success( $result );
}
