<?php
if ( ! defined( 'ABSPATH' ) ) { exit; }

/**
 * Menu OSTE + alias con stesso callback
 */
add_action( 'admin_menu', function () {
	add_menu_page(
		__( 'OSTE Manager', 'order-status-tracking-emails-for-woocommerce' ),
		__( 'OSTE Manager', 'order-status-tracking-emails-for-woocommerce' ),
		'manage_woocommerce',
		'oste-order-manager',
		'oste_status_manager_page',
		'dashicons-clipboard',
		56
	);

	add_submenu_page(
		'oste-order-manager',
		__( 'States Manager', 'order-status-tracking-emails-for-woocommerce' ),
		__( 'States Manager', 'order-status-tracking-emails-for-woocommerce' ),
		'manage_woocommerce',
		'oste-order-manager',
		'oste_status_manager_page'
	);
}, 10 );

/**
 * Enqueue solo nella pagina OSTE Manager
 */
add_action( 'admin_enqueue_scripts', function ( $hook_suffix ) {
	// La pagina può essere toplevel o sottomenu: verifica entrambi i possibili ID.
	$targets = array(
		'toplevel_page_oste-order-manager',
		'woocommerce_page_oste-order-manager',
	);

	if ( ! in_array( $hook_suffix, $targets, true ) ) {
		return;
	}

	// registrati nel main; qui solo enqueue
	wp_enqueue_style( 'oste-admin' );
	wp_enqueue_style( 'oste-admin-ui' );
	wp_enqueue_script( 'oste-admin' );

	// JS: aggiungi/rimuovi righe + rename live checkbox "Actions?" -> slug
	wp_add_inline_script( 'oste-admin', "
(function(){
  function sanitizeSlug(v){
    return String(v||'').toLowerCase().replace(/[^a-z0-9\\-]/g, '');
  }

  function wireRow(tr){
    if(!tr) return;
    var slugInput = tr.querySelector('input[name^=\"oste_statuses\"][name$=\"[slug]\"]');
    var chk       = tr.querySelector('input[type=\"checkbox\"][name^=\"oste_show_action[\"]');
    if(!slugInput || !chk) return;

    function updateName(){
      var s = sanitizeSlug(slugInput.value);
      slugInput.value = s;
      if(s){
        chk.name = 'oste_show_action['+ s +']';
      }
    }

    // prima impostazione
    updateName();

    // rename live ad ogni input
    slugInput.addEventListener('input', updateName, { passive:true });
    slugInput.addEventListener('change', updateName);
  }

  // handler 'Aggiungi riga'
  window.oste_AddRow = function(){
    var tbody = document.querySelector('#oste-status-table tbody');
    if(!tbody) return;
    var idx = Date.now();
    var html = `
      <tr>
        <td><input name=\"oste_statuses[new\${idx}][slug]\" required pattern=\"[a-z0-9\\-]+\" oninput=\"this.value=this.value.toLowerCase().replace(/[^a-z0-9\\-]/g,'');\"></td>
        <td><input name=\"oste_statuses[new\${idx}][label]\" required></td>
        <td><input type=\"color\" name=\"oste_statuses[new\${idx}][color]\" value=\"#cccccc\"></td>
        <td style=\"text-align:center;\"><label class=\"oste-toggle\"><input type=\"checkbox\" name=\"oste_show_action[new\${idx}]\" value=\"1\" checked><span class=\"oste-toggle-slider\" aria-hidden=\"true\"></span></label></td>
        <td><button type=\"button\" class=\"button oste-remove-status\">". esc_js( __( 'Remove', 'order-status-tracking-emails-for-woocommerce' ) ) ."</button></td>
      </tr>`;
    tbody.insertAdjacentHTML('beforeend', html);
    wireRow(tbody.lastElementChild);
  };

  // remove row (delegato)
  document.addEventListener('click', function(e){
    if(e.target && e.target.classList && e.target.classList.contains('oste-remove-status')){
      var tr = e.target.closest('tr'); if(tr) tr.remove();
    }
  });

  // al load: aggancia a tutte le righe esistenti
  document.addEventListener('DOMContentLoaded', function(){
    document.querySelectorAll('#oste-status-table tbody tr').forEach(wireRow);
  });
})();
" );
} );

/**
 * Pagina gestione stati
 */
function oste_status_manager_page() {

	// Permessi
	if ( ! current_user_can( 'manage_woocommerce' ) ) {
		wp_die( esc_html__( 'You do not have permission to access this page.', 'order-status-tracking-emails-for-woocommerce' ) );
	}

	/* ---------- SALVATAGGIO ---------- */
	if (
		isset( $_SERVER['REQUEST_METHOD'] ) && 'POST' === $_SERVER['REQUEST_METHOD']
		&& isset( $_POST['oste_statuses_nonce'] )
		&& wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['oste_statuses_nonce'] ) ), 'oste_save_statuses' )
	) {
		$raw_post = wp_unslash( $_POST );

		// Base statuses (label/color)
		$base_labels_raw = [];
		$raw_base = ( isset( $raw_post['oste_base_status_labels'] ) && is_array( $raw_post['oste_base_status_labels'] ) )
			? $raw_post['oste_base_status_labels']
			: [];

		foreach ( $raw_base as $slug => $values ) {
			$slug  = sanitize_key( $slug );
			$label = isset( $values['label'] ) ? sanitize_text_field( $values['label'] ) : '';

			$c_raw = isset( $values['color'] ) ? $values['color'] : '';
			$color = sanitize_hex_color( $c_raw );
			if ( ! $color ) { $color = '#cccccc'; }

			$base_labels_raw[ $slug ] = [
				'label' => $label,
				'color' => $color,
			];
		}
		update_option( 'oste_base_status_labels', $base_labels_raw );

		// Custom statuses (slug/label/color)
		$statuses_raw = [];
		if ( isset( $raw_post['oste_statuses'] ) && is_array( $raw_post['oste_statuses'] ) ) {
			foreach ( $raw_post['oste_statuses'] as $key => $row ) {
				$c_raw = isset( $row['color'] ) ? $row['color'] : '';
				$color = sanitize_hex_color( $c_raw );
				if ( ! $color ) { $color = '#cccccc'; }

				$statuses_raw[ $key ] = [
					'slug'  => isset( $row['slug'] )  ? sanitize_title( $row['slug'] )      : '',
					'label' => isset( $row['label'] ) ? sanitize_text_field( $row['label'] ) : '',
					'color' => $color,
				];
			}
		}

		$custom_ok = [];
		foreach ( $statuses_raw as $row ) {
			if ( empty( $row['slug'] ) || empty( $row['label'] ) ) { continue; }
			$slug   = sanitize_title( $row['slug'] );
			$label  = sanitize_text_field( $row['label'] );
			$c_raw  = isset( $row['color'] ) ? $row['color'] : '';
			$color  = sanitize_hex_color( $c_raw );
			if ( ! $color ) { $color = '#cccccc'; }

			$custom_ok[] = compact( 'slug', 'label', 'color' );
		}
		update_option( 'oste_custom_statuses', $custom_ok );

		// Azioni da mostrare (chiave = slug). Grazie al rename live, anche le righe nuove
		// arrivano già come oste_show_action[slug]
		$show_action_raw = ( isset( $raw_post['oste_show_action'] ) && is_array( $raw_post['oste_show_action'] ) )
			? array_map( 'sanitize_text_field', $raw_post['oste_show_action'] )
			: [];

		$all_slugs = array_merge(
			array_keys( oste_get_base_statuses() ),
			array_column( $custom_ok, 'slug' ),
			[ 'complete', 'processing', 'cancel', 'refund' ]
		);
		$all_slugs = array_unique( array_map( 'sanitize_key', $all_slugs ) );

		$show = [];
		foreach ( $all_slugs as $s ) {
			$show[ $s ] = ( ! empty( $show_action_raw[ $s ] ) && '1' === $show_action_raw[ $s ] ) ? 1 : 0;
		}
		update_option( 'oste_order_actions_settings', $show );

		// Nascondi/mostra bottone azioni core
		$hide_core_raw = isset( $raw_post['oste_hide_core_action'] ) ? sanitize_text_field( $raw_post['oste_hide_core_action'] ) : '';
		update_option( 'oste_hide_core_action', ( '1' === $hide_core_raw ) ? 1 : 0 );

		echo '<div class=\"updated\"><p>' . esc_html__( 'Settings saved.', 'order-status-tracking-emails-for-woocommerce' ) . '</p></div>';
	}

	// Dati per il form
	$base_statuses   = oste_get_base_statuses();
	$base_labels     = oste_get_base_labels();
	$custom_statuses = oste_get_custom_statuses();
	$show_actions    = oste_get_actions_settings();

	?>
	<div class="wrap oste-admin-wrap">
		<h1><?php esc_html_e( 'WooCommerce Order Status Management', 'order-status-tracking-emails-for-woocommerce' ); ?></h1>

		<form method="post">
			<?php wp_nonce_field( 'oste_save_statuses', 'oste_statuses_nonce' ); ?>

			<h2><?php esc_html_e( 'Other', 'order-status-tracking-emails-for-woocommerce' ); ?></h2>
			<table class="form-table">
				<tr>
					<th><?php esc_html_e( 'Hide the "Actions" menu button (three dots in the order table)', 'order-status-tracking-emails-for-woocommerce' ); ?></th>
					<td>
						<label class="oste-toggle-label">
							<span class="oste-toggle">
								<input type="checkbox" name="oste_hide_core_action" value="1" <?php checked( get_option( 'oste_hide_core_action', 0 ), 1 ); ?>>
								<span class="oste-toggle-slider" aria-hidden="true"></span>
							</span>
							<span><?php esc_html_e( 'If selected, the button will be shown.', 'order-status-tracking-emails-for-woocommerce' ); ?></span>
						</label>
					</td>
				</tr>
			</table>

			<h2><?php esc_html_e( 'Default WooCommerce Statuses', 'order-status-tracking-emails-for-woocommerce' ); ?></h2>
			<table class="widefat striped">
				<thead>
					<tr>
						<th><?php esc_html_e( 'Status', 'order-status-tracking-emails-for-woocommerce' ); ?></th>
						<th><?php esc_html_e( 'Label', 'order-status-tracking-emails-for-woocommerce' ); ?></th>
						<th><?php esc_html_e( 'Color', 'order-status-tracking-emails-for-woocommerce' ); ?></th>
						<th><?php esc_html_e( 'Actions?', 'order-status-tracking-emails-for-woocommerce' ); ?></th>
					</tr>
				</thead>
				<tbody>
				<?php foreach ( $base_statuses as $slug => $def_label ) :
					$row     = $base_labels[ $slug ] ?? [];
					$label   = isset( $row['label'] ) ? $row['label'] : $def_label;
					$color   = isset( $row['color'] ) ? $row['color'] : '#cccccc';
					$checked = ! empty( $show_actions[ $slug ] ); ?>
					<tr>
						<td><?php echo esc_html( $def_label ); ?></td>
						<td><input type="text"  name="oste_base_status_labels[<?php echo esc_attr( $slug ); ?>][label]" value="<?php echo esc_attr( $label ); ?>"></td>
						<td><input type="color" name="oste_base_status_labels[<?php echo esc_attr( $slug ); ?>][color]" value="<?php echo esc_attr( $color ); ?>"></td>
						<td style="text-align:center;">
							<label class="oste-toggle">
								<input type="checkbox" name="oste_show_action[<?php echo esc_attr( $slug ); ?>]" value="1" <?php checked( $checked ); ?>>
								<span class="oste-toggle-slider" aria-hidden="true"></span>
							</label>
						</td>
					</tr>
				<?php endforeach; ?>
				</tbody>
			</table>

			<h2 style="margin-top:40px;"><?php esc_html_e( 'Custom Statuses', 'order-status-tracking-emails-for-woocommerce' ); ?></h2>
			<table class="widefat striped" id="oste-status-table">
				<thead>
					<tr>
						<th><?php esc_html_e( 'Slug', 'order-status-tracking-emails-for-woocommerce' ); ?></th>
						<th><?php esc_html_e( 'Label', 'order-status-tracking-emails-for-woocommerce' ); ?></th>
						<th><?php esc_html_e( 'Color', 'order-status-tracking-emails-for-woocommerce' ); ?></th>
						<th><?php esc_html_e( 'Actions?', 'order-status-tracking-emails-for-woocommerce' ); ?></th>
						<th></th>
					</tr>
				</thead>
				<tbody>
				<?php foreach ( (array) $custom_statuses as $i => $st ) :
					$slug    = isset( $st['slug'] )  ? $st['slug']  : '';
					$label   = isset( $st['label'] ) ? $st['label'] : '';
					$color   = isset( $st['color'] ) ? $st['color'] : '#cccccc';
					$checked = ! empty( $show_actions[ $slug ] ); ?>
					<tr>
						<td>
							<input type="text"
								   name="oste_statuses[<?php echo esc_attr( $i ); ?>][slug]"
								   value="<?php echo esc_attr( $slug ); ?>"
								   required
								   pattern="[a-z0-9\-]+"
								   oninput="this.value=this.value.toLowerCase().replace(/[^a-z0-9\-]/g,'');">
						</td>
						<td>
							<input type="text"
								   name="oste_statuses[<?php echo esc_attr( $i ); ?>][label]"
								   value="<?php echo esc_attr( $label ); ?>"
								   required>
						</td>
						<td>
							<input type="color"
								   name="oste_statuses[<?php echo esc_attr( $i ); ?>][color]"
								   value="<?php echo esc_attr( $color ); ?>">
						</td>
						<td style="text-align:center;">
							<label class="oste-toggle">
								<input type="checkbox"
									   name="oste_show_action[<?php echo esc_attr( $slug ); ?>]"
									   value="1"
									   <?php checked( $checked ); ?>>
								<span class="oste-toggle-slider" aria-hidden="true"></span>
							</label>
						</td>
						<td>
							<button type="button" class="button oste-remove-status">
								<?php esc_html_e( 'Remove', 'order-status-tracking-emails-for-woocommerce' ); ?>
							</button>
						</td>
					</tr>
				<?php endforeach; ?>
				</tbody>
			</table>

			<p><button type="button" class="button" onclick="oste_AddRow()"><?php esc_html_e( 'Add Status', 'order-status-tracking-emails-for-woocommerce' ); ?></button></p>
			<p><input type="submit" class="button-primary" value="<?php esc_attr_e( 'Save All Settings', 'order-status-tracking-emails-for-woocommerce' ); ?>"></p>
		</form>
	</div>
	<?php
}
