<?php
class Waba_Chat_Woocommerce {
	protected $loader;
	protected $plugin_name;
	protected $version;
	protected $api;

	public function __construct( $plugin_name, $version ) {
		$this->plugin_name = $plugin_name;
		$this->version = $version;
		$this->api = new Waba_Chat_Api();
	}

	public function handle_order_status_change( $order_id, $old_status, $new_status ) {
		$order = wc_get_order( $order_id );
		if ( ! $order ) return;

		$phone = $order->get_billing_phone();
		if ( empty( $phone ) ) return;

		// Here you would fetch the template ID mapped to this status from settings
		// For MVP, we will assume a generic notification or look up setting
		$options = get_option( 'waba_chat_settings' );
		// Example: 'template_processing'
		$template_key = 'template_' . $new_status; 
		$template_id = isset( $options[$template_key] ) ? $options[$template_key] : '';

		if ( ! empty( $template_id ) ) {
			// Variables: {name}, {order_id}, {total}
			$variables = array(
				$order->get_formatted_billing_full_name(),
				$order_id,
				$order->get_formatted_order_total()
			);
			
			// Get language for this template
			$language_key = $template_key . '_language';
			$language = isset( $options[$language_key] ) ? $options[$language_key] : 'en_US';

			$this->api->send_message( $phone, $template_id, $language, $variables );
		}
	}

	public function capture_cart_update() {
		if ( is_user_logged_in() || ! empty( WC()->session->get( 'billing_phone' ) ) ) {
			$cart = WC()->cart;
			if ( $cart->is_empty() ) return;

			$user_id = get_current_user_id();
			$billing_phone = WC()->session->get( 'billing_phone' );
			$billing_email = WC()->session->get( 'billing_email' );
			
			if ( ! $user_id && ! empty( $billing_phone ) ) {
				// Guest with phone captured
				// Store in a custom table or transient identified by session ID
				$this->save_abandoned_cart( $cart, $billing_phone, $billing_email );
			} elseif ( $user_id ) {
				// Logged in user
				$this->save_abandoned_cart( $cart, get_user_meta( $user_id, 'billing_phone', true ), get_user_meta( $user_id, 'billing_email', true ), $user_id );
			}
		}
	}

	private function save_abandoned_cart( $cart, $phone, $email, $user_id = 0 ) {
		if ( empty( $phone ) ) return;

		// Use transient for simplicity in MVP, or custom table in full version
		// Key: waba_cart_{hash}
		$cart_hash = md5( json_encode( $cart->get_cart() ) . $phone );
		$data = array(
			'user_id' => $user_id,
			'phone' => $phone,
			'email' => $email,
			'cart_contents' => $cart->get_cart(),
			'total' => $cart->total,
			'timestamp' => time(),
			'recovery_sent' => false
		);

		// Store for 2 days
		set_transient( 'waba_cart_' . $cart_hash, $data, 2 * DAY_IN_SECONDS );
		
		// Indexing: we need a way to find these transients. 
		// For MVP/Demo, we might use a dedicated Option array to store keys of active carts
		$active_carts = get_option( 'waba_active_carts', array() );
		$active_carts[$cart_hash] = time();
		update_option( 'waba_active_carts', $active_carts );
	}

	// Called by Cron
	public function process_abandoned_carts() {
		$active_carts = get_option( 'waba_active_carts', array() );
		$options = get_option( 'waba_chat_settings' );
		$template_id = isset( $options['template_abandoned_cart'] ) ? $options['template_abandoned_cart'] : '';
		
		if ( empty( $template_id ) ) return;

		$cutoff_time = time() - ( 15 * 60 ); // 15 minutes ago
		$updated_carts = $active_carts;

		foreach ( $active_carts as $cart_hash => $timestamp ) {
			if ( $timestamp < $cutoff_time ) {
				$cart_data = get_transient( 'waba_cart_' . $cart_hash );
				
				if ( $cart_data && ! $cart_data['recovery_sent'] ) {
					// Variables: {name}, {checkout_url}
					$variables = array(
						'Customer', // Placeholder if name not captured
						wc_get_checkout_url()
					);
					
					// Get language for abandoned cart template
					$language = isset( $options['template_abandoned_cart_language'] ) ? $options['template_abandoned_cart_language'] : 'en_US';

					$result = $this->api->send_message( $cart_data['phone'], $template_id, $language, $variables );
					
					if ( ! is_wp_error( $result ) ) {
						// Mark sent
						$cart_data['recovery_sent'] = true;
						set_transient( 'waba_cart_' . $cart_hash, $cart_data, 2 * DAY_IN_SECONDS );
						// Remove from active list to avoid re-checking constantly? 
						// Or actally we can remove it because we sent the recovery.
						unset( $updated_carts[$cart_hash] );
					}
				} elseif ( ! $cart_data ) {
					// Expired
					unset( $updated_carts[$cart_hash] );
				}
			}
		}
		
		if ( $updated_carts !== $active_carts ) {
			update_option( 'waba_active_carts', $updated_carts );
		}
	}
}
