<?php
if ( ! defined( 'ABSPATH' ) ) exit;

class Waba_Chat_Admin {
	private $plugin_name;
	private $version;
	private $option_name = 'waba_chat_settings';

	public function __construct( $plugin_name, $version ) {
		$this->plugin_name = $plugin_name;
		$this->version = $version;
		add_action( 'wp_ajax_waba_fetch_templates', array( $this, 'ajax_fetch_templates' ) );
		add_action( 'wp_ajax_waba_send_test_message', array( $this, 'ajax_send_test_message' ) );
	}

	public function enqueue_styles() {
		wp_enqueue_style( $this->plugin_name, plugin_dir_url( __FILE__ ) . 'css/waba-admin.css', array(), $this->version );
	}
	
	public function enqueue_scripts() {
		wp_enqueue_script( $this->plugin_name, plugin_dir_url( __FILE__ ) . 'js/waba-admin.js', array( 'jquery' ), $this->version, false );
		wp_localize_script( $this->plugin_name, 'wabaAdmin', array(
			'ajax_url' => admin_url( 'admin-ajax.php' ),
			'nonce' => wp_create_nonce( 'waba_fetch_templates' )
		) );
	}

	public function register_settings() {
		register_setting( $this->option_name, $this->option_name, array( $this, 'sanitize_settings' ) );
		
		add_settings_section(
			'waba_general_section',
			__( 'General Settings', 'order-notification-waba-nxc' ),
			null,
			'waba-chat-admin'
		);

		add_settings_field(
			'api_url',
			__( 'Waba Panel URL', 'order-notification-waba-nxc' ),
			array( $this, 'render_api_url_field' ),
			'waba-chat-admin',
			'waba_general_section'
		);

		add_settings_field(
			'app_key',
			__( 'App Key', 'order-notification-waba-nxc' ),
			array( $this, 'render_app_key_field' ),
			'waba-chat-admin',
			'waba_general_section'
		);

		add_settings_field(
			'auth_key',
			__( 'Auth Key', 'order-notification-waba-nxc' ),
			array( $this, 'render_auth_key_field' ),
			'waba-chat-admin',
			'waba_general_section'
		);

		add_settings_field(
			'whatsapp_number',
			__( 'WhatsApp Number (for Chat Widget)', 'order-notification-waba-nxc' ),
			array( $this, 'render_whatsapp_number_field' ),
			'waba-chat-admin',
			'waba_general_section'
		);

		add_settings_field(
			'test_phone_number',
			__( 'Test Phone Number', 'order-notification-waba-nxc' ),
			array( $this, 'render_test_phone_field' ),
			'waba-chat-admin',
			'waba_general_section'
		);

		// WooCommerce Template Mapping Section
		if ( class_exists( 'WooCommerce' ) ) {
			add_settings_section(
				'waba_template_section',
				__( 'WooCommerce Template Mapping', 'order-notification-waba-nxc' ),
				array( $this, 'render_template_section_description' ),
				'waba-chat-admin'
			);

			add_settings_field(
				'template_abandoned_cart',
				__( 'Abandoned Cart Template ID', 'order-notification-waba-nxc' ),
				array( $this, 'render_template_abandoned_cart_field' ),
				'waba-chat-admin',
				'waba_template_section'
			);

			add_settings_field(
				'template_processing',
				__( 'Order Processing Template ID', 'order-notification-waba-nxc' ),
				array( $this, 'render_template_processing_field' ),
				'waba-chat-admin',
				'waba_template_section'
			);

			add_settings_field(
				'template_completed',
				__( 'Order Completed Template ID', 'order-notification-waba-nxc' ),
				array( $this, 'render_template_completed_field' ),
				'waba-chat-admin',
				'waba_template_section'
			);

			add_settings_field(
				'template_on-hold',
				__( 'Order On-Hold Template ID', 'order-notification-waba-nxc' ),
				array( $this, 'render_template_onhold_field' ),
				'waba-chat-admin',
				'waba_template_section'
			);
		}
	}

	public function add_admin_menu() {
		add_menu_page(
			__( 'WABA Notifications', 'order-notification-waba-nxc' ),
			__( 'WABA Notifications', 'order-notification-waba-nxc' ),
			'manage_options',
			'waba-chat-admin',
			array( $this, 'render_admin_page' ),
			'dashicons-whatsapp',
			56
		);
	}

	public function render_admin_page() {
		?>
		<div class="wrap">
			<h1><?php echo esc_html( get_admin_page_title() ); ?></h1>
			<form action="options.php" method="post">
				<?php
				settings_fields( $this->option_name );
				do_settings_sections( 'waba-chat-admin' );
				submit_button();
				?>
			</form>
		</div>
		<?php
	}

	public function render_api_url_field() {
		$options = get_option( $this->option_name );
		$value = isset( $options['api_url'] ) ? $options['api_url'] : '';
		if ( empty( $value ) ) $value = 'https://waba.nxccontrols.in'; // Example default
		echo '<input type="url" name="' . esc_attr( $this->option_name ) . '[api_url]" value="' . esc_attr( $value ) . '" class="regular-text" placeholder="https://your-panel.com">';
		echo '<p class="description">' . esc_html__( 'The base URL of your Waba Panel.', 'order-notification-waba-nxc' ) . '</p>';
	}

	public function render_app_key_field() {
		$options = get_option( $this->option_name );
		$value = isset( $options['app_key'] ) ? $options['app_key'] : '';
		echo '<input type="text" name="' . esc_attr( $this->option_name ) . '[app_key]" value="' . esc_attr( $value ) . '" class="regular-text">';
	}

	public function render_auth_key_field() {
		$options = get_option( $this->option_name );
		$value = isset( $options['auth_key'] ) ? $options['auth_key'] : '';
		echo '<input type="password" name="' . esc_attr( $this->option_name ) . '[auth_key]" value="' . esc_attr( $value ) . '" class="regular-text">';
	}

	public function render_whatsapp_number_field() {
		$options = get_option( $this->option_name );
		$value = isset( $options['whatsapp_number'] ) ? $options['whatsapp_number'] : '';
		echo '<input type="text" name="' . esc_attr( $this->option_name ) . '[whatsapp_number]" value="' . esc_attr( $value ) . '" class="regular-text" placeholder="1234567890">';
	}

	public function render_test_phone_field() {
		$options = get_option( $this->option_name );
		$value = isset( $options['test_phone_number'] ) ? $options['test_phone_number'] : '';
		echo '<input type="text" name="' . esc_attr( $this->option_name ) . '[test_phone_number]" value="' . esc_attr( $value ) . '" class="regular-text" placeholder="919876543210">';
		echo '<p class="description">' . esc_html__( 'Phone number for testing templates (with country code, no + or spaces). Example: 919876543210', 'order-notification-waba-nxc' ) . '</p>';
	}

	public function render_template_section_description() {
		echo '<p>' . esc_html__( 'Configure template IDs from your Waba Panel for WooCommerce notifications. Leave empty to disable specific notifications.', 'order-notification-waba-nxc' ) . '</p>';
	}

	public function render_template_abandoned_cart_field() {
		$this->render_template_dropdown_field( 'template_abandoned_cart', __( 'Template for cart abandonment recovery. Supported variables: {name}, {checkout_url}', 'order-notification-waba-nxc' ) );
	}

	public function render_template_processing_field() {
		$this->render_template_dropdown_field( 'template_processing', __( 'Template when order status changes to Processing. Supported variables: {name}, {order_id}, {total}', 'order-notification-waba-nxc' ) );
	}

	public function render_template_completed_field() {
		$this->render_template_dropdown_field( 'template_completed', __( 'Template when order status changes to Completed. Supported variables: {name}, {order_id}, {total}', 'order-notification-waba-nxc' ) );
	}

	public function render_template_onhold_field() {
		$this->render_template_dropdown_field( 'template_on-hold', __( 'Template when order status changes to On-Hold. Supported variables: {name}, {order_id}, {total}', 'order-notification-waba-nxc' ) );
	}

	private function render_template_dropdown_field( $field_name, $description ) {
		$options = get_option( $this->option_name );
		$value = isset( $options[$field_name] ) ? $options[$field_name] : '';
		
		echo '<div class="waba-template-field-wrapper">';
		echo '<select name="' . esc_attr( $this->option_name ) . '[' . esc_attr( $field_name ) . ']" class="waba-template-select" data-field="' . esc_attr( $field_name ) . '">';
		echo '<option value="">' . esc_html__( '-- Select Template --', 'order-notification-waba-nxc' ) . '</option>';
		if ( ! empty( $value ) ) {
			echo '<option value="' . esc_attr( $value ) . '" selected>' . esc_html( $value ) . '</option>';
		}
		echo '</select>';
		// Hidden field to store template language
		$language_value = isset( $options[$field_name . '_language'] ) ? $options[$field_name . '_language'] : 'en_US';
		echo '<input type="hidden" name="' . esc_attr( $this->option_name ) . '[' . esc_attr( $field_name ) . '_language]" value="' . esc_attr( $language_value ) . '" class="waba-template-language">';
		echo ' <button type="button" class="button waba-refresh-templates">' . esc_html__( 'Refresh Templates', 'order-notification-waba-nxc' ) . '</button>';
		echo ' <button type="button" class="button button-secondary waba-send-test" data-field="' . esc_attr( $field_name ) . '">' . esc_html__( 'Send Test', 'order-notification-waba-nxc' ) . '</button>';
		echo '<span class="waba-template-loading" style="display:none;"><span class="spinner is-active" style="float:none;margin:0 10px;"></span></span>';
		echo '<p class="description">' . esc_html( $description ) . '</p>';
		echo '<p class="waba-template-error" style="color:#dc3232;display:none;"></p>';
		echo '</div>';
	}

	public function ajax_fetch_templates() {
		check_ajax_referer( 'waba_fetch_templates', 'nonce' );
		
		$options = get_option( $this->option_name );
		$api_url = isset( $options['api_url'] ) ? $options['api_url'] : '';
		$app_key = isset( $options['app_key'] ) ? $options['app_key'] : '';
		$auth_key = isset( $options['auth_key'] ) ? $options['auth_key'] : '';
		
		if ( empty( $api_url ) || empty( $app_key ) || empty( $auth_key ) ) {
			wp_send_json_error( array( 'message' => __( 'Please configure API credentials first.', 'order-notification-waba-nxc' ) ) );
		}
		
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-waba-api.php';
		$api = new Waba_Chat_Api( $api_url, $app_key, $auth_key );
		$templates = $api->get_templates();
		
		if ( is_wp_error( $templates ) ) {
			wp_send_json_error( array( 'message' => $templates->get_error_message() ) );
		}
		
		wp_send_json_success( array( 'templates' => $templates ) );
	}

	public function ajax_send_test_message() {
		check_ajax_referer( 'waba_fetch_templates', 'nonce' );
		
		$template_id = isset( $_POST['template_id'] ) ? sanitize_text_field( wp_unslash( $_POST['template_id'] ) ) : '';
		$phone_number = isset( $_POST['phone_number'] ) ? sanitize_text_field( wp_unslash( $_POST['phone_number'] ) ) : '';
		$field_name = isset( $_POST['field_name'] ) ? sanitize_text_field( wp_unslash( $_POST['field_name'] ) ) : '';
		$language = isset( $_POST['language'] ) ? sanitize_text_field( wp_unslash( $_POST['language'] ) ) : 'en_US';
		
		if ( empty( $template_id ) ) {
			wp_send_json_error( array( 'message' => __( 'Please select a template first.', 'order-notification-waba-nxc' ) ) );
		}
		
		if ( empty( $phone_number ) ) {
			wp_send_json_error( array( 'message' => __( 'Please enter a test phone number in settings.', 'order-notification-waba-nxc' ) ) );
		}
		
		$options = get_option( $this->option_name );
		$api_url = isset( $options['api_url'] ) ? $options['api_url'] : '';
		$app_key = isset( $options['app_key'] ) ? $options['app_key'] : '';
		$auth_key = isset( $options['auth_key'] ) ? $options['auth_key'] : '';
		
		if ( empty( $api_url ) || empty( $app_key ) || empty( $auth_key ) ) {
			wp_send_json_error( array( 'message' => __( 'Please configure API credentials first.', 'order-notification-waba-nxc' ) ) );
		}
		
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-waba-api.php';
		$api = new Waba_Chat_Api( $api_url, $app_key, $auth_key );
		
		// Set variables based on template field type
		$variables = array();
		switch ( $field_name ) {
			case 'template_abandoned_cart':
				// Variables: {name}, {checkout_url}
				$variables = array( 'Test User', site_url( '/checkout' ) );
				break;
			case 'template_processing':
			case 'template_completed':
			case 'template_on-hold':
				// Variables: {name}, {order_id}, {total}
				$variables = array( 'Test User', '12345', '$100.00' );
				break;
			default:
				// Fallback: provide all common variables
				$variables = array( 'Test User', '12345', '$100.00', site_url( '/checkout' ) );
				break;
		}
		
		$result = $api->send_message( $phone_number, $template_id, $language, $variables );
		
		if ( is_wp_error( $result ) ) {
			wp_send_json_error( array( 'message' => $result->get_error_message() ) );
		}
		
		wp_send_json_success( array( 'message' => __( 'Test message sent successfully!', 'order-notification-waba-nxc' ) ) );
	}

	public function sanitize_settings( $input ) {
		$new_input = array();
		if( isset( $input['api_url'] ) ) $new_input['api_url'] = esc_url_raw( $input['api_url'] );
		if( isset( $input['app_key'] ) ) $new_input['app_key'] = sanitize_text_field( $input['app_key'] );
		if( isset( $input['auth_key'] ) ) $new_input['auth_key'] = sanitize_text_field( $input['auth_key'] );
		if( isset( $input['whatsapp_number'] ) ) $new_input['whatsapp_number'] = sanitize_text_field( $input['whatsapp_number'] );
		if( isset( $input['test_phone_number'] ) ) $new_input['test_phone_number'] = sanitize_text_field( $input['test_phone_number'] );
		// Template IDs and Languages
		if( isset( $input['template_abandoned_cart'] ) ) $new_input['template_abandoned_cart'] = sanitize_text_field( $input['template_abandoned_cart'] );
		if( isset( $input['template_abandoned_cart_language'] ) ) $new_input['template_abandoned_cart_language'] = sanitize_text_field( $input['template_abandoned_cart_language'] );
		if( isset( $input['template_processing'] ) ) $new_input['template_processing'] = sanitize_text_field( $input['template_processing'] );
		if( isset( $input['template_processing_language'] ) ) $new_input['template_processing_language'] = sanitize_text_field( $input['template_processing_language'] );
		if( isset( $input['template_completed'] ) ) $new_input['template_completed'] = sanitize_text_field( $input['template_completed'] );
		if( isset( $input['template_completed_language'] ) ) $new_input['template_completed_language'] = sanitize_text_field( $input['template_completed_language'] );
		if( isset( $input['template_on-hold'] ) ) $new_input['template_on-hold'] = sanitize_text_field( $input['template_on-hold'] );
		if( isset( $input['template_on-hold_language'] ) ) $new_input['template_on-hold_language'] = sanitize_text_field( $input['template_on-hold_language'] );
		return $new_input;
	}
}
