<?php
if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

class Oradel_Image_Accordion_Widget extends \Elementor\Widget_Base
{


    public function get_name()
    {
        return 'oradel_image_accordion';
    }


    public function get_title()
    {
        return esc_html__('Image Accordion', 'orange-addons-for-elementor');
    }


    public function get_icon()
    {
        return 'eicon-photo-library';
    }


    public function get_categories()
    {
        return ['oradel-plugin'];
    }


    public function get_keywords()
    {
        return ['image_accordion'];
    }


    public function get_custom_help_url()
    {
        return 'https://orange-addons.com/docs/';
    }

    public function get_style_depends()
    {
        return ['oea-image-accordion'];
    }

    public function get_script_depends()
    {
        return ['oea-hover-accordion', 'oea-click-accordion', 'oea-reset-transitions'];
    }

    protected function register_controls()
    {

        $this->start_controls_section(
            'content_section',
            [
                'label' => esc_html__('Content', 'orange-addons-for-elementor'),
                'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
            ]
        );

        $repeater = new \Elementor\Repeater();

        $repeater->add_control(
            'image',
            [
                'label' => esc_html__('Choose Image', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::MEDIA,
                'default' => [
                    'url' => \Elementor\Utils::get_placeholder_image_src(),
                ],
            ]
        );

        $repeater->add_control(
            'item_title',
            [
                'label' => esc_html__('Title', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::TEXT,
                'default' => esc_html__('Default title', 'orange-addons-for-elementor'),
                'placeholder' => esc_html__('Type your title here', 'orange-addons-for-elementor'),
            ]
        );

        $repeater->add_control(
            'item_content',
            [
                'label' => esc_html__('Description', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::TEXTAREA,
                'rows' => 10,
                'default' => esc_html__('Default description', 'orange-addons-for-elementor'),
                'placeholder' => esc_html__('Type your description here', 'orange-addons-for-elementor'),
            ]
        );

        $repeater->add_control(
            'item_button_text',
            [
                'label' => esc_html__('Button', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::TEXT,
                'default' => esc_html__('Button', 'orange-addons-for-elementor'),
                'placeholder' => esc_html__('Type your title here', 'orange-addons-for-elementor'),
            ]
        );

        $repeater->add_control(
            'item_link',
            [
                'label' => esc_html__('Link', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::URL,
                'options' => ['url', 'is_external', 'nofollow'],
                'default' => [
                    'url' => '',
                    'is_external' => true,
                    'nofollow' => true,
                ],
                'label_block' => true,
            ]
        );

        $this->add_control(
            'list',
            [
                'label' => esc_html__('Repeater List', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::REPEATER,
                'fields' => $repeater->get_controls(),
                'default' => [
                    [
                        'item_title' => esc_html__('Title #1', 'orange-addons-for-elementor'),
                        'item_content' => esc_html__('Item content. Click the edit button to change this text.', 'orange-addons-for-elementor'),
                    ],
                    [
                        'item_title' => esc_html__('Title #2', 'orange-addons-for-elementor'),
                        'item_content' => esc_html__('Item content. Click the edit button to change this text.', 'orange-addons-for-elementor'),
                    ],
                    [
                        'item_title' => esc_html__('Title #3', 'orange-addons-for-elementor'),
                        'item_content' => esc_html__('Item content. Click the edit button to change this text.', 'orange-addons-for-elementor'),
                    ],
                    [
                        'item_title' => esc_html__('Title #4', 'orange-addons-for-elementor'),
                        'item_content' => esc_html__('Item content. Click the edit button to change this text.', 'orange-addons-for-elementor'),
                    ],
                ],
                'title_field' => '{{{ item_title }}}',
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            'settings_section',
            [
                'label' => esc_html__('Settings', 'orange-addons-for-elementor'),
                'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'image_clickable',
            [
                'label' => esc_html__('Make Image Clickable', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'label_on' => esc_html__('Yes', 'orange-addons-for-elementor'),
                'label_off' => esc_html__('No', 'orange-addons-for-elementor'),
                'return_value' => 'yes',
                'default' => 'yes',
            ]
        );

        $this->add_control(
            'active_image',
            [
                'label' => esc_html__('Active Image', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::NUMBER,
                'min' => 0,
                'default' => 0,
            ]
        );

        $this->add_control(
            'image_accordion_hover_interaction',
            [
                'label' => esc_html__('Hover Interaction', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'label_on' => esc_html__('Yes', 'orange-addons-for-elementor'),
                'label_off' => esc_html__('No', 'orange-addons-for-elementor'),
                'return_value' => 'yes',
                'default' => 'no',
            ]
        );

        $this->add_responsive_control(
            'item_gap',
            [
                'label' => esc_html__('Item gap', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::SLIDER,
                'size_units' => ['px', 'rem', 'em'],
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 100,
                    ],
                ],
                'default' => [
                    'unit' => 'rem',
                    'size' => 0.5,
                ],
                'selectors' => [
                    '{{WRAPPER}} .oea-image-accordion' => 'gap: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} .oea-image-accordion-hover-interaction' => 'gap: {{SIZE}}{{UNIT}};',
                ],
                'separator' => 'before',
            ]
        );

        $this->add_responsive_control(
            'image_accordion_padding',
            [
                'label' => esc_html__('Padding', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%', 'em', 'rem'],
                'default' => [
                    'top' => 0,
                    'right' => 0,
                    'bottom' => 0,
                    'left' => 0,
                    'unit' => 'rem',
                    'isLinked' => true,
                ],
                'selectors' => [
                    '{{WRAPPER}} .oea-image-accordion-hover-interaction' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                    '{{WRAPPER}} .oea-image-accordion' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'image_accordion_height',
            [
                'label' => esc_html__('Height', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::SLIDER,
                'size_units' => ['px', 'rem', '%'],
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 1000,
                    ],
                ],
                'default' => [
                    'unit' => 'px',
                    'size' => 600,
                ],
                'selectors' => [
                    '{{WRAPPER}} .oea-image-accordion' => 'height: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} .oea-image-accordion-hover-interaction' => 'height: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'accordion_item_border_type',
            [
                'label' => esc_html__('Border Type', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::SELECT,
                'options' => [
                    'outer' => esc_html__('Outer', 'orange-addons-for-elementor'),
                    'individual' => esc_html__('Individual', 'orange-addons-for-elementor')
                ],
                'default' => 'individual',
            ]
        );

        $this->add_control(
            'accordion_item_border_radius',
            [
                'label' => esc_html__('Border Radius', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'default' => [
                    'top' => 0,
                    'right' => 0,
                    'bottom' => 0,
                    'left' => 0,
                ],
                'selectors' => [
                    '{{WRAPPER}} .oea-image-accordion-hover-interaction .item' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                    '{{WRAPPER}} .oea-image-accordion .item' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ]
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Border::get_type(),
            [
                'name' => 'border',
                'selector' => '{{WRAPPER}} .oea-image-accordion, {{WRAPPER}} .oea-image-accordion-hover-interaction',
                'fields_options' => [
                    'border' => [
                        'label' => esc_html__('Accordion Border', 'orange-addons-for-elementor'),
                    ],
                ],
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Border::get_type(),
            [
                'name' => 'hover_interaction_item_border',
                'selector' => '{{WRAPPER}} .oea-image-accordion-hover-interaction .item, {{WRAPPER}} .oea-image-accordion .item',
                'fields_options' => [
                    'border' => [
                        'label' => esc_html__('Item Border', 'orange-addons-for-elementor'),
                    ],
                ],
            ]
        );

        $this->add_control(
            'item_animation_duration',
            [
                'label' => esc_html__('Grow Duration', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::NUMBER,
                'min' => 0.0,
                'max' => 30,
                'step' => 0.1,
                'default' => 0.6,
                'selectors' => [
                    '{{WRAPPER}} .item' => '--grow-transition: {{VALUE}}s;',
                ]
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            'textblock_style_section',
            [
                'label' => esc_html__('Text Block', 'orange-addons-for-elementor'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_responsive_control(
            'image_accordion_text_vertical_align',
            [
                'label' => esc_html__('Vertical Alignment', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::CHOOSE,
                'options' => [
                    'top' => [
                        'top' => esc_html__('Top', 'orange-addons-for-elementor'),
                        'icon' => 'eicon-v-align-top',
                    ],
                    'middle' => [
                        'title' => esc_html__('Middle', 'orange-addons-for-elementor'),
                        'icon' => 'eicon-v-align-middle',
                    ],
                    'bottom' => [
                        'title' => esc_html__('Bottom', 'orange-addons-for-elementor'),
                        'icon' => 'eicon-v-align-bottom',
                    ],
                ],
                'default' => 'middle',
                'toggle' => true,
            ]
        );

        $this->add_responsive_control(
            'image_accordion_text_horizontal_align',
            [
                'label' => esc_html__('Alignment', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::CHOOSE,
                'options' => [
                    'left' => [
                        'title' => esc_html__('Left', 'orange-addons-for-elementor'),
                        'icon' => 'eicon-text-align-left',
                    ],
                    'center' => [
                        'title' => esc_html__('Center', 'orange-addons-for-elementor'),
                        'icon' => 'eicon-text-align-center',
                    ],
                    'right' => [
                        'title' => esc_html__('Right', 'orange-addons-for-elementor'),
                        'icon' => 'eicon-text-align-right',
                    ],
                ],
                'default' => 'center',
                'toggle' => true,
                'selectors' => [
                    '{{WRAPPER}} .image-accordion-text' => 'text-align: {{VALUE}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'image_accordion_text_padding',
            [
                'label' => esc_html__('Padding', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%', 'em', 'rem'],
                'default' => [
                    'top' => 2,
                    'right' => 2,
                    'bottom' => 2,
                    'left' => 2,
                    'unit' => 'rem',
                    'isLinked' => true,
                ],
                'selectors' => [
                    '{{WRAPPER}} .image-accordion-text' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            'title_style_section',
            [
                'label' => esc_html__('Title', 'orange-addons-for-elementor'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Typography::get_type(),
            [
                'name' => 'title_typography',
                'selector' => '{{WRAPPER}} .item h2',
            ]
        );

        $this->add_control(
            'text_color',
            [
                'label' => esc_html__('Title Color', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .item h2' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Background::get_type(),
            [
                'name' => 'title_background',
                'types' => ['classic', 'gradient'],
                'selector' => '{{WRAPPER}} .item h2',
            ]
        );

        $this->add_control(
            'title_border_popover',
            [
                'label' => esc_html__('Title Border', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::POPOVER_TOGGLE,
                'label_off' => esc_html__('Default', 'orange-addons-for-elementor'),
                'label_on' => esc_html__('Custom', 'orange-addons-for-elementor'),
                'return_value' => 'yes',
            ]
        );

        $this->start_popover();

        $this->add_group_control(
            \Elementor\Group_Control_Border::get_type(),
            [
                'name' => 'title_border',
                'selector' => '{{WRAPPER}} .item h2',
            ]
        );

        $this->add_control(
            'title_border_radius',
            [
                'label' => esc_html__('Border Radius', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'default' => [
                    'top' => 0,
                    'right' => 0,
                    'bottom' => 0,
                    'left' => 0,
                ],
                'selectors' => [
                    '{{WRAPPER}} .item h2' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->end_popover();

        $this->add_responsive_control(
            'title_padding',
            [
                'label' => esc_html__('Padding', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%', 'em', 'rem'],
                'default' => [
                    'top' => 0,
                    'right' => 0,
                    'bottom' => 0,
                    'left' => 0,
                    'unit' => 'px',
                    'isLinked' => true,
                ],
                'selectors' => [
                    '{{WRAPPER}} .item h2' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'title_margin',
            [
                'label' => esc_html__('Margin', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%', 'em', 'rem'],
                'default' => [
                    'top' => 0,
                    'right' => 0,
                    'bottom' => 0,
                    'left' => 0,
                    'unit' => 'px',
                    'isLinked' => true,
                ],
                'selectors' => [
                    '{{WRAPPER}} .item h2' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'title_animation_effect',
            [
                'label' => esc_html__('Animation Effect', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::SELECT,
                'default' => 'fade',
                'options' => [
                    'fade' => esc_html__('Fade', 'orange-addons-for-elementor'),
                    'slide-up' => esc_html__('Slide Up', 'orange-addons-for-elementor'),
                    'slide-down' => esc_html__('Slide Down', 'orange-addons-for-elementor'),
                    'slide-left' => esc_html__('Slide Left', 'orange-addons-for-elementor'),
                    'slide-right' => esc_html__('Slide Right', 'orange-addons-for-elementor'),
                    'zoom-in' => esc_html__('Zoom In', 'orange-addons-for-elementor'),
                    'zoom-out' => esc_html__('Zoom Out', 'orange-addons-for-elementor'),
                    'flip-x' => esc_html__('Flip X', 'orange-addons-for-elementor'),
                    'flip-y' => esc_html__('Flip Y', 'orange-addons-for-elementor'),
                    'rotate' => esc_html__('Rotate', 'orange-addons-for-elementor'),
                    'bounce' => esc_html__('Bounce', 'orange-addons-for-elementor'),
                    'elastic' => esc_html__('Elastic', 'orange-addons-for-elementor'),
                    'swing' => esc_html__('Swing', 'orange-addons-for-elementor'),
                    'roll-in' => esc_html__('Roll In', 'orange-addons-for-elementor'),
                    'roll-out' => esc_html__('Roll Out', 'orange-addons-for-elementor'),
                    'slide-rotate' => esc_html__('Slide & Rotate', 'orange-addons-for-elementor'),
                    'zoom-twist' => esc_html__('Zoom & Twist', 'orange-addons-for-elementor'),
                    'flip-scale' => esc_html__('Flip & Scale', 'orange-addons-for-elementor'),
                    'spiral' => esc_html__('Spiral', 'orange-addons-for-elementor'),
                    'skew-fade' => esc_html__('Skew Fade', 'orange-addons-for-elementor'),
                ],
                'selectors_dictionary' => [
                    'fade' => 'opacity: 0;',
                    'slide-up' => 'opacity: 0; transform: translateY(20px);',
                    'slide-down' => 'opacity: 0; transform: translateY(-20px);',
                    'slide-left' => 'opacity: 0; transform: translateX(20px);',
                    'slide-right' => 'opacity: 0; transform: translateX(-20px);',
                    'zoom-in' => 'opacity: 0; transform: scale(0.8);',
                    'zoom-out' => 'opacity: 0; transform: scale(1.2);',
                    'flip-x' => 'opacity: 0; transform: rotateX(90deg);',
                    'flip-y' => 'opacity: 0; transform: rotateY(90deg);',
                    'rotate' => 'opacity: 0; transform: rotate(-180deg);',
                    'bounce' => 'opacity: 0; transform: scale(0.3); transform-origin: bottom;',
                    'elastic' => 'opacity: 0; transform: scaleX(2) translateX(100px);',
                    'swing' => 'opacity: 0; transform: rotateZ(45deg) translateY(-30px);',
                    'roll-in' => 'opacity: 0; transform: rotate(720deg) translateX(-100px);',
                    'roll-out' => 'opacity: 0; transform: rotate(-720deg) translateX(100px);',
                    'slide-rotate' => 'opacity: 0; transform: translateX(-100px) rotate(-90deg);',
                    'zoom-twist' => 'opacity: 0; transform: scale(0.5) rotate(180deg) skew(10deg, 10deg);',
                    'flip-scale' => 'opacity: 0; transform: scale(0.7) rotateY(180deg) translateZ(50px);',
                    'spiral' => 'opacity: 0; transform: rotate(720deg) scale(0) translate(-100%, -100%);',
                    'skew-fade' => 'opacity: 0; transform: skew(-25deg, -25deg) translateX(-50px);',
                ],
                'selectors' => [
                    '{{WRAPPER}} .item:not(.animation) h2' => '{{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            'title_animation_duration',
            [
                'label' => esc_html__('Animation Duration', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::SLIDER,
                'size_units' => ['s'],
                'range' => [
                    's' => [
                        'min' => 0.1,
                        'max' => 3,
                        'step' => 0.1,
                    ],
                ],
                'default' => [
                    'unit' => 's',
                    'size' => 0.6,
                ],
                'selectors' => [
                    '{{WRAPPER}} .item h2' => 'transition-duration: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            'description_style_section',
            [
                'label' => esc_html__('Description', 'orange-addons-for-elementor'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Typography::get_type(),
            [
                'name' => 'description_typography',
                'selector' => '{{WRAPPER}} .item p',
            ]
        );

        $this->add_control(
            'description_text_color',
            [
                'label' => esc_html__('Description Color', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .item p' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Background::get_type(),
            [
                'name' => 'description_background',
                'types' => ['classic', 'gradient'],
                'selector' => '{{WRAPPER}} .item p',
            ]
        );

        $this->add_control(
            'description_border_popover',
            [
                'label' => esc_html__('Description Border', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::POPOVER_TOGGLE,
                'label_off' => esc_html__('Default', 'orange-addons-for-elementor'),
                'label_on' => esc_html__('Custom', 'orange-addons-for-elementor'),
                'return_value' => 'yes',
            ]
        );

        $this->start_popover();

        $this->add_group_control(
            \Elementor\Group_Control_Border::get_type(),
            [
                'name' => 'description_border',
                'selector' => '{{WRAPPER}} .item p',
            ]
        );

        $this->add_control(
            'description_border_radius',
            [
                'label' => esc_html__('Border Radius', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'default' => [
                    'top' => 0,
                    'right' => 0,
                    'bottom' => 0,
                    'left' => 0,
                ],
                'selectors' => [
                    '{{WRAPPER}} .item p' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->end_popover();

        $this->add_responsive_control(
            'description_padding',
            [
                'label' => esc_html__('Padding', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%', 'em', 'rem'],
                'default' => [
                    'top' => 0,
                    'right' => 0,
                    'bottom' => 0,
                    'left' => 0,
                    'unit' => 'px',
                    'isLinked' => true,
                ],
                'selectors' => [
                    '{{WRAPPER}} .item p' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'description_margin',
            [
                'label' => esc_html__('Margin', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%', 'em', 'rem'],
                'default' => [
                    'top' => 0,
                    'right' => 0,
                    'bottom' => 0,
                    'left' => 0,
                    'unit' => 'px',
                    'isLinked' => true,
                ],
                'selectors' => [
                    '{{WRAPPER}} .item p' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'description_animation_effect',
            [
                'label' => esc_html__('Animation Effect', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::SELECT,
                'default' => 'fade',
                'options' => [
                    'fade' => esc_html__('Fade', 'orange-addons-for-elementor'),
                    'slide-up' => esc_html__('Slide Up', 'orange-addons-for-elementor'),
                    'slide-down' => esc_html__('Slide Down', 'orange-addons-for-elementor'),
                    'slide-left' => esc_html__('Slide Left', 'orange-addons-for-elementor'),
                    'slide-right' => esc_html__('Slide Right', 'orange-addons-for-elementor'),
                    'zoom-in' => esc_html__('Zoom In', 'orange-addons-for-elementor'),
                    'zoom-out' => esc_html__('Zoom Out', 'orange-addons-for-elementor'),
                    'flip-x' => esc_html__('Flip X', 'orange-addons-for-elementor'),
                    'flip-y' => esc_html__('Flip Y', 'orange-addons-for-elementor'),
                    'rotate' => esc_html__('Rotate', 'orange-addons-for-elementor'),
                    'bounce' => esc_html__('Bounce', 'orange-addons-for-elementor'),
                    'elastic' => esc_html__('Elastic', 'orange-addons-for-elementor'),
                    'swing' => esc_html__('Swing', 'orange-addons-for-elementor'),
                    'roll-in' => esc_html__('Roll In', 'orange-addons-for-elementor'),
                    'roll-out' => esc_html__('Roll Out', 'orange-addons-for-elementor'),
                    'slide-rotate' => esc_html__('Slide & Rotate', 'orange-addons-for-elementor'),
                    'zoom-twist' => esc_html__('Zoom & Twist', 'orange-addons-for-elementor'),
                    'flip-scale' => esc_html__('Flip & Scale', 'orange-addons-for-elementor'),
                    'spiral' => esc_html__('Spiral', 'orange-addons-for-elementor'),
                    'skew-fade' => esc_html__('Skew Fade', 'orange-addons-for-elementor'),
                ],
                'selectors_dictionary' => [
                    'fade' => 'opacity: 0;',
                    'slide-up' => 'opacity: 0; transform: translateY(20px);',
                    'slide-down' => 'opacity: 0; transform: translateY(-20px);',
                    'slide-left' => 'opacity: 0; transform: translateX(20px);',
                    'slide-right' => 'opacity: 0; transform: translateX(-20px);',
                    'zoom-in' => 'opacity: 0; transform: scale(0.8);',
                    'zoom-out' => 'opacity: 0; transform: scale(1.2);',
                    'flip-x' => 'opacity: 0; transform: rotateX(90deg);',
                    'flip-y' => 'opacity: 0; transform: rotateY(90deg);',
                    'rotate' => 'opacity: 0; transform: rotate(-180deg);',
                    'bounce' => 'opacity: 0; transform: scale(0.3); transform-origin: bottom;',
                    'elastic' => 'opacity: 0; transform: scaleX(2) translateX(100px);',
                    'swing' => 'opacity: 0; transform: rotateZ(45deg) translateY(-30px);',
                    'roll-in' => 'opacity: 0; transform: rotate(720deg) translateX(-100px);',
                    'roll-out' => 'opacity: 0; transform: rotate(-720deg) translateX(100px);',
                    'slide-rotate' => 'opacity: 0; transform: translateX(-100px) rotate(-90deg);',
                    'zoom-twist' => 'opacity: 0; transform: scale(0.5) rotate(180deg) skew(10deg, 10deg);',
                    'flip-scale' => 'opacity: 0; transform: scale(0.7) rotateY(180deg) translateZ(50px);',
                    'spiral' => 'opacity: 0; transform: rotate(720deg) scale(0) translate(-100%, -100%);',
                    'skew-fade' => 'opacity: 0; transform: skew(-25deg, -25deg) translateX(-50px);',
                ],
                'selectors' => [
                    '{{WRAPPER}} .item:not(.animation) p' => '{{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            'description_animation_duration',
            [
                'label' => esc_html__('Animation Duration', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::SLIDER,
                'size_units' => ['s'],
                'range' => [
                    's' => [
                        'min' => 0.1,
                        'max' => 3,
                        'step' => 0.1,
                    ],
                ],
                'default' => [
                    'unit' => 's',
                    'size' => 0.6,
                ],
                'selectors' => [
                    '{{WRAPPER}} .item p' => 'transition-duration: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            'media_style_section',
            [
                'label' => esc_html__('Media', 'orange-addons-for-elementor'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            'enable_media_background',
            [
                'label' => esc_html__('Enable Active Item Background', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'label_on' => esc_html__('Yes', 'orange-addons-for-elementor'),
                'label_off' => esc_html__('No', 'orange-addons-for-elementor'),
                'return_value' => 'yes',
                'default' => 'yes',
                'separator' => 'before',
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Background::get_type(),
            [
                'name' => 'media_background',
                'types' => ['classic', 'gradient'],
                'selector' => '{{WRAPPER}} .item::before',
                'condition' => [
                    'enable_media_background' => 'yes',
                ],
                'fields_options' => [
                    'background' => [
                        'default' => 'classic',
                    ],
                    'color' => [
                        'default' => 'rgba(0, 0, 0, 0.5)',
                    ],
                ],
            ]
        );

        $this->add_control(
            'enable_inactive_overlay',
            [
                'label' => esc_html__('Enable Inactive Item Overlay', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'label_on' => esc_html__('Yes', 'orange-addons-for-elementor'),
                'label_off' => esc_html__('No', 'orange-addons-for-elementor'),
                'return_value' => 'yes',
                'default' => 'no',
                'separator' => 'before',
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Background::get_type(),
            [
                'name' => 'inactive_item_overlay',
                'types' => ['classic', 'gradient'],
                'selector' => '{{WRAPPER}} .item::after',
                'condition' => [
                    'enable_inactive_overlay' => 'yes',
                ],
                'fields_options' => [
                    'background' => [
                        'default' => 'classic',
                    ],
                    'color' => [
                        'default' => 'rgba(0, 0, 0, 0.3)',
                    ],
                ],
            ]
        );

        $this->add_control(
            'image_animation_duration',
            [
                'label' => esc_html__('Image Transform Duration', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::NUMBER,
                'min' => 0.0,
                'max' => 30,
                'step' => 0.1,
                'default' => 0.7,
                'selectors' => [
                    '{{WRAPPER}} .oea-image-accordion-img' => '--transform-transition: {{VALUE}}s;'
                ]
            ]
        );

        $this->add_control(
            'background_animation_duration',
            [
                'label' => esc_html__('Background Animation Duration', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::NUMBER,
                'min' => 0.0,
                'max' => 30,
                'step' => 0.1,
                'default' => 0.7,
                'selectors' => [
                    '{{WRAPPER}} .item' => '--animation-transition: {{VALUE}}s;',
                    '{{WRAPPER}} .oea-image-accordion-img' => '--opacity-transition: {{VALUE}}s;'
                ]
            ]
        );

        $this->add_control(
            'background_animation_delay',
            [
                'label' => esc_html__('Background Animation Delay', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::NUMBER,
                'min' => 0,
                'max' => 30,
                'step' => 0.1,
                'default' => 0,
                'selectors' => [
                    '{{WRAPPER}} .oea-image-accordion-img' => '--opacity-delay: {{VALUE}}s;'
                ]
            ]
        );

        $this->add_control(
            'media_shadow_popover',
            [
                'type' => \Elementor\Controls_Manager::POPOVER_TOGGLE,
                'label' => esc_html__('Box Shadow', 'orange-addons-for-elementor'),
                'label_off' => esc_html__('Default', 'orange-addons-for-elementor'),
                'label_on' => esc_html__('Custom', 'orange-addons-for-elementor'),
                'return_value' => 'yes',
            ]
        );

        $this->start_popover();

        $this->add_control(
            'media_box_shadow',
            [
                'label' => esc_html__('Box Shadow', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::BOX_SHADOW,
                'selectors' => [
                    '{{SELECTOR}} .item' => 'box-shadow: {{HORIZONTAL}}px {{VERTICAL}}px {{BLUR}}px {{SPREAD}}px {{COLOR}};',
                ],
                'default' => [
                    'horizontal' => 6,
                    'vertical' => 5,
                    'blur' => 13,
                    'spread' => 0,
                    'color' => 'rgba(0, 0, 0, 0)',
                ],
            ]
        );

        $this->end_popover();

        $this->add_control(
            'image_opacity',
            [
                'label' => esc_html__('Active Image Opacity', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::NUMBER,
                'min' => 0.0,
                'max' => 1,
                'step' => 0.1,
                'default' => 0.6,
                'selectors' => [
                    '{{WRAPPER}} .oea-image-accordion-hover-interaction .item:hover img' => 'opacity: {{VALUE}};',
                    '{{WRAPPER}} .animation.item .oea-image-accordion-img' => 'opacity: {{VALUE}};'
                ],
                'condition' => [
                    'enable_media_background' => 'yes',
                ],
            ]
        );

        $this->add_control(
            'inactive_image_opacity',
            [
                'label' => esc_html__('Inactive Image Opacity', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::NUMBER,
                'min' => 0.0,
                'max' => 1,
                'step' => 0.1,
                'default' => 0.6,
                'selectors' => [
                    '{{WRAPPER}}' => '--inactive-image-opacity: {{VALUE}};',
                ],
                'condition' => [
                    'enable_inactive_overlay' => 'yes',
                ],
            ]
        );

        $this->add_control(
            'animation_styles',
            [
                'label' => esc_html__('Animation Styles', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::HIDDEN,
                'default' => 'yes',
                'selectors' => [
                    '{{WRAPPER}} .item h2, {{WRAPPER}} .item p' => 'transition-property: opacity, transform; transform-origin: center; transition-timing-function: cubic-bezier(0.4, 0, 0.2, 1); backface-visibility: hidden; transform-style: preserve-3d; will-change: opacity, transform;',
                    '{{WRAPPER}} .item.animation h2, {{WRAPPER}} .item.animation p' => 'opacity: 1; transform: none;',
                    '{{WRAPPER}} .item:not(.animation) h2, {{WRAPPER}} .item:not(.animation) p' => 'pointer-events: none;',
                    '{{WRAPPER}} [data-animation="bounce"] h2, {{WRAPPER}} [data-animation="bounce"] p' => 'transition-timing-function: cubic-bezier(0.36, 0, 0.66, -0.56);',
                    '{{WRAPPER}} [data-animation="elastic"] h2, {{WRAPPER}} [data-animation="elastic"] p' => 'transition-timing-function: cubic-bezier(0.68, -0.55, 0.265, 1.55);',
                    '{{WRAPPER}} [data-animation="swing"] h2, {{WRAPPER}} [data-animation="swing"] p' => 'transition-timing-function: cubic-bezier(0.34, 1.56, 0.64, 1);',
                    '{{WRAPPER}} [data-animation="spiral"] h2, {{WRAPPER}} [data-animation="spiral"] p' => 'transition-timing-function: cubic-bezier(0.175, 0.885, 0.32, 1.275);',
                    '{{WRAPPER}} .item::after' => 'content: ""; position: absolute; top: 0; left: 0; width: 100%; height: 100%; z-index: 0; pointer-events: none; opacity: 0; transition: opacity var(--animation-transition) ease;',
                    '{{WRAPPER}} .item:not(.animation)::after' => 'opacity: 1;',
                ],
            ]
        );

        $this->end_controls_section();
    }

    protected function render()
    {
        $settings = $this->get_settings_for_display();
        $active_image = $settings['active_image'];
        $alignment_class = 'text-align-' . $this->get_settings_for_display('image_accordion_text_vertical_align');

        $disable_classes = '';
        if ($settings['enable_media_background'] !== 'yes') {
            $disable_classes .= ' disable-active-background';
        }
        if ($settings['enable_inactive_overlay'] !== 'yes') {
            $disable_classes .= ' disable-inactive-overlay';
        }

        $accordion_class = ($settings['image_accordion_hover_interaction'] === 'yes' ? 'oea-image-accordion-hover-interaction' : 'oea-image-accordion') .
            ($settings['accordion_item_border_type'] === 'outer' ? ' border-outer' : '') .
            $disable_classes;

        $title_duration = !empty($settings['title_animation_duration']['size']) ? $settings['title_animation_duration']['size'] . $settings['title_animation_duration']['unit'] : '0.6s';
        $description_duration = !empty($settings['description_animation_duration']['size']) ? $settings['description_animation_duration']['size'] . $settings['description_animation_duration']['unit'] : '0.6s';

        $style = sprintf(
            '--title-animation-duration: %s; --description-animation-duration: %s;',
            esc_attr($title_duration),
            esc_attr($description_duration)
        );
        ?>

        <ul class='<?php echo esc_attr(trim($accordion_class)); ?>' data-active-image="<?php echo esc_attr($active_image); ?>"
            style="<?php echo esc_attr($style); ?>">
            <?php if ($settings['list']) {
                foreach ($settings['list'] as $index => $item) {
                    $isActive = ($index + 1 == $active_image) ? 'animation' : '';
                    if (!empty($item['item_link']['url']) && $settings['image_clickable'] === 'yes') {
                        $this->add_link_attributes('item_link', $item['item_link']);
                    }
                    ?>
                    <li class='item <?php echo esc_attr($isActive); ?>' data-index="<?php echo esc_attr($index); ?>">
                        <?php if ($settings['image_clickable'] === 'yes' && !empty($item['item_link']['url'])): ?>
                            <a <?php $this->print_render_attribute_string('item_link'); ?>>
                            <?php endif; ?>
                            <img class="oea-image-accordion-img" src='<?php echo esc_attr($item['image']['url']); ?>'>
                            <div class="image-accordion-text <?php echo esc_attr($alignment_class); ?>">
                                <h2><?php echo esc_attr($item['item_title']); ?></h2>
                                <p><?php echo wp_kses_post($item['item_content']); ?></p>
                            </div>

                            <?php if ($settings['image_clickable'] === 'yes' && !empty($item['item_link']['url'])): ?>
                            </a>
                        <?php endif; ?>
                    </li>
                    <?php
                    $this->remove_render_attribute('item_link');
                }
            } ?>
        </ul>
        <?php

    }

}