<?php

if ( !defined( 'ABSPATH' ) ) {
    exit;
    // Exit if accessed directly.
}
require_once ORADEL_PATH . 'includes/helpers.php';
class Oradel_Icon_Menu_Widget extends \Elementor\Widget_Base {
    public static function get_premium_options_for_styling() {
        return [
            'menu_version' => ['v2'],
            'dots_style'   => ['line', 'ring', 'line-dot'],
            'hover_effect' => [
                'oea-hover-rotate-ccw',
                'oea-hover-shadow-pop',
                'oea-hover-glow-main',
                'oea-hover-pulse-gentle',
                'oea-hover-shake-gentle',
                'oea-hover-float-gentle',
                'oea-hover-fade',
                'oea-hover-skew',
                'oea-hover-flip',
                'oea-hover-wobble',
                'oea-hover-tada',
                'oea-hover-swing',
                'oea-hover-rubberband',
                'oea-hover-heartbeat',
                'oea-hover-jello',
                'oea-hover-bounce',
                'oea-hover-flash'
            ],
        ];
    }

    public function get_name() {
        return 'oradel_icon_menu';
    }

    public function get_title() {
        return esc_html__( 'Icon Menu', 'orange-addons-for-elementor' );
    }

    public function get_icon() {
        return 'eicon-apps';
    }

    public function get_categories() {
        return ['oradel-plugin'];
    }

    public function get_keywords() {
        return ['icon menu, custom menu'];
    }

    public function get_custom_help_url() {
        return 'https://orange-addons.com/docs/';
    }

    public function get_style_depends() {
        return ['oea-icon-menu'];
    }

    public function get_script_depends() {
        return ['oea-icon-menu', 'oea-scroll-animations'];
    }

    protected function register_controls() {
        $mark_premium = function ( $options ) {
            $result = [];
            foreach ( $options as $key => $label ) {
                $result[$key] = $label . ' (Premium)';
            }
            return $result;
        };
        $this->start_controls_section( 'general_settings_section', [
            'label' => esc_html__( 'General Settings', 'orange-addons-for-elementor' ),
            'tab'   => \Elementor\Controls_Manager::TAB_CONTENT,
        ] );
        $menu_version_options = apply_filters( 'oradel_menu_version_options', [
            'v1' => esc_html__( 'Auto (Grid Based)', 'orange-addons-for-elementor' ),
            'v2' => esc_html__( 'Manual (Fixed Slides) (Premium)', 'orange-addons-for-elementor' ),
        ] );
        $this->add_control( 'menu_version', [
            'label'   => esc_html__( 'Menu Logic Version', 'orange-addons-for-elementor' ),
            'type'    => \Elementor\Controls_Manager::SELECT,
            'options' => $menu_version_options,
            'default' => 'v1',
        ] );
        $premium_notice = apply_filters( 'oradel_menu_version_premium_notice', Oradel_Plugin_Helper::create_premium_notice( 
            /* translators: %s: Premium upgrade URL */
            __( 'Manual (Fixed Slides) option is available in the <a href="%s" target="_blank">Premium version</a>', 'orange-addons-for-elementor' ),
            'icon-menu',
            'info'
         ) );
        if ( $premium_notice !== null ) {
            $this->add_control( 'oea_premium_menu_version_notice', array_merge( $premium_notice, [
                'condition' => [
                    'menu_version' => 'v2',
                ],
            ] ) );
        }
        $this->add_control( 'total_slides_v2', [
            'label'       => esc_html__( 'Total Slides', 'orange-addons-for-elementor' ),
            'type'        => \Elementor\Controls_Manager::NUMBER,
            'default'     => 3,
            'min'         => 1,
            'max'         => 20,
            'description' => esc_html__( 'Set the total number of slides to distribute items across.', 'orange-addons-for-elementor' ),
            'condition'   => [
                'menu_version' => 'v2',
            ],
        ] );
        $this->add_control( 'v2_editor_notice', [
            'type'        => \Elementor\Controls_Manager::NOTICE,
            'notice_type' => 'info',
            'show_icon'   => true,
            'content'     => esc_html__( 'Editor preview may differ slightly from the live site due to JavaScript-based slide calculations. Please check the preview on the frontend for the most accurate representation.', 'orange-addons-for-elementor' ),
            'condition'   => [
                'menu_version' => 'v2',
            ],
        ] );
        $this->add_responsive_control( 'columns', [
            'label'          => esc_html__( 'Columns', 'orange-addons-for-elementor' ),
            'type'           => \Elementor\Controls_Manager::SELECT,
            'default'        => '5',
            'tablet_default' => '4',
            'mobile_default' => '3',
            'options'        => [
                '1' => '1',
                '2' => '2',
                '3' => '3',
                '4' => '4',
                '5' => '5',
                '6' => '6',
                '7' => '7',
                '8' => '8',
            ],
            'selectors'      => [
                '{{WRAPPER}} .slide' => 'grid-template-columns: repeat({{VALUE}}, 1fr);',
            ],
        ] );
        $this->add_responsive_control( 'rows', [
            'label'          => esc_html__( 'Rows', 'orange-addons-for-elementor' ),
            'type'           => \Elementor\Controls_Manager::SELECT,
            'default'        => '2',
            'tablet_default' => '2',
            'mobile_default' => '3',
            'options'        => [
                '1' => '1',
                '2' => '2',
                '3' => '3',
                '4' => '4',
                '5' => '5',
                '6' => '6',
            ],
        ] );
        $this->end_controls_section();
        $this->start_controls_section( 'menu_items_section', [
            'label' => esc_html__( 'Menu Items', 'orange-addons-for-elementor' ),
            'tab'   => \Elementor\Controls_Manager::TAB_CONTENT,
        ] );
        $this->add_control( 'menu_items', [
            'label'       => esc_html__( 'Menu Items', 'orange-addons-for-elementor' ),
            'type'        => \Elementor\Controls_Manager::REPEATER,
            'fields'      => [
                [
                    'name'    => 'icon_type',
                    'label'   => esc_html__( 'Type', 'orange-addons-for-elementor' ),
                    'type'    => \Elementor\Controls_Manager::CHOOSE,
                    'options' => [
                        'icon'  => [
                            'title' => esc_html__( 'Icon', 'orange-addons-for-elementor' ),
                            'icon'  => 'eicon-star',
                        ],
                        'image' => [
                            'title' => esc_html__( 'Image', 'orange-addons-for-elementor' ),
                            'icon'  => 'eicon-image',
                        ],
                    ],
                    'default' => 'icon',
                    'toggle'  => true,
                ],
                [
                    'name'      => 'item_icon',
                    'label'     => esc_html__( 'Icon', 'orange-addons-for-elementor' ),
                    'type'      => \Elementor\Controls_Manager::ICONS,
                    'default'   => [
                        'value'   => 'fas fa-star',
                        'library' => 'fa-solid',
                    ],
                    'condition' => [
                        'icon_type' => 'icon',
                    ],
                ],
                [
                    'name'      => 'item_image',
                    'label'     => esc_html__( 'Image', 'orange-addons-for-elementor' ),
                    'type'      => \Elementor\Controls_Manager::MEDIA,
                    'default'   => [
                        'url' => \Elementor\Utils::get_placeholder_image_src(),
                    ],
                    'condition' => [
                        'icon_type' => 'image',
                    ],
                ],
                [
                    'name'        => 'item_title',
                    'label'       => esc_html__( 'Title', 'orange-addons-for-elementor' ),
                    'type'        => \Elementor\Controls_Manager::TEXT,
                    'default'     => esc_html__( 'Menu Item', 'orange-addons-for-elementor' ),
                    'placeholder' => esc_html__( 'Enter menu item title', 'orange-addons-for-elementor' ),
                ],
                [
                    'name'        => 'item_link',
                    'label'       => esc_html__( 'Link', 'orange-addons-for-elementor' ),
                    'type'        => \Elementor\Controls_Manager::URL,
                    'placeholder' => esc_html__( 'https://your-link.com', 'orange-addons-for-elementor' ),
                    'default'     => [
                        'url'         => '',
                        'is_external' => true,
                        'nofollow'    => true,
                    ],
                ]
            ],
            'default'     => [
                [
                    'icon_type'  => 'icon',
                    'item_title' => esc_html__( 'Home', 'orange-addons-for-elementor' ),
                    'item_icon'  => [
                        'value'   => 'fas fa-home',
                        'library' => 'fa-solid',
                    ],
                ],
                [
                    'icon_type'  => 'icon',
                    'item_title' => esc_html__( 'About', 'orange-addons-for-elementor' ),
                    'item_icon'  => [
                        'value'   => 'fas fa-user',
                        'library' => 'fa-solid',
                    ],
                ],
                [
                    'icon_type'  => 'icon',
                    'item_title' => esc_html__( 'Services', 'orange-addons-for-elementor' ),
                    'item_icon'  => [
                        'value'   => 'fas fa-cogs',
                        'library' => 'fa-solid',
                    ],
                ],
                [
                    'icon_type'  => 'icon',
                    'item_title' => esc_html__( 'Contact', 'orange-addons-for-elementor' ),
                    'item_icon'  => [
                        'value'   => 'fas fa-envelope',
                        'library' => 'fa-solid',
                    ],
                ],
                [
                    'icon_type'  => 'icon',
                    'item_title' => esc_html__( 'Blog', 'orange-addons-for-elementor' ),
                    'item_icon'  => [
                        'value'   => 'fas fa-blog',
                        'library' => 'fa-solid',
                    ],
                ]
            ],
            'title_field' => '{{{ item_title }}}',
        ] );
        $this->end_controls_section();
        $this->start_controls_section( 'general_section', [
            'label' => esc_html__( 'General', 'orange-addons-for-elementor' ),
            'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
        ] );
        $this->add_control( 'icon_size', [
            'label'      => esc_html__( 'Icon Size', 'orange-addons-for-elementor' ),
            'type'       => \Elementor\Controls_Manager::SLIDER,
            'size_units' => ['px'],
            'range'      => [
                'px' => [
                    'min'  => 20,
                    'max'  => 100,
                    'step' => 5,
                ],
            ],
            'default'    => [
                'unit' => 'px',
                'size' => 40,
            ],
            'selectors'  => [
                '{{WRAPPER}} .menu-icon svg'         => 'width: {{SIZE}}{{UNIT}};',
                '{{WRAPPER}} .menu-icon .menu-image' => 'height: {{SIZE}}{{UNIT}}; width: {{SIZE}}{{UNIT}};',
            ],
        ] );
        $this->add_group_control( \Elementor\Group_Control_Typography::get_type(), [
            'name'     => 'title_typography',
            'label'    => esc_html__( 'Title Typography', 'orange-addons-for-elementor' ),
            'selector' => '{{WRAPPER}} .menu-title',
        ] );
        $dots_style_options = [
            'default' => esc_html__( 'Default', 'orange-addons-for-elementor' ),
            'rounded' => esc_html__( 'Rounded', 'orange-addons-for-elementor' ),
            'square'  => esc_html__( 'Square', 'orange-addons-for-elementor' ),
            'diamond' => esc_html__( 'Diamond', 'orange-addons-for-elementor' ),
        ];
        $premium_dots_styles = [
            'line'     => esc_html__( 'Line', 'orange-addons-for-elementor' ),
            'ring'     => esc_html__( 'Ring', 'orange-addons-for-elementor' ),
            'line-dot' => esc_html__( 'Line-Dot', 'orange-addons-for-elementor' ),
        ];
        $dots_style_options = array_merge( $dots_style_options, $mark_premium( $premium_dots_styles ) );
        $dots_style_options = apply_filters( 'oradel_dots_style_options', $dots_style_options );
        $this->add_control( 'dots_style', [
            'label'        => esc_html__( 'Dots Style', 'orange-addons-for-elementor' ),
            'type'         => \Elementor\Controls_Manager::SELECT,
            'default'      => 'default',
            'options'      => $dots_style_options,
            'prefix_class' => 'oea-dots-style-',
        ] );
        $premium_dots_keys = array_keys( $premium_dots_styles );
        $premium_notice = apply_filters( 'oradel_dots_style_premium_notice', Oradel_Plugin_Helper::create_premium_notice( 
            /* translators: %s: Premium upgrade URL */
            __( 'Advanced dot styles are available in the <a href="%s" target="_blank">Premium version</a>', 'orange-addons-for-elementor' ),
            'icon-menu',
            'info'
         ) );
        if ( $premium_notice !== null ) {
            $this->add_control( 'oea_premium_dots_style_notice', array_merge( $premium_notice, [
                'condition' => [
                    'dots_style' => $premium_dots_keys,
                ],
            ] ) );
        }
        $this->add_control( 'show_navigation_arrows', [
            'label'       => esc_html__( 'Show Navigation Arrows', 'orange-addons-for-elementor' ),
            'type'        => \Elementor\Controls_Manager::SWITCHER,
            'label_on'    => esc_html__( 'Yes', 'orange-addons-for-elementor' ),
            'label_off'   => esc_html__( 'No', 'orange-addons-for-elementor' ),
            'default'     => 'no',
            'description' => esc_html__( 'Show arrow navigation buttons on desktop devices. Arrows inherit colors from dots settings.', 'orange-addons-for-elementor' ),
        ] );
        $this->add_control( 'arrows_size', [
            'label'      => esc_html__( 'Arrows Size', 'orange-addons-for-elementor' ),
            'type'       => \Elementor\Controls_Manager::SLIDER,
            'size_units' => ['px'],
            'range'      => [
                'px' => [
                    'min'  => 24,
                    'max'  => 64,
                    'step' => 2,
                ],
            ],
            'default'    => [
                'unit' => 'px',
                'size' => 36,
            ],
            'selectors'  => [
                '{{WRAPPER}} .nav-arrow'     => 'width: {{SIZE}}{{UNIT}}; height: {{SIZE}}{{UNIT}};',
                '{{WRAPPER}} .nav-arrow svg' => 'width: calc({{SIZE}}{{UNIT}} * 0.55); height: calc({{SIZE}}{{UNIT}} * 0.55);',
            ],
            'condition'  => [
                'show_navigation_arrows' => 'yes',
            ],
        ] );
        $this->start_controls_tabs( 'general_style_tabs' );
        $this->start_controls_tab( 'general_normal', [
            'label' => esc_html__( 'Normal', 'orange-addons-for-elementor' ),
        ] );
        $this->add_control( 'icon_color', [
            'label'     => esc_html__( 'Icon Color', 'orange-addons-for-elementor' ),
            'type'      => \Elementor\Controls_Manager::COLOR,
            'default'   => '#444',
            'selectors' => [
                '{{WRAPPER}} .menu-icon svg' => 'fill: {{VALUE}};',
                '{{WRAPPER}}'                => '--oea-icon-color: {{VALUE}};',
            ],
        ] );
        $this->add_control( 'title_color', [
            'label'     => esc_html__( 'Title Color', 'orange-addons-for-elementor' ),
            'type'      => \Elementor\Controls_Manager::COLOR,
            'default'   => '#333',
            'selectors' => [
                '{{WRAPPER}} .menu-title' => 'color: {{VALUE}};',
                '{{WRAPPER}}'             => '--oea-title-color: {{VALUE}};',
            ],
        ] );
        $this->add_control( 'dots_color', [
            'label'     => esc_html__( 'Dots Color', 'orange-addons-for-elementor' ),
            'type'      => \Elementor\Controls_Manager::COLOR,
            'default'   => '#ccc',
            'selectors' => [
                '{{WRAPPER}}' => '--oea-dots-color: {{VALUE}};',
            ],
        ] );
        $this->add_control( 'dots_active_color', [
            'label'     => esc_html__( 'Active Dot Color', 'orange-addons-for-elementor' ),
            'type'      => \Elementor\Controls_Manager::COLOR,
            'default'   => '#555',
            'selectors' => [
                '{{WRAPPER}}' => '--oea-dots-active-color: {{VALUE}};',
            ],
        ] );
        $this->end_controls_tab();
        $this->start_controls_tab( 'general_hover', [
            'label' => esc_html__( 'Hover', 'orange-addons-for-elementor' ),
        ] );
        $this->add_control( 'icon_hover_color', [
            'label'     => esc_html__( 'Icon Hover Color', 'orange-addons-for-elementor' ),
            'type'      => \Elementor\Controls_Manager::COLOR,
            'default'   => '#d31616',
            'selectors' => [
                '{{WRAPPER}} .menu-item a[data-hover-effect]:hover .menu-icon'           => 'color: {{VALUE}};',
                '{{WRAPPER}} .menu-item a[data-hover-effect]:hover .menu-icon svg'       => 'fill: {{VALUE}};',
                '{{WRAPPER}} .menu-item a:not([data-hover-effect]):hover .menu-icon'     => 'color: {{VALUE}};',
                '{{WRAPPER}} .menu-item a:not([data-hover-effect]):hover .menu-icon svg' => 'fill: {{VALUE}};',
            ],
        ] );
        $this->add_control( 'title_hover_color', [
            'label'     => esc_html__( 'Title Hover Color', 'orange-addons-for-elementor' ),
            'type'      => \Elementor\Controls_Manager::COLOR,
            'default'   => '#d31616',
            'selectors' => [
                '{{WRAPPER}} .menu-item a[data-hover-effect]:hover .menu-title'       => 'color: {{VALUE}};',
                '{{WRAPPER}} .menu-item a:not([data-hover-effect]):hover .menu-title' => 'color: {{VALUE}};',
            ],
        ] );
        $this->add_control( 'dots_hover_color', [
            'label'     => esc_html__( 'Dots Hover Color', 'orange-addons-for-elementor' ),
            'type'      => \Elementor\Controls_Manager::COLOR,
            'default'   => '#999',
            'selectors' => [
                '{{WRAPPER}}' => '--oea-dots-hover-color: {{VALUE}};',
            ],
        ] );
        $this->end_controls_tab();
        $this->end_controls_tabs();
        $this->end_controls_section();
        $this->start_controls_section( 'animation_section', [
            'label' => esc_html__( 'Transitions', 'orange-addons-for-elementor' ),
            'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
        ] );
        $this->add_control( 'slide_transition_speed', [
            'label'      => esc_html__( 'Slide Transition Speed', 'orange-addons-for-elementor' ),
            'type'       => \Elementor\Controls_Manager::SLIDER,
            'size_units' => ['s', 'ms'],
            'range'      => [
                's'  => [
                    'min'  => 0.1,
                    'max'  => 2.0,
                    'step' => 0.1,
                ],
                'ms' => [
                    'min'  => 100,
                    'max'  => 2000,
                    'step' => 50,
                ],
            ],
            'default'    => [
                'unit' => 's',
                'size' => 0.5,
            ],
            'selectors'  => [
                '{{WRAPPER}} .slides' => '--slide-transition-speed: {{SIZE}}{{UNIT}};',
            ],
        ] );
        $this->add_control( 'height_transition_speed', [
            'label'      => esc_html__( 'Height Transition Speed', 'orange-addons-for-elementor' ),
            'type'       => \Elementor\Controls_Manager::SLIDER,
            'size_units' => ['s', 'ms'],
            'range'      => [
                's'  => [
                    'min'  => 0.1,
                    'max'  => 2.0,
                    'step' => 0.1,
                ],
                'ms' => [
                    'min'  => 100,
                    'max'  => 2000,
                    'step' => 50,
                ],
            ],
            'default'    => [
                'unit' => 's',
                'size' => 0.6,
            ],
            'selectors'  => [
                '{{WRAPPER}} .slides' => '--height-transition-speed: {{SIZE}}{{UNIT}};',
            ],
        ] );
        $this->end_controls_section();
        $this->start_controls_section( 'animations_section', [
            'label' => esc_html__( 'Animations', 'orange-addons-for-elementor' ),
            'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
        ] );
        $this->add_control( 'hover_effect_heading', [
            'label'     => esc_html__( 'Hover Animation (Entire Menu)', 'orange-addons-for-elementor' ),
            'type'      => \Elementor\Controls_Manager::HEADING,
            'separator' => 'before',
        ] );
        $hover_effect_options = [
            'none'             => esc_html__( 'None', 'orange-addons-for-elementor' ),
            'oea-hover-lift'   => esc_html__( 'Lift Up', 'orange-addons-for-elementor' ),
            'oea-hover-grow'   => esc_html__( 'Grow', 'orange-addons-for-elementor' ),
            'oea-hover-rotate' => esc_html__( 'Rotate Clockwise', 'orange-addons-for-elementor' ),
        ];
        $premium_hover_options = [
            'oea-hover-rotate-ccw'   => esc_html__( 'Rotate Counter-Clockwise', 'orange-addons-for-elementor' ),
            'oea-hover-shadow-pop'   => esc_html__( 'Shadow Pop', 'orange-addons-for-elementor' ),
            'oea-hover-glow-main'    => esc_html__( 'Glow (Main Color)', 'orange-addons-for-elementor' ),
            'oea-hover-pulse-gentle' => esc_html__( 'Gentle Pulse (on hover)', 'orange-addons-for-elementor' ),
            'oea-hover-shake-gentle' => esc_html__( 'Gentle Shake (on hover)', 'orange-addons-for-elementor' ),
            'oea-hover-float-gentle' => esc_html__( 'Gentle Float (on hover)', 'orange-addons-for-elementor' ),
            'oea-hover-fade'         => esc_html__( 'Fade', 'orange-addons-for-elementor' ),
            'oea-hover-skew'         => esc_html__( 'Skew', 'orange-addons-for-elementor' ),
            'oea-hover-flip'         => esc_html__( 'Flip', 'orange-addons-for-elementor' ),
            'oea-hover-wobble'       => esc_html__( 'Wobble', 'orange-addons-for-elementor' ),
            'oea-hover-tada'         => esc_html__( 'Tada', 'orange-addons-for-elementor' ),
            'oea-hover-swing'        => esc_html__( 'Swing', 'orange-addons-for-elementor' ),
            'oea-hover-rubberband'   => esc_html__( 'RubberBand', 'orange-addons-for-elementor' ),
            'oea-hover-heartbeat'    => esc_html__( 'HeartBeat', 'orange-addons-for-elementor' ),
            'oea-hover-jello'        => esc_html__( 'Jello', 'orange-addons-for-elementor' ),
            'oea-hover-bounce'       => esc_html__( 'Bounce', 'orange-addons-for-elementor' ),
            'oea-hover-flash'        => esc_html__( 'Flash', 'orange-addons-for-elementor' ),
        ];
        $hover_effect_options = array_merge( $hover_effect_options, $mark_premium( $premium_hover_options ) );
        $hover_effect_options = apply_filters( 'oradel_hover_effect_options', $hover_effect_options );
        $this->add_control( 'hover_effect', [
            'label'   => esc_html__( 'Hover Animation Type', 'orange-addons-for-elementor' ),
            'type'    => \Elementor\Controls_Manager::SELECT,
            'default' => 'none',
            'options' => $hover_effect_options,
        ] );
        $premium_hover_keys = array_keys( $premium_hover_options );
        $premium_notice = apply_filters( 'oradel_hover_effect_premium_notice', Oradel_Plugin_Helper::create_premium_notice( 
            /* translators: %s: Premium upgrade URL */
            __( 'Advanced hover animations are available in the <a href="%s" target="_blank">Premium version</a>', 'orange-addons-for-elementor' ),
            'icon-menu',
            'info'
         ) );
        if ( $premium_notice !== null ) {
            $this->add_control( 'oea_premium_hover_effect_notice', array_merge( $premium_notice, [
                'condition' => [
                    'hover_effect' => $premium_hover_keys,
                ],
            ] ) );
        }
        $this->add_control( 'hover_effect_duration', [
            'label'      => esc_html__( 'Hover Animation Duration (s)', 'orange-addons-for-elementor' ),
            'type'       => \Elementor\Controls_Manager::SLIDER,
            'size_units' => ['s'],
            'range'      => [
                's' => [
                    'min'  => 0.1,
                    'max'  => 3,
                    'step' => 0.1,
                ],
            ],
            'default'    => [
                'unit' => 's',
                'size' => 0.5,
            ],
            'condition'  => [
                'hover_effect!' => 'none',
            ],
            'selectors'  => [
                '{{WRAPPER}} .menu-item a[data-hover-effect]:not([data-hover-effect="none"])'                        => '--oea-hover-animation-duration: {{SIZE}}{{UNIT}};',
                '{{WRAPPER}} .menu-item a[data-hover-effect]:not([data-hover-effect="none"]) .menu-icon'             => '--oea-hover-animation-duration: {{SIZE}}{{UNIT}};',
                '{{WRAPPER}} .menu-item a[data-hover-effect]:not([data-hover-effect="none"]) .menu-icon svg'         => '--oea-hover-animation-duration: {{SIZE}}{{UNIT}};',
                '{{WRAPPER}} .menu-item a[data-hover-effect]:not([data-hover-effect="none"]) .menu-icon .menu-image' => '--oea-hover-animation-duration: {{SIZE}}{{UNIT}};',
                '{{WRAPPER}} .menu-item a[data-hover-effect]:not([data-hover-effect="none"]) .menu-title'            => '--oea-hover-animation-duration: {{SIZE}}{{UNIT}};',
            ],
        ] );
        $this->add_control( 'hover_effect_delay', [
            'label'      => esc_html__( 'Hover Animation Delay (s)', 'orange-addons-for-elementor' ),
            'type'       => \Elementor\Controls_Manager::SLIDER,
            'size_units' => ['s'],
            'range'      => [
                's' => [
                    'min'  => 0,
                    'max'  => 3,
                    'step' => 0.1,
                ],
            ],
            'default'    => [
                'unit' => 's',
                'size' => 0,
            ],
            'condition'  => [
                'hover_effect!' => 'none',
            ],
            'selectors'  => [
                '{{WRAPPER}} .menu-item a[data-hover-effect]:not([data-hover-effect="none"])'                        => '--oea-hover-animation-delay: {{SIZE}}{{UNIT}};',
                '{{WRAPPER}} .menu-item a[data-hover-effect]:not([data-hover-effect="none"]) .menu-icon'             => '--oea-hover-animation-delay: {{SIZE}}{{UNIT}};',
                '{{WRAPPER}} .menu-item a[data-hover-effect]:not([data-hover-effect="none"]) .menu-icon svg'         => '--oea-hover-animation-delay: {{SIZE}}{{UNIT}};',
                '{{WRAPPER}} .menu-item a[data-hover-effect]:not([data-hover-effect="none"]) .menu-icon .menu-image' => '--oea-hover-animation-delay: {{SIZE}}{{UNIT}};',
                '{{WRAPPER}} .menu-item a[data-hover-effect]:not([data-hover-effect="none"]) .menu-title'            => '--oea-hover-animation-delay: {{SIZE}}{{UNIT}};',
            ],
        ] );
        $this->add_control( 'item_animation_heading', [
            'label'     => esc_html__( 'Menu Item Animation', 'orange-addons-for-elementor' ),
            'type'      => \Elementor\Controls_Manager::HEADING,
            'separator' => 'before',
        ] );
        $this->add_control( 'item_animation', [
            'label'   => esc_html__( 'Animation', 'orange-addons-for-elementor' ),
            'type'    => \Elementor\Controls_Manager::SELECT,
            'default' => 'none',
            'options' => [
                'none'              => esc_html__( 'None', 'orange-addons-for-elementor' ),
                'oea-fade-in'       => esc_html__( 'Fade In', 'orange-addons-for-elementor' ),
                'oea-fade-in-up'    => esc_html__( 'Fade In Up', 'orange-addons-for-elementor' ),
                'oea-fade-in-down'  => esc_html__( 'Fade In Down', 'orange-addons-for-elementor' ),
                'oea-fade-in-left'  => esc_html__( 'Fade In Left', 'orange-addons-for-elementor' ),
                'oea-fade-in-right' => esc_html__( 'Fade In Right', 'orange-addons-for-elementor' ),
                'oea-zoom-in'       => esc_html__( 'Zoom In', 'orange-addons-for-elementor' ),
                'oea-bounce-in'     => esc_html__( 'Bounce In', 'orange-addons-for-elementor' ),
                'oea-flip-in'       => esc_html__( 'Flip In', 'orange-addons-for-elementor' ),
                'oea-rotate-in'     => esc_html__( 'Rotate In', 'orange-addons-for-elementor' ),
            ],
        ] );
        $this->add_control( 'item_animation_duration', [
            'label'      => esc_html__( 'Animation Duration (s)', 'orange-addons-for-elementor' ),
            'type'       => \Elementor\Controls_Manager::SLIDER,
            'size_units' => ['s'],
            'range'      => [
                's' => [
                    'min'  => 0.1,
                    'max'  => 5,
                    'step' => 0.1,
                ],
            ],
            'default'    => [
                'unit' => 's',
                'size' => 0.8,
            ],
            'condition'  => [
                'item_animation!' => 'none',
            ],
        ] );
        $this->add_control( 'item_animation_delay', [
            'label'      => esc_html__( 'Animation Delay (s)', 'orange-addons-for-elementor' ),
            'type'       => \Elementor\Controls_Manager::SLIDER,
            'size_units' => ['s'],
            'range'      => [
                's' => [
                    'min'  => 0,
                    'max'  => 5,
                    'step' => 0.1,
                ],
            ],
            'default'    => [
                'unit' => 's',
                'size' => 0.1,
            ],
            'condition'  => [
                'item_animation!' => 'none',
            ],
        ] );
        $this->add_control( 'animation_on_scroll', [
            'label'       => esc_html__( 'Animate Only When Visible', 'orange-addons-for-elementor' ),
            'type'        => \Elementor\Controls_Manager::SWITCHER,
            'label_on'    => esc_html__( 'Yes', 'orange-addons-for-elementor' ),
            'label_off'   => esc_html__( 'No', 'orange-addons-for-elementor' ),
            'default'     => '',
            'description' => esc_html__( 'Animation will start only when element enters the center of the viewport.', 'orange-addons-for-elementor' ),
            'condition'   => [
                'item_animation!' => 'none',
            ],
        ] );
        $stagger_animation_config = apply_filters( 'oradel_stagger_animation_control_config', [
            'label'       => esc_html__( 'Stagger Animation', 'orange-addons-for-elementor' ) . ' <i class="eicon-pro-icon"></i>',
            'type'        => \Elementor\Controls_Manager::SWITCHER,
            'default'     => 'no',
            'description' => esc_html__( 'Apply animation delay to each item sequentially', 'orange-addons-for-elementor' ),
            'condition'   => [
                'item_animation!' => 'none',
            ],
            'classes'     => 'elementor-control-disabled',
        ] );
        $this->add_control( 'stagger_animation', $stagger_animation_config );
        $premium_notice = apply_filters( 'oradel_stagger_animation_premium_notice', Oradel_Plugin_Helper::create_premium_notice( 
            /* translators: %s: Premium upgrade URL */
            __( 'Stagger animation is available in the <a href="%s" target="_blank">Premium version</a>', 'orange-addons-for-elementor' ),
            'icon-menu',
            'info'
         ) );
        if ( $premium_notice !== null ) {
            $this->add_control( 'oea_premium_stagger_animation_notice', array_merge( $premium_notice, [
                'condition' => [
                    'item_animation!' => 'none',
                ],
            ] ) );
        }
        $this->add_control( 'disable_load_animation_mobile', [
            'label'       => esc_html__( 'Disable Load Animation on Mobile', 'orange-addons-for-elementor' ),
            'type'        => \Elementor\Controls_Manager::SWITCHER,
            'label_on'    => esc_html__( 'Yes', 'orange-addons-for-elementor' ),
            'label_off'   => esc_html__( 'No', 'orange-addons-for-elementor' ),
            'default'     => 'yes',
            'description' => esc_html__( 'Turn off load animations (fade-in, zoom-in, etc.) on mobile devices for better performance.', 'orange-addons-for-elementor' ),
            'condition'   => [
                'item_animation!' => 'none',
            ],
            'separator'   => 'before',
        ] );
        $this->add_control( 'disable_hover_animation_mobile', [
            'label'       => esc_html__( 'Disable Hover Animation on Mobile', 'orange-addons-for-elementor' ),
            'type'        => \Elementor\Controls_Manager::SWITCHER,
            'label_on'    => esc_html__( 'Yes', 'orange-addons-for-elementor' ),
            'label_off'   => esc_html__( 'No', 'orange-addons-for-elementor' ),
            'default'     => 'yes',
            'description' => esc_html__( 'Turn off hover effects (lift, grow, rotate, etc.) on mobile devices for better performance.', 'orange-addons-for-elementor' ),
            'condition'   => [
                'hover_effect!' => 'none',
            ],
        ] );
        $this->end_controls_section();
    }

    protected function render() {
        $settings = $this->get_settings_for_display();
        // Применяем фильтр для menu_version
        $settings['menu_version'] = apply_filters(
            'oradel_enable_menu_version_render',
            $settings['menu_version'],
            $settings['menu_version'],
            $settings
        );
        $premium_dots_styles = ['line', 'ring', 'line-dot'];
        $premium_hover_effects = [
            'oea-hover-rotate-ccw',
            'oea-hover-shadow-pop',
            'oea-hover-glow-main',
            'oea-hover-pulse-gentle',
            'oea-hover-shake-gentle',
            'oea-hover-float-gentle',
            'oea-hover-fade',
            'oea-hover-skew',
            'oea-hover-flip',
            'oea-hover-wobble',
            'oea-hover-tada',
            'oea-hover-swing',
            'oea-hover-rubberband',
            'oea-hover-heartbeat',
            'oea-hover-jello',
            'oea-hover-bounce',
            'oea-hover-flash'
        ];
        // Применяем фильтр для dots_style
        $settings['dots_style'] = apply_filters(
            'oradel_enable_dots_style_render',
            $settings['dots_style'],
            $settings['dots_style'],
            $settings,
            $premium_dots_styles
        );
        // Применяем фильтр для hover_effect
        $settings['hover_effect'] = apply_filters(
            'oradel_enable_hover_effect_render',
            $settings['hover_effect'],
            $settings['hover_effect'],
            $settings,
            $premium_hover_effects
        );
        $menu_items = $settings['menu_items'];
        $menu_version = $settings['menu_version'] ?? 'v1';
        if ( empty( $menu_items ) ) {
            return;
        }
        $menu_container_classes = ['menu-container'];
        $menu_container_data_attributes = [];
        if ( !empty( $settings['disable_load_animation_mobile'] ) && $settings['disable_load_animation_mobile'] === 'yes' ) {
            $menu_container_classes[] = 'oea-disable-load-animation-mobile';
        }
        if ( !empty( $settings['disable_hover_animation_mobile'] ) && $settings['disable_hover_animation_mobile'] === 'yes' ) {
            $menu_container_classes[] = 'oea-disable-hover-animation-mobile';
        }
        if ( !empty( $settings['animation_on_scroll'] ) && $settings['animation_on_scroll'] === 'yes' ) {
            $menu_container_classes[] = 'oea-animate-on-scroll';
        }
        $menu_container_classes = array_unique( $menu_container_classes );
        $menu_container_data_attr_string = implode( ' ', $menu_container_data_attributes );
        $responsive_config = $this->calculate_responsive_config( $settings );
        $use_v2_slides = apply_filters(
            'oradel_use_v2_menu_slides',
            false,
            $menu_version === 'v2',
            $settings
        );
        if ( $use_v2_slides ) {
            $slides = $this->generate_v2_slides( $menu_items, $settings, $responsive_config );
        } else {
            $slides = $this->generate_v1_slides( $menu_items, $responsive_config );
        }
        $total_slides = count( $slides );
        $widget_wrapper_classes = ['oea-icon-menu-wrapper'];
        if ( !empty( $settings['show_navigation_arrows'] ) && $settings['show_navigation_arrows'] === 'yes' ) {
            $widget_wrapper_classes[] = 'oea-show-arrows';
        }
        ?>
        <div class="<?php 
        echo esc_attr( implode( ' ', $widget_wrapper_classes ) );
        ?>">
            <?php 
        $this->render_menu_container_with_animations(
            $slides,
            $responsive_config,
            $menu_version,
            count( $menu_items ),
            $menu_container_classes,
            $menu_container_data_attr_string,
            $settings
        );
        ?>
            <?php 
        $this->render_navigation_arrows( $settings );
        ?>
        </div>
        <?php 
    }

    private function calculate_responsive_config( $settings ) {
        return [
            'desktop' => (int) ($settings['columns'] ?? 5) * (int) ($settings['rows'] ?? 2),
            'tablet'  => (int) ($settings['columns_tablet'] ?? 4) * (int) ($settings['rows_tablet'] ?? 2),
            'mobile'  => (int) ($settings['columns_mobile'] ?? 3) * (int) ($settings['rows_mobile'] ?? 3),
        ];
    }

    private function generate_v1_slides( $menu_items, $responsive_config ) {
        return array_chunk( $menu_items, $responsive_config['desktop'] );
    }

    private function generate_v2_slides( $menu_items, $settings, $responsive_config ) {
        return [];
    }

    private function generate_device_slides( $menu_items, $items_per_slide, $total_slides_setting ) {
        $slides = [];
        $remaining_items = $menu_items;
        for ($i = 0; $i < $total_slides_setting && !empty( $remaining_items ); $i++) {
            if ( $i === $total_slides_setting - 1 ) {
                $slides[] = $remaining_items;
                break;
            } else {
                $slide_items = array_slice( $remaining_items, 0, $items_per_slide );
                $slides[] = $slide_items;
                $remaining_items = array_slice( $remaining_items, $items_per_slide );
            }
        }
        return $slides;
    }

    private function render_menu_container_with_animations(
        $slides,
        $responsive_config,
        $menu_version,
        $total_items,
        $menu_container_classes,
        $menu_container_data_attr_string,
        $settings
    ) {
        $use_v2_container = apply_filters(
            'oradel_use_v2_menu_container',
            false,
            $menu_version === 'v2',
            $settings
        );
        if ( $use_v2_container ) {
            $this->render_v2_container_with_animations(
                $slides,
                $responsive_config,
                $total_items,
                $menu_container_classes,
                $menu_container_data_attr_string,
                $settings
            );
        } else {
            $this->render_v1_container_with_animations(
                $slides,
                $responsive_config,
                $total_items,
                $menu_container_classes,
                $menu_container_data_attr_string,
                $settings
            );
        }
    }

    private function render_menu_container(
        $slides,
        $responsive_config,
        $menu_version,
        $total_items
    ) {
        $use_v2_container = apply_filters(
            'oradel_use_v2_menu_container',
            false,
            $menu_version === 'v2',
            []
        );
        if ( $use_v2_container ) {
            $this->render_v2_container( $slides, $responsive_config, $total_items );
        } else {
            $this->render_v1_container( $slides, $responsive_config, $total_items );
        }
    }

    private function render_v1_container_with_animations(
        $slides,
        $responsive_config,
        $total_items,
        $menu_container_classes,
        $menu_container_data_attr_string,
        $settings
    ) {
        $total_slides = count( $slides );
        ?>
        <div class="<?php 
        echo esc_attr( implode( ' ', $menu_container_classes ) );
        ?>" <?php 
        echo wp_kses_post( $menu_container_data_attr_string );
        ?> data-total-slides="<?php 
        echo esc_attr( $total_slides );
        ?>"
            data-items-desktop="<?php 
        echo esc_attr( $responsive_config['desktop'] );
        ?>"
            data-items-tablet="<?php 
        echo esc_attr( $responsive_config['tablet'] );
        ?>"
            data-items-mobile="<?php 
        echo esc_attr( $responsive_config['mobile'] );
        ?>"
            data-total-items="<?php 
        echo esc_attr( $total_items );
        ?>" data-menu-version="v1"
            style="--tablet-limit: <?php 
        echo esc_attr( $responsive_config['tablet'] + 1 );
        ?>; --mobile-limit: <?php 
        echo esc_attr( $responsive_config['mobile'] + 1 );
        ?>">

            <div class="slides" id="slides">
                <?php 
        foreach ( $slides as $slide_index => $slide_items ) {
            ?>
                    <div class="slide" data-slide="<?php 
            echo esc_attr( $slide_index );
            ?>">
                        <?php 
            foreach ( $slide_items as $item_index => $item ) {
                ?>
                            <div class="menu-item<?php 
                echo wp_kses_post( $this->get_item_animation_class( $settings, $item_index ) );
                ?>"
                                <?php 
                echo wp_kses_post( $this->get_item_animation_style( $settings, $item_index ) );
                ?>>
                                <?php 
                $this->render_menu_item( $item, $settings );
                ?>
                            </div>
                        <?php 
            }
            ?>
                    </div>
                <?php 
        }
        ?>
            </div>

            <?php 
        $this->render_navigation_dots( $total_slides );
        ?>
        </div>
        <?php 
    }

    private function render_v2_container_with_animations(
        $slides,
        $responsive_config,
        $total_items,
        $menu_container_classes,
        $menu_container_data_attr_string,
        $settings
    ) {
    }

    private function get_item_animation_class( $settings, $item_index ) {
        if ( empty( $settings['item_animation'] ) || $settings['item_animation'] === 'none' ) {
            return '';
        }
        $class = ' ' . $settings['item_animation'];
        return $class;
    }

    private function get_item_animation_style( $settings, $item_index ) {
        if ( empty( $settings['item_animation'] ) || $settings['item_animation'] === 'none' ) {
            return '';
        }
        $style = '';
        if ( !empty( $settings['item_animation_duration']['size'] ) ) {
            $style .= 'animation-duration:' . $settings['item_animation_duration']['size'] . 's;';
        }
        $delay = 0;
        if ( !empty( $settings['item_animation_delay']['size'] ) ) {
            $delay = $settings['item_animation_delay']['size'];
        }
        if ( $delay > 0 ) {
            $style .= 'animation-delay:' . $delay . 's;';
        }
        return ( $style ? ' style="' . esc_attr( $style ) . '"' : '' );
    }

    private function render_v1_container( $slides, $responsive_config, $total_items ) {
        $total_slides = count( $slides );
        ?>
        <div class="menu-container" data-total-slides="<?php 
        echo esc_attr( $total_slides );
        ?>"
            data-items-desktop="<?php 
        echo esc_attr( $responsive_config['desktop'] );
        ?>"
            data-items-tablet="<?php 
        echo esc_attr( $responsive_config['tablet'] );
        ?>"
            data-items-mobile="<?php 
        echo esc_attr( $responsive_config['mobile'] );
        ?>"
            data-total-items="<?php 
        echo esc_attr( $total_items );
        ?>" data-menu-version="v1"
            style="--tablet-limit: <?php 
        echo esc_attr( $responsive_config['tablet'] + 1 );
        ?>; --mobile-limit: <?php 
        echo esc_attr( $responsive_config['mobile'] + 1 );
        ?>">

            <div class="slides" id="slides">
                <?php 
        foreach ( $slides as $slide_index => $slide_items ) {
            ?>
                    <div class="slide" data-slide="<?php 
            echo esc_attr( $slide_index );
            ?>">
                        <?php 
            foreach ( $slide_items as $item ) {
                ?>
                            <div class="menu-item">
                                <?php 
                $this->render_menu_item( $item, [] );
                ?>
                            </div>
                        <?php 
            }
            ?>
                    </div>
                <?php 
        }
        ?>
            </div>

            <?php 
        $this->render_navigation_dots( $total_slides );
        ?>
        </div>
        <?php 
    }

    private function render_v2_container( $slides, $responsive_config, $total_items ) {
    }

    private function render_v2_navigation_dots( $max_slides ) {
    }

    private function render_menu_item( $item, $settings = [] ) {
        $target = ( $item['item_link']['is_external'] ? ' target="_blank"' : '' );
        $nofollow = ( $item['item_link']['nofollow'] ? ' rel="nofollow"' : '' );
        $link_url = ( !empty( $item['item_link']['url'] ) ? $item['item_link']['url'] : '#' );
        $hover_effect_attr = '';
        if ( !empty( $settings['hover_effect'] ) && $settings['hover_effect'] !== 'none' ) {
            $hover_effect_attr = ' data-hover-effect="' . esc_attr( $settings['hover_effect'] ) . '"';
        }
        ?>
        <a href="<?php 
        echo esc_url( $link_url );
        ?>" <?php 
        echo wp_kses_post( $target . $nofollow . $hover_effect_attr );
        ?>>
            <div class="menu-icon">
                <?php 
        $this->render_menu_icon( $item );
        ?>
            </div>
            <div class="menu-title"><?php 
        echo esc_html( $item['item_title'] );
        ?></div>
        </a>
        <?php 
    }

    private function render_menu_icon( $item ) {
        $icon_type = ( isset( $item['icon_type'] ) ? $item['icon_type'] : 'icon' );
        if ( $icon_type === 'image' && !empty( $item['item_image']['url'] ) ) {
            echo '<img src="' . esc_url( $item['item_image']['url'] ) . '" alt="' . esc_attr( $item['item_title'] ) . '" class="menu-image">';
        } elseif ( $icon_type === 'icon' && !empty( $item['item_icon']['value'] ) ) {
            \Elementor\Icons_Manager::render_icon( $item['item_icon'], [
                'aria-hidden' => 'true',
            ] );
        }
    }

    private function render_navigation_dots( $total_slides ) {
        ?>
        <div class="dots" id="dots">
            <?php 
        if ( $total_slides > 1 ) {
            ?>
                <?php 
            for ($i = 0; $i < $total_slides; $i++) {
                ?>
                    <span class="dot<?php 
                echo ( $i === 0 ? ' active' : '' );
                ?>" data-slide="
<?php 
                echo esc_attr( $i );
                ?>"></span>
                <?php 
            }
            ?>
            <?php 
        }
        ?>
        </div>
        <?php 
    }

    private function render_navigation_arrows( $settings = [] ) {
        $show_arrows = !empty( $settings['show_navigation_arrows'] ) && $settings['show_navigation_arrows'] === 'yes';
        if ( !$show_arrows ) {
            return;
        }
        ?>
        <div class="navigation-arrows">
            <button class="nav-arrow nav-arrow-prev"
                aria-label="<?php 
        echo esc_attr__( 'Previous slide', 'orange-addons-for-elementor' );
        ?>">
                <svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                    <path d="M15 18L9 12L15 6" stroke="currentColor" stroke-width="2" stroke-linecap="round"
                        stroke-linejoin="round" />
                </svg>
            </button>
            <button class="nav-arrow nav-arrow-next"
                aria-label="<?php 
        echo esc_attr__( 'Next slide', 'orange-addons-for-elementor' );
        ?>">
                <svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                    <path d="M9 18L15 12L9 6" stroke="currentColor" stroke-width="2" stroke-linecap="round"
                        stroke-linejoin="round" />
                </svg>
            </button>
        </div>
        <?php 
    }

}
