<?php
if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

require_once ORADEL_PATH . 'includes/helpers.php';

class Oradel_Button_Widget extends \Elementor\Widget_Base
{

    public static function get_premium_options_for_styling()
    {
        return [
            'oea_button_hover_effect' => [
                'hover-effect-5',
                'hover-effect-6',
                'hover-effect-7',
                'hover-effect-8',
                'hover-effect-9',
            ],
        ];
    }

    public function get_name()
    {
        return 'button';
    }


    public function get_title()
    {
        return esc_html__('Button', 'orange-addons-for-elementor');
    }


    public function get_icon()
    {
        return 'eicon-button';
    }


    public function get_categories()
    {
        return ['oradel-plugin'];
    }


    public function get_keywords()
    {
        return ['button'];
    }


    public function get_custom_help_url()
    {
        return 'https://orange-addons.com/docs/';
    }

    public function get_style_depends()
    {
        $styles = ['oea-button'];

        if (Oradel_Plugin_Helper::is_premium()) {
            $styles[] = 'oea-button-premium-effects';
        }

        return $styles;
    }

    public function get_script_depends()
    {
        return ['oea-button-effects'];
    }

    private function get_premium_hover_effects()
    {
        $premium_effects = [
            'hover-effect-5' => __('Hover Effect 5 (Premium)', 'orange-addons-for-elementor'),
            'hover-effect-6' => __('Hover Effect 6 (Premium)', 'orange-addons-for-elementor'),
            'hover-effect-7' => __('Hover Effect 7 (Premium)', 'orange-addons-for-elementor'),
            'hover-effect-8' => __('Hover Effect 8 (Premium)', 'orange-addons-for-elementor'),
            'hover-effect-9' => __('Hover Effect 9 (Premium)', 'orange-addons-for-elementor'),
        ];

        return apply_filters('oradel_button_premium_hover_effects_options', $premium_effects);
    }

    protected function register_controls()
    {

        $this->start_controls_section(
            'content_section',
            [
                'label' => esc_html__('Content', 'orange-addons-for-elementor'),
                'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'oea_button',
            [
                'label' => esc_html__('Your Button', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::TEXT,
                'default' => esc_html__('Button', 'orange-addons-for-elementor'),
                'placeholder' => esc_html__('Type your text here', 'orange-addons-for-elementor'),
            ]
        );

        $this->add_control(
            'button_link',
            [
                'label' => esc_html__('Link', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::URL,
                'options' => ['url', 'is_external', 'nofollow'],
                'default' => [
                    'url' => '',
                    'is_external' => true,
                    'nofollow' => true,
                ],
                'label_block' => true,
            ]
        );

        $this->add_control(
            'oea_button_hover_effect',
            [
                'label' => __('Button Hover Effect', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::SELECT,
                'options' => [
                    '' => __('None', 'orange-addons-for-elementor'),
                    'hover-effect-1' => __('Hover Effect 1', 'orange-addons-for-elementor'),
                    'hover-effect-2' => __('Hover Effect 2', 'orange-addons-for-elementor'),
                    'hover-effect-3' => __('Hover Effect 3', 'orange-addons-for-elementor'),
                    'hover-effect-4' => __('Hover Effect 4', 'orange-addons-for-elementor'),
                ] + $this->get_premium_hover_effects(),
                'default' => '',
            ]
        );

        $premium_notice = apply_filters(
            'oradel_button_hover_effects_premium_notice',
            Oradel_Plugin_Helper::create_premium_notice(
                /* translators: %s: Premium upgrade URL */
                __('More hover effects are available in the <a href="%s" target="_blank">Premium version</a>', 'orange-addons-for-elementor'),
                'button',
                'info'
            )
        );

        if ($premium_notice !== null) {
            $this->add_control('oea_premium_notice', $premium_notice);
        }

        $this->add_control(
            'oea_button_click_effect',
            [
                'label' => __('Button Click Effect', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::SELECT,
                'options' => [
                    '' => __('None', 'orange-addons-for-elementor'),
                    'click-effect-1' => __('Click Effect 1', 'orange-addons-for-elementor'),
                ],
                'default' => '',
                'condition' => [
                    'oea_button_hover_effect!' => ['hover-effect-2', 'hover-effect-4', 'hover-effect-5', 'hover-effect-9'],
                ],
            ]
        );

        $this->add_control(
            'oea_click_color',
            [
                'label' => __('Click Color', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#ff8800',
                'selectors' => [
                    '{{WRAPPER}} .oea-button-link' => '--click-color: {{VALUE}};',
                ],
                'condition' => [
                    'oea_button_hover_effect!' => ['hover-effect-2', 'hover-effect-4', 'hover-effect-5', 'hover-effect-9'],
                ],
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            'oea_button_style',
            [
                'label' => __('Button', 'orange-addons-for-elementor'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_responsive_control(
            'oea_button_alignment',
            [
                'label' => esc_html__('Button Alignment', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::CHOOSE,
                'options' => [
                    'flex-start' => [
                        'title' => esc_html__('Left', 'orange-addons-for-elementor'),
                        'icon' => 'eicon-h-align-left',
                    ],
                    'center' => [
                        'title' => esc_html__('Center', 'orange-addons-for-elementor'),
                        'icon' => 'eicon-h-align-center',
                    ],
                    'flex-end' => [
                        'title' => esc_html__('Right', 'orange-addons-for-elementor'),
                        'icon' => 'eicon-h-align-right',
                    ],
                ],
                'default' => 'center',
                'selectors' => [
                    '{{WRAPPER}} .oea-button-wrapper' => 'justify-content: {{VALUE}};',
                ],
            ]
        );

        $this->start_controls_tabs('oea_button_tabs');

        $this->start_controls_tab(
            'oea_button_normal',
            [
                'label' => __('Normal', 'orange-addons-for-elementor'),
            ]
        );

        $this->add_control(
            'oea_button_bg_color',
            [
                'label' => __('Background Color', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .oea-button-link' => '--background-color: {{VALUE}};',
                ],
                'condition' => [
                    'oea_button_hover_effect' => ['hover-effect-1', 'hover-effect-4', 'hover-effect-5', 'hover-effect-9'],
                ],
            ]
        );

        $this->add_control(
            'oea_button_bg_type',
            [
                'label' => __('Background Type', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::SELECT,
                'default' => 'color',
                'options' => [
                    'color' => __('Solid Color', 'orange-addons-for-elementor'),
                    'gradient' => __('Gradient', 'orange-addons-for-elementor'),
                ],
                'condition' => [
                    'oea_button_hover_effect!' => ['hover-effect-1', 'hover-effect-4', 'hover-effect-5', 'hover-effect-9'],
                ],
            ]
        );

        $this->add_control(
            'oea_button_solid_bg_color',
            [
                'label' => __('Solid Color', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'condition' => [
                    'oea_button_bg_type' => 'color',
                    'oea_button_hover_effect!' => ['hover-effect-1', 'hover-effect-4', 'hover-effect-5', 'hover-effect-9'],
                ],
                'selectors' => [
                    '{{WRAPPER}} .oea-button-link' => '--background-color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Background::get_type(),
            [
                'name' => 'oea_button_gradient_bg_color',
                'types' => ['gradient'],
                'condition' => [
                    'oea_button_bg_type' => 'gradient',
                    'oea_button_hover_effect!' => ['hover-effect-1', 'hover-effect-4', 'hover-effect-5', 'hover-effect-9'],
                ],
                'selector' => '{{WRAPPER}} .oea-button-link',
                'fields_options' => [
                    'background' => [
                        'label' => __('Gradient Background', 'orange-addons-for-elementor'),
                    ],
                ],
            ]
        );

        $this->add_control(
            'oea_button_text_color',
            [
                'label' => __('Text Color', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .oea-button-link span' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Typography::get_type(),
            [
                'name' => 'oea_button_typography',
                'label' => __('Typography', 'orange-addons-for-elementor'),
                'selector' => '{{WRAPPER}} .oea-button-link',
            ]
        );

        $this->add_control(
            'oea_button_padding',
            [
                'label' => __('Padding', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%', 'em'],
                'selectors' => [
                    '{{WRAPPER}} .oea-button-link' => '--padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'oea_button_border_color',
            [
                'label' => __('Border Color', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .oea-button-link' => '--border-color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'oea_button_border_width',
            [
                'label' => __('Border Width', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 20,
                        'step' => 1,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .oea-button-link' => '--border-width: {{SIZE}}{{UNIT}};',
                ],
                'condition' => [
                    'oea_button_hover_effect!' => 'hover-effect-9',
                ],
            ]
        );

        $this->add_control(
            'oea_button_before_after_border_width',
            [
                'label' => __('Border Width (Decorative Elements)', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::SLIDER,
                'selectors' => [
                    '{{WRAPPER}} i::before, {{WRAPPER}} i::after' => 'border-width: {{SIZE}}{{UNIT}};',
                ],
                'condition' => [
                    'oea_button_hover_effect' => 'hover-effect-2',
                ],
            ]
        );

        $this->add_control(
            'oea_button_before_after_border_radius',
            [
                'label' => __('Border Radius (Decorative Elements)', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::SLIDER,
                'selectors' => [
                    '{{WRAPPER}} i::before, {{WRAPPER}} i::after' => 'border-radius: {{SIZE}}{{UNIT}};',
                ],
                'condition' => [
                    'oea_button_hover_effect' => 'hover-effect-2',
                ],
            ]
        );

        $this->add_control(
            'oea_button_border_style',
            [
                'label' => __('Border Style', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::SELECT,
                'options' => [
                    'solid' => __('Solid', 'orange-addons-for-elementor'),
                    'dashed' => __('Dashed', 'orange-addons-for-elementor'),
                    'dotted' => __('Dotted', 'orange-addons-for-elementor'),
                    'double' => __('Double', 'orange-addons-for-elementor'),
                    'groove' => __('Groove', 'orange-addons-for-elementor'),
                    'ridge' => __('Ridge', 'orange-addons-for-elementor'),
                    'inset' => __('Inset', 'orange-addons-for-elementor'),
                    'outset' => __('Outset', 'orange-addons-for-elementor'),
                    'none' => __('None', 'orange-addons-for-elementor'),
                ],
                'default' => 'solid',
                'selectors' => [
                    '{{WRAPPER}} .oea-button-link' => '--border-style: {{VALUE}};',
                ],
                'condition' => [
                    'oea_button_hover_effect!' => ['hover-effect-1', 'hover-effect-4'],
                ],
            ]
        );

        $this->add_control(
            'oea_button_gradient_color_1',
            [
                'label' => __('Gradient Color 1', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#00ccff',
                'condition' => [
                    'oea_button_hover_effect' => 'hover-effect-4',
                ],
                'selectors' => [
                    '{{WRAPPER}} .oea-button-link.hover-effect-4 i' => '--gradient-color-1: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'oea_button_gradient_color_2',
            [
                'label' => __('Gradient Color 2', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#0e1538',
                'condition' => [
                    'oea_button_hover_effect' => 'hover-effect-4',
                ],
                'selectors' => [
                    '{{WRAPPER}} .oea-button-link.hover-effect-4 i' => '--gradient-color-2: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'oea_button_gradient_color_3',
            [
                'label' => __('Gradient Color 3', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#0e1538',
                'condition' => [
                    'oea_button_hover_effect' => 'hover-effect-4',
                ],
                'selectors' => [
                    '{{WRAPPER}} .oea-button-link.hover-effect-4 i' => '--gradient-color-3: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'oea_button_gradient_color_4',
            [
                'label' => __('Gradient Color 4', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#d400d4',
                'condition' => [
                    'oea_button_hover_effect' => 'hover-effect-4',
                ],
                'selectors' => [
                    '{{WRAPPER}} .oea-button-link.hover-effect-4 i' => '--gradient-color-4: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'oea_button_border_radius',
            [
                'label' => __('Border Radius', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'selectors' => [
                    '{{WRAPPER}} .oea-button-link' =>
                        '--border-radius-top-left: {{TOP}}{{UNIT}}; ' .
                        '--border-radius-top-right: {{RIGHT}}{{UNIT}}; ' .
                        '--border-radius-bottom-right: {{BOTTOM}}{{UNIT}}; ' .
                        '--border-radius-bottom-left: {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Box_Shadow::get_type(),
            [
                'name' => 'oea_button_box_shadow',
                'label' => __('Box Shadow', 'orange-addons-for-elementor'),
                'selector' => '{{WRAPPER}} .oea-button-link',
            ]
        );

        $this->end_controls_tab();

        $this->start_controls_tab(
            'oea_button_hover',
            [
                'label' => __('Hover', 'orange-addons-for-elementor'),
            ]
        );

        $this->add_control(
            'oea_button_bg_hover_color',
            [
                'label' => __('Background Color', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .oea-button-link:hover' => '--background-hover-color: {{VALUE}};',
                ],
                'condition' => [
                    'oea_button_hover_effect' => ['hover-effect-1', 'hover-effect-4', 'hover-effect-5', 'hover-effect-9'],
                ],
            ]
        );

        $this->add_control(
            'oea_button_bg_hover_type',
            [
                'label' => __('Background Type', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::SELECT,
                'default' => 'color',
                'options' => [
                    'color' => __('Solid Color', 'orange-addons-for-elementor'),
                    'gradient' => __('Gradient', 'orange-addons-for-elementor'),
                ],
                'condition' => [
                    'oea_button_hover_effect!' => ['hover-effect-1', 'hover-effect-4', 'hover-effect-5', 'hover-effect-9'],
                ],
            ]
        );

        $this->add_control(
            'oea_button_solid_bg_hover_color',
            [
                'label' => __('Solid Color', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'condition' => [
                    'oea_button_bg_hover_type' => 'color',
                    'oea_button_hover_effect!' => ['hover-effect-1', 'hover-effect-4', 'hover-effect-5', 'hover-effect-9'],
                ],
                'selectors' => [
                    '{{WRAPPER}} .oea-button-link' => '--background-hover-color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Background::get_type(),
            [
                'name' => 'oea_button_gradient_bg_hover_color',
                'types' => ['gradient'],
                'condition' => [
                    'oea_button_bg_hover_type' => 'gradient',
                    'oea_button_hover_effect!' => ['hover-effect-1', 'hover-effect-4', 'hover-effect-5', 'hover-effect-9'],
                ],
                'selector' => '{{WRAPPER}} .oea-button-link:hover',
                'fields_options' => [
                    'background' => [
                        'label' => __('Gradient Background', 'orange-addons-for-elementor'),
                    ],
                ],
            ]
        );

        $this->add_control(
            'oea_button_text_hover_color',
            [
                'label' => __('Text Color', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .oea-button-link:hover span' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Typography::get_type(),
            [
                'name' => 'oea_button_typography_hover',
                'label' => __('Typography', 'orange-addons-for-elementor'),
                'selector' => '{{WRAPPER}} .oea-button-link:hover',
            ]
        );

        $this->add_control(
            'oea_text_shadow_color',
            [
                'label' => esc_html__('Text Shadow Color', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .oea-button-link:hover' => '--text-shadow-hover-color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'oea_button_padding_hover',
            [
                'label' => __('Padding', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%', 'em'],
                'selectors' => [
                    '{{WRAPPER}} .oea-button-link:hover' => '--padding-hover: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'oea_button_border_hover_color',
            [
                'label' => __('Border Color', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .oea-button-link:hover' => '--border-hover-color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'oea_button_border_hover_width',
            [
                'label' => __('Border Width', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 20,
                        'step' => 1,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .oea-button-link:hover' => '--border-hover-width: {{SIZE}}{{UNIT}};',
                ],
                'condition' => [
                    'oea_button_hover_effect!' => 'hover-effect-9',
                ],
            ]
        );


        $this->add_control(
            'oea_button_border_hover_style',
            [
                'label' => __('Border Style', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::SELECT,
                'options' => [
                    'solid' => __('Solid', 'orange-addons-for-elementor'),
                    'dashed' => __('Dashed', 'orange-addons-for-elementor'),
                    'dotted' => __('Dotted', 'orange-addons-for-elementor'),
                    'double' => __('Double', 'orange-addons-for-elementor'),
                    'groove' => __('Groove', 'orange-addons-for-elementor'),
                    'ridge' => __('Ridge', 'orange-addons-for-elementor'),
                    'inset' => __('Inset', 'orange-addons-for-elementor'),
                    'outset' => __('Outset', 'orange-addons-for-elementor'),
                    'none' => __('None', 'orange-addons-for-elementor'),
                ],
                'default' => 'solid',
                'selectors' => [
                    '{{WRAPPER}} .oea-button-link:hover' => '--border-hover-style: {{VALUE}};',
                ],
                'condition' => [
                    'oea_button_hover_effect!' => ['hover-effect-1', 'hover-effect-4'],
                ],
            ]
        );

        $this->add_control(
            'oea_button_border_hover_radius',
            [
                'label' => __('Border Hover Radius', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'selectors' => [
                    '{{WRAPPER}} .oea-button-link:hover' =>
                        '--border-radius-top-left-hover: {{TOP}}{{UNIT}}; ' .
                        '--border-radius-top-right-hover: {{RIGHT}}{{UNIT}}; ' .
                        '--border-radius-bottom-right-hover: {{BOTTOM}}{{UNIT}}; ' .
                        '--border-radius-bottom-left-hover: {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Box_Shadow::get_type(),
            [
                'name' => 'oea_button_hover_box_shadow',
                'label' => __('Box Shadow', 'orange-addons-for-elementor'),
                'selector' => '{{WRAPPER}} .oea-button-link:hover',
            ]
        );

        $this->end_controls_tab();
        $this->end_controls_tabs();

        $this->add_control(
            'wave_color',
            [
                'label' => esc_html__('Wave Color', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#00aeff',
                'selectors' => [
                    '{{WRAPPER}} .wave' => '--wave-color: {{VALUE}};',
                ],
                'condition' => [
                    'oea_button_hover_effect' => 'hover-effect-3',
                ],
            ]
        );


        $this->add_control(
            'oea_button_transition_duration',
            [
                'label' => __('Transition Duration', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::SLIDER,
                'size_units' => ['s', 'ms'],
                'range' => [
                    's' => [
                        'min' => 0,
                        'max' => 5,
                        'step' => 0.1,
                    ],
                    'ms' => [
                        'min' => 0,
                        'max' => 5000,
                        'step' => 100,
                    ],
                ],
                'default' => [
                    'unit' => 's',
                ],
                'selectors' => [
                    '{{WRAPPER}} .oea-button-link' => '--transition-duration: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'oea_button_transition_timing',
            [
                'label' => __('Transition Timing Function', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::SELECT,
                'options' => [
                    'ease' => __('Ease', 'orange-addons-for-elementor'),
                    'linear' => __('Linear', 'orange-addons-for-elementor'),
                    'ease-in' => __('Ease-In', 'orange-addons-for-elementor'),
                    'ease-out' => __('Ease-Out', 'orange-addons-for-elementor'),
                    'ease-in-out' => __('Ease-In-Out', 'orange-addons-for-elementor'),
                    'cubic-bezier(0.25, 1, 0.5, 1)' => __('Custom Cubic Bezier', 'orange-addons-for-elementor'),
                ],
                'default' => 'ease',
                'selectors' => [
                    '{{WRAPPER}} .oea-button-link' => '--transition-timing: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'oea_button_transition_delay',
            [
                'label' => __('Transition Delay', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::SLIDER,
                'size_units' => ['s', 'ms'],
                'range' => [
                    's' => [
                        'min' => 0,
                        'max' => 5,
                        'step' => 0.1,
                    ],
                    'ms' => [
                        'min' => 0,
                        'max' => 5000,
                        'step' => 100,
                    ],
                ],
                'default' => [
                    'unit' => 's',
                    'size' => 0,
                ],
                'selectors' => [
                    '{{WRAPPER}} .oea-button-link' => '--transition-delay: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'gradient_divider',
            [
                'type' => \Elementor\Controls_Manager::DIVIDER,
                'condition' => [
                    'oea_button_hover_effect' => ['hover-effect-5'],
                ],
            ]
        );

        $this->add_control(
            'gradient_heading',
            [
                'label' => __('Customize Gradient Colors', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::HEADING,
                'condition' => [
                    'oea_button_hover_effect' => ['hover-effect-5'],
                ],
            ]
        );

        $gradient_premium_notice = apply_filters(
            'oradel_button_gradient_effect_premium_notice',
            Oradel_Plugin_Helper::create_premium_notice(
                /* translators: %s: Premium upgrade URL */
                __('This gradient effect is available only in the <a href="%s" target="_blank">Premium version</a>. Settings below will not work without premium license.', 'orange-addons-for-elementor'),
                'button',
                'warning'
            )
        );

        if ($gradient_premium_notice !== null) {
            $this->add_control(
                'gradient_premium_notice',
                array_merge($gradient_premium_notice, [
                    'condition' => [
                        'oea_button_hover_effect' => ['hover-effect-5'],
                    ],
                ])
            );
        }

        $this->add_control(
            'gradient_color_1',
            [
                'label' => __('Color 1', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#ff0000',
                'selectors' => [
                    '{{WRAPPER}} .hover-effect-5' => '--bg-gradient-color-1: {{VALUE}};',
                ],
                'condition' => [
                    'oea_button_hover_effect' => ['hover-effect-5'],
                ],
            ]
        );

        $this->add_control(
            'gradient_color_2',
            [
                'label' => __('Color 2', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#ff7300',
                'selectors' => [
                    '{{WRAPPER}} .hover-effect-5' => '--bg-gradient-color-2: {{VALUE}};',
                ],
                'condition' => [
                    'oea_button_hover_effect' => ['hover-effect-5'],
                ],
            ]
        );

        $this->add_control(
            'gradient_color_3',
            [
                'label' => __('Color 3', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#fffb00',
                'selectors' => [
                    '{{WRAPPER}} .hover-effect-5' => '--bg-gradient-color-3: {{VALUE}};',
                ],
                'condition' => [
                    'oea_button_hover_effect' => ['hover-effect-5'],
                ],
            ]
        );

        $this->add_control(
            'gradient_color_4',
            [
                'label' => __('Color 4', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#48ff00',
                'selectors' => [
                    '{{WRAPPER}} .hover-effect-5' => '--bg-gradient-color-4: {{VALUE}};',
                ],
                'condition' => [
                    'oea_button_hover_effect' => ['hover-effect-5'],
                ],
            ]
        );

        $this->add_control(
            'gradient_color_5',
            [
                'label' => __('Color 5', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#00ffd5',
                'selectors' => [
                    '{{WRAPPER}} .hover-effect-5' => '--bg-gradient-color-5: {{VALUE}};',
                ],
                'condition' => [
                    'oea_button_hover_effect' => ['hover-effect-5'],
                ],
            ]
        );

        $this->add_control(
            'gradient_color_6',
            [
                'label' => __('Color 6', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#002bff',
                'selectors' => [
                    '{{WRAPPER}} .hover-effect-5' => '--bg-gradient-color-6: {{VALUE}};',
                ],
                'condition' => [
                    'oea_button_hover_effect' => ['hover-effect-5'],
                ],
            ]
        );

        $this->add_control(
            'gradient_color_7',
            [
                'label' => __('Color 7', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#ff00c8',
                'selectors' => [
                    '{{WRAPPER}} .hover-effect-5' => '--bg-gradient-color-7: {{VALUE}};',
                ],
                'condition' => [
                    'oea_button_hover_effect' => ['hover-effect-5'],
                ],
            ]
        );

        $this->add_control(
            'gradient_color_8',
            [
                'label' => __('Color 8', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#ff0000',
                'selectors' => [
                    '{{WRAPPER}} .hover-effect-5' => '--bg-gradient-color-8: {{VALUE}};',
                ],
                'condition' => [
                    'oea_button_hover_effect' => ['hover-effect-5'],
                ],
            ]
        );

        $this->end_controls_section();
    }

    protected function render()
    {
        $settings = $this->get_settings_for_display();
        $button_hover_effect = !empty($settings['oea_button_hover_effect']) ? $settings['oea_button_hover_effect'] : '';
        $button_click_effect = !empty($settings['oea_button_click_effect']) ? $settings['oea_button_click_effect'] : '';

        $premium_effects = ['hover-effect-5', 'hover-effect-6', 'hover-effect-7', 'hover-effect-8', 'hover-effect-9'];

        $button_hover_effect = apply_filters(
            'oradel_enable_button_hover_effect_render',
            $button_hover_effect,
            $button_hover_effect,
            $settings,
            $premium_effects
        );

        $button_classes = array_filter([
            'oea-button-link',
            $button_hover_effect,
            $button_click_effect
        ]);

        if (!empty($settings['button_link']['url'])) {
            $this->add_link_attributes('button_link', $settings['button_link']);
        }

        ?>
        <div class="oea-button-wrapper">
            <a href="<?php echo esc_url($settings['button_link']['url']); ?>"
                class="<?php echo esc_attr(implode(' ', $button_classes)); ?>" <?php $this->print_render_attribute_string('button_link'); ?>><span><?php echo esc_attr($settings['oea_button']); ?></span><i></i><i></i>
                <div class="wave" style="display:none;"></div>
            </a>
        </div>
        <?php

    }

}