<?php
if (!defined('ABSPATH')) {
    exit;
}

require_once ORADEL_PATH . 'includes/helpers.php';

class Oradel_Benefits_Widget extends \Elementor\Widget_Base
{
    public static function get_premium_options_for_styling()
    {
        return [
            'card_hover_transform' => [
                'rotate_neg_5',
                'skewX_10',
                'skewY_10',
                'scale_1_15',
                'translateY_neg_15',
                'rotate_10',
                'rotate_neg_10',
                'scale_and_up',
                'scale_and_rotate',
                'tilt_effect',
                'bounce_effect',
                'pulse_effect',
            ],
        ];
    }

    public function get_name()
    {
        return 'oradel_benefits';
    }

    public function get_title()
    {
        return __('Benefits', 'orange-addons-for-elementor');
    }

    public function get_icon()
    {
        return 'eicon-check-circle';
    }

    public function get_categories()
    {
        return ['oradel-plugin'];
    }

    public function get_custom_help_url()
    {
        return 'https://orange-addons.com/docs/';
    }

    public function get_style_depends()
    {
        return ['oea-benefits'];
    }

    public function get_script_depends()
    {
        return ['oea-scroll-animations'];
    }

    private function get_premium_hover_transforms()
    {
        return [
            'rotate_neg_5' => __('Rotate -5deg (Premium)', 'orange-addons-for-elementor'),
            'skewX_10' => __('Skew X 10deg (Premium)', 'orange-addons-for-elementor'),
            'skewY_10' => __('Skew Y 10deg (Premium)', 'orange-addons-for-elementor'),
            'scale_1_15' => __('Scale 1.15 (Premium)', 'orange-addons-for-elementor'),
            'translateY_neg_15' => __('Move Up 15px (Premium)', 'orange-addons-for-elementor'),
            'rotate_10' => __('Rotate 10deg (Premium)', 'orange-addons-for-elementor'),
            'rotate_neg_10' => __('Rotate -10deg (Premium)', 'orange-addons-for-elementor'),
            'scale_and_up' => __('Scale and Move Up (Premium)', 'orange-addons-for-elementor'),
            'scale_and_rotate' => __('Scale and Rotate (Premium)', 'orange-addons-for-elementor'),
            'tilt_effect' => __('Tilt Effect (Premium)', 'orange-addons-for-elementor'),
            'bounce_effect' => __('Bounce Effect (Premium)', 'orange-addons-for-elementor'),
            'pulse_effect' => __('Pulse Effect (Premium)', 'orange-addons-for-elementor'),
        ];
    }

    protected function register_controls()
    {

        $this->start_controls_section(
            'content_section',
            [
                'label' => __('Content', 'orange-addons-for-elementor'),
                'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'enable_links',
            [
                'label' => __('Enable Links', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'label_on' => __('Yes', 'orange-addons-for-elementor'),
                'label_off' => __('No', 'orange-addons-for-elementor'),
                'return_value' => 'yes',
                'default' => 'no',
            ]
        );

        $this->add_control(
            'show_title',
            [
                'label' => __('Show Title', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'default' => 'yes',
            ]
        );

        $this->add_control(
            'title',
            [
                'label' => __('Title', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::TEXT,
                'default' => __('Why Orange Addons?', 'orange-addons-for-elementor'),
                'condition' => [
                    'show_title' => 'yes',
                ],
            ]
        );

        $this->add_control(
            'columns',
            [
                'label' => __('Columns', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::NUMBER,
                'default' => 3,
                'min' => 1,
                'max' => 6,
            ]
        );

        $this->add_control(
            'mobile_columns',
            [
                'label' => __('Mobile Columns', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::SELECT,
                'default' => '2',
                'options' => [
                    '1' => __('1 Column', 'orange-addons-for-elementor'),
                    '2' => __('2 Columns', 'orange-addons-for-elementor'),
                ],
                'description' => __('Select number of columns for mobile devices (screen width <= 768px)', 'orange-addons-for-elementor'),
            ]
        );

        $repeater = new \Elementor\Repeater();

        $repeater->add_control(
            'icon',
            [
                'label' => __('Icon', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::ICONS,
                'default' => [
                    'value' => 'fas fa-lightbulb',
                    'library' => 'fa-solid',
                ],
            ]
        );

        $repeater->add_control(
            'benefit_title',
            [
                'label' => __('Benefit Title', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::TEXT,
                'default' => __('Vivid Design', 'orange-addons-for-elementor'),
            ]
        );

        $repeater->add_control(
            'benefit_description',
            [
                'label' => __('Description', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::TEXTAREA,
                'default' => __('Bright, eye-catching widgets', 'orange-addons-for-elementor'),
            ]
        );

        $repeater->add_control(
            'benefit_link',
            [
                'label' => __('Link', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::URL,
                'description' => __('Note: Links will only work if "Enable Links" option is turned on in the main settings', 'orange-addons-for-elementor'),
                'placeholder' => __('https://your-link.com', 'orange-addons-for-elementor'),
                'show_external' => true,
                'default' => [
                    'url' => '',
                    'is_external' => true,
                    'nofollow' => true,
                ],
            ]
        );

        $this->add_control(
            'benefits',
            [
                'label' => __('Benefits', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::REPEATER,
                'fields' => $repeater->get_controls(),
                'default' => [
                    [
                        'icon' => ['value' => 'fas fa-lightbulb', 'library' => 'fa-solid'],
                        'benefit_title' => __('Vivid Design', 'orange-addons-for-elementor'),
                        'benefit_description' => __('Bright, eye-catching widgets', 'orange-addons-for-elementor'),
                    ],
                    [
                        'icon' => ['value' => 'fas fa-bolt', 'library' => 'fa-solid'],
                        'benefit_title' => __('Fast Performance', 'orange-addons-for-elementor'),
                        'benefit_description' => __('Lightweight and quick', 'orange-addons-for-elementor'),
                    ],
                    [
                        'icon' => ['value' => 'fas fa-plug', 'library' => 'fa-solid'],
                        'benefit_title' => __('Seamless Fit', 'orange-addons-for-elementor'),
                        'benefit_description' => __('Perfect for Elementor', 'orange-addons-for-elementor'),
                    ],
                ],
                'title_field' => '{{{ benefit_title }}}',
            ]
        );

        $this->add_control(
            'show_icons',
            [
                'label' => __('Show Icons', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'default' => 'yes',
            ]
        );

        $this->add_control(
            'show_benefit_titles',
            [
                'label' => __('Show Benefit Titles', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'default' => 'yes',
            ]
        );

        $this->add_control(
            'show_descriptions',
            [
                'label' => __('Show Descriptions', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'default' => 'yes',
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            'section_style',
            [
                'label' => __('Section', 'orange-addons-for-elementor'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            'background_color',
            [
                'label' => __('Background Color', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#F9FAFB',
                'selectors' => [
                    '{{WRAPPER}} .benefit-section' => 'background-color: {{VALUE}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'section_padding',
            [
                'label' => __('Section Padding', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'default' => [
                    'top' => '80',
                    'right' => '0',
                    'bottom' => '80',
                    'left' => '0',
                    'unit' => 'px',
                    'isLinked' => false,
                ],
                'selectors' => [
                    '{{WRAPPER}} .benefit-section' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'section_margin',
            [
                'label' => __('Section Margin', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors' => [
                    '{{WRAPPER}} .benefit-section' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_section();

        // Grid Layout
        $this->start_controls_section(
            'grid_style',
            [
                'label' => __('Grid Layout', 'orange-addons-for-elementor'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_responsive_control(
            'grid_gap',
            [
                'label' => __('Grid Gap', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::SLIDER,
                'size_units' => ['px', 'em'],
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 100,
                        'step' => 1,
                    ],
                    'em' => [
                        'min' => 0,
                        'max' => 10,
                        'step' => 0.1,
                    ],
                ],
                'default' => [
                    'unit' => 'px',
                    'size' => 30,
                ],
                'selectors' => [
                    '{{WRAPPER}} .benefit-grid' => 'gap: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_section();

        // Section Title
        $this->start_controls_section(
            'title_style',
            [
                'label' => __('Section Title', 'orange-addons-for-elementor'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
                'condition' => [
                    'show_title' => 'yes',
                ],
            ]
        );

        $this->add_responsive_control(
            'title_margin_bottom',
            [
                'label' => __('Bottom Margin', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::SLIDER,
                'size_units' => ['px', 'em'],
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 100,
                        'step' => 1,
                    ],
                    'em' => [
                        'min' => 0,
                        'max' => 10,
                        'step' => 0.1,
                    ],
                ],
                'default' => [
                    'unit' => 'px',
                    'size' => 50,
                ],
                'selectors' => [
                    '{{WRAPPER}} .benefit-section h2' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Typography::get_type(),
            [
                'name' => 'title_typography',
                'label' => __('Typography', 'orange-addons-for-elementor'),
                'selector' => '{{WRAPPER}} .benefit-section h2',
            ]
        );

        $this->add_control(
            'title_color',
            [
                'label' => __('Text Color', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#ff8800',
                'selectors' => [
                    '{{WRAPPER}} .benefit-section h2' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'title_alignment',
            [
                'label' => __('Alignment', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::CHOOSE,
                'options' => [
                    'left' => [
                        'title' => __('Left', 'orange-addons-for-elementor'),
                        'icon' => 'eicon-text-align-left',
                    ],
                    'center' => [
                        'title' => __('Center', 'orange-addons-for-elementor'),
                        'icon' => 'eicon-text-align-center',
                    ],
                    'right' => [
                        'title' => __('Right', 'orange-addons-for-elementor'),
                        'icon' => 'eicon-text-align-right',
                    ],
                ],
                'default' => 'center',
                'selectors' => [
                    '{{WRAPPER}} .benefit-section h2' => 'text-align: {{VALUE}};',
                ],
            ]
        );

        $this->end_controls_section();

        // Card Style
        $this->start_controls_section(
            'card_style',
            [
                'label' => __('Card', 'orange-addons-for-elementor'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_responsive_control(
            'card_padding',
            [
                'label' => __('Padding', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'default' => [
                    'top' => '30',
                    'right' => '30',
                    'bottom' => '30',
                    'left' => '30',
                    'unit' => 'px',
                    'isLinked' => true,
                ],
                'selectors' => [
                    '{{WRAPPER}} .benefit-card' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'card_alignment',
            [
                'label' => __('Content Alignment', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::CHOOSE,
                'options' => [
                    'left' => [
                        'title' => __('Left', 'orange-addons-for-elementor'),
                        'icon' => 'eicon-text-align-left',
                    ],
                    'center' => [
                        'title' => __('Center', 'orange-addons-for-elementor'),
                        'icon' => 'eicon-text-align-center',
                    ],
                    'right' => [
                        'title' => __('Right', 'orange-addons-for-elementor'),
                        'icon' => 'eicon-text-align-right',
                    ],
                ],
                'default' => 'center',
                'selectors' => [
                    '{{WRAPPER}} .benefit-card' => 'text-align: {{VALUE}};',
                ],
            ]
        );

        $this->start_controls_tabs('card_style_tabs');

        $this->start_controls_tab(
            'card_normal',
            [
                'label' => __('Normal', 'orange-addons-for-elementor'),
            ]
        );

        $this->add_control(
            'card_background',
            [
                'label' => __('Background Color', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#FFFFFF',
                'selectors' => [
                    '{{WRAPPER}} .benefit-card' => 'background-color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Border::get_type(),
            [
                'name' => 'card_border',
                'selector' => '{{WRAPPER}} .benefit-card',
            ]
        );

        $this->add_control(
            'card_border_radius',
            [
                'label' => __('Border Radius', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'default' => [
                    'top' => 10,
                    'right' => 10,
                    'bottom' => 10,
                    'left' => 10,
                    'unit' => 'px',
                ],
                'selectors' => [
                    '{{WRAPPER}} .benefit-card' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Box_Shadow::get_type(),
            [
                'name' => 'card_shadow',
                'selector' => '{{WRAPPER}} .benefit-card',
                'default' => [
                    'horizontal' => 0,
                    'vertical' => 2,
                    'blur' => 8,
                    'spread' => 0,
                    'color' => 'rgba(0,0,0,0.05)',
                ],
            ]
        );

        $this->end_controls_tab();

        $this->start_controls_tab(
            'card_hover',
            [
                'label' => __('Hover', 'orange-addons-for-elementor'),
            ]
        );

        $this->add_control(
            'card_background_hover',
            [
                'label' => __('Background Color', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .benefit-card:hover' => 'background-color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Border::get_type(),
            [
                'name' => 'card_border_hover',
                'selector' => '{{WRAPPER}} .benefit-card:hover',
            ]
        );

        $this->add_control(
            'card_border_radius_hover',
            [
                'label' => __('Border Radius', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'default' => [
                    'top' => 10,
                    'right' => 10,
                    'bottom' => 10,
                    'left' => 10,
                    'unit' => 'px',
                ],
                'selectors' => [
                    '{{WRAPPER}} .benefit-card:hover' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Box_Shadow::get_type(),
            [
                'name' => 'card_shadow_hover',
                'selector' => '{{WRAPPER}} .benefit-card:hover',
                'default' => [
                    'horizontal' => 0,
                    'vertical' => 0,
                    'blur' => 15,
                    'spread' => 0,
                    'color' => 'rgba(255,136,0,0.3)',
                ],
            ]
        );

        $hover_transform_options = [
            'none' => __('None', 'orange-addons-for-elementor'),
            'scale_1_05' => __('Scale 1.05', 'orange-addons-for-elementor'),
            'scale_1_1' => __('Scale 1.1', 'orange-addons-for-elementor'),
            'translateY_neg_5' => __('Move Up 5px', 'orange-addons-for-elementor'),
            'translateY_neg_10' => __('Move Up 10px', 'orange-addons-for-elementor'),
            'translateX_5' => __('Move Right 5px', 'orange-addons-for-elementor'),
            'translateX_neg_5' => __('Move Left 5px', 'orange-addons-for-elementor'),
            'rotate_5' => __('Rotate 5deg', 'orange-addons-for-elementor'),
        ] + $this->get_premium_hover_transforms();

        $hover_transform_options = apply_filters('oradel_benefit_hover_transform_options', $hover_transform_options);

        $this->add_control(
            'card_hover_transform',
            [
                'label' => __('Hover Animation', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::SELECT,
                'options' => $hover_transform_options,
                'default' => 'none',
                'prefix_class' => 'oea-benefit-hover-',
                'render_type' => 'template',
            ]
        );

        $premium_notice = apply_filters(
            'oradel_benefit_hover_transform_premium_notice',
            Oradel_Plugin_Helper::create_premium_notice(
                /* translators: %s: Premium upgrade URL */
                __('More hover animations are available in the <a href="%s" target="_blank">Premium version</a>', 'orange-addons-for-elementor'),
                'benefits'
            )
        );

        if ($premium_notice !== null) {
            $this->add_control('oea_premium_hover_notice', $premium_notice);
        }

        $this->end_controls_tab();
        $this->end_controls_tabs();

        $this->add_control(
            'card_load_animation',
            [
                'label' => __('Load Animation', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::SELECT,
                'options' => [
                    'none' => __('None', 'orange-addons-for-elementor'),
                    'fadeIn' => __('Fade In', 'orange-addons-for-elementor'),
                    'fadeInUp' => __('Fade In Up', 'orange-addons-for-elementor'),
                    'fadeInDown' => __('Fade In Down', 'orange-addons-for-elementor'),
                    'fadeInLeft' => __('Fade In Left', 'orange-addons-for-elementor'),
                    'fadeInRight' => __('Fade In Right', 'orange-addons-for-elementor'),
                    'scaleIn' => __('Scale In', 'orange-addons-for-elementor'),
                    'flipIn' => __('Flip In', 'orange-addons-for-elementor'),
                    'rotateIn' => __('Rotate In', 'orange-addons-for-elementor'),
                    'swingIn' => __('Swing In', 'orange-addons-for-elementor'),
                    'bounceIn' => __('Bounce In', 'orange-addons-for-elementor'),
                    'glideIn' => __('Glide In', 'orange-addons-for-elementor'),
                    'zoomInBlur' => __('Zoom In Blur', 'orange-addons-for-elementor'),
                ],
                'default' => 'fadeInUp',
                'prefix_class' => 'oea-benefit-load-anim-',
                'separator' => 'before',
            ]
        );

        $this->add_control(
            'card_load_animation_duration',
            [
                'label' => __('Animation Duration', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::SLIDER,
                'size_units' => ['s'],
                'range' => [
                    's' => [
                        'min' => 0.1,
                        'max' => 5,
                        'step' => 0.1,
                    ],
                ],
                'default' => [
                    'unit' => 's',
                    'size' => 0.6,
                ],
                'selectors' => [
                    '{{WRAPPER}} .benefit-card' => 'animation-duration: {{SIZE}}s;',
                ],
                'condition' => [
                    'card_load_animation!' => 'none',
                ],
            ]
        );

        $this->add_control(
            'card_load_animation_delay',
            [
                'label' => __('Animation Delay (Staggered)', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::SLIDER,
                'description' => __('Delay between each card animation starting.', 'orange-addons-for-elementor'),
                'size_units' => ['s'],
                'range' => [
                    's' => [
                        'min' => 0,
                        'max' => 2,
                        'step' => 0.05,
                    ],
                ],
                'default' => [
                    'unit' => 's',
                    'size' => 0.1,
                ],
                'condition' => [
                    'card_load_animation!' => 'none',
                ],
            ]
        );

        $this->add_control(
            'card_transition_duration',
            [
                'label' => __('Transition Duration', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::SLIDER,
                'size_units' => ['s'],
                'range' => [
                    's' => [
                        'min' => 0,
                        'max' => 5,
                        'step' => 0.1,
                    ],
                ],
                'default' => [
                    'unit' => 's',
                    'size' => 0.3,
                ],
                'selectors' => [
                    '{{WRAPPER}} .benefit-card' => 'transition: all {{SIZE}}s;',
                    '{{WRAPPER}} .benefit-card svg' => 'transition: all {{SIZE}}s;',
                    '{{WRAPPER}} .benefit-card h3' => 'transition: all {{SIZE}}s;',
                    '{{WRAPPER}} .benefit-card p' => 'transition: all {{SIZE}}s;',
                ],
                'separator' => 'before',
            ]
        );

        $this->add_control(
            'card_transition_delay',
            [
                'label' => __('Transition Delay', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::SLIDER,
                'size_units' => ['s'],
                'range' => [
                    's' => [
                        'min' => 0,
                        'max' => 5,
                        'step' => 0.1,
                    ],
                ],
                'default' => [
                    'unit' => 's',
                    'size' => 0,
                ],
                'selectors' => [
                    '{{WRAPPER}} .benefit-card' => 'transition-delay: {{SIZE}}s;',
                    '{{WRAPPER}} .benefit-card svg' => 'transition-delay: {{SIZE}}s;',
                    '{{WRAPPER}} .benefit-card h3' => 'transition-delay: {{SIZE}}s;',
                    '{{WRAPPER}} .benefit-card p' => 'transition-delay: {{SIZE}}s;',
                ],
            ]
        );

        $this->add_control(
            'animation_on_scroll',
            [
                'label' => __('Animate Only When Visible', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'label_on' => __('Yes', 'orange-addons-for-elementor'),
                'label_off' => __('No', 'orange-addons-for-elementor'),
                'default' => 'yes',
                'description' => __('Animation will start only when element enters the viewport.', 'orange-addons-for-elementor'),
                'condition' => [
                    'card_load_animation!' => 'none',
                ],
                'separator' => 'before',
            ]
        );

        $this->add_control(
            'disable_animation_mobile',
            [
                'label' => __('Disable Animation on Mobile', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'label_on' => __('Yes', 'orange-addons-for-elementor'),
                'label_off' => __('No', 'orange-addons-for-elementor'),
                'default' => 'yes',
                'description' => __('Turn off animation effects on mobile devices for better performance.', 'orange-addons-for-elementor'),
                'condition' => [
                    'card_load_animation!' => 'none',
                ],
            ]
        );

        $this->end_controls_section();

        // Icon Style
        $this->start_controls_section(
            'icon_style',
            [
                'label' => __('Icon', 'orange-addons-for-elementor'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
                'condition' => [
                    'show_icons' => 'yes',
                ],
            ]
        );

        $this->start_controls_tabs('icon_style_tabs');

        $this->start_controls_tab(
            'icon_normal_tab',
            [
                'label' => __('Normal', 'orange-addons-for-elementor'),
            ]
        );

        $this->add_control(
            'icon_color',
            [
                'label' => __('Icon Color', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => 'var(--main-color)',
                'selectors' => [
                    '{{WRAPPER}} .benefit-card i' => 'color: {{VALUE}};',
                    '{{WRAPPER}} .benefit-card svg' => 'fill: {{VALUE}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'icon_size',
            [
                'label' => __('Size', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::SLIDER,
                'size_units' => ['px', 'em', 'rem'],
                'range' => [
                    'px' => [
                        'min' => 10,
                        'max' => 100,
                        'step' => 1,
                    ],
                    'em' => [
                        'min' => 0.5,
                        'max' => 5,
                        'step' => 0.1,
                    ],
                    'rem' => [
                        'min' => 0.5,
                        'max' => 5,
                        'step' => 0.1,
                    ],
                ],
                'default' => [
                    'unit' => 'px',
                    'size' => 40,
                ],
                'selectors' => [
                    '{{WRAPPER}} .benefit-card i' => 'font-size: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} .benefit-card svg' => 'width: {{SIZE}}{{UNIT}}; height: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'icon_margin_bottom',
            [
                'label' => __('Bottom Margin', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::SLIDER,
                'size_units' => ['px', 'em'],
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 50,
                        'step' => 1,
                    ],
                    'em' => [
                        'min' => 0,
                        'max' => 5,
                        'step' => 0.1,
                    ],
                ],
                'default' => [
                    'unit' => 'px',
                    'size' => 20,
                ],
                'selectors' => [
                    '{{WRAPPER}} .benefit-card i, {{WRAPPER}} .benefit-card svg' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_tab();

        $this->start_controls_tab(
            'icon_hover_tab',
            [
                'label' => __('Hover', 'orange-addons-for-elementor'),
            ]
        );

        $this->add_control(
            'icon_color_hover',
            [
                'label' => __('Icon Color', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .benefit-card:hover i' => 'color: {{VALUE}};',
                    '{{WRAPPER}} .benefit-card:hover svg' => 'fill: {{VALUE}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'icon_size_hover',
            [
                'label' => __('Size', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::SLIDER,
                'size_units' => ['px', 'em', 'rem'],
                'range' => [
                    'px' => [
                        'min' => 10,
                        'max' => 100,
                        'step' => 1,
                    ],
                    'em' => [
                        'min' => 0.5,
                        'max' => 5,
                        'step' => 0.1,
                    ],
                    'rem' => [
                        'min' => 0.5,
                        'max' => 5,
                        'step' => 0.1,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .benefit-card:hover i' => 'font-size: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} .benefit-card:hover svg' => 'width: {{SIZE}}{{UNIT}}; height: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'icon_margin_bottom_hover',
            [
                'label' => __('Bottom Margin', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::SLIDER,
                'size_units' => ['px', 'em'],
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 50,
                        'step' => 1,
                    ],
                    'em' => [
                        'min' => 0,
                        'max' => 5,
                        'step' => 0.1,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .benefit-card:hover i, {{WRAPPER}} .benefit-card:hover svg' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_tab();
        $this->end_controls_tabs();

        $this->end_controls_section();

        $this->start_controls_section(
            'benefit_title_style',
            [
                'label' => __('Benefit Title', 'orange-addons-for-elementor'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
                'condition' => [
                    'show_benefit_titles' => 'yes',
                ],
            ]
        );

        $this->start_controls_tabs('benefit_title_tabs');

        $this->start_controls_tab(
            'benefit_title_normal_tab',
            [
                'label' => __('Normal', 'orange-addons-for-elementor'),
            ]
        );

        $this->add_responsive_control(
            'benefit_title_margin',
            [
                'label' => __('Margin', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'default' => [
                    'top' => '0',
                    'right' => '0',
                    'bottom' => '15',
                    'left' => '0',
                    'unit' => 'px',
                    'isLinked' => false,
                ],
                'selectors' => [
                    '{{WRAPPER}} .benefit-card h3' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Typography::get_type(),
            [
                'name' => 'benefit_title_typography',
                'label' => __('Typography', 'orange-addons-for-elementor'),
                'selector' => '{{WRAPPER}} .benefit-card h3',
            ]
        );

        $this->add_control(
            'benefit_title_color',
            [
                'label' => __('Text Color', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#1F2937',
                'selectors' => [
                    '{{WRAPPER}} .benefit-card h3' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->end_controls_tab();

        $this->start_controls_tab(
            'benefit_title_hover_tab',
            [
                'label' => __('Hover', 'orange-addons-for-elementor'),
            ]
        );

        $this->add_responsive_control(
            'benefit_title_margin_hover',
            [
                'label' => __('Margin', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors' => [
                    '{{WRAPPER}} .benefit-card:hover h3' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Typography::get_type(),
            [
                'name' => 'benefit_title_typography_hover',
                'label' => __('Typography', 'orange-addons-for-elementor'),
                'selector' => '{{WRAPPER}} .benefit-card:hover h3',
            ]
        );

        $this->add_control(
            'benefit_title_color_hover',
            [
                'label' => __('Text Color', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .benefit-card:hover h3' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->end_controls_tab();
        $this->end_controls_tabs();

        $this->end_controls_section();

        $this->start_controls_section(
            'description_style',
            [
                'label' => __('Description', 'orange-addons-for-elementor'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
                'condition' => [
                    'show_descriptions' => 'yes',
                ],
            ]
        );

        $this->start_controls_tabs('description_tabs');

        $this->start_controls_tab(
            'description_normal_tab',
            [
                'label' => __('Normal', 'orange-addons-for-elementor'),
            ]
        );

        $this->add_responsive_control(
            'description_margin',
            [
                'label' => __('Margin', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'default' => [
                    'top' => '0',
                    'right' => '0',
                    'bottom' => '0',
                    'left' => '0',
                    'unit' => 'px',
                    'isLinked' => false,
                ],
                'selectors' => [
                    '{{WRAPPER}} .benefit-card p' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Typography::get_type(),
            [
                'name' => 'description_typography',
                'label' => __('Typography', 'orange-addons-for-elementor'),
                'selector' => '{{WRAPPER}} .benefit-card p',
            ]
        );

        $this->add_control(
            'description_color',
            [
                'label' => __('Text Color', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#1F2937',
                'selectors' => [
                    '{{WRAPPER}} .benefit-card p' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->end_controls_tab();

        $this->start_controls_tab(
            'description_hover_tab',
            [
                'label' => __('Hover', 'orange-addons-for-elementor'),
            ]
        );

        $this->add_responsive_control(
            'description_margin_hover',
            [
                'label' => __('Margin', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors' => [
                    '{{WRAPPER}} .benefit-card:hover p' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Typography::get_type(),
            [
                'name' => 'description_typography_hover',
                'label' => __('Typography', 'orange-addons-for-elementor'),
                'selector' => '{{WRAPPER}} .benefit-card:hover p',
            ]
        );

        $this->add_control(
            'description_color_hover',
            [
                'label' => __('Text Color', 'orange-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .benefit-card:hover p' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->end_controls_tab();
        $this->end_controls_tabs();

        $this->end_controls_section();
    }

    protected function render()
    {
        $settings = $this->get_settings_for_display();
        $mobile_columns = isset($settings['mobile_columns']) ? intval($settings['mobile_columns']) : 2;

        $grid_classes = 'benefit-grid';
        if (!empty($settings['animation_on_scroll']) && $settings['animation_on_scroll'] === 'yes' && !empty($settings['card_load_animation']) && $settings['card_load_animation'] !== 'none') {
            $grid_classes .= ' oea-animate-on-scroll';
        }
        if (!empty($settings['disable_animation_mobile']) && $settings['disable_animation_mobile'] === 'yes') {
            $grid_classes .= ' oea-disable-animation-mobile';
        }
        ?>
        <section class="benefit-section">
            <div class="container">
                <?php if ($settings['show_title'] === 'yes'): ?>
                    <h2><?php echo esc_html($settings['title']); ?></h2>
                <?php endif; ?>
                <div class="<?php echo esc_attr($grid_classes); ?>"
                    style="grid-template-columns: repeat(<?php echo esc_attr($settings['columns']); ?>, 1fr);"
                    data-mobile-columns="<?php echo esc_attr($mobile_columns); ?>">
                    <?php
                    foreach ($settings['benefits'] as $index => $benefit):
                        $has_link = $settings['enable_links'] === 'yes' && !empty($benefit['benefit_link']['url']);
                        $link_attrs = '';
                        if ($has_link) {
                            $link_attrs = 'href="' . esc_url($benefit['benefit_link']['url']) . '"';
                            if ($benefit['benefit_link']['is_external']) {
                                $link_attrs .= ' target="_blank"';
                            }
                            if ($benefit['benefit_link']['nofollow']) {
                                $link_attrs .= ' rel="nofollow"';
                            }
                        }

                        $animation_delay_value = 0;
                        if (!empty($settings['card_load_animation']) && $settings['card_load_animation'] !== 'none' && !empty($settings['card_load_animation_delay']['size'])) {
                            $base_delay = floatval($settings['card_load_animation_delay']['size']);
                            $animation_delay_value = $index * $base_delay;
                        }
                        $animation_delay_style = ($animation_delay_value > 0) ? 'animation-delay: ' . esc_attr($animation_delay_value) . 's;' : '';

                        ?>
                        <?php if ($has_link): ?>
                            <a <?php echo wp_kses_post($link_attrs); ?>>
                            <?php endif; ?>
                            <div class="benefit-card" style="<?php echo esc_attr($animation_delay_style); ?>">
                                <?php if ($settings['show_icons'] === 'yes'): ?>
                                    <?php \Elementor\Icons_Manager::render_icon($benefit['icon'], ['aria-hidden' => 'true']); ?>
                                <?php endif; ?>
                                <?php if ($settings['show_benefit_titles'] === 'yes'): ?>
                                    <h3><?php echo esc_html($benefit['benefit_title']); ?></h3>
                                <?php endif; ?>
                                <?php if ($settings['show_descriptions'] === 'yes'): ?>
                                    <p><?php echo esc_html($benefit['benefit_description']); ?></p>
                                <?php endif; ?>
                            </div>
                            <?php if ($has_link): ?>
                            </a>
                        <?php endif; ?>
                        <?php
                    endforeach;
                    ?>
                </div>
            </div>
        </section>
        <?php
    }
}