<?php

namespace Oradel_Plugin;

if (!defined('ABSPATH')) {
    exit;
}

final class Oradel_Plugin
{
    const VERSION = '1.0.0';
    const CSS_VERSION = '1.0.72';
    const JS_VERSION = '1.0.72';
    const MINIMUM_ELEMENTOR_VERSION = '3.21.0';
    const MINIMUM_PHP_VERSION = '7.4';
    private static $_instance = null;

    public static function instance()
    {
        if (is_null(self::$_instance)) {
            self::$_instance = new self();
        }
        return self::$_instance;
    }

    public function __construct()
    {
        if ($this->is_compatible()) {
            add_action('elementor/init', [$this, 'init']);
        }
    }

    public function is_compatible()
    {
        if (!did_action('elementor/loaded')) {
            add_action('admin_notices', [$this, 'admin_notice_missing_main_plugin']);
            return false;
        }
        if (!version_compare(ELEMENTOR_VERSION, self::MINIMUM_ELEMENTOR_VERSION, '>=')) {
            add_action('admin_notices', [$this, 'admin_notice_minimum_elementor_version']);
            return false;
        }
        if (version_compare(PHP_VERSION, self::MINIMUM_PHP_VERSION, '<')) {
            add_action('admin_notices', [$this, 'admin_notice_minimum_php_version']);
            return false;
        }
        return true;
    }

    public function admin_notice_missing_main_plugin()
    {
        if (isset($_GET['activate'])) // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Reading activation URL parameter, not processing form data
            unset($_GET['activate']); // phpcs:ignore WordPress.Security.NonceVerification.Recommended
        $message = sprintf(
            /* translators: %1$s: Plugin name, %2$s: Required plugin name */
            esc_html__('" %1$s" requires "%2$s" to be installed and activated.', 'orange-addons-for-elementor'),
            '<strong>' . esc_html__('Orange Addons for Elementor', 'orange-addons-for-elementor') . '</strong>',
            '<strong>' . esc_html__('Elementor', 'orange-addons-for-elementor') . '</strong>'
        );
        printf('<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', wp_kses_post($message));
    }

    public function admin_notice_minimum_elementor_version()
    {
        if (isset($_GET['activate'])) // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Reading activation URL parameter, not processing form data
            unset($_GET['activate']); // phpcs:ignore WordPress.Security.NonceVerification.Recommended
        $message = sprintf(
            /* translators: %1$s: Plugin name, %2$s: Required plugin name, %3$s: Required version number */
            esc_html__('" %1$s" requires "%2$s" version %3$s or greater.', 'orange-addons-for-elementor'),
            '<strong>' . esc_html__('Orange Addons for Elementor', 'orange-addons-for-elementor') . '</strong>',
            '<strong>' . esc_html__('Elementor', 'orange-addons-for-elementor') . '</strong>',
            self::MINIMUM_ELEMENTOR_VERSION
        );
        printf('<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', wp_kses_post($message));
    }

    public function admin_notice_minimum_php_version()
    {
        if (isset($_GET['activate'])) // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Reading activation URL parameter, not processing form data
            unset($_GET['activate']); // phpcs:ignore WordPress.Security.NonceVerification.Recommended
        $message = sprintf(
            /* translators: %1$s: Plugin name, %2$s: Required software name, %3$s: Required version number */
            esc_html__('" %1$s" requires "%2$s" version %3$s or greater.', 'orange-addons-for-elementor'),
            '<strong>' . esc_html__('Orange Addons for Elementor', 'orange-addons-for-elementor') . '</strong>',
            '<strong>' . esc_html__('PHP', 'orange-addons-for-elementor') . '</strong>',
            self::MINIMUM_PHP_VERSION
        );
        printf('<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', wp_kses_post($message));
    }

    public function init()
    {
        add_action('elementor/widgets/register', [$this, 'register_widgets']);
        add_action('elementor/controls/register', [$this, 'register_controls']);
        add_action('elementor/frontend/after_enqueue_styles', [$this, 'frontend_styles']);
        add_action('elementor/editor/after_enqueue_styles', [$this, 'editor_styles']);
        add_action('elementor/frontend/after_register_scripts', [$this, 'frontend_scripts']);
        add_action('elementor/elements/categories_registered', [$this, 'add_elementor_widget_categories']);
    }

    public function get_widgets()
    {
        return [
            'oradel-accordion' => [
                'title' => __('Accordion', 'orange-addons-for-elementor'),
                'class' => '\Oradel_Accordion_Widget',
            ],
            'oradel-image-accordion' => [
                'title' => __('Image Accordion', 'orange-addons-for-elementor'),
                'class' => '\Oradel_Image_Accordion_Widget',
            ],
            'oradel-offer-listing' => [
                'title' => __('Offer Listing', 'orange-addons-for-elementor'),
                'class' => '\Oradel_Offer_Listing_Widget',
            ],
            'oradel-pros-cons' => [
                'title' => __('Pros Cons', 'orange-addons-for-elementor'),
                'class' => '\Oradel_Pros_Cons_Widget',
            ],
            'oradel-verdict' => [
                'title' => __('Verdict', 'orange-addons-for-elementor'),
                'class' => '\Oradel_Verdict_Widget',
            ],
            'oradel-comparison-table' => [
                'title' => __('Comparison Table', 'orange-addons-for-elementor'),
                'class' => '\Oradel_Comparison_Table_Widget',
            ],
            'oradel-posts-grid' => [
                'title' => __('Posts Grid', 'orange-addons-for-elementor'),
                'class' => '\Oradel_Posts_Grid_Widget',
            ],
            'oradel-button' => [
                'title' => __('Button', 'orange-addons-for-elementor'),
                'class' => '\Oradel_Button_Widget',
            ],
            'oradel-advanced-heading' => [
                'title' => __('Advanced Heading', 'orange-addons-for-elementor'),
                'class' => '\Oradel_Heading_Widget',
            ],
            'oradel-testimonials' => [
                'title' => __('Testimonials', 'orange-addons-for-elementor'),
                'class' => '\Oradel_Testimonials_Widget',
            ],
            'oradel-benefits' => [
                'title' => __('Benefits', 'orange-addons-for-elementor'),
                'class' => '\Oradel_Benefits_Widget',
            ],
            'oradel-iconbox' => [
                'title' => __('Iconbox', 'orange-addons-for-elementor'),
                'class' => '\Oradel_Iconbox_Widget',
            ],
            'oradel-pricing-plans' => [
                'title' => __('Pricing Plans', 'orange-addons-for-elementor'),
                'class' => '\Oradel_Pricing_Plans_Widget',
            ],
            'oradel-hero-block' => [
                'title' => __('Hero Block', 'orange-addons-for-elementor'),
                'class' => '\Oradel_Hero_Widget',
            ],
            'oradel-icon-menu' => [
                'title' => __('Icon Menu', 'orange-addons-for-elementor'),
                'class' => '\Oradel_Icon_Menu_Widget',
            ],
        ];
    }

    public function get_widget_statuses()
    {
        $widgets = $this->get_widgets();
        $saved_statuses = get_option('oradel_elementor_widgets_status', []);
        $statuses = [];
        foreach ($widgets as $name => $widget) {
            $statuses[$name] = isset($saved_statuses[$name]) ? $saved_statuses[$name] : '1';
        }
        return $statuses;
    }

    private function get_asset_suffix()
    {
        return (defined('WP_DEBUG') && WP_DEBUG) ? '' : '.min';
    }

    public function register_widgets($widgets_manager)
    {
        require_once(__DIR__ . '/widgets/accordion.php');
        require_once(__DIR__ . '/widgets/image-accordion.php');
        require_once(__DIR__ . '/widgets/offer-listing.php');
        require_once(__DIR__ . '/widgets/pros-cons.php');
        require_once(__DIR__ . '/widgets/verdict.php');
        require_once(__DIR__ . '/widgets/comparison-table.php');
        require_once(__DIR__ . '/widgets/posts-grid.php');
        require_once(__DIR__ . '/widgets/button.php');
        require_once(__DIR__ . '/widgets/advanced-heading.php');
        require_once(__DIR__ . '/widgets/testimonials.php');
        require_once(__DIR__ . '/widgets/benefits.php');
        require_once(__DIR__ . '/widgets/iconbox.php');
        require_once(__DIR__ . '/widgets/pricing-plans.php');
        require_once(__DIR__ . '/widgets/hero-block.php');
        require_once(__DIR__ . '/widgets/icon-menu.php');

        $widgets = $this->get_widgets();
        $widgets_status = $this->get_widget_statuses();

        foreach ($widgets as $name => $widget) {
            if ($widgets_status[$name] == '1') {
                $widgets_manager->register(new $widget['class']());
            }
        }
    }

    public function register_controls($controls_manager)
    {
    }

    public function frontend_styles()
    {
        $css_url = ORADEL_URL . 'assets/css/widgets/';
        $css_version = self::CSS_VERSION;
        $suffix = $this->get_asset_suffix();

        wp_register_style('oea-common', $css_url . 'common' . $suffix . '.css', array(), $css_version);
        wp_register_style('oea-accordion', $css_url . 'accordion' . $suffix . '.css', array(), $css_version);
        wp_register_style('oea-image-accordion', $css_url . 'image-accordion' . $suffix . '.css', array(), $css_version);
        wp_register_style('oea-offer-listing', $css_url . 'offer-listing' . $suffix . '.css', array(), $css_version);
        wp_register_style('oea-pros-cons', $css_url . 'pros-cons' . $suffix . '.css', array(), $css_version);
        wp_register_style('oea-verdict', $css_url . 'verdict' . $suffix . '.css', array(), $css_version);
        wp_register_style('oea-comparison-table', $css_url . 'comparison-table' . $suffix . '.css', array(), $css_version);
        wp_register_style('oea-posts-grid', $css_url . 'posts-grid' . $suffix . '.css', array(), $css_version);
        wp_register_style('oea-button', $css_url . 'button' . $suffix . '.css', array(), $css_version);
        wp_register_style('oea-advanced-heading', $css_url . 'advanced-heading' . $suffix . '.css', array(), $css_version);
        wp_register_style('oea-testimonials', $css_url . 'testimonials' . $suffix . '.css', array(), $css_version);
        wp_register_style('oea-benefits', $css_url . 'benefits' . $suffix . '.css', array(), $css_version);
        wp_register_style('oea-iconbox', $css_url . 'iconbox' . $suffix . '.css', array(), $css_version);
        wp_register_style('oea-pricing', $css_url . 'pricing' . $suffix . '.css', array(), $css_version);
        wp_register_style('oea-hero-block', $css_url . 'hero-block' . $suffix . '.css', array(), $css_version);
        wp_register_style('oea-animations', $css_url . 'animations' . $suffix . '.css', array(), $css_version);
        wp_register_style('oea-icon-menu', $css_url . 'icon-menu' . $suffix . '.css', array(), $css_version);

        wp_enqueue_style('oea-common');
        wp_enqueue_style('oea-animations');
    }

    public function editor_styles()
    {
        $css_url = ORADEL_URL . 'assets/css/';
        $css_version = self::CSS_VERSION;
        $suffix = $this->get_asset_suffix();

        wp_register_style('oradel-editor-style', $css_url . 'editor-style' . $suffix . '.css', array(), $css_version);

        wp_enqueue_style('oradel-editor-style');

        if (!\Oradel_Plugin_Helper::is_premium()) {
            $free_version_styles = $this->generate_premium_options_styles();
            if (!empty($free_version_styles)) {
                wp_add_inline_style('oradel-editor-style', wp_strip_all_tags($free_version_styles));
            }
        }
    }

    private function generate_premium_options_styles()
    {
        $css_rules = [];
        $disabled_option_style = ' {
            color: #999 !important;
            background-color: #f5f5f5 !important;
            cursor: not-allowed !important;
            opacity: 0.6 !important;
        }';

        $widgets = [
            'Oradel_Icon_Menu_Widget',
            'Oradel_Posts_Grid_Widget',
            'Oradel_Benefits_Widget',
            'Oradel_Offer_Listing_Widget',
            'Oradel_Button_Widget',
        ];

        foreach ($widgets as $widget_class) {
            if (class_exists($widget_class)) {
                $widget_options = call_user_func([$widget_class, 'get_premium_options_for_styling']);

                foreach ($widget_options as $control_name => $premium_values) {
                    $selectors = [];
                    foreach ($premium_values as $value) {
                        $selectors[] = '.elementor-control-' . esc_attr($control_name) . ' select option[value="' . esc_attr($value) . '"]';
                    }

                    if (!empty($selectors)) {
                        $css_rules[] = implode(',', $selectors) . $disabled_option_style;
                    }
                }
            }
        }

        return !empty($css_rules) ? implode("\n", $css_rules) : '';
    }

    public function frontend_scripts()
    {
        $js_url = ORADEL_URL . 'assets/js/';
        $js_version = self::JS_VERSION;
        $suffix = $this->get_asset_suffix();

        wp_register_script('oea-utils', $js_url . 'utils' . $suffix . '.js', array('jquery'), $js_version, true);
        wp_register_script('oea-scroll-animations', $js_url . 'scroll-animations' . $suffix . '.js', array('jquery', 'oea-utils'), $js_version, true);
        wp_register_script('oea-custom-details', $js_url . 'custom-details' . $suffix . '.js', array('jquery'), $js_version, true);
        wp_register_script('oea-accordion', $js_url . 'accordion' . $suffix . '.js', array('jquery'), $js_version, true);
        wp_register_script('oea-hover-accordion', $js_url . 'hover-accordion' . $suffix . '.js', array('jquery'), $js_version, true);
        wp_register_script('oea-click-accordion', $js_url . 'click-accordion' . $suffix . '.js', array('jquery'), $js_version, true);
        wp_register_script('oea-equal-height', $js_url . 'equal-height' . $suffix . '.js', array('jquery'), $js_version, true);
        wp_register_script('oea-post-grid', $js_url . 'post-grid' . $suffix . '.js', array('jquery'), $js_version, true);
        wp_register_script('oea-taxonomy-menu', $js_url . 'taxonomy-menu' . $suffix . '.js', array('jquery'), $js_version, true);
        wp_register_script('oea-button-effects', $js_url . 'button-effects' . $suffix . '.js', array('jquery'), $js_version, true);
        wp_register_script('oea-testimonials-slider', $js_url . 'testimonials-slider' . $suffix . '.js', array('jquery'), $js_version, true);
        wp_register_script('oea-testimonial-gallery', $js_url . 'testimonial-gallery' . $suffix . '.js', array('jquery'), $js_version, true);
        wp_register_script('oea-pricing-toggle', $js_url . 'pricing-toggle' . $suffix . '.js', array('jquery'), $js_version, true);
        wp_register_script('oea-hero-slider', $js_url . 'hero-slider' . $suffix . '.js', array('jquery'), $js_version, true);
        wp_register_script('oea-mobile-pricing-slider', $js_url . 'mobile-pricing-slider' . $suffix . '.js', array('jquery'), $js_version, true);
        wp_register_script('oea-offer-listing', $js_url . 'offer-listing' . $suffix . '.js', array('jquery'), $js_version, true);
        wp_register_script('oea-reset-transitions', $js_url . 'reset-transitions' . $suffix . '.js', array('jquery'), $js_version, true);
        wp_register_script('oea-icon-menu', $js_url . 'icon-menu' . $suffix . '.js', array('jquery'), $js_version, true);
        wp_register_script('oea-comparison-table', $js_url . 'comparison-table' . $suffix . '.js', array('jquery'), $js_version, true);

        wp_enqueue_script('oea-utils');

        wp_localize_script('oea-post-grid', 'oradel_ajax_object', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'security' => wp_create_nonce('load_more_posts_nonce')
        ));
    }

    public function add_elementor_widget_categories($elements_manager)
    {
        $elements_manager->add_category(
            'oradel-plugin',
            [
                'title' => esc_html__('Orange Addons for Elementor', 'orange-addons-for-elementor'),
                'icon' => 'fa fa-plug',
            ]
        );
    }
}