(function ($) {
    'use strict';

    $(document).ready(function () {
        const $window = $(window);
        const $cards = $('.price-card');
        const $pricingGrid = $('.pricing-grid');
        const $nextBtn = $('.pricing-next');
        const $prevBtn = $('.pricing-prev');
        const $dots = $('.pricing-dot');

        const MOBILE_BREAKPOINT = 768;
        const ANIMATION_DURATION = 500;
        const DEBOUNCE_DELAY = 250;
        const SWIPE_THRESHOLD = 50;

        let currentIndex = 0;
        let isAnimating = false;
        let isMobileMode = false;

        function isMobile() {
            return window.innerWidth <= MOBILE_BREAKPOINT;
        }

        function setEqualHeight() {
            if (!isMobile()) return;

            let maxHeight = 0;
            $cards.css('height', 'auto');

            $cards.each(function () {
                const height = $(this).outerHeight();
                maxHeight = Math.max(maxHeight, height);
            });

            if (maxHeight > 0) {
                $cards.css('height', maxHeight + 'px');
            }
        }

        function updateActiveStates(index) {
            $cards.removeClass('active');
            $cards.eq(index).addClass('active');
            $dots.removeClass('active');
            $dots.eq(index).addClass('active');
        }

        function showCard(newIndex, direction = 'next') {
            if (isAnimating || newIndex === currentIndex) return;

            isAnimating = true;
            const $currentCard = $cards.eq(currentIndex);
            const $nextCard = $cards.eq(newIndex);

            $cards.removeClass('prev next');

            if (direction === 'next') {
                $currentCard.addClass('prev');
                $nextCard.addClass('next');
            } else {
                $currentCard.addClass('next');
                $nextCard.addClass('prev');
            }

            setTimeout(() => {
                updateActiveStates(newIndex);
                $nextCard.removeClass('prev next');

                setTimeout(() => {
                    isAnimating = false;
                }, ANIMATION_DURATION);
            }, 50);

            currentIndex = newIndex;
        }

        function getNextIndex(direction) {
            const totalCards = $cards.length;

            if (direction === 'next') {
                return currentIndex === totalCards - 1 ? 0 : currentIndex + 1;
            } else {
                return currentIndex === 0 ? totalCards - 1 : currentIndex - 1;
            }
        }

        function handleNavigation(direction) {
            if (isAnimating) return;
            const newIndex = getNextIndex(direction);
            showCard(newIndex, direction);
        }

        function initSwipeHandlers() {
            let touchStartX = 0;
            let touchEndX = 0;

            $pricingGrid.off('touchstart.pricingSlider touchend.pricingSlider')
                .on('touchstart.pricingSlider', function (e) {
                    if (isAnimating) return;
                    touchStartX = e.originalEvent.touches[0].clientX;
                })
                .on('touchend.pricingSlider', function (e) {
                    if (isAnimating) return;
                    touchEndX = e.originalEvent.changedTouches[0].clientX;

                    const diff = touchStartX - touchEndX;
                    if (Math.abs(diff) > SWIPE_THRESHOLD) {
                        const direction = diff > 0 ? 'next' : 'prev';
                        handleNavigation(direction);
                    }
                });
        }

        function initClickHandlers() {
            $nextBtn.off('click.pricingSlider').on('click.pricingSlider', function () {
                handleNavigation('next');
            });

            $prevBtn.off('click.pricingSlider').on('click.pricingSlider', function () {
                handleNavigation('prev');
            });

            $dots.off('click.pricingSlider').on('click.pricingSlider', function () {
                if (isAnimating) return;

                const newIndex = $(this).data('index');
                const direction = newIndex > currentIndex ? 'next' : 'prev';
                showCard(newIndex, direction);
            });
        }

        function enableMobileMode() {
            if (isMobileMode) return;

            isMobileMode = true;
            setEqualHeight();
            updateActiveStates(0);
            initSwipeHandlers();
            initClickHandlers();

            $window.on('resize.pricingSlider', debounce(function () {
                if (isMobile()) {
                    setEqualHeight();
                } else {
                    disableMobileMode();
                }
            }, DEBOUNCE_DELAY));
        }

        function disableMobileMode() {
            if (!isMobileMode) return;

            isMobileMode = false;
            $cards.css('height', 'auto').removeClass('active prev next');
            $dots.removeClass('active');

            $window.off('resize.pricingSlider');
            $pricingGrid.off('touchstart.pricingSlider touchend.pricingSlider');
            $nextBtn.off('click.pricingSlider');
            $prevBtn.off('click.pricingSlider');
            $dots.off('click.pricingSlider');

            currentIndex = 0;
            isAnimating = false;
        }

        function initPricingSlider() {
            if (isMobile()) {
                enableMobileMode();
            } else {
                disableMobileMode();
            }
        }

        if ($cards.length > 0) {
            initPricingSlider();
            $window.on('resize.pricingSliderInit', debounce(initPricingSlider, DEBOUNCE_DELAY));
        }
    });
})(jQuery);