(function ($) {
    'use strict';

    let isInitialized = false;
    let debounceTimer = null;

    function initializeButtonEffects() {
        if (isInitialized) return;

        $(document)
            .off('.oeaButtonEffects')
            .on('mouseenter.oeaButtonEffects', '.oea-button-link.hover-effect-1', handleHoverEnter)
            .on('mousemove.oeaButtonEffects', '.oea-button-link.hover-effect-1', handleHoverMove)
            .on('mouseleave.oeaButtonEffects', '.oea-button-link.hover-effect-1', handleHoverLeave)
            .on('click.oeaButtonEffects', '.oea-button-link.click-effect-1', handleClickEffect)
            .on('mousemove.oeaButtonEffects', '.oea-button-link.hover-effect-4', handleHoverEffect4);

        isInitialized = true;
    }

    function handleHoverEnter() {
        $(this).addClass('active');
    }

    function handleHoverMove(e) {
        const rect = this.getBoundingClientRect();
        const x = Math.max(0, Math.min(rect.width, e.clientX - rect.left));
        const y = Math.max(0, Math.min(rect.height, e.clientY - rect.top));

        this.style.setProperty('--x', `${x}px`);
        this.style.setProperty('--y', `${y}px`);
    }

    function handleHoverLeave() {
        const $this = $(this);
        $this.removeClass('active');
        requestAnimationFrame(() => {
            this.style.setProperty('--x', '50%');
            this.style.setProperty('--y', '50%');
        });
    }

    function handleClickEffect(e) {
        e.preventDefault();

        const rect = this.getBoundingClientRect();
        const x = e.clientX - rect.left;
        const y = e.clientY - rect.top;
        const href = this.getAttribute('href');
        const target = this.getAttribute('target');

        const $clickEffect = $('<span>', {
            class: 'click-effect-one',
            css: {
                left: `${x}px`,
                top: `${y}px`,
                position: 'absolute',
                width: '50px',
                height: '50px',
                background: 'var(--click-color, #ff8800)',
                borderRadius: '50%',
                transform: 'translate(-50%, -50%)',
                pointerEvents: 'none',
                animation: 'clickEffectOneAnimation 1s ease-out forwards',
                zIndex: 100,
                opacity: 0.6
            }
        });

        $(this).append($clickEffect);

        setTimeout(() => $clickEffect.remove(), 1000);

        if (href && href !== '#') {
            setTimeout(() => {
                target === '_blank' ? window.open(href, '_blank') : window.location.href = href;
            }, 300);
        }
    }

    function handleHoverEffect4(e) {
        const rect = e.target.getBoundingClientRect();
        const x = e.clientX * 3 - rect.left;
        this.style.setProperty('--x', `${x}deg`);
    }

    function debounceReinit() {
        clearTimeout(debounceTimer);
        debounceTimer = setTimeout(() => {
            isInitialized = false;
            initializeButtonEffects();
        }, 150);
    }

    function setupMutationObserver() {
        if (typeof MutationObserver === 'undefined') return;

        const observer = new MutationObserver(mutations => {
            const hasButtonNodes = mutations.some(mutation =>
                mutation.type === 'childList' &&
                [...mutation.addedNodes].some(node =>
                    node.nodeType === 1 && (
                        node.classList?.contains('oea-button-link') ||
                        node.querySelector?.('.oea-button-link')
                    )
                )
            );

            if (hasButtonNodes) debounceReinit();
        });

        observer.observe(document.body, {
            childList: true,
            subtree: true
        });
    }

    $(() => initializeButtonEffects());

    $(window).on('elementor/frontend/init', () => {
        if (typeof elementorFrontend !== 'undefined') {
            elementorFrontend.hooks.addAction('frontend/element_ready/widget', debounceReinit);
        }
    });

    setupMutationObserver();

})(jQuery);