<?php

/**
 * @package OpenUserMapPlugin
 */
namespace OpenUserMapPlugin\Pages;

use OpenUserMapPlugin\Base\BaseController;
class Frontend extends BaseController {
    public function register() {
        // Shortcodes
        add_action( 'init', array($this, 'set_shortcodes') );
    }

    /**
     * Setup Shortcodes
     * 
     * Note: Page builder detection is handled in the individual rendering methods
     * via the is_page_builder_active() helper function. Shortcodes are always
     * registered, but will return a styled placeholder when rendered inside a
     * page builder interface.
     */
    public function set_shortcodes() {
        // Render Map
        add_shortcode( 'open-user-map', array($this, 'render_block_map') );
        // Shortcode: "Add Location" Form (only)
        add_shortcode( 'open-user-map-form', array($this, 'render_block_form') );
        // Inject Complianz attributes and the tile provider attribute on all OUM scripts
        add_filter(
            'script_loader_tag',
            function ( $tag, $handle, $source ) {
                if ( strpos( $handle, 'oum' ) === false ) {
                    return $tag;
                }
                // Parse existing attributes from the tag
                $existing_attrs = array();
                if ( preg_match_all(
                    '/(\\w+(?:-\\w+)*)=["\']([^"\']*)["\']/',
                    $tag,
                    $matches,
                    PREG_SET_ORDER
                ) ) {
                    foreach ( $matches as $match ) {
                        $existing_attrs[$match[1]] = $match[2];
                    }
                }
                // Extract src if present (fallback to $source parameter)
                $src = ( isset( $existing_attrs['src'] ) ? $existing_attrs['src'] : $source );
                // Build our custom attributes (these will override existing ones if present)
                $custom_attrs = array(
                    'src'           => esc_url( $src ),
                    'data-category' => 'functional',
                    'class'         => 'cmplz-native',
                    'id'            => esc_attr( $handle ) . '-js',
                );
                // Append the tile provider attribute if we stored a value for this handle during enqueue.
                if ( function_exists( 'wp_scripts' ) ) {
                    $scripts = wp_scripts();
                    if ( $scripts instanceof \WP_Scripts ) {
                        $tile_provider = $scripts->get_data( $handle, 'data-oum-tile-provider' );
                        if ( !empty( $tile_provider ) ) {
                            $custom_attrs['data-oum-tile-provider'] = esc_attr( $tile_provider );
                        }
                    }
                }
                // Merge: existing attributes first, then our custom ones (custom overrides)
                $all_attrs = array_merge( $existing_attrs, $custom_attrs );
                // Build attribute string
                $attr_parts = array();
                foreach ( $all_attrs as $key => $value ) {
                    $attr_parts[] = sprintf( '%s="%s"', esc_attr( $key ), esc_attr( $value ) );
                }
                return sprintf( '<script %s></script>', implode( ' ', $attr_parts ) );
            },
            10,
            3
        );
        // Prevent shortcode parsing by All In One SEO plugin
        add_filter( 'aioseo_disable_shortcode_parsing', '__return_true' );
        // Prevent shortcode parsing by Slim SEO plugin
        add_filter( 'slim_seo_skipped_shortcodes', function ( $shortcodes ) {
            $shortcodes[] = 'open-user-map';
            $shortcodes[] = 'open-user-map-location';
            $shortcodes[] = 'open-user-map-gallery';
            $shortcodes[] = 'open-user-map-list';
            $shortcodes[] = 'open-user-map-form';
            return $shortcodes;
        } );
        // Prevent block parsing by Slim SEO plugin
        add_filter( 'slim_seo_skipped_blocks', function ( $blocks ) {
            $blocks[] = 'open-user-map/map';
            return $blocks;
        } );
    }

}
