import './editor.scss';
import './style.scss';

const {__} = wp.i18n;
const {RawHTML} = wp.element;
const {registerBlockType} = wp.blocks;
const {SelectControl, Dashicon, Spinner} = wp.components;
const {apiFetch} = wp;
const {registerStore, withSelect} = wp.data;

const iconEl = wp.element.RawHTML({
    children: '<svg style="max-height:25px;max-width:25px;min-height:25px;min-width:25px;width:25px" version="1.1" viewBox="0 0 654.48 510" xml:space="preserve" xmlns="http://www.w3.org/2000/svg"> <path transform="translate(-75.6 -13.58)" d="m292.93 105.89a21.48 21.48 0 0 1-11.11-2.67 17.22 17.22 0 0 1-6.82-7.47 26.7 26.7 0 0 1-2.43-11.44 50.49 50.49 0 0 1 2.19-14.56 77.43 77.43 0 0 1 3.09-8.75 87.82 87.82 0 0 1 4.46-9 94.43 94.43 0 0 1 5.69-8.75 83.36 83.36 0 0 1 6.71-8.12 74.34 74.34 0 0 1 7.62-7.05 54.06 54.06 0 0 1 8.43-5.56 45.43 45.43 0 0 1 9.1-3.65 35.49 35.49 0 0 1 9.65-1.31 23.8 23.8 0 0 1 11.15 2.44 16.36 16.36 0 0 1 6.92 6.71 22.73 22.73 0 0 1 2.58 10.29 42.55 42.55 0 0 1-1.86 13.18 83.82 83.82 0 0 1-4.7 12.32 104.34 104.34 0 0 1-7.05 12.6 99.84 99.84 0 0 1-9 11.79 72.46 72.46 0 0 1-10.55 9.79 52 52 0 0 1-11.69 6.72 32.41 32.41 0 0 1-12.38 2.49zm4-10.25a18.68 18.68 0 0 0 7-1.38 30.34 30.34 0 0 0 6.69-3.79 48.54 48.54 0 0 0 6.26-5.64 72.23 72.23 0 0 0 5.73-6.93q2.68-3.69 5-7.66t4.26-7.85c1.28-2.58 2.38-5.07 3.32-7.46s1.68-4.57 2.25-6.53a35.13 35.13 0 0 0 1.38-8 16.26 16.26 0 0 0-0.93-6.55 9.14 9.14 0 0 0-3.48-4.39 10.91 10.91 0 0 0-6.2-1.61 20.43 20.43 0 0 0-8.44 2 43.25 43.25 0 0 0-8.7 5.34 68.22 68.22 0 0 0-8.39 7.8 88.75 88.75 0 0 0-7.46 9.34 84.48 84.48 0 0 0-6 10 55.38 55.38 0 0 0-3.81 9.67 33.51 33.51 0 0 0-1.41 9.35 18 18 0 0 0 1.47 7.51 11.31 11.31 0 0 0 4.32 5 13.17 13.17 0 0 0 7.14 1.78z"/> <path transform="translate(-75.6 -13.58)" d="M342.35,88.16q.84-2.91,2.29-6.74t3.29-8.4q1.85-4.58,3.56-8.24t3.09-6.28a10.24,10.24,0,0,1,2.85-3.41,6.72,6.72,0,0,1,4.18-1.37c1.59,0,2.56.49,2.9,1.46a6.54,6.54,0,0,1-.19,3.89,18.15,18.15,0,0,1-.64,1.79c-.31.75-.65,1.58-1,2.48s-.8,1.83-1.22,2.78-.88,2-1.36,3.28-.86,2.13-1.12,2.74-.42,1-.48,1.21h.88c.27-.32.7-.84,1.29-1.57S362,70,362.85,68.67a39.91,39.91,0,0,1,2.65-3.61c.88-1,1.7-2.05,2.48-3s1.48-1.81,2.1-2.55,1.09-1.28,1.4-1.63a12.14,12.14,0,0,1,4.17-3.27,10.76,10.76,0,0,1,4.28-.89,2.93,2.93,0,0,1,3,1.86c.52,1.24.46,3-.16,5.13a15.46,15.46,0,0,1-.67,1.63c-.35.77-.79,1.68-1.3,2.75s-1.13,2.43-1.86,4.11l-2,4.51c-.59,1.32-1.14,2.64-1.64,3.93s-.91,2.49-1.23,3.58a12.77,12.77,0,0,0-.61,4.51c.12,1.18.67,1.77,1.64,1.77a5.13,5.13,0,0,0,1.62-.4,7.85,7.85,0,0,0,2.13-.93,19.32,19.32,0,0,1,2.45-1.45A30.63,30.63,0,0,0,384.77,83q4.19-2.27,10-5.53l-2.89,9.9L389.23,89q-.33.23-4.82,2.3-6.48,3-8.08,3.63-3.88,1.54-4.8,1.77a17.17,17.17,0,0,1-4.84.7c-1.63,0-2.65-1.17-3.09-3.51s0-5.77,1.29-10.28a58.42,58.42,0,0,1,2-5.73c.82-2,1.75-4.25,2.78-6.6H369l-18.19,23a8.68,8.68,0,0,1-2.7,2.47,6,6,0,0,1-2.75.66,4,4,0,0,1-2.7-.77,3.42,3.42,0,0,1-1.11-2.08,9,9,0,0,1,.05-3A32.21,32.21,0,0,1,342.35,88.16Z"/> <path transform="translate(-75.6 -13.58)" d="M392.44,97.4a4.48,4.48,0,0,1-3.84-2.06,12.94,12.94,0,0,1-2-5.55,33.09,33.09,0,0,1-.06-8.07,53.81,53.81,0,0,1,1.91-9.64,85,85,0,0,1,4.6-11.59q3.35-7.25,3.35-7.24l4-7.23a109.54,109.54,0,0,1,7.21-11.61q3.19-4.36,6.56-8.17a68,68,0,0,1,6.68-6.62,33.82,33.82,0,0,1,6.5-4.43,13.45,13.45,0,0,1,6-1.61,5.46,5.46,0,0,1,3.16.78,3.83,3.83,0,0,1,1.5,2.06,7.59,7.59,0,0,1,.24,2.91,18.11,18.11,0,0,1-.64,3.3,46.73,46.73,0,0,1-3.43,8.52,88,88,0,0,1-5.17,8.71q-2.94,4.38-6.41,8.78t-7.1,8.74q-3.65,4.34-7.26,8.53c-2.41,2.8-4.69,5.54-6.81,8.22a12.43,12.43,0,0,0-.8,1.16c-.3.47-.6,1-.91,1.54s-.62,1.13-.95,1.76-.62,1.23-.88,1.82l-.18.63a10.88,10.88,0,0,0-.4,2.25,10.4,10.4,0,0,0,0,2,4.75,4.75,0,0,0,.35,1.39c.17.34.37.51.61.51a7.38,7.38,0,0,0,2.07-.36,25.61,25.61,0,0,0,2.64-1c.94-.4,1.93-.86,2.95-1.38s2-1,3-1.58q3.39-1.87,7.22-4.33l1.24,9.32q-4.83,2.7-9.4,4.79c-1.32.59-2.68,1.17-4.07,1.75s-2.78,1.09-4.16,1.53A38.65,38.65,0,0,1,395.88,97,17.18,17.18,0,0,1,392.44,97.4Zm10.1-35q3-3.57,6.54-8T416,45.15q3.36-4.73,6.17-9.43A57.56,57.56,0,0,0,426.5,27c.25-.87.44-1.6.57-2.21s.09-.92-.14-.92a5.39,5.39,0,0,0-3.1,1.68,34.7,34.7,0,0,0-4.11,4.55,93.79,93.79,0,0,0-6,9.32l-3.81,6.46-2.91,6A93.42,93.42,0,0,0,402.54,62.36Z"/> <path transform="translate(-75.6 -13.58)" d="M436,88q-2,1.45-4.85,3.11a61.16,61.16,0,0,1-6,3.06,54.19,54.19,0,0,1-6.26,2.32,20.06,20.06,0,0,1-5.69.92,8.89,8.89,0,0,1-3.69-.64,4.27,4.27,0,0,1-2-1.79,5.32,5.32,0,0,1-.59-2.8,15.29,15.29,0,0,1,.63-3.65A47.43,47.43,0,0,1,410,82c2.31-5.12,3.69-8,4.11-8.63Q415.29,71.5,419,65q3.19-5.28,4.05-6.87a5.3,5.3,0,0,1,1.8-2,11.28,11.28,0,0,1,2.44-1.32,13.14,13.14,0,0,1,2.61-.76,12.17,12.17,0,0,1,2.31-.25q1.81,0,2.28.69a2.16,2.16,0,0,1,.12,1.83,14.67,14.67,0,0,1-1.24,2.75l-2.2,4.05Q429.9,65.4,427.74,69a58.32,58.32,0,0,0-3.31,6.09q-1.14,2.52-2.49,5A22.62,22.62,0,0,0,420,84.59a9.14,9.14,0,0,0-.47,3.15c.07.79.57,1.18,1.48,1.18a9.15,9.15,0,0,0,3.86-1.11,48.3,48.3,0,0,0,4.7-2.67q2.44-1.58,4.83-3.25T438.59,79Zm-6.46-46.4a8.89,8.89,0,0,1,1.11-2.32,10.87,10.87,0,0,1,2.13-2.43A11.69,11.69,0,0,1,436,34.92a11.27,11.27,0,0,1,4.36-.78,10.65,10.65,0,0,1,2.81.35,5.1,5.1,0,0,1,2.08,1.1,3.53,3.53,0,0,1,1.06,1.94,6,6,0,0,1-.22,2.85,11.3,11.3,0,0,1-1.34,2.95,10,10,0,0,1-2.28,2.54,11.25,11.25,0,0,1-7.26,2.5c-2.6,0-4.34-.61-5.22-1.85S429,43.64,429.56,41.59Z"/> <path transform="translate(-75.6 -13.58)" d="M432.87,88.16q.84-2.91,2.29-6.74t3.3-8.4c1.23-3,2.41-5.8,3.56-8.24s2.17-4.54,3.08-6.28A10.27,10.27,0,0,1,448,55.09a6.67,6.67,0,0,1,4.17-1.37c1.6,0,2.56.49,2.9,1.46a6.53,6.53,0,0,1-.18,3.89c-.13.44-.35,1-.65,1.79s-.65,1.58-1,2.48-.8,1.83-1.22,2.78-.88,2-1.36,3.28-.86,2.13-1.12,2.74-.42,1-.48,1.21h.88l1.3-1.57c.59-.72,1.33-1.76,2.22-3.11A38.08,38.08,0,0,1,456,65.06c.87-1,1.7-2.05,2.47-3s1.48-1.81,2.1-2.55l1.4-1.63a12.14,12.14,0,0,1,4.17-3.27,10.8,10.8,0,0,1,4.28-.89,2.94,2.94,0,0,1,3,1.86,8,8,0,0,1-.17,5.13,13.5,13.5,0,0,1-.67,1.63c-.35.77-.78,1.68-1.29,2.75s-1.14,2.43-1.87,4.11-1.39,3.18-2,4.51-1.13,2.64-1.64,3.93-.91,2.49-1.22,3.58a12.32,12.32,0,0,0-.61,4.51c.12,1.18.66,1.77,1.64,1.77a5.15,5.15,0,0,0,1.61-.4,7.7,7.7,0,0,0,2.13-.93,20.22,20.22,0,0,1,2.45-1.45A30.63,30.63,0,0,0,475.29,83q4.2-2.27,10-5.53l-2.88,9.9L479.75,89q-.33.23-4.82,2.3-6.48,3-8.07,3.63-3.9,1.54-4.8,1.77a17.24,17.24,0,0,1-4.85.7q-2.43,0-3.09-3.51t1.3-10.28a58.2,58.2,0,0,1,2-5.73c.83-2,1.75-4.25,2.79-6.6h-.71l-18.19,23a8.91,8.91,0,0,1-2.7,2.47,6,6,0,0,1-2.76.66,4,4,0,0,1-2.7-.77,3.36,3.36,0,0,1-1.1-2.08,8.73,8.73,0,0,1,0-3A29.84,29.84,0,0,1,432.87,88.16Z"/> <path transform="translate(-75.6 -13.58)" d="M488,97.4a13.38,13.38,0,0,1-6.79-1.53,9.84,9.84,0,0,1-4-4.11,13.56,13.56,0,0,1-1.38-6,24.44,24.44,0,0,1,1-7.13,28.42,28.42,0,0,1,2.6-5.83q1.89-3.37,3.63-6a41.12,41.12,0,0,1,3.92-5,30.81,30.81,0,0,1,4.7-4.19A23.53,23.53,0,0,1,497,54.75a16,16,0,0,1,5.83-1.07,11.3,11.3,0,0,1,4.38.79,7.79,7.79,0,0,1,3.06,2.23,7.1,7.1,0,0,1,1.46,3.47,10.84,10.84,0,0,1-.42,4.55,17.5,17.5,0,0,1-3.52,6,36.75,36.75,0,0,1-5.83,5.7,38.86,38.86,0,0,1-6.91,4.24,41,41,0,0,1-8,2.9,9.6,9.6,0,0,0,.75,1.94,3.78,3.78,0,0,0,1,1.21,3.23,3.23,0,0,0,1.23.61,6.42,6.42,0,0,0,1.52.17,19.29,19.29,0,0,0,5.9-1,49.31,49.31,0,0,0,6.47-2.59q3.24-1.56,6.24-3.39c2-1.23,3.74-2.39,5.2-3.48l4,5.79A89.27,89.27,0,0,1,509,90.12q-2.29,1.36-4.86,2.69a50.74,50.74,0,0,1-5.28,2.34,39.87,39.87,0,0,1-5.47,1.64A24.88,24.88,0,0,1,488,97.4Zm.14-21.05a10.8,10.8,0,0,0,3.71-.8,22.81,22.81,0,0,0,4.1-2.06,25.36,25.36,0,0,0,4.14-3.37,9.25,9.25,0,0,0,2.56-3.5,5.83,5.83,0,0,0,.19-3.2,1.47,1.47,0,0,0-1.44-1.25,8.42,8.42,0,0,0-3.46.69,10.43,10.43,0,0,0-2.78,1.79A13.9,13.9,0,0,0,493,67.16q-1,1.41-2.28,3.48a17.61,17.61,0,0,0-1.81,3.42C488.55,75,488.3,75.71,488.12,76.35Z"/> <path transform="translate(-75.6 -13.58)" d="M333.65,430.89c-.93,2.52-1.87,5.11-2.85,7.76l-2.63,7.2q-1.43,3.78-2.68,7.3,4.53,0,8.43-.27a38.8,38.8,0,0,0,7.23-1.13,25.11,25.11,0,0,0,6.18-2.47,20.33,20.33,0,0,0,5.2-4.29,29.73,29.73,0,0,0,4.28-6.57,56.2,56.2,0,0,0,3.49-9.32c.42-1.48.76-2.65,1-3.52a5,5,0,0,1,1.06-2,3.91,3.91,0,0,1,1.93-1A28.09,28.09,0,0,1,368,422a6.25,6.25,0,0,1,2.68.47,3.08,3.08,0,0,1,1.35,1.12,2.59,2.59,0,0,1,.41,1.35,5.23,5.23,0,0,1-.08,1.13c-.13.68-.37,1.57-.7,2.67s-.6,2.13-.81,3.08a73.34,73.34,0,0,1-3.26,9.13,39.47,39.47,0,0,1-4.27,7.45,29.3,29.3,0,0,1-5.6,5.79,29.65,29.65,0,0,1-7.27,4.13,42.42,42.42,0,0,1-9.22,2.48,74.05,74.05,0,0,1-11.5.82q1.63,12.51,1.68,12.64,1.2,5,2.47,12.38a10.56,10.56,0,0,0,2.79,5.39,5.67,5.67,0,0,0,4.42,1.86,18.61,18.61,0,0,0,3.85-.46,33.29,33.29,0,0,0,4.52-1.33,48,48,0,0,0,4.77-2.06,44.34,44.34,0,0,0,4.59-2.6l2.65,8.66a6.74,6.74,0,0,0-1,.51c-.66.37-1.58.84-2.75,1.41s-2.53,1.2-4.09,1.86-3.25,1.29-5.08,1.88a47.83,47.83,0,0,1-5.66,1.46,30.34,30.34,0,0,1-5.8.58,12.89,12.89,0,0,1-5.36-1,9.93,9.93,0,0,1-3.6-2.72,17.17,17.17,0,0,1-2.64-5.81q-1.23-4.1-1.74-6.75t-1-5.72q-.51-3.07-1-5.86a51.42,51.42,0,0,0-1.18-5.37,19.15,19.15,0,0,0-2-4.71,116.48,116.48,0,0,1-9.92,15.45,64,64,0,0,1-10,10.55,36.9,36.9,0,0,1-10.09,6,28.05,28.05,0,0,1-10,1.92c-1.15,0-2.49-.07-4-.21a26,26,0,0,1-4.27-.76,10.88,10.88,0,0,1-3.26-1.37,1.77,1.77,0,0,1-.91-2.08,5.79,5.79,0,0,1,1.2-2.18l1.73-2a10.52,10.52,0,0,1,.81-.79,10,10,0,0,1,1.2-.95,8.68,8.68,0,0,1,1.38-.8,3.17,3.17,0,0,1,1.33-.33,1.65,1.65,0,0,1,.39,0,14.06,14.06,0,0,0,2.79,1.13,10.45,10.45,0,0,0,2.91.42,14.92,14.92,0,0,0,7.91-2.43,36,36,0,0,0,7.7-6.63,74.21,74.21,0,0,0,7.36-9.88q3.6-5.69,6.93-12.21t6.3-13.53l5.86-13.89,1.55-3.68a27.7,27.7,0,0,0-7.3.93,29.74,29.74,0,0,0-6.34,2.52,29,29,0,0,0-5.33,3.69,34.76,34.76,0,0,0-4.22,4.4,28,28,0,0,0-3.08,4.71,25.57,25.57,0,0,0-1.87,4.58,4,4,0,0,1-.37.75,3.87,3.87,0,0,1-1,1.08,6.65,6.65,0,0,1-1.91,1,9.65,9.65,0,0,1-3.11.42,12.68,12.68,0,0,1-3.79-.44,4.48,4.48,0,0,1-1.87-1,2,2,0,0,1-.59-1.26,3.3,3.3,0,0,1,.1-1.1l.11-.4a.32.32,0,0,1,.08-.13A37.84,37.84,0,0,1,289,437.08a40.16,40.16,0,0,1,8-7.28,43.89,43.89,0,0,1,9.09-4.77,63.06,63.06,0,0,1,9.27-2.79,66.55,66.55,0,0,1,8.58-1.3q4.05-.33,7-.33a7.25,7.25,0,0,1,2.7.4,3.27,3.27,0,0,1,1.41,1,2.38,2.38,0,0,1,.44,1.38,5.49,5.49,0,0,1-.2,1.47,11,11,0,0,1-.61,3.26Z"/> <path transform="translate(-75.6 -13.58)" d="M401.2,486.05q-1.43,1.43-3.68,3.35a50.12,50.12,0,0,1-4.88,3.67,35.61,35.61,0,0,1-5.4,2.95,13.12,13.12,0,0,1-5.22,1.21,5.17,5.17,0,0,1-4.37-1.86,10.86,10.86,0,0,1-1.83-5.41,33.55,33.55,0,0,1-3.35,2.5,35,35,0,0,1-4.07,2.29,29.69,29.69,0,0,1-4.34,1.69,14.64,14.64,0,0,1-4.21.65,8.63,8.63,0,0,1-3.88-.86,7.18,7.18,0,0,1-2.85-2.5,8.82,8.82,0,0,1-1.34-4,13.93,13.93,0,0,1,.59-5.29,32.33,32.33,0,0,1,2.88-6.78,68.54,68.54,0,0,1,11.27-14.54,47.92,47.92,0,0,1,6.4-5.16,35.35,35.35,0,0,1,6.63-3.5,17.58,17.58,0,0,1,6.41-1.3,2.31,2.31,0,0,1,1.49.4,2.75,2.75,0,0,1,.69.92c.15.34.3.68.45,1a1.37,1.37,0,0,0,.68.72,3.1,3.1,0,0,0,1,.29,8.07,8.07,0,0,0,1.11.07l1,0,1,0a5,5,0,0,1,1.21.14,1.42,1.42,0,0,1,.85.58,2.21,2.21,0,0,1,.33,1.25,8.19,8.19,0,0,1-.38,2.19,34.27,34.27,0,0,1-1.79,4.69q-1.19,2.57-2.53,5.28L388,476.83a35.49,35.49,0,0,0-2.4,5.83,10.75,10.75,0,0,0-.54,3.17q.06,1.14,1.29,1.14a5.62,5.62,0,0,0,2.08-.46,17.89,17.89,0,0,0,2.4-1.2c.84-.5,1.69-1.06,2.57-1.69s1.71-1.28,2.53-1.95q2.84-2.37,5.91-5.35Zm-37-3.46a15.69,15.69,0,0,0-.48,2.21,6.1,6.1,0,0,0,0,1.91,2.61,2.61,0,0,0,.67,1.37,2.08,2.08,0,0,0,1.54.53,5.42,5.42,0,0,0,3.38-1.43,21.12,21.12,0,0,0,3.28-3.43,37.42,37.42,0,0,0,4.56-7.7l7.43-14.37a9.19,9.19,0,0,0-3.8.9,19.8,19.8,0,0,0-3.95,2.43,33.47,33.47,0,0,0-3.82,3.51,46.23,46.23,0,0,0-3.91,4.86,43.79,43.79,0,0,0-3.21,5.09A22.31,22.31,0,0,0,364.24,482.59Z"/> <path transform="translate(-75.6 -13.58)" d="M444.12,486.05q-1.42,1.43-3.68,3.35a50.12,50.12,0,0,1-4.88,3.67,35.7,35.7,0,0,1-5.41,2.95,13,13,0,0,1-5.21,1.21,5.16,5.16,0,0,1-4.37-1.86,11,11,0,0,1-1.84-5.41,31.38,31.38,0,0,1-3.35,2.5,34.17,34.17,0,0,1-4.07,2.29,29.26,29.26,0,0,1-4.34,1.69,14.52,14.52,0,0,1-4.2.65,8.7,8.7,0,0,1-3.89-.86,7.22,7.22,0,0,1-2.84-2.5,8.82,8.82,0,0,1-1.34-4,13.73,13.73,0,0,1,.59-5.29,32.33,32.33,0,0,1,2.88-6.78,67.68,67.68,0,0,1,11.27-14.54,46.68,46.68,0,0,1,6.4-5.16,35.35,35.35,0,0,1,6.63-3.5,17.54,17.54,0,0,1,6.41-1.3,2.28,2.28,0,0,1,1.48.4,2.78,2.78,0,0,1,.7.92l.44,1a1.39,1.39,0,0,0,.69.72,3.1,3.1,0,0,0,1,.29,8,8,0,0,0,1.11.07l1,0,1,0a4.94,4.94,0,0,1,1.2.14,1.44,1.44,0,0,1,.86.58,2.21,2.21,0,0,1,.33,1.25,7.89,7.89,0,0,1-.39,2.19,34.11,34.11,0,0,1-1.78,4.69q-1.19,2.57-2.53,5.28L431,476.83a36.58,36.58,0,0,0-2.41,5.83,11,11,0,0,0-.53,3.17q0,1.14,1.29,1.14a5.62,5.62,0,0,0,2.08-.46,19.16,19.16,0,0,0,2.4-1.2c.84-.5,1.69-1.06,2.56-1.69s1.72-1.28,2.53-1.95q2.85-2.37,5.92-5.35Zm-37-3.46a17.15,17.15,0,0,0-.47,2.21,5.81,5.81,0,0,0,0,1.91,2.54,2.54,0,0,0,.67,1.37,2.06,2.06,0,0,0,1.53.53,5.39,5.39,0,0,0,3.38-1.43,20.75,20.75,0,0,0,3.29-3.43,34.64,34.64,0,0,0,2.76-4.14,32.2,32.2,0,0,0,1.8-3.56l7.42-14.37a9.14,9.14,0,0,0-3.79.9,19.52,19.52,0,0,0-3.95,2.43,32.65,32.65,0,0,0-3.82,3.51,47.51,47.51,0,0,0-3.92,4.86,42.25,42.25,0,0,0-3.2,5.09A22.39,22.39,0,0,0,407.15,482.59Z"/> <path transform="translate(-75.6 -13.58)" d="M440.09,487.77q.56-1.95,1.52-4.54c.64-1.72,1.37-3.54,2.17-5.46s1.65-3.86,2.56-5.85,2-4.12,3.13-6.37,2.18-4.13,3-5.64,1.56-2.74,2.22-3.71a12.73,12.73,0,0,1,1.15-1.55,4.93,4.93,0,0,1,1.08-.91,3.66,3.66,0,0,1,1.28-.44,12.06,12.06,0,0,1,1.73-.11,10.9,10.9,0,0,1,1.76.11,1,1,0,0,1,.8.53,2,2,0,0,1,0,1.22c-.11.53-.29,1.25-.56,2.16-.31,1.1-.83,2.64-1.55,4.65a54.3,54.3,0,0,0-1.64,5.26h.49q1.38-1.77,3.54-4.64A37.21,37.21,0,0,1,467,457.7a23,23,0,0,1,4.23-3.2,9.15,9.15,0,0,1,4.67-1.31,4.2,4.2,0,0,1,1.56.29,2.78,2.78,0,0,1,1.15.82,2.67,2.67,0,0,1,.58,1.3,3.87,3.87,0,0,1-.15,1.75,15.61,15.61,0,0,1-.82,2.1c-.41.9-1,2.07-1.72,3.51s-1.31,2.63-1.73,3.56a17.72,17.72,0,0,0-.88,2.32,9.37,9.37,0,0,0-.45,2.66,3.9,3.9,0,0,0,.32,1.7,1.87,1.87,0,0,0,.9.9,2.89,2.89,0,0,0,1.27.27,12.42,12.42,0,0,0,4.37-.89,37.09,37.09,0,0,0,4.46-2,56.51,56.51,0,0,0,4.91-2.92L486,478.92l-3.1,1.68c-1.15.62-2.35,1.2-3.62,1.75a30.32,30.32,0,0,1-3.84,1.37,13.87,13.87,0,0,1-3.81.55,7.74,7.74,0,0,1-5-1.35,6.56,6.56,0,0,1-2.21-3.53,14.92,14.92,0,0,1-.36-4.91c.13-1.81.35-3.65.65-5.5H464L449.77,491c-.74,1.15-1.41,2.1-2,2.85a14.11,14.11,0,0,1-1.66,1.83,4.68,4.68,0,0,1-1.52,1,4.29,4.29,0,0,1-1.54.29,4,4,0,0,1-2.7-.78,3.42,3.42,0,0,1-1.11-2.08,8.75,8.75,0,0,1,.06-3A29.69,29.69,0,0,1,440.09,487.77Z"/> <path transform="translate(-75.6 -13.58)" d="M515.19,487.1q-5,2.47-8.82,4.34c-2.57,1.23-4.84,2.27-6.81,3.09a45.25,45.25,0,0,1-5.14,1.86,14.48,14.48,0,0,1-3.83.62,7,7,0,0,1-5.31-2,10.63,10.63,0,0,1-2.59-5.86,33.12,33.12,0,0,1,.08-9.26,84.81,84.81,0,0,1,2.66-12.25c.51-1.76,1.12-3.65,1.84-5.65s1.5-4,2.33-6c-.85-.13-1.51-.26-2-.37s-.94-.23-1.45-.35-1.18-.23-2-.37-2-.29-3.54-.49a2.76,2.76,0,0,1-1.48-.56,1.72,1.72,0,0,1-.56-1,2.94,2.94,0,0,1,0-1.18c.09-.41.18-.76.26-1.06a1.72,1.72,0,0,1,1.17-1.06,10.43,10.43,0,0,1,2.39-.55,27.91,27.91,0,0,1,3-.21c1.09,0,2.13,0,3.11,0s1.87.06,2.65.07a9.85,9.85,0,0,0,1.64-.06q2.91-6,6.34-11.53a93,93,0,0,1,6.95-9.92,41.61,41.61,0,0,1,7-7,10.84,10.84,0,0,1,6.4-2.62,4.74,4.74,0,0,1,3.17,1.06,6.11,6.11,0,0,1,1.87,2.83,11.4,11.4,0,0,1,.52,4.12,20.2,20.2,0,0,1-.81,4.93,22.19,22.19,0,0,1-1.91,4.48,45.47,45.47,0,0,1-3.06,4.8q-1.76,2.44-3.92,4.95c-1.44,1.67-2.92,3.33-4.46,5a3.71,3.71,0,0,1,1.35-.05c.6.06,1.3.1,2.11.12l2.61.07,2.72.08,2.45.09,1.88.09a1,1,0,0,1,1,1,5.71,5.71,0,0,1-.28,2.36,5.41,5.41,0,0,1-1.66,2.69,3.45,3.45,0,0,1-2.24,1h-17q-1.8,1.77-3.6,3.47c-1.2,1.14-2.3,2.16-3.3,3.08q-1,2.73-1.86,5.39c-.25.77-.5,1.55-.77,2.36s-.52,1.62-.78,2.41-.5,1.57-.72,2.3-.43,1.42-.61,2a16.39,16.39,0,0,0-.71,4,5.53,5.53,0,0,0,.52,2.67,3.22,3.22,0,0,0,1.64,1.51,6.57,6.57,0,0,0,2.6.46,9.07,9.07,0,0,0,2.45-.44,32,32,0,0,0,3.4-1.22c1.26-.51,2.6-1.12,4-1.83s2.86-1.48,4.3-2.32,2.83-1.73,4.19-2.65,2.62-1.88,3.79-2.86Zm1.9-56a8.91,8.91,0,0,0,.25-1.07,3.6,3.6,0,0,0,0-1.13,1.5,1.5,0,0,0-.43-.88,1.75,1.75,0,0,0-1.21-.35c-.77,0-1.75.67-2.92,2a43.05,43.05,0,0,0-3.74,5.16q-2,3.13-4,6.91c-1.36,2.52-2.64,5-3.86,7.33l1.3.12,1.35.11q2-2.22,4.16-4.83t4.06-5.11c1.25-1.67,2.34-3.24,3.26-4.69A14.62,14.62,0,0,0,517.09,431.11Z"/> <path transform="translate(-75.6 -13.58)" d="M155.66,387.29H122V155.42h48.24L201.61,322l30.07-166.56H278V387.29H244.41V241.59l-29.17,145.7h-27L155.66,242Z"/> <path transform="translate(-75.6 -13.58)" d="M308.08,387.29V155.42h95.7v35.14H341.7v63.23h45.36v35.13H341.7v63.24h62.08v35.13Z"/> <path transform="translate(-75.6 -13.58)" d="M464.64,155.42l44.61,155.83V155.42h33.62V387.29H506.76L460.51,238.81V387.29H426.9V155.42Z"/> <path transform="translate(-75.6 -13.58)" d="m625.92 389.87a50.33 50.33 0 0 1-37.7-16.08q-15.48-16.1-15.48-39v-179.18h32.91v178.8a19.46 19.46 0 0 0 5.69 14.14 18.46 18.46 0 0 0 13.7 5.82 18.17 18.17 0 0 0 13.6-5.82 19.63 19.63 0 0 0 5.61-14.14v-178.8h35v179.17q0 23.3-15.65 39.19t-37.68 15.9z"/> </svg>'
});

const actions = {
    setMenuCards(menuCards) {
        return {
            type: 'SET_MENU_CARDS',
            menuCards,
        };
    },
    receiveMenuCards(path) {
        return {
            type: 'RECEIVE_MENU_CARDS',
            path,
        };
    },
};

const store = registerStore('onlinemenukaart', {
    reducer(state = {menuCards: {}}, action) {
        switch (action.type) {
            case 'SET_MENU_CARDS':
                return {
                    ...state,
                    menuCards: action.menuCards,
                    loaded: true,
                };
        }
        return state;
    },

    actions,

    selectors: {
        receiveMenuCards(state) {
            const {menuCards} = state;
            return menuCards;
        },
        isLoaded(state) {
            const {loaded} = state;
            return loaded;
        }
    },

    controls: {
        RECEIVE_MENU_CARDS(action) {
            return apiFetch({path: action.path});
        },
    },

    resolvers: {
        * receiveMenuCards(state) {
            const menuCards = yield actions.receiveMenuCards('onlinemenukaart/wordpress-integrations');
            return actions.setMenuCards(menuCards);
        },
    },
});

registerBlockType('onlinemenukaart/block-menukaart-embed', {
    title: __('Online Menukaart'),
    icon: iconEl,
    category: 'embed',
    keywords: [
        __('Online Menukaart'),
        __('onlinemenukaart.nl'),
        __('Menukaart'),
        __('Menu Card'),
    ],

    attributes: {
        uuid: {
            type: 'string'
        }
    },

    edit: withSelect((select) => {
        return {
            loaded: select('onlinemenukaart').isLoaded(),
            menuCards: select('onlinemenukaart').receiveMenuCards(),
        };
    })(({className, loaded, menuCards, attributes, setAttributes}) => {
        let token = omkGlobal.api_token;

        if (!attributes.uuid && menuCards.length > 0) {
            setAttributes({uuid: menuCards[0].value});
        }

        console.log(menuCards);

        if (!token) {
            return (
                <div className={className}>
                    <strong>Online Menukaart is nog niet geauthoriseerd.</strong>
                    {/* TODO: meer uitleg + knopje om naar settings pagina te gaan */}
                </div>
            )
        } else if (!menuCards.length && !loaded) {
            return (
                <p className={className}>
                    <Spinner/>
                    {__('Loading Data', 'omk')}
                </p>
            );
        } else if (!menuCards.length && loaded) {
            return (
                <div className={className}>
                    <div>
                        <div className="components-placeholder__label">
                            <span className="block-editor-block-icon has-colors">{iconEl}</span>
                            Onlinemenukaart.nl
                        </div>

                        <div className="components-placeholder__instructions">
                            {__('Je hebt nog geen gepubliceerde menukaarten.')}
                        </div>

                        <div className="components-placeholder__instructions">
                            {__('Ga naar onlinemenukaart.nl om je menukaart te publiceren.')}
                        </div>
                    </div>
                </div>
            );
        } else {
            return (
                <div className={className}>
                    <div>
                        <div className="components-placeholder__label">
                            <span className="block-editor-block-icon has-colors">{iconEl}</span>
                            Onlinemenukaart.nl
                        </div>

                        <div className="components-placeholder__instructions">
                            {__('Selecteer de menukaart die je wilt weergeven op deze pagina.')}
                        </div>

                        <div className="components-placeholder__fieldset">
                            <form>
                                <SelectControl
                                    value={attributes.uuid}
                                    onChange={(uuid) => {
                                        setAttributes({uuid: uuid})
                                    }}
                                    options={menuCards}
                                />
                            </form>

                            <div className="components-placeholder__learn-more">
                                <a className="components-external-link"
                                   href="https://onlinemenukaart.nl"
                                   target="_blank" rel="external noreferrer noopener">
                                    Meer informatie over onlinemenukaart.nl
                                    <span className="components-visually-hidden">(opent in nieuwe tab)</span>
                                    <Dashicon icon="external" className="components-external-link__icon" size={20}/>
                                </a>
                            </div>
                        </div>
                    </div>
                </div>
            );
        }
    }),
    save({className, attributes}) {
        let shortcode = `[menukaart uuid="${attributes.uuid}"]`;

        return (
            <div className={className}>
                <RawHTML>{shortcode}</RawHTML>
            </div>
        );
    }
});
