<?php
/**
 * Handles the form submission for the Service Provider Setup tab.
 *
 * @package onid-sso-by-saml-2-0\handlers
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * The Form handler class for the Service Provider Setup tab.
 */
class Onid_SAML_Service_Provider_Settings_Handler {

	/**
	 * Parses the Service Provider Setup form data and saves the configuration in the database.
	 *
	 * @param array           $post_array Contains the complete form post data.
	 * @param DatabaseHandler $db_handler The database handler for saving the data.
	 * @return void
	 */
	public static function onid_saml_service_provider_save_settings( $post_array, $db_handler ) {

		if ( ! self::onid_saml_validate_service_provider_fields( $post_array ) ) {
			return;
		}

		$save_array = array();
		$save_array[ Onid_Saml_Options_Enum_Service_Provider::SP_ENTITY_ID ]     = sanitize_text_field( trim( $post_array[ Onid_Saml_Options_Enum_Service_Provider::SP_ENTITY_ID ] ) );
		$save_array[ Onid_Saml_Options_Enum_Service_Provider::CONSUMER_URL ]     = sanitize_text_field( trim( $post_array[ Onid_Saml_Options_Enum_Service_Provider::CONSUMER_URL ] ) );
		$save_array[ Onid_Saml_Options_Enum_Service_Provider::IDENTITY_NAME ]    = sanitize_text_field( trim( $post_array[ Onid_Saml_Options_Enum_Service_Provider::IDENTITY_NAME ] ) );
		$save_array[ Onid_Saml_Options_Enum_Service_Provider::LOGIN_URL ]        = sanitize_text_field( trim( $post_array[ Onid_Saml_Options_Enum_Service_Provider::LOGIN_URL ] ) );
		$save_array[ Onid_Saml_Options_Enum_Service_Provider::ISSUER ]           = sanitize_text_field( trim( $post_array[ Onid_Saml_Options_Enum_Service_Provider::ISSUER ] ) );
		$save_array[ Onid_Saml_Options_Enum_Service_Provider::X509_CERTIFICATE ] = $post_array[ Onid_Saml_Options_Enum_Service_Provider::X509_CERTIFICATE ];
		$save_array[ Onid_Saml_Options_Enum_Service_Provider::X509_CERTIFICATE ] = self::onid_saml_parse_saml_certificates( $save_array[ Onid_Saml_Options_Enum_Service_Provider::X509_CERTIFICATE ] );

		if ( ! $save_array[ Onid_Saml_Options_Enum_Service_Provider::X509_CERTIFICATE ] ) {
			return;
		}

		if ( ! empty( $post_array[ Onid_Saml_Options_Enum_Service_Provider::IDENTITY_PROVIDER_NAME ] ) ) {
			$save_array[ Onid_Saml_Options_Enum_Service_Provider::IDENTITY_PROVIDER_NAME ] = sanitize_text_field( $post_array[ Onid_Saml_Options_Enum_Service_Provider::IDENTITY_PROVIDER_NAME ] );
		}

		if ( ! empty( $post_array[ Onid_Saml_Options_Enum_Service_Provider::IS_ENCODING_ENABLED ] ) ) {
			$save_array[ Onid_Saml_Options_Enum_Service_Provider::IS_ENCODING_ENABLED ] = 'checked';
		} else {
			$save_array[ Onid_Saml_Options_Enum_Service_Provider::IS_ENCODING_ENABLED ] = 'unchecked';
		}

		if ( ! empty( $post_array[ Onid_Saml_Options_Enum_Service_Provider::ASSERTION_TIME_VALIDITY ] ) ) {
			$save_array[ Onid_Saml_Options_Enum_Service_Provider::ASSERTION_TIME_VALIDITY ] = 'checked';
		} else {
			$save_array[ Onid_Saml_Options_Enum_Service_Provider::ASSERTION_TIME_VALIDITY ] = 'unchecked';
		}
		$save_array[ Onid_Saml_Options_Enum_Sso_Login::SSO_BUTTON ] = 'true';
		$db_handler->onid_saml_save_options( $save_array );
		$post_save = new Onid_SAML_Post_Save_Handler( Onid_Saml_Save_Status_Constants::SUCCESS, Onid_Saml_Messages::onid_saml_translate( 'IDP_DETAILS_SUCCESS' ), 'SERVICE_PROVIDER_CONF', $save_array );
		$post_save->onid_saml_post_save_action();

		$onid_saml_identity_provider_identifier_name = get_option( Onid_Saml_Options_Enum_Service_Provider::IDENTITY_PROVIDER_NAME );

		if ( ! empty( $onid_saml_identity_provider_identifier_name ) ) {

			// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Reading GET parameter to check the current tab name doesn't require nonce verification.
			if ( ! empty( Onid_Saml_Options_Plugin_Idp::$idp_list[ $onid_saml_identity_provider_identifier_name ] ) && ( ! isset( $_GET['tab'] ) || 'addons' !== $_GET['tab'] ) ) {
				update_option( Onid_Saml_Sso_Constants::ONID_SAML_CLOSE_NOTICE, '0' );
			} else {
				update_option( Onid_Saml_Sso_Constants::ONID_SAML_CLOSE_NOTICE, '1' );
			}
		}
	}

	/**
	 * This function takes care of validating and sanitizing the X.509 Certificate.
	 *
	 * @param array $onid_saml_x509_certificate An array of certificates configured in the Service Provider Setup tab.
	 * @return mixed
	 */
	public static function onid_saml_parse_saml_certificates( $onid_saml_x509_certificate ) {
		foreach ( $onid_saml_x509_certificate as $key => $value ) {
			if ( empty( $value ) ) {
				unset( $onid_saml_x509_certificate[ $key ] );
			} else {
				$onid_saml_x509_certificate[ $key ] = Onid_SAML_Utilities::onid_saml_sanitize_certificate( $value );

				// phpcs:ignore WordPress.PHP.NoSilencedErrors.Discouraged -- Silenced this to avoid any additional warning messages if an invalid certificate is uploaded.
				if ( ! @openssl_x509_read( $onid_saml_x509_certificate[ $key ] ) ) {
					$post_save = new Onid_SAML_Post_Save_Handler( Onid_Saml_Save_Status_Constants::ERROR, Onid_Saml_Messages::onid_saml_translate( 'INVALID_CERT' ), 'INVALID_CERT' );
					$post_save->onid_saml_post_save_action();
					return false;
				}
			}
		}
		$onid_saml_x509_certificate = maybe_serialize( $onid_saml_x509_certificate );

		return $onid_saml_x509_certificate;
	}

	/**
	 * Takes care of validating the input fields from the Service Provider Setup tab.
	 *
	 * @param array $post_array Contains the form post data.
	 * @return boolean
	 */
	public static function onid_saml_validate_service_provider_fields( $post_array ) {
		$validate_fields_array = array(
			$post_array[ Onid_Saml_Options_Enum_Service_Provider::SP_ENTITY_ID ],
			$post_array[ Onid_Saml_Options_Enum_Service_Provider::CONSUMER_URL ],
			$post_array[ Onid_Saml_Options_Enum_Service_Provider::LOGIN_URL ],
			$post_array[ Onid_Saml_Options_Enum_Service_Provider::ISSUER ],
			$post_array[ Onid_Saml_Options_Enum_Service_Provider::X509_CERTIFICATE ],
			);

		if ( '0' !== $post_array[ Onid_Saml_Options_Enum_Service_Provider::IDENTITY_NAME ] ) {
			array_push( $validate_fields_array, $post_array[ Onid_Saml_Options_Enum_Service_Provider::IDENTITY_NAME ] );
		}
		if ( Onid_SAML_Utilities::onid_saml_check_empty_or_null( $validate_fields_array ) ) {
			$log_object = array(				
				Onid_Saml_Options_Enum_Service_Provider::SP_ENTITY_ID  => $post_array[ Onid_Saml_Options_Enum_Service_Provider::SP_ENTITY_ID ],
				Onid_Saml_Options_Enum_Service_Provider::CONSUMER_URL  => $post_array[ Onid_Saml_Options_Enum_Service_Provider::CONSUMER_URL ],
				Onid_Saml_Options_Enum_Service_Provider::IDENTITY_NAME => $post_array[ Onid_Saml_Options_Enum_Service_Provider::IDENTITY_NAME ],
				Onid_Saml_Options_Enum_Service_Provider::LOGIN_URL     => $post_array[ Onid_Saml_Options_Enum_Service_Provider::LOGIN_URL ],
				Onid_Saml_Options_Enum_Service_Provider::ISSUER        => $post_array[ Onid_Saml_Options_Enum_Service_Provider::ISSUER ],
			);
			$post_save  = new Onid_SAML_Post_Save_Handler( Onid_Saml_Save_Status_Constants::ERROR, Onid_Saml_Messages::onid_saml_translate( 'FIELDS_EMPTY' ).json_encode($post_array[ Onid_Saml_Options_Enum_Service_Provider::X509_CERTIFICATE ]), 'INVALID_CONFIGURATION_SETTING', $log_object );
		} elseif ( ! preg_match( '/^\w*$/', $post_array[ Onid_Saml_Options_Enum_Service_Provider::IDENTITY_NAME ] ) ) {
			$log_object = array( Onid_Saml_Options_Enum_Service_Provider::IDENTITY_NAME => $post_array[ Onid_Saml_Options_Enum_Service_Provider::IDENTITY_NAME ] );
			$post_save  = new Onid_SAML_Post_Save_Handler( Onid_Saml_Save_Status_Constants::ERROR, Onid_Saml_Messages::onid_saml_translate( 'INVALID_FORMAT' ), 'INVALID_IDP_NAME_FORMAT', $log_object );
		}
		if ( isset( $post_save ) ) {
			$post_save->onid_saml_post_save_action();
			return false;
		}
		return true;
	}
}
