<?php
/**
 * This file is part of onID SAML plugin.
 *
 * The onID SAML plugin is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * onID SAML plugin is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with onID SAML plugin.  If not, see <http://www.gnu.org/licenses/>.
 *
 * @package onid-sso-by-saml-2-0
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

require 'class-onid-saml-assertion.php';

/**
 * Class for SAML2 Response messages.
 */
class Onid_SAML_Response {

	/**
	 * The assertions in this response.
	 *
	 * @var array $assertion
	 */
	private $assertions;

	/**
	 * The destination URL in this response.
	 *
	 * @var string $destination
	 */
	private $destination;

	/**
	 * The nonce in this response - GET parameter of $destination
	 *
	 * @var string $nonce
	 */
	private $nonce;

	/**
	 * The certificate in this response.
	 *
	 * @var string $certificates
	 */
	private $certificates;

	/**
	 * The signature data in this response.
	 *
	 * @var array $signature_data
	 */
	private $signature_data;

	/**
	 * Constructor for SAML 2 response messages.
	 *
	 * @param DOMElement|NULL $xml The input message.
	 */
	public function __construct( DOMElement $xml = null ) {

		$this->assertions   = array();
		$this->certificates = array();

		if ( null === $xml ) {
			return;
		}

		$sig = Onid_SAML_Utilities::onid_saml_validate_element( $xml );
		if ( false !== $sig ) {
			$this->certificates   = $sig['Certificates'];
			$this->signature_data = $sig;
		}

		/* set the destination from saml response, and extract 'nonce' to be used in nonceVerification against 'onid_saml_nonce' */
		if ( $xml->hasAttribute( 'Destination' ) ) {
			$this->destination = $xml->getAttribute( 'Destination' );
			$getAttributes = strpos($this->destination, "?");
			if($getAttributes != false) {
				$parts = explode("?", $this->destination);
				$this->destination = $parts[0];
				$args = explode("&", $parts[1]);
				foreach($args as $a) {
					$p = explode("=", $a);
					if(strpos($p[0], "nonce")) { 
						$this->nonce=$p[1]; 
						//error_log("zio can ECCOLO >> ".$this->nonce);
					}
				}


			}
		}

		// phpcs:ignore WordPress.NamingConventions.ValidVariableName.UsedPropertyNotSnakeCase -- $xml is the object of PHP's predefined class
		for ( $node = $xml->firstChild; null !== $node; $node = $node->nextSibling ) {
			// phpcs:ignore WordPress.NamingConventions.ValidVariableName.UsedPropertyNotSnakeCase -- $xml is the object of PHP's predefined class
			if ( 'urn:oasis:names:tc:SAML:2.0:assertion' !== $node->namespaceURI ) {
				continue;
			}

			try {
				// phpcs:ignore WordPress.NamingConventions.ValidVariableName.UsedPropertyNotSnakeCase -- $xml is the object of PHP's predefined class
				if ( 'Assertion' === $node->localName || 'EncryptedAssertion' === $node->localName ) {
$nonce = wp_create_nonce('Assertion_EncryptedAssertion');

					$this->assertions[] = new Onid_SAML_Assertion( $node, $nonce );
				}
			} catch ( Exception $exception ) {
				wp_die( 'We could not sign you in. Please contact your administrator.', 'Missing Issuer in Assertion' );
			}
		}
	}

	/**
	 * Retrieve the assertions in this response.
	 *
	 * @return Onid_SAML_Assertion[]|SAML2_EncryptedAssertion[]
	 */
	public function onid_saml_get_assertions() {
		return $this->assertions;
	}

	/**
	 * Set the assertions that should be included in this response.
	 *
	 * @param array $assertions The assertions.
	 */
	public function onid_saml_set_assertions( array $assertions ) {
		$this->assertions = $assertions;
	}

	/**
	 * Get the destination for the message in response.
	 *
	 * @return string destination
	 */
	public function onid_saml_get_destination() {
		return $this->destination;
	}

	/**
	 * Get the nonce for the message in response.
	 *
	 * @return string nonce
	 */
	public function onid_saml_get_nonce() {
		return $this->nonce;
	}

	/**
	 * Get the certificates in response.
	 *
	 * @return string certificates
	 */
	public function onid_saml_get_certificates() {
		return $this->certificates;
	}

	/**
	 * Get the signature data in response.
	 *
	 * @return array signature_data
	 */
	public function onid_saml_get_signature_data() {
		return $this->signature_data;
	}
}
