<?php
/**
 * File to handle SAML response and generate saml request.
 *
 * @package onid-sso-by-saml-2-0
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit();
}

require_once 'onid-saml-import-export.php';
require_once __DIR__ . '/includes/lib/class-onid-saml-options-enum.php';
require_once __DIR__ . '/includes/lib/class-onid-saml-options-enum-error-codes.php';
require_once __DIR__ . '/class-onid-saml-response.php';
require_once __DIR__ . '/class-onid-saml-utilities.php';
require_once 'onid-saml-xmlseclibs.php';

use onid_RobRichards\XMLSecLibs\Onid_SAML_XML_Security_Key;

/**
 * This class handles the Single-Sign-On.
 */
class Onid_SAML_Login_Validate {

	/**
	 * The Constructor for the Onid_SAML_Login_Validate class. This takes care of initializing the hooks used by the plugin.
	 */
	public function __construct() {
		add_action( 'init', array( $this, 'onid_saml_login_validate' ) );
	}


	/**
	 * Function to handle all incoming request with 'option' & "SAMLResponse Parameter"
	 *
	 * @return void
	 */
	public function onid_saml_login_validate() {

		//phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Ignore the nonce verification for SSO initiation.
		if ( isset( $_REQUEST['option'] ) && ( 'saml_user_login' === $_REQUEST['option'] || 'testConfig' === $_REQUEST['option'] ) ) {
			//phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Ignore the nonce verification for extensions installation checks.
			Onid_SAML_Utilities::onid_saml_check_required_extensions_installed( sanitize_text_field( wp_unslash( $_REQUEST['option'] ) ) );
			//phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Ignore the nonce verification for test config operation.
			if ( 'testConfig' === $_REQUEST['option'] ) {
				if ( ! is_user_logged_in() || ( is_user_logged_in() && ! current_user_can( 'manage_options' ) ) ) {
					return;
				}
			} elseif ( is_user_logged_in() ) {
					return;
			}

			if ( Onid_SAML_Utilities::onid_saml_is_sp_configured() ) {
				//phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Ignore the nonce verification for test config operation.
				if ( 'testConfig' === $_REQUEST['option'] ) {
					$send_relay_state = 'testValidate';
					//phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Ignore the nonce verification for SSO redirect parameter.
				} elseif ( isset( $_REQUEST['redirect_to'] ) ) {
					//phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Ignore the nonce verification for SSO redirect parameter.
					$send_relay_state = sanitize_text_field( wp_unslash( $_REQUEST['redirect_to'] ) );
				} else {
					$send_relay_state = Onid_SAML_Utilities::onid_saml_get_current_page_url();
				}

				$send_relay_state = $this->onid_saml_get_relay_state( $send_relay_state );
				$send_relay_state = empty( $send_relay_state ) ? '/' : $send_relay_state;

				$send_relay_state = rawurlencode( $send_relay_state );
				$sp_base_url      = get_option( Onid_Saml_Options_Enum_Service_Provider::SP_BASE_URL );
				if ( empty( $sp_base_url ) ) {
					$sp_base_url = site_url();
				}

				$sso_url      = htmlspecialchars_decode( get_option( Onid_Saml_Options_Enum_Service_Provider::LOGIN_URL ) );
				$acs_url      = site_url() . '/';
				$issuer       = site_url() . '/wp-content/plugins/onid-sso-by-saml-2-0/';
				$sp_entity_id = get_option( Onid_Saml_Options_Enum_Service_Provider::SP_ENTITY_ID );
				if ( empty( $sp_entity_id ) ) {
					$sp_entity_id = $sp_base_url . '/wp-content/plugins/onid-sso-by-saml-2-0/';
				}

				$log_message = array(
					'ssoUrl'         => $sso_url,
					'acsUrl'         => $acs_url,
					'spEntityId'     => $sp_entity_id,
					'sendRelayState' => $send_relay_state,
				);
				$saml_request = Onid_SAML_Utilities::onid_saml_create_authn_request( $acs_url, $sp_entity_id );

				$redirect = $sso_url;

				if ( strpos( $sso_url, '?' ) !== false ) {
					$redirect .= '&';
				} else {
					$redirect .= '?';
				}
				$redirect .= 'SAMLRequest=' . $saml_request . '&RelayState=' . $send_relay_state;
				
				// append 'onid_saml_nonce', to be received later on for verification
				$redirect .= "&onid_saml_nonce=".wp_create_nonce('onid_saml_nonce');

				header( 'Location: ' . $redirect );
				exit();
			}
		}
		//nonce -- done in 'class-onid-saml-utilities.php', at function onid_saml_process_response()
		if ( ! empty( $_POST['SAMLResponse'] ) ) {
			// Check to verify nonce correctness is done in 'class-onid-saml-utilities.php,
			//		within function 'onid_saml_process_response', as long as after base64 decode and XML conversion.
			// 		Moreover, the 'onid_saml_nonce' is sent back to SP as a GET-like value in the Destination within SAML assertion
			//wp_verify_nonce($response->onid_saml_get_nonce(), 'onid_saml_nonce') == true


			//nonce -- done in 'class-onid-saml-utilities.php', at function onid_saml_process_response()
			$saml_response = sanitize_text_field( wp_unslash( $_POST['SAMLResponse'] ) );

			//nonce -- done in 'class-onid-saml-utilities.php', at function onid_saml_process_response()
			if ( ! empty( $_POST['RelayState'] ) && '/' !== $_POST['RelayState'] ) {
				//nonce -- done in 'class-onid-saml-utilities.php', at function onid_saml_process_response()
				$relay_state = sanitize_text_field( wp_unslash( $_POST['RelayState'] ) );
			} else {
				$relay_state = '';
			}

			update_option( Onid_Saml_Options_Test_Configuration::SAML_RESPONSE, $saml_response );
			//phpcs:ignore WordPress.PHP.DiscouragedPHPFunctions.obfuscation_base64_decode -- SAML response is base64 encoded.
			$saml_response = base64_decode( $saml_response );

			$document              = new DOMDocument();
			$load_saml_resp_result = $document->loadXML( $saml_response );
			if ( ! $load_saml_resp_result ) {
				$error_code = Onid_Saml_Options_Enum_Error_Codes::$error_codes['WPSAMLERR017'];
				Onid_SAML_Utilities::onid_saml_die( $error_code );
			}
			//phpcs:ignore WordPress.NamingConventions.ValidVariableName.UsedPropertyNotSnakeCase -- firstChild property is Method of DOMDocument.
			$saml_response_xml = $document->firstChild;
			//phpcs:ignore WordPress.NamingConventions.ValidVariableName.UsedPropertyNotSnakeCase -- documentElement property is Method of DOMDocument.
			$doc   = $document->documentElement;
			$xpath = new DOMXpath( $document );
			$xpath->registerNamespace( 'samlp', 'urn:oasis:names:tc:SAML:2.0:protocol' );
			$xpath->registerNamespace( 'saml', 'urn:oasis:names:tc:SAML:2.0:assertion' );

			$status         = $xpath->query( '/samlp:Response/samlp:Status/samlp:StatusCode', $doc );
			$status_string  = $status->item( 0 )->getAttribute( 'Value' );
			$status_message = $xpath->query( '/samlp:Response/samlp:Status/samlp:StatusMessage', $doc )->item( 0 );
			if ( ! empty( $status_message ) ) {
				//phpcs:ignore WordPress.NamingConventions.ValidVariableName.UsedPropertyNotSnakeCase -- nodeValue property is Method of DOMDocument.
				$status_message = $status_message->nodeValue;
			}

			$status_array = explode( ':', $status_string );
			if ( ! empty( $status_array[7] ) ) {
				$status = $status_array[7];
			}
			if ( 'Success' !== $status ) {
				$this->onid_saml_show_status_error( $status, $relay_state, $status_message );
			}

			$cert_from_plugin = maybe_unserialize( get_option( Onid_Saml_Options_Enum_Service_Provider::X509_CERTIFICATE ) );

			$acs_url                  = site_url() . '/';
			$saml_response            = new Onid_SAML_Response( $saml_response_xml );
			$response_signature_data  = $saml_response->onid_saml_get_signature_data();
			$assertion_signature_data = current( $saml_response->onid_saml_get_assertions() )->onid_saml_get_signature_data();

			if ( empty( $assertion_signature_data ) && empty( $response_signature_data ) ) {

				$error_code = Onid_Saml_Options_Enum_Error_Codes::$error_codes['WPSAMLERR003'];
				if ( 'testValidate' === $relay_state ) {
					onid_saml_display_test_config_error_page( $error_code );
					exit;
				} else {
					Onid_SAML_Utilities::onid_saml_die( $error_code );
				}
			}
			if ( is_array( $cert_from_plugin ) ) {
				foreach ( $cert_from_plugin as $key => $value ) {
					$plugin_cert         = $value;
					$cert_fp_from_plugin = Onid_SAML_XML_Security_Key::onid_saml_get_raw_thumbprint( $value );

					$cert_fp_from_plugin = $this->onid_saml_convert_to_windows_iconv( $cert_fp_from_plugin );
					$cert_fp_from_plugin = preg_replace( '/\s+/', '', $cert_fp_from_plugin );
					if ( ! empty( $response_signature_data ) ) {
						$valid_signature = Onid_SAML_Utilities::onid_saml_process_response( $acs_url, $cert_fp_from_plugin, $response_signature_data, $saml_response, $key, $relay_state );
					}
					if ( ! empty( $assertion_signature_data ) ) {
						$valid_signature = Onid_SAML_Utilities::onid_saml_process_response( $acs_url, $cert_fp_from_plugin, $assertion_signature_data, $saml_response, $key, $relay_state );
					}
					if ( $valid_signature ) {
						break;
					}
				}
			} else {
				$plugin_cert         = $cert_from_plugin;
				$cert_fp_from_plugin = Onid_SAML_XML_Security_Key::onid_saml_get_raw_thumbprint( $cert_from_plugin );
				$cert_fp_from_plugin = $this->onid_saml_convert_to_windows_iconv( $cert_fp_from_plugin );
				$cert_fp_from_plugin = preg_replace( '/\s+/', '', $cert_fp_from_plugin );
				if ( ! empty( $response_signature_data ) ) {
					$valid_signature = Onid_SAML_Utilities::onid_saml_process_response( $acs_url, $cert_fp_from_plugin, $response_signature_data, $saml_response, 0, $relay_state );
				}

				if ( ! empty( $assertion_signature_data ) ) {
					$valid_signature = Onid_SAML_Utilities::onid_saml_process_response( $acs_url, $cert_fp_from_plugin, $assertion_signature_data, $saml_response, 0, $relay_state );
				}
			}
			if ( $response_signature_data ) {
				$saml_required_certificate = $response_signature_data['Certificates'][0];
			} elseif ( $assertion_signature_data ) {
				$saml_required_certificate = $assertion_signature_data['Certificates'][0];
			}
			update_option( Onid_Saml_Sso_Constants::ONID_SAML_REQUIRED_CERTIFICATE, $saml_required_certificate );
			$saml_is_encoding_enabled = get_option( Onid_Saml_Options_Enum_Service_Provider::IS_ENCODING_ENABLED ) ? get_option( Onid_Saml_Options_Enum_Service_Provider::IS_ENCODING_ENABLED ) : 'checked';
			if ( ! $valid_signature ) {

				$desanitized_certificate = Onid_SAML_Utilities::onid_saml_desanitize_certificate( $plugin_cert );
				if ( $saml_required_certificate !== $desanitized_certificate ) {
					$error_code = Onid_Saml_Options_Enum_Error_Codes::$error_codes['WPSAMLERR004'];
					if ( 'testValidate' === $relay_state ) {
						$pem                       = '-----BEGIN CERTIFICATE-----<br>' . chunk_split( $saml_required_certificate, 64 ) . '<br>-----END CERTIFICATE-----';
						$display_metadata_mismatch = '<p><strong>Certificate found in SAML Response: </strong><font face="Courier New";font-size:10pt><br><br>' . $pem . '</p></font>';
						onid_saml_display_test_config_error_page( $error_code, $display_metadata_mismatch );
						exit;
					} else {
						Onid_SAML_Utilities::onid_saml_die( $error_code );
					}
				} elseif ( 'checked' === $saml_is_encoding_enabled ) {
					$error_code = Onid_Saml_Options_Enum_Error_Codes::$error_codes['WPSAMLERR012'];
					if ( 'testValidate' === $relay_state ) {
						onid_saml_display_test_config_error_page( $error_code );
						exit;
					} else {
						Onid_SAML_Utilities::onid_saml_die( $error_code );
					}
				} else {
					wp_die( 'Unable to process the SAML response' );
				}
			}

			$sp_base_url = get_option( Onid_Saml_Options_Enum_Service_Provider::SP_BASE_URL );
			if ( empty( $sp_base_url ) ) {
				$sp_base_url = site_url();
			}
			// verify the issuer and audience from saml response.
			$issuer      = get_option( Onid_Saml_Options_Enum_Service_Provider::ISSUER );
			$sp_enity_id = get_option( Onid_Saml_Options_Enum_Service_Provider::SP_ENTITY_ID );
			if ( empty( $sp_enity_id ) ) {
				$sp_enity_id = $sp_base_url . '/wp-content/plugins/onid-sso-by-saml-2-0/';
			}
			Onid_SAML_Utilities::onid_saml_validate_issuer_and_audience( $saml_response, $sp_enity_id, $issuer, $relay_state );

			try {
				$ssoemail = current( current( $saml_response->onid_saml_get_assertions() )->onid_saml_get_name_id() );
			} catch ( Exception $exception ) {
				wp_die( 'We could not sign you in. Please contact your administrator.', 'Encrypted NameID' );
			}
			$attrs           = current( $saml_response->onid_saml_get_assertions() )->onid_saml_get_attributes();
			$attrs['NameID'] = array( '0' => sanitize_text_field( $ssoemail ) );
			$session_index   = current( $saml_response->onid_saml_get_assertions() )->onid_saml_get_session_index();
			$this->onid_saml_check_mapping( $attrs, $relay_state );
		}
	}

	/**
	 * Map SAML response to WP user data.
	 *
	 * @param array  $attrs array of attribute node of SAML response.
	 *
	 * @param string $relay_state  redirect url for post SSO flow.
	 *
	 * @return void
	 */
	private function onid_saml_check_mapping( $attrs, $relay_state ) {
		try {
			// Get encrypted user_email.
			$email_attribute                           = get_option( Onid_Saml_Options_Enum_Attribute_Mapping::ATTRIBUTE_EMAIL );
			$onid_saml_identity_provider_identifier_name = get_option( Onid_Saml_Options_Enum_Service_Provider::IDENTITY_PROVIDER_NAME ) ? get_option( Onid_Saml_Options_Enum_Service_Provider::IDENTITY_PROVIDER_NAME ) : '';
			if ( ! empty( $onid_saml_identity_provider_identifier_name ) && 'Azure B2C' === $onid_saml_identity_provider_identifier_name ) {
				$email_attribute = 'http://schemas.xmlsoap.org/ws/2005/05/identity/claims/emailaddress';
			}
			$username_attribute = get_option( Onid_Saml_Options_Enum_Attribute_Mapping::ATTRIBUTE_USERNAME );
			$first_name         = get_option( Onid_Saml_Options_Enum_Attribute_Mapping::ATTRIBUTE_FIRST_NAME );
			$last_name          = get_option( Onid_Saml_Options_Enum_Attribute_Mapping::ATTRIBUTE_LAST_NAME );
			$group_name         = get_option( Onid_Saml_Options_Enum_Attribute_Mapping::ATTRIBUTE_GROUP_NAME );
			$default_role       = get_option( Onid_Saml_Options_Enum_Role_Mapping::ROLE_DEFAULT_ROLE );
			$check_if_match_by  = get_option( Onid_Saml_Options_Enum_Attribute_Mapping::ATTRIBUTE_ACCOUNT_MATCHER );
			$user_email         = '';
			$user_name          = '';

			// Attribute mapping. Check if Match/Create user is by username/email:.
			if ( ! empty( $attrs ) ) {
				if ( ! empty( $attrs[ $first_name ] ) ) {
					$first_name = $attrs[ $first_name ][0];
				} else {
					$first_name = '';
				}

				if ( ! empty( $attrs[ $last_name ] ) ) {
					$last_name = $attrs[ $last_name ][0];
				} else {
					$last_name = '';
				}

				if ( ! empty( $attrs[ $username_attribute ] ) ) {
					$user_name = $attrs[ $username_attribute ][0];
				} else {
					$user_name = $attrs['NameID'][0];
				}

				if ( ! empty( $attrs[ $email_attribute ] ) ) {
					$user_email = $attrs[ $email_attribute ][0];
				} else {
					$user_email = $attrs['NameID'][0];
				}

				if ( ! empty( $attrs[ $group_name ] ) ) {
					$group_name = $attrs[ $group_name ];
				} else {
					$group_name = array();
				}

				if ( empty( $check_if_match_by ) ) {
					$check_if_match_by = 'email';
				}
			}

			if ( 'testValidate' === $relay_state ) {
				update_option( Onid_Saml_Options_Test_Configuration::TEST_CONFIG_ERROR_LOG, 'Test successful' );
				update_option( Onid_Saml_Sso_Constants::ONID_SAML_TEST_STATUS, 1 );
				$this->onid_saml_show_test_result( $first_name, $last_name, $user_email, $group_name, $attrs );
			} else {
				$this->onid_saml_login_user( $user_email, $first_name, $last_name, $user_name, $group_name, $default_role, $relay_state, $check_if_match_by );
			}
		} catch ( Exception $e ) {
			printf( 'An error occurred while processing the SAML Response.' );
			exit;
		}
	}

	/**
	 * Show test configuration window after SAML response is processed.
	 *
	 * @param string       $first_name first name of user from the SAML response.
	 *
	 * @param string       $last_name last name of user from the SAML response.
	 *
	 * @param string       $user_email email of user from the SAML response.
	 *
	 * @param array|string $group_name mapped group name of user from the SAML response.
	 *
	 * @param array        $attrs array of attributes received in the attributes node of the SAML response.
	 *
	 * @return void
	 */
	private function onid_saml_show_test_result( $first_name, $last_name, $user_email, $group_name, $attrs ) {
		if ( ob_get_contents() ) {
			ob_end_clean();
		}
		echo '<div style="font-family:Calibri;padding:0 3%;">';
		$name_id = $attrs['NameID'][0];
		if ( ! empty( $user_email ) ) {
			update_option( Onid_Saml_Options_Test_Configuration::TEST_CONFIG_ATTRS, $attrs );
			echo '<div style="color: #3c763d;
					background-color: #dff0d8; padding:2%;margin-bottom:20px;text-align:center; border:1px solid #AEDB9A; font-size:18pt; border-radius:10px;margin-top:17px;">TEST SUCCESSFUL</div>
					<div style="display:block;text-align:center;margin-bottom:4%;"><svg class="animate" width="100" height="100">
					<filter id="dropshadow" height="">
					<feGaussianBlur in="SourceAlpha" stdDeviation="3" result="blur"></feGaussianBlur>
					<feFlood flood-color="rgba(76, 175, 80, 1)" flood-opacity="0.5" result="color"></feFlood>
					<feComposite in="color" in2="blur" operator="in" result="blur"></feComposite>
					<feMerge> 
						<feMergeNode></feMergeNode>
						<feMergeNode in="SourceGraphic"></feMergeNode>
					</feMerge>
					</filter>
					
					<circle cx="50" cy="50" r="46.5" fill="none" stroke="rgba(76, 175, 80, 0.5)" stroke-width="5"></circle>
					
					<path d="M67,93 A46.5,46.5 0,1,0 7,32 L43,67 L88,19" fill="none" stroke="rgba(76, 175, 80, 1)" stroke-width="5" stroke-linecap="round" stroke-dasharray="80 1000" stroke-dashoffset="-220" style="filter:url(#dropshadow)"></path>
				</svg></div>';
		} else {
			echo '<div style="color: #a94442;background-color: #f2dede;padding: 15px;margin-bottom: 20px;text-align:center;border:1px solid #E6B3B2;font-size:18pt;">TEST FAILED</div>
					<div style="color: #a94442;font-size:14pt; margin-bottom:20px;">WARNING: Some Attributes Did Not Match.</div>
					<div style="display:block;text-align:center;margin-bottom:4%;"><img style="width:15%;"src="' . esc_url( plugin_dir_url( __FILE__ ) ) . 'images/wrong.webp"></div>';
		}
		$match_account_by = get_option( Onid_Saml_Options_Enum_Attribute_Mapping::ATTRIBUTE_ACCOUNT_MATCHER ) ? get_option( Onid_Saml_Options_Enum_Attribute_Mapping::ATTRIBUTE_ACCOUNT_MATCHER ) : 'email';
		if ( strlen( $name_id ) > 60 ) {
			echo '<p><font color="#FF0000" style="font-size:14pt;font-weight:bold">Warning: The NameID value is longer than 60 characters. User will not be created during SSO.</font></p>';
		} elseif ( 'email' === $match_account_by && ! filter_var( $name_id, FILTER_VALIDATE_EMAIL ) ) {
			echo '<p><font color="#FF0000" style="font-size:14pt;font-weight:bold">Warning: The NameID value is not a valid Email ID</font></p>';
		}
		echo '<span style="font-size:14pt;"><b>Hello</b>, ' . esc_html( $user_email ) . '</span>';

		echo '<br/><p style="font-weight:bold;font-size:14pt;margin-left:1%;">Attributes Received:</p>
					<table style="border-collapse:collapse;border-spacing:0; display:table;width:100%; font-size:14pt;word-break:break-all;">
					<tr style="text-align:center;background:#d3e1ff;border:2.5px solid #ffffff";word-break:break-all;><td style="font-weight:bold;padding:2%;border-top-left-radius: 10px;border:2.5px solid #ffffff">ATTRIBUTE NAME</td><td style="font-weight:bold;padding:2%;border:2.5px solid #ffffff; word-wrap:break-word;border-top-right-radius:10px">ATTRIBUTE VALUE</td></tr>';

		if ( ! empty( $attrs ) ) {
			foreach ( $attrs as $key => $value ) {
				if ( is_array( $value ) ) {
					$attr_values = implode( '<hr>', $value );
				} else {
					$attr_values = esc_html( $value );
				}
				$allowed_html = array( 'hr' => array() );
				echo "<tr><td style='border:2.5px solid #ffffff;padding:2%;background:#e9f0ff;'>" . esc_html( $key ) . "</td><td style='padding:2%;border:2.5px solid #ffffff;background:#e9f0ff;word-wrap:break-word;'>" . wp_kses( $attr_values, $allowed_html ) . '</td></tr>';
			}
		} else {
			echo 'No Attributes Received.';
		}
		echo '</table></div>';
		exit;
	}

	/**
	 * Change cert fingerprint to correct encoding.
	 *
	 * @param string $cert_fp_from_plugin IdP certificate from the plugin.
	 * @return string
	 */
	private function onid_saml_convert_to_windows_iconv( $cert_fp_from_plugin ) {
		$encoding_enabled = get_option( Onid_Saml_Options_Enum_Service_Provider::IS_ENCODING_ENABLED ) ? get_option( Onid_Saml_Options_Enum_Service_Provider::IS_ENCODING_ENABLED ) : 'checked';

		if ( 'checked' === $encoding_enabled && Onid_SAML_Utilities::onid_saml_is_iconv_installed() ) {
			// phpcs:ignore WordPress.PHP.NoSilencedErrors.Discouraged -- Used to suppress iconv warning.
			return @iconv( Onid_Saml_Options_Enum_Encoding::ENCODING_UTF_8, Onid_Saml_Options_Enum_Encoding::ENCODING_CP1252, $cert_fp_from_plugin );
		}
		return $cert_fp_from_plugin;
	}

	/**
	 * Process SAML response data and Create authentication cookie for users.
	 *
	 * This functions will Process SAML response data, map the user data received from the SAML response to WP User.
	 * If all the parameter are correct then it will check if user exists based on the username and email address received in the SAML response.
	 * If user doesn't exists then it will create a new user.
	 * If SAML response data is not verified then it will exit the process with WP die.
	 *
	 * @param string       $user_email email for user from the SAML response ( NameID or Subject Node in the SAML response).
	 *
	 * @param string       $first_name first name of user from SAML response.
	 *
	 * @param string       $last_name last name of user from SAML response.
	 *
	 * @param string       $user_name user name of user from SAML response.
	 *
	 * @param array|string $group_name group name of user from SAML response.
	 *
	 * @param string       $default_role default role from the plugin configurations saved in DB.
	 *
	 * @param string       $relay_state relay state parameter passed by IDP.
	 *
	 * @param string       $check_if_match_by default username, parameter from which users will be matched.
	 *
	 * @return void
	 */
	private function onid_saml_login_user( $user_email, $first_name, $last_name, $user_name, $group_name, $default_role, $relay_state, $check_if_match_by ) {
		$user_id = null;
		if ( ( 'username' === $check_if_match_by && username_exists( $user_name ) ) || username_exists( $user_name ) ) {
			$user    = get_user_by( 'login', $user_name );
			$user_id = $user->ID;
		} elseif ( email_exists( $user_email ) ) {

			$user    = get_user_by( 'email', $user_email );
			$user_id = $user->ID;
		} elseif ( ! username_exists( $user_name ) && ! email_exists( $user_email ) ) {
			
wp_die( 'We couldn\'t sign you in.<br><b>Please close all browser instances and reopen to retry.</b><br>[this is a security measure to discourage brute force attacks<br>and to keep your account secure and protected]');
exit();	

		}
		if($user_id!=null) { $this->onid_saml_add_firstlast_name( $user_id, $first_name, $last_name, $relay_state ); }
	}

	/**
	 * Check if given user is administrator or not.
	 *
	 * @param wp_user $user wp_user object.
	 * @return bool
	 */
	private function onid_saml_is_administrator_user( $user ) {
		if ( ! is_null( $user->roles ) && in_array( 'administrator', $user->roles, true ) ) {
			return true;
		} else {
			return false;
		}
	}

	/**
	 * Add first and last name of user.
	 *
	 * @param int    $user_id ID of the wp_user.
	 *
	 * @param string $first_name first name of the user.
	 *
	 * @param string $last_name last name of the user.
	 *
	 * @param string $relay_state relay state parameter. URL where the user should be redirected to after authentication.
	 * @return void
	 */
	private function onid_saml_add_firstlast_name( $user_id, $first_name, $last_name, $relay_state ) {
		if ( ! empty( $first_name ) ) {
			$user_id = wp_update_user(
				array(
					'ID'         => $user_id,
					'first_name' => $first_name,
				)
			);
		}
		if ( ! empty( $last_name ) ) {
			$user_id = wp_update_user(
				array(
					'ID'        => $user_id,
					'last_name' => $last_name,
				)
			);
		}

		wp_set_auth_cookie( $user_id, true );

		if ( ! empty( $relay_state ) ) {
			$redirect_url = $relay_state;
		} else {
			$redirect_url = site_url();
		}

		wp_safe_redirect( $redirect_url );
		
		exit;
	}

	/**
	 * Function to show status error code and status message.
	 *
	 * @param string $status_code status code from the SAML response.
	 *
	 * @param string $relay_state relay state parameter from the SAML response.
	 *
	 * @param string $statusmessage status message returned from the IDP in the SAML response.
	 *
	 * @return void
	 */
	private function onid_saml_show_status_error( $status_code, $relay_state, $statusmessage ) {
		$status_code   = sanitize_text_field( $status_code );
		$statusmessage = sanitize_text_field( $statusmessage );
		$error_code    = Onid_Saml_Options_Enum_Error_Codes::$error_codes['WPSAMLERR006'];
		if ( 'testValidate' === $relay_state ) {
			$statusmessage = sprintf( $error_code['testconfig_msg'], $status_code );
			onid_saml_display_test_config_error_page( $error_code, '', $statusmessage );
			exit;
		} else {
			Onid_SAML_Utilities::onid_saml_die( $error_code );
		}
	}

	/**
	 * Parse relay state parameter.
	 *
	 * @param string $relay_state Relay state from the IDP.
	 *
	 * @return array|bool|int|null|string
	 */
	private function onid_saml_get_relay_state( $relay_state ) {

		if ( 'testValidate' === $relay_state ) {
			return $relay_state;
		}

		$relay_path = wp_parse_url( $relay_state, PHP_URL_PATH );
		if ( wp_parse_url( $relay_state, PHP_URL_QUERY ) ) {
			$relay_query_paramter = wp_parse_url( $relay_state, PHP_URL_QUERY );
			$relay_path           = $relay_path . '?' . $relay_query_paramter;
		}
		if ( wp_parse_url( $relay_state, PHP_URL_FRAGMENT ) ) {
			$relay_fragment_identifier = wp_parse_url( $relay_state, PHP_URL_FRAGMENT );
			$relay_path                = $relay_path . '#' . $relay_fragment_identifier;
		}

		return $relay_path;
	}
}

$onid_saml_login_validate = new Onid_SAML_Login_Validate();
