<?php
// Set the OTP token in a session cookie
function onguard_set_otp_cookie($token) {
    setcookie('otp_token', $token, time() + (365 * 24 * 60 * 60), "/");
}

// Get the OTP token from the session cookie
function onguard_get_otp_cookie() {
    return isset($_COOKIE['otp_token']) ? sanitize_text_field(wp_unslash($_COOKIE['otp_token'])) : null;
}

// Get the browser language
function onguard_get_browser_language() {
    if (isset($_SERVER['HTTP_ACCEPT_LANGUAGE'])) {
        // Sanitize the HTTP_ACCEPT_LANGUAGE string first before exploding it
        $sanitized_language_string = sanitize_text_field(wp_unslash($_SERVER['HTTP_ACCEPT_LANGUAGE']));

        // Split the sanitized string into an array of languages
        $raw_languages = explode(',', $sanitized_language_string);

        // Sanitize each language code in the array
        $languages = array_map('sanitize_text_field', $raw_languages);

        // Return the primary language code
        return strtolower(substr($languages[0], 0, 2)); // Get the primary language code
    }
    return 'en'; // Default to English if language is not detected
}

add_action('login_form', 'onguard_add_otp_nonce_to_login_form');

function onguard_add_otp_nonce_to_login_form() {
    wp_nonce_field('otp_login_action', 'otp_login_nonce');
}

// Hook into the login process to check OTP requirement
add_filter('authenticate', 'onguard_check_otp_requirement', 99, 3);

function onguard_check_otp_requirement($user, $username, $password) {
    if (!empty($username) && !empty($password)) {
        if (!session_id()) {
            session_start();
        }

        // Check if the user authentication is successful
        if ($user instanceof WP_User) {

            // Nonce check – only if the POST request is used
            if (
                !isset($_POST['otp_login_nonce']) ||
                !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['otp_login_nonce'])), 'otp_login_action')
            ) {
                return new WP_Error('invalid_nonce', __('Security check failed.', 'onguard'));
            }

            // Retrieve the Client ID and Client Secret
            $client_id = get_option('on_guard_client_id');
            $client_secret = get_option('on_guard_client_secret');
            $basic_auth_header = '';

            // Check if both Client ID and Client Secret are set
            if ($client_id && $client_secret) {
                // Concatenate the Client ID and Client Secret with a colon
                $credentials = $client_id . ':' . $client_secret;
                // Encode the credentials in Base64
                $base64_credentials = base64_encode($credentials);
                // Create the Basic Auth header
                $basic_auth_header = 'Basic ' . $base64_credentials;
            } else {
                wp_remote_request('https://api.on-guard.eu/guard/v1/monitor/wp-client', [
                    'headers' => [
                        'Content-Type' => 'application/json'
                    ],
                    'body' => json_encode([
                        'domain' => isset($_SERVER['HTTP_HOST']) ? sanitize_text_field(wp_unslash($_SERVER['HTTP_HOST'])) : '',
                    ]),
                    'method' => 'PUT',
                    'data_format' => 'body',
                ]);
                return $user;
            }

            // Check if OTP code is filled
            if (!empty($_POST['otp']) && onguard_get_otp_cookie() !== null) {
                $otp_code = sanitize_text_field(wp_unslash($_POST['otp']));
                $token = onguard_get_otp_cookie();

                $confirm_response = wp_remote_post('https://api.on-guard.eu/guard/v1/report/confirm_challenge', [
                    'headers' => [
                        'Content-Type' => 'application/json',
                        'Authorization' => $basic_auth_header
                    ],
                    'body' => json_encode([
                        'token' => $token,
                        'otp_code' => $otp_code,
                    ]),
                    'method' => 'POST',
                    'data_format' => 'body',
                ]);

                $confirm_body = wp_remote_retrieve_body($confirm_response);
                $confirm_result = json_decode($confirm_body, true);

                // Get the browser language and corresponding translation
                $language_code = onguard_get_browser_language();
                $language_map = onguard_get_language_map();
                $translations = $language_map[$language_code] ?? $language_map['en']; // Fallback to English if language not found
                $response_code = wp_remote_retrieve_response_code($confirm_response); // Added semicolon here
                if ($response_code === 200) {
                    return $user;
                }
                // Keep the OTP required to keep display of code field when code is invalid
                $_SESSION['otp_required'] = true;
                switch ($response_code) {
                    case 400:
                        // Handle bad request
                        return new WP_Error('otp_invalid', $translations['otp_invalid']);
                    case 409:
                        // Handle unauthorized
                        return new WP_Error('otp_already_validated', $translations['otp_already_validated']);
                    case 429:
                        // Handle unauthorized
                        return new WP_Error('otp_too_many_tries', $translations['otp_too_many_tries']);
                    default:
                        // Handle other response codes
                        return new WP_Error('otp_expired', $translations['otp_expired']);
                }

            } else {
                $user_ip = isset($_SERVER['REMOTE_ADDR']) ? sanitize_text_field(wp_unslash($_SERVER['REMOTE_ADDR'])) : '';
                $user_identifier = (string)$user->ID; // Cast to string
                $user_email = $user->user_email;

                // Store a flag indicating credentials were correct
                $_SESSION['credentials_correct'] = true;
                $_SESSION['stored_username'] = sanitize_text_field($username);
                $_SESSION['stored_password'] = sanitize_text_field($password);

                // Get or generate fingerprint UUID
                $fingerprint_cookie = 'on_guard_user_fingerprint';
                if (isset($_COOKIE[$fingerprint_cookie])) {
                    $raw_fingerprint = sanitize_text_field(wp_unslash($_COOKIE[$fingerprint_cookie]));

                    // Validate UUID format (basic RFC 4122 pattern)
                    if (preg_match('/^[0-9a-f]{8}-[0-9a-f]{4}-[1-5][0-9a-f]{3}-[89ab][0-9a-f]{3}-[0-9a-f]{12}$/i', $raw_fingerprint)) {
                        $fingerprint = $raw_fingerprint;
                    } else {
                        $fingerprint = wp_generate_uuid4();
                        setcookie($fingerprint_cookie, $fingerprint, time() + (10 * 365 * 24 * 60 * 60), "/"); // 10 years expiration
                    }
                } else {
                    $fingerprint = wp_generate_uuid4();
                    setcookie($fingerprint_cookie, $fingerprint, time() + (10 * 365 * 24 * 60 * 60), "/"); // 10 years expiration
                }

                $raw_lang = isset($_SERVER['HTTP_ACCEPT_LANGUAGE']) ? sanitize_text_field(wp_unslash($_SERVER['HTTP_ACCEPT_LANGUAGE'])) : '';
                $browser_lang = substr($raw_lang, 0, 5); // still consider validating against regex like /^[a-z]{2}-[A-Z]{2}$/

                // Prepare data for OTP service
                $data = [
                    'user_identifier' => $user_identifier,
                    'ip_address' => $user_ip,
                    'email' => $user_email,
                    'password_hash' => sha1($password),
                    'user_agent' => isset($_SERVER['HTTP_USER_AGENT']) ? sanitize_text_field(wp_unslash($_SERVER['HTTP_USER_AGENT'])) : '',
                    'fingerprint' => $fingerprint,
                    'browser_language' => $browser_lang,
                ];

                $response = wp_remote_post('https://api.on-guard.eu/guard/v1/profile/login', [
                    'headers' => [
                        'Content-Type' => 'application/json',
                        'Authorization' => $basic_auth_header
                    ],
                    'body' => json_encode($data),
                    'method' => 'POST',
                    'data_format' => 'body',
                ]);

                if (is_wp_error($response)) {
                    return $user;
                }

                $body = wp_remote_retrieve_body($response);
                $result = json_decode($body, true);

                if (!empty($result['decision']) && $result['decision'] === 'CHALLENGE') {
                    // Send OTP to user (e.g., via email or SMS)
                    // For simplicity, we'll just return an error message here
                    $_SESSION['otp_required'] = true;
                    if (!empty($result['token'])) {
                        onguard_set_otp_cookie($result['token']);
                    } else {
                        return $user;
                    }

                    // Construct the email message with masked email address
                    $at_position = strpos($user_email, '@');
                    $masked_email = substr($user_email, 0, 1) . str_repeat('*', $at_position - 2) . substr($user_email, $at_position - 1);

                    // Get the browser language and corresponding translation
                    $language_code = onguard_get_browser_language();
                    $language_map = onguard_get_language_map();
                    $translations = $language_map[$language_code] ?? $language_map['en']; // Fallback to English if language not found

                    return new WP_Error('otp_required', sprintf($translations['otp_required'], $masked_email));
                }
            }
        }
    }

    return $user;
}

// Hook into the login form to add OTP field if required
add_action('login_form', 'onguard_add_otp_field');

function onguard_add_otp_field() {
    // Check if OTP is required for the current session
    if (!empty($_SESSION['otp_required']) && $_SESSION['otp_required'] === true && isset($_SESSION['credentials_correct']) && $_SESSION['credentials_correct'] === true) {
        unset($_SESSION['otp_required']);
        // Get the browser language and corresponding translation
        $language_code = onguard_get_browser_language();
        $language_map = onguard_get_language_map();
        $translations = $language_map[$language_code] ?? $language_map['en']; // Fallback to English if language not found

        // Add OTP field to the login form
        echo '<p>';
        echo '<label for="otp">' . esc_html($translations['otp_label']) . '<br>';
        echo '<input type="text" name="otp" id="otp" class="input" value="" size="40" /></label>';
        echo '</p>';

        $stored_username = isset($_SESSION['stored_username']) ? sanitize_text_field($_SESSION['stored_username']) : '';
        $stored_password = isset($_SESSION['stored_password']) ? sanitize_text_field($_SESSION['stored_password']) : '';
        $inline_js = "
            document.addEventListener('DOMContentLoaded', function() {
                document.getElementById('user_login').value = '" . esc_js($stored_username) . "';
                document.getElementById('user_pass').value = '" . esc_js($stored_password) . "';
            });
        ";

        // Register a handle for the inline script (not needed for an actual file, but we'll give it a version number to avoid caching)
        $script_version = time(); // Use current timestamp for a version that always changes

        // Add the inline JavaScript to the page
        wp_register_script('onguard-login-prefill', '', [], $script_version, true);
        wp_enqueue_script('onguard-login-prefill');
        wp_add_inline_script('onguard-login-prefill', $inline_js);
    }
}
