<?php
/**
 * Plugin Name: One Click Delete Page and Post
 * Description: Adds a "Delete" dropdown item to the frontend admin bar "Edit" menu.
 * Version: 1.1.1
 * Author: magnathorax
 * License: GPLv2 or later
 * Text Domain: oneclick-delete-page-and-post
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/* include settings page if present */
if ( file_exists( plugin_dir_path( __FILE__ ) . 'dpp-settings.php' ) ) {
	require_once plugin_dir_path( __FILE__ ) . 'dpp-settings.php';
}

class DPP_Delete_Page_Post {

	public function __construct() {
		add_action( 'admin_bar_menu', array( $this, 'maybe_add_admin_bar_item' ), 100 );
		add_action( 'wp_enqueue_scripts', array( $this, 'enqueue_scripts' ) );
		add_action( 'wp_footer', array( $this, 'print_hidden_form' ) );
		add_action( 'admin_post_delete_page_post', array( $this, 'handle_delete_request' ) );
	}

	public function maybe_add_admin_bar_item( $wp_admin_bar ) {
		if ( ! is_admin_bar_showing() || is_admin() ) {
			return;
		}
		if ( ! is_singular() ) {
			return;
		}
		$post = get_queried_object();
		if ( ! $post || empty( $post->ID ) ) {
			return;
		}
		$post_id = (int) $post->ID;
		if ( ! current_user_can( 'delete_post', $post_id ) ) {
			return;
		}

		$parent = 'edit';
		if ( ! $wp_admin_bar->get_node( $parent ) ) {
			$parent = false;
		}

		$wp_admin_bar->add_node( array(
			'id'     => 'dpp-delete',
			'parent' => $parent,
			'title'  => esc_html__( 'Delete', 'oneclick-delete-page-and-post' ),
			'href'   => '#dpp-delete',
			'meta'   => array(
				'class'        => 'dpp-delete-link',
				'data-post-id' => (string) $post_id,
				'data-nonce'   => wp_create_nonce( 'delete_page_post_action' ),
				'title'        => esc_attr__( 'Delete this page/post', 'oneclick-delete-page-and-post' ),
			),
		) );
	}

	public function enqueue_scripts() {
		if ( ! is_admin_bar_showing() || is_admin() || ! is_singular() ) {
			return;
		}
		$post = get_queried_object();
		if ( ! $post || empty( $post->ID ) ) {
			return;
		}
		$post_id = (int) $post->ID;
		if ( ! current_user_can( 'delete_post', $post_id ) ) {
			return;
		}

		$fm  = filemtime( __FILE__ );
		$ver = ( $fm !== false ) ? (int) $fm : time();

		wp_register_script( 'dpp-delete-script', '', array(), $ver, true );
		wp_enqueue_script( 'dpp-delete-script' );
		wp_register_style( 'dpp-delete-style', '', array(), $ver );
		wp_enqueue_style( 'dpp-delete-style' );

		$css = "
#wp-admin-bar-dpp-delete > a.ab-item {
	display: inline-flex;
	align-items: center;
	gap: .5em;
	white-space: nowrap;
}
#wp-admin-bar-dpp-delete > a.ab-item::before {
	content: \"\\f182\";
	display: inline-block;
	height: auto;
}
#wpadminbar .ab-top-menu > li#wp-admin-bar-dpp-delete > a.ab-item,
#wpadminbar .ab-top-secondary > li#wp-admin-bar-dpp-delete > a.ab-item {
	display: inline-flex;
	align-items: center;
}
#wp-admin-bar-dpp-delete .ab-item,
#wp-admin-bar-dpp-delete > a.ab-item {
	vertical-align: middle;
}
";
		wp_add_inline_style( 'dpp-delete-style', $css );

		$show_confirm = get_option( 'dpp_show_confirm', '1' );
		$show_confirm = ( $show_confirm === '1' || $show_confirm === 1 || $show_confirm === true );

		if ( $show_confirm ) {
			$confirm_text = __( 'Are you sure you want to delete this page/post?', 'oneclick-delete-page-and-post' );
			$confirm_js   = wp_json_encode( $confirm_text );
		} else {
			$confirm_js = 'null';
		}

		$admin_post_url = esc_url_raw( admin_url( 'admin-post.php' ) );
		$admin_post_js  = wp_json_encode( $admin_post_url );

		$inline_js = '(function(){' .
			'function closestByClass(el, className){' .
				'while(el && el !== document.documentElement){' .
					'if(el.classList && el.classList.contains(className)) return el;' .
					'el = el.parentNode;' .
				'}' .
				'return null;' .
			'}' .
			'document.addEventListener("click", function(e){' .
				'var link = closestByClass(e.target, "dpp-delete-link");' .
				'if(!link) return;' .
				'e.preventDefault();' .
				'var confirmText = ' . $confirm_js . ';' .
				'if (confirmText !== null) {' .
					'if (! window.confirm(confirmText)) { return; }' .
				'}' .
				'var form = document.getElementById("dpp-delete-form");' .
				'if (! form ) {' .
					'form = document.createElement("form");' .
					'form.style.display = "none";' .
					'form.method = "post";' .
					'form.action = ' . $admin_post_js . ';' .
					'var actionField = document.createElement("input");' .
					'actionField.type = "hidden";' .
					'actionField.name = "action";' .
					'actionField.value = "delete_page_post";' .
					'form.appendChild(actionField);' .
					'var pidField = document.createElement("input");' .
					'pidField.type = "hidden";' .
					'pidField.name = "post_id";' .
					'pidField.value = link.getAttribute("data-post-id") || "";' .
					'form.appendChild(pidField);' .
					'var nonce = link.getAttribute("data-nonce") || "";' .
					'if (nonce) {' .
						'var nonceField = document.createElement("input");' .
						'nonceField.type = "hidden";' .
						'nonceField.name = "delete_page_post_nonce";' .
						'nonceField.value = nonce;' .
						'form.appendChild(nonceField);' .
					'}' .
					'document.body.appendChild(form);' .
					'form.submit();' .
					'return;' .
				'}' .
				'var idInput = document.getElementById("dpp-delete-post-id");' .
				'if (idInput) {' .
					'idInput.value = link.getAttribute("data-post-id") || idInput.value;' .
				'} else {' .
					'var pidField = document.createElement("input");' .
					'pidField.type = "hidden";' .
					'pidField.name = "post_id";' .
					'pidField.id = "dpp-delete-post-id";' .
					'pidField.value = link.getAttribute("data-post-id") || "";' .
					'form.appendChild(pidField);' .
				'}' .
				'form.submit();' .
			'}, false);' .
		'})();';

		wp_add_inline_script( 'dpp-delete-script', $inline_js );
	}

	public function print_hidden_form() {
		if ( is_admin() ) {
			return;
		}
		if ( ! is_singular() || ! is_admin_bar_showing() ) {
			return;
		}
		$post = get_queried_object();
		if ( ! $post || empty( $post->ID ) ) {
			return;
		}
		$post_id = (int) $post->ID;
		if ( ! current_user_can( 'delete_post', $post_id ) ) {
			return;
		}
		$action_url = esc_url( admin_url( 'admin-post.php' ) );
		?>
		<form id="dpp-delete-form" method="post" action="<?php echo esc_attr( $action_url ); ?>" style="display:none;">
			<input type="hidden" name="action" value="delete_page_post" />
			<input type="hidden" name="post_id" id="dpp-delete-post-id" value="<?php echo esc_attr( $post_id ); ?>" />
			<?php wp_nonce_field( 'delete_page_post_action', 'delete_page_post_nonce' ); ?>
		</form>
		<?php
	}

	public function handle_delete_request() {
		if ( ! is_user_logged_in() ) {
			wp_die(
				esc_html__( 'You must be logged in to perform that action.', 'oneclick-delete-page-and-post' ),
				esc_html__( 'Error', 'oneclick-delete-page-and-post' ),
				array( 'response' => 403 )
			);
		}

		$nonce_raw = filter_input( INPUT_POST, 'delete_page_post_nonce', FILTER_SANITIZE_FULL_SPECIAL_CHARS );
		$nonce     = sanitize_text_field( (string) $nonce_raw );

		if ( '' === $nonce ) {
			wp_die(
				esc_html__( 'Missing nonce.', 'oneclick-delete-page-and-post' ),
				esc_html__( 'Error', 'oneclick-delete-page-and-post' ),
				array( 'response' => 400 )
			);
		}

		if ( ! wp_verify_nonce( $nonce, 'delete_page_post_action' ) ) {
			wp_die(
				esc_html__( 'Security check failed.', 'oneclick-delete-page-and-post' ),
				esc_html__( 'Error', 'oneclick-delete-page-and-post' ),
				array( 'response' => 403 )
			);
		}

		$post_id_raw = filter_input( INPUT_POST, 'post_id', FILTER_SANITIZE_NUMBER_INT );
		$post_id = absint( $post_id_raw );
		if ( $post_id <= 0 ) {
			wp_die(
				esc_html__( 'Invalid post ID.', 'oneclick-delete-page-and-post' ),
				esc_html__( 'Error', 'oneclick-delete-page-and-post' ),
				array( 'response' => 400 )
			);
		}

		if ( ! current_user_can( 'delete_post', $post_id ) ) {
			wp_die(
				esc_html__( 'You do not have permission to delete this post.', 'oneclick-delete-page-and-post' ),
				esc_html__( 'Error', 'oneclick-delete-page-and-post' ),
				array( 'response' => 403 )
			);
		}

		$deleted = wp_delete_post( $post_id, false );

		$redirect = wp_get_referer();
		if ( ! $redirect ) {
			$redirect = home_url();
		}
		$redirect = wp_validate_redirect( $redirect, home_url() );

		if ( $deleted ) {
			$redirect = add_query_arg( 'dpp_deleted', '1', $redirect );
		} else {
			$redirect = add_query_arg( 'dpp_deleted', '0', $redirect );
		}

		wp_safe_redirect( $redirect );
		exit;
	}
}

new DPP_Delete_Page_Post();
