<?php
/**
 * Import handlers for NR Post Exporter plugin.
 *
 * @package NR_Post_Exporter
 */

namespace Nikolareljin\NrPostExporter\Post;

/**
 * Handles post import form and reconstruction of posts from JSON.
 */
class Import {

	const ENCODE_OPTIONS = JSON_HEX_APOS; // JSON_HEX_TAG | JSON_HEX_APOS | JSON_HEX_QUOT | JSON_HEX_AMP | JSON_UNESCAPED_UNICODE.

	/**
	 * Renders the button that will trigger the Import.
	 * Create a new post from a JSON file data.
	 *
	 * @return void
	 */
	public static function import_post_button() {
			// Render HTML form for uploading the JSON file in the format generated by Export::class.
		?>
		<form
			action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>"
			id="nrpexp_import_form"
			enctype="multipart/form-data"
			method="post">
			<input type="hidden" name="action" value="nrpexp_import"/>
			<input type="file" accept="application/json" id="upload" name="upload"/>
			<?php
			wp_nonce_field( 'nrpexp_import', 'post_import_nonce' );
			submit_button( __( 'Import Post', 'nr-post-exporter' ), 'secondary', 'submit', false );
			?>
		</form>
		<?php
	}

	/**
	 * Imports the post data from the JSON file.
	 *
	 * @return void
	 */
	public static function post_import() {
		if ( isset( $_SERVER['REQUEST_METHOD'] ) && 'POST' === $_SERVER['REQUEST_METHOD'] ) {
			// Capability check: require ability to edit posts.
			if ( ! current_user_can( 'edit_posts' ) ) {
				wp_die( esc_html__( 'Insufficient permissions.', 'nr-post-exporter' ) );
			}

			// Check if form was submitted. Then process the file.
			if ( ! isset( $_POST['post_import_nonce'] ) || ! check_admin_referer( 'nrpexp_import', 'post_import_nonce' ) ) {
				wp_die( esc_html__( 'Sorry, your nonce did not verify.', 'nr-post-exporter' ) );
			}

			$action = isset( $_POST['action'] ) ? sanitize_key( wp_unslash( $_POST['action'] ) ) : '';
			$submit = isset( $_POST['submit'] );

			if ( 'nrpexp_import' === $action && $submit ) {
				// Now, parse the JSON file and create a new post.
				$upload_name = isset( $_FILES['upload']['name'] ) ? sanitize_text_field( wp_unslash( $_FILES['upload']['name'] ) ) : '';
				if ( $upload_name ) {
					if ( empty( $_FILES['upload']['error'] ) ) {
							// Validate the file.
						$new_file_name = isset( $_FILES['upload']['tmp_name'] ) ? sanitize_text_field( wp_unslash( $_FILES['upload']['tmp_name'] ) ) : '';
						if ( ! file_exists( $new_file_name ) ) {
							wp_die( esc_html__( 'File does not exist!', 'nr-post-exporter' ) . ' ' . esc_html( $new_file_name ) );
						}
							// Can't be larger than ~1MB.
						if ( isset( $_FILES['upload']['size'] ) && (int) $_FILES['upload']['size'] > 1000000 ) {
										wp_die( esc_html__( 'Your file size is too large.', 'nr-post-exporter' ) );
						} else {
							try {
								// Get current user's ID. We will update each reference of the author with our current user ID.
								$author_id = get_current_user_id();

								// Process the file.
								// phpcs:ignore WordPress.WP.AlternativeFunctions.file_get_contents_file_get_contents -- Reading local uploaded JSON file.
								$post_json      = file_get_contents( $new_file_name );
								$full_post_data = json_decode( $post_json, true, 30, JSON_THROW_ON_ERROR );

								// This is the data we will save to the post.
								$post_data                 = self::get_post_data_set( $full_post_data );
								$post_data['post_author']  = $author_id;
								$tmp                       = $post_data['post_content'];
								$post_data['post_content'] = self::process_json( $tmp );
								// Further normalize to eliminate double-encoded UTF-8 sequences.
								$post_data['post_content'] = self::normalize_encoding( $post_data['post_content'] );

								// Import Post meta (stored in $post_data['post_meta']).
								if ( isset( $full_post_data['post_meta'] ) ) {
									$post_meta               = $full_post_data['post_meta'];
									$post_data['meta_input'] = $post_meta;
								}

								$initial_post_data = self::get_post_data_set( $post_data );

								if ( isset( $full_post_data['revisions'] ) && array() !== $full_post_data['revisions'] ) {
									$ordered_revisions = self::order_revisions( $full_post_data['revisions'] );
									$initial_post_data = self::get_post_data_set( $ordered_revisions[0] ?? $post_data );
								}

								// Initial step - no real data in the post.
								$initial_post_data['post_parent'] = 0;
								$initial_post_data['post_author'] = $author_id;

								$post_id = wp_insert_post( $initial_post_data, false, false );
								wp_save_post_revision( $post_id );

								// If containing the 'revisions', process them first.
								if ( isset( $full_post_data['revisions'] ) && array() !== $full_post_data['revisions'] ) {
									$revisions = $full_post_data['revisions'];

									// Go in reverse order, so that the latest revision is the parent.
									$revisions = array_reverse( $revisions );
									foreach ( $revisions as $revision ) {
										$post_meta = $revision['post_meta'] ?? null;
										unset( $revision['post_meta'] );

										$tmp                      = $revision['post_content'];
										$tmp                      = self::process_json( $tmp );
										$tmp                      = self::normalize_encoding( $tmp );
										$revision['post_content'] = $tmp;

										// Override to the parent.
										unset( $revision['post_parent'] );
										// Needed to pass all of the changes in the document.
										// And not just for the final document type.
										unset( $revision['post_type'] );

										$revision['ID']          = $post_id;
										$revision['post_author'] = $author_id;

										// Import Post meta (stored in $post_data['post_meta']).
										if ( isset( $post_meta ) ) {
											$revision['meta_input'] = $post_meta;
										}

										wp_update_post( $revision, true );
										wp_save_post_revision( $post_id );

										// Update revision dates directly for proper ordering.
										global $wpdb;
										$wp_posts_table = $wpdb->prefix . 'posts';
										// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Updating revision timestamps for proper ordering.
										$wpdb->update(
											$wp_posts_table,
											array(
												'post_date'         => $revision['post_date'],
												'post_date_gmt'     => $revision['post_date_gmt'],
												'post_modified'     => $revision['post_modified'],
												'post_modified_gmt' => $revision['post_modified_gmt'],
											),
											array(
												'post_parent' => $post_id,
												'post_type'   => 'revision',
												'post_status' => 'inherit',
											)
										);
									}
								}

								// Store the current post data.
								$post_data['ID'] = $post_id;
								// Add prefix to the title - to allow easier detection of the imported posts.
								/* translators: %s: original post title. */
								$post_data['post_title'] = sprintf( __( 'Imported: %s', 'nr-post-exporter' ), $post_data['post_title'] );
								wp_update_post( $post_data );
								wp_save_post_revision( $post_id );

								// Import taxonomy terms if provided in the export payload.
								if ( isset( $full_post_data['terms'] ) && is_array( $full_post_data['terms'] ) ) {
									foreach ( $full_post_data['terms'] as $taxonomy => $slugs ) {
										if ( empty( $slugs ) || ! taxonomy_exists( $taxonomy ) ) {
											continue;
										}

										$term_ids = array();
										foreach ( (array) $slugs as $slug ) {
											$existing = get_term_by( 'slug', $slug, $taxonomy );
											if ( $existing && ! is_wp_error( $existing ) ) {
												$term_ids[] = (int) $existing->term_id;
												continue;
											}

											$inserted = wp_insert_term( $slug, $taxonomy, array( 'slug' => $slug ) );
											if ( ! is_wp_error( $inserted ) ) {
												$term_ids[] = (int) ( $inserted['term_id'] ?? 0 );
											}
										}

										if ( ! empty( $term_ids ) ) {
											wp_set_object_terms( $post_id, $term_ids, $taxonomy, false );
										}
									}
								}
							} catch ( \Exception $e ) {
								wp_die( esc_html( $e->getMessage() ) );
							}
						}

						// Open the dashboard and list of all the articles.
						wp_safe_redirect( admin_url( 'edit.php' ) );
						exit;
					}
				} else {
					wp_die( esc_html__( 'No file was uploaded.', 'nr-post-exporter' ) );
				}
			}
		}
	}

	/**
	 * Prepare the needed fields for the wp_insert_post function.
	 *
	 * @param array $post_data Post data parsed from import file.
	 * @return array Prepared data for wp_insert_post().
	 */
	public static function get_post_data_set( $post_data ) {
		$data = array(
			'post_author'       => $post_data['post_author'] ?? '',
			'post_content'      => $post_data['post_content'] ?? '',
			'post_title'        => $post_data['post_title'] ?? '',
			'post_excerpt'      => $post_data['post_excerpt'] ?? '',
			'post_status'       => $post_data['post_status'] ?? '',
			'comment_status'    => $post_data['comment_status'] ?? '',
			'ping_status'       => $post_data['ping_status'] ?? '',
			'post_password'     => $post_data['post_password'] ?? '',
			'post_name'         => $post_data['post_name'] ?? '',
			'to_ping'           => $post_data['to_ping'] ?? '',
			'pinged'            => $post_data['pinged'] ?? '',
			'post_parent'       => $post_data['post_parent'] ?? '',
			'guid'              => $post_data['guid'] ?? '',
			'menu_order'        => $post_data['menu_order'] ?? '',
			'post_type'         => $post_data['post_type'] ?? '',
			'post_mime_type'    => $post_data['post_mime_type'] ?? '',
			'filter'            => $post_data['filter'] ?? '',
			'meta_input'        => $post_data['meta_input'] ?? ( $post_data['post_meta'] ?? '' ),
			// Post dates.
			'post_date'         => $post_data['post_date'] ?? '',
			'post_date_gmt'     => $post_data['post_date_gmt'] ?? '',
			'post_modified'     => $post_data['post_modified'] ?? '',
			'post_modified_gmt' => $post_data['post_modified_gmt'] ?? '',
		);

		return $data;
	}

	/**
	 * Process the JSON data and add slashes where needed.
	 * This is needed because the JSON data is double-encoded.
	 * Reference: https://core.trac.wordpress.org/ticket/47420.
	 *
	 * @param mixed $input Raw JSON string or nested array.
	 * @return mixed Sanitized and normalized content.
	 */
	public static function process_json( $input ) {
		$output = $input;

				// We need to convert the double-encoded characters to the original characters.
				// Reference: https://core.trac.wordpress.org/ticket/47420.
		try {
			// First pass: generic decode/escape.
			if ( is_array( $output ) ) {
				array_walk_recursive(
					$output,
					function ( &$item ) {
						if ( is_string( $item ) ) {
							$item = wp_slash( wp_specialchars_decode( $item, ENT_QUOTES ) );
						}
					}
				);
			} elseif ( is_string( $output ) ) {
				$output = wp_slash( wp_specialchars_decode( $output, ENT_QUOTES ) );
			}

					// Second pass: handle DJ-prefixed custom blocks, if present.
			if ( is_string( $output ) ) {
				$output = preg_replace_callback(
					'#<!-- wp:dj\/([a-zA-Z0-9\-]*) ([{].*?[}]) (\/)?-->(.*<!-- /wp:dj\/\1 -->)?#',
					function ( $matches ) {
						$block            = $matches[0];
						$block_name       = $matches[1];
						$block_attributes = $matches[2];
							// Optional elements.
						$block_internal_closure          = $matches[3] ?? '';
						$block_content_and_final_closure = $matches[4] ?? '';
						$block_attributes                = json_decode( $block_attributes, true );

						if ( $block_attributes ) {
							// Process each of the attributes of the keys.
							foreach ( $block_attributes as $key => $value ) {
								if ( is_string( $value ) ) {
									$block_attributes[ $key ] = self::unserialize_attributes( $value );
								}
								if ( is_array( $value ) ) {
									foreach ( $value as $key2 => $value2 ) {
										if ( is_string( $value2 ) ) {
											$block_attributes[ $key ][ $key2 ] = self::unserialize_attributes( $value2 );
										}
									}
								}
							}
							$block_attributes = wp_json_encode( $block_attributes, self::ENCODE_OPTIONS );

							$block = "<!-- wp:dj/{$block_name} {$block_attributes} {$block_internal_closure}-->{$block_content_and_final_closure}";

							return $block;
						}

						return $block;
					},
					$output
				);
			}
		} catch ( \Exception $e ) {
								// Fall back to original payload.
								do_action( 'nrpexp_import_error', $e );
		}

		return $output;
	}

	/**
	 * Order revisions by the `post_date` value ascending; ensure oldest first.
	 *
	 * @param array $revisions List of revision arrays.
	 * @return array Ordered revisions from oldest to newest.
	 */
	public static function order_revisions( array $revisions ): array {
		$ordered_revisions = array();
		foreach ( $revisions as $revision ) {
			if ( isset( $revision['post_date_gmt'] ) ) {
				$ordered_revisions[ $revision['post_date_gmt'] ] = $revision;
			}
		}
		ksort( $ordered_revisions );
		return array_values( $ordered_revisions );
	}

	/**
	 * Normalize the encoding of content to eliminate double-encoded Unicode.
	 * Converts HTML entities to proper UTF-8 and applies slashes for DB writes.
	 *
	 * @param string $content Content to normalize.
	 * @return string Normalized content string.
	 */
	public static function normalize_encoding( $content ) {
		$result = (string) $content;
		$result = addslashes( $result );
		$result = html_entity_decode( $result, ENT_QUOTES | ENT_HTML5, 'UTF-8' );
		return $result;
	}

	/**
	 * Unserialize double-encoded attributes within block comments.
	 * Used as a reference: https://core.trac.wordpress.org/ticket/47420.
	 *
	 * @param string $content Block attributes content.
	 * @return string Unserialized attribute string.
	 */
	public static function unserialize_attributes( $content ) {
		$output = $content;

		// Unserialize attributes (previously done by Gutenberg's serializeAttributes()).
		$output = preg_replace_callback(
			'#\&quot;([a-zA-Z0-9\-]*)\&quot;:(\&quot;.*?\&quot;|\[.*?\])#',
			function ( $matches ) {
				$attribute_name  = $matches[1];
				$attribute_value = $matches[2];
				$attribute_value = str_replace( '&quot;', '"', $attribute_value );
				$attribute_value = str_replace( '\\', '', $attribute_value );

				return "\"{$attribute_name}\":{$attribute_value}";
			},
			$output
		);

		$output = addslashes( $output );

		return $output;
	}
}
