<?php
/**
 * Settings Page
 *
 * @package Email_Safe_View_Fallback_Viewer
 */

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class NOBSMV_Settings
 *
 * Handles plugin settings page and options
 */
class NOBSMV_Settings {

	/**
	 * Option group name
	 *
	 * @var string
	 */
	const OPTION_GROUP = 'nobsmv_settings';

	/**
	 * Settings page slug
	 *
	 * @var string
	 */
	const PAGE_SLUG = 'nobs-mail-view-email-fallback-viewer-in-browser';

	/**
	 * Initialize the class
	 */
	public function __construct() {
		add_action( 'admin_menu', [ $this, 'add_settings_page' ] );
		add_action( 'admin_init', [ $this, 'register_settings' ] );
	}

	/**
	 * Add settings page to WordPress admin menu
	 */
	public function add_settings_page() {
		add_options_page(
			__( 'NoBS-Mail View - Email Fallback Viewer in Browser Settings', 'nobs-mail-view-email-fallback-viewer-in-browser' ),
			__( 'NoBS-Mail View', 'nobs-mail-view-email-fallback-viewer-in-browser' ),
			'manage_options',
			self::PAGE_SLUG,
			[ $this, 'render_settings_page' ]
		);
	}

	/**
	 * Register plugin settings
	 */
	public function register_settings() {
		// Register settings
		register_setting(
			self::OPTION_GROUP,
			'nobsmv_enabled',
			[
				'type'              => 'boolean',
				'default'           => true,
				'sanitize_callback' => [ $this, 'sanitize_checkbox' ],
			]
		);

		register_setting(
			self::OPTION_GROUP,
			'nobsmv_link_position',
			[
				'type'              => 'string',
				'default'           => 'prepend',
				'sanitize_callback' => [ $this, 'sanitize_position' ],
			]
		);

		register_setting(
			self::OPTION_GROUP,
			'nobsmv_custom_message',
			[
				'type'              => 'string',
				'default'           => $this->get_default_message(),
				'sanitize_callback' => [ $this, 'sanitize_message' ],
			]
		);

		// Add settings section
		add_settings_section(
			'nobsmv_main_section',
			__( 'General Settings', 'nobs-mail-view-email-fallback-viewer-in-browser' ),
			[ $this, 'render_section_description' ],
			self::PAGE_SLUG
		);

		// Add settings fields
		add_settings_field(
			'nobsmv_enabled',
			__( 'Enable Email Fallback Viewer', 'nobs-mail-view-email-fallback-viewer-in-browser' ),
			[ $this, 'render_enabled_field' ],
			self::PAGE_SLUG,
			'nobsmv_main_section'
		);

		add_settings_field(
			'nobsmv_link_position',
			__( 'Link Position', 'nobs-mail-view-email-fallback-viewer-in-browser' ),
			[ $this, 'render_position_field' ],
			self::PAGE_SLUG,
			'nobsmv_main_section'
		);

		add_settings_field(
			'nobsmv_custom_message',
			__( 'Custom Message', 'nobs-mail-view-email-fallback-viewer-in-browser' ),
			[ $this, 'render_message_field' ],
			self::PAGE_SLUG,
			'nobsmv_main_section'
		);
	}

	/**
	 * Render settings page
	 */
	public function render_settings_page() {
		// Check user capabilities
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}
		?>
		<div class="wrap">
			<h1><?php echo esc_html( get_admin_page_title() ); ?></h1>
			<form action="options.php" method="post">
				<?php
				settings_fields( self::OPTION_GROUP );
				do_settings_sections( self::PAGE_SLUG );
				submit_button( __( 'Save Settings', 'nobs-mail-view-email-fallback-viewer-in-browser' ) );
				?>
			</form>
			<div style="margin-top: 24px; padding: 10px 16px; background: #f0f0f1; border-left: 3px solid #2271b1; border-radius: 2px; font-size: 13px; color: #50575e;">
				<?php
				printf(
					/* translators: %1$s: opening anchor tag, %2$s: closing anchor tag */
					esc_html__( 'Not working as expected, or something missing? %1$sReach out to us%2$s so we can help.', 'nobs-mail-view-email-fallback-viewer-in-browser' ),
					'<a href="' . esc_url( 'mailto:hello@wpspins.com?subject=NoBS-Mail View — Support Request' ) . '" style="color: #2271b1; text-decoration: none;">',
					'</a>'
				);
				?>
			</div>
		</div>
		<?php
	}

	/**
	 * Render section description
	 */
	public function render_section_description() {
		echo '<p>' . esc_html__( 'Configure how the email fallback viewer works.', 'nobs-mail-view-email-fallback-viewer-in-browser' ) . '</p>';
	}

	/**
	 * Render enabled checkbox field
	 */
	public function render_enabled_field() {
		$enabled = get_option( 'nobsmv_enabled', true );
		?>
		<label>
			<input
				type="checkbox"
				name="nobsmv_enabled"
				value="1"
				<?php checked( $enabled, true ); ?>
			/>
			<?php esc_html_e( 'Add browser view link to all outgoing emails', 'nobs-mail-view-email-fallback-viewer-in-browser' ); ?>
		</label>
		<p class="description">
			<?php esc_html_e( 'Uncheck to disable the plugin functionality without deactivating it.', 'nobs-mail-view-email-fallback-viewer-in-browser' ); ?>
		</p>
		<?php
	}

	/**
	 * Render position dropdown field
	 */
	public function render_position_field() {
		$position = get_option( 'nobsmv_link_position', 'prepend' );
		?>
		<select name="nobsmv_link_position">
			<option value="prepend" <?php selected( $position, 'prepend' ); ?>>
				<?php esc_html_e( 'Prepend (Add before email content)', 'nobs-mail-view-email-fallback-viewer-in-browser' ); ?>
			</option>
			<option value="append" <?php selected( $position, 'append' ); ?>>
				<?php esc_html_e( 'Append (Add after email content)', 'nobs-mail-view-email-fallback-viewer-in-browser' ); ?>
			</option>
		</select>
		<p class="description">
			<?php esc_html_e( 'Choose where to add the browser view link in the email.', 'nobs-mail-view-email-fallback-viewer-in-browser' ); ?>
		</p>
		<?php
	}

	/**
	 * Render custom message field
	 */
	public function render_message_field() {
		$message = get_option( 'nobsmv_custom_message', $this->get_default_message() );
		?>
		<textarea
			name="nobsmv_custom_message"
			rows="3"
			cols="50"
			class="large-text"
		><?php echo esc_textarea( $message ); ?></textarea>
		<p class="description">
			<?php
			echo wp_kses_post(
				sprintf(
					/* translators: %s: shortcode tag */
					__( 'Use %s as a placeholder for the email browser link. HTML tags are not allowed.', 'nobs-mail-view-email-fallback-viewer-in-browser' ),
					'<code>{email_link}</code>'
				)
			);
			?>
		</p>
		<p class="description">
			<strong><?php esc_html_e( 'Default:', 'nobs-mail-view-email-fallback-viewer-in-browser' ); ?></strong>
			<code><?php echo esc_html( $this->get_default_message() ); ?></code>
		</p>
		<?php
	}

	/**
	 * Sanitize checkbox value
	 *
	 * @param mixed $value Input value
	 * @return bool
	 */
	public function sanitize_checkbox( $value ) {
		return (bool) $value;
	}

	/**
	 * Sanitize position value
	 *
	 * @param string $value Input value
	 * @return string
	 */
	public function sanitize_position( $value ) {
		$allowed = [ 'prepend', 'append' ];
		return in_array( $value, $allowed, true ) ? $value : 'prepend';
	}

	/**
	 * Sanitize message value
	 *
	 * @param string $value Input value
	 * @return string
	 */
	public function sanitize_message( $value ) {
		// Remove HTML tags and trim
		$sanitized = wp_strip_all_tags( $value );
		$sanitized = trim( $sanitized );

		// Ensure the message contains the shortcode
		if ( empty( $sanitized ) || strpos( $sanitized, '{email_link}' ) === false ) {
			return $this->get_default_message();
		}

		return $sanitized;
	}

	/**
	 * Get default message template
	 *
	 * @return string
	 */
	public function get_default_message() {
		return 'Not seeing the email correctly. Use this link to read: {email_link}';
	}

	/**
	 * Get option with default fallback
	 * Static helper method for use in other classes
	 *
	 * @param string $option_name Option name
	 * @param mixed  $default Default value
	 * @return mixed
	 */
	public static function get_option( $option_name, $default = null ) {
		return get_option( $option_name, $default );
	}
}
