<?php
/**
 * Email Viewer
 *
 * @package Email_Safe_View_Fallback_Viewer
 */

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class NOBSMV_Email_Viewer
 *
 * Handles frontend email viewing
 */
class NOBSMV_Email_Viewer {

	/**
	 * Email storage instance
	 *
	 * @var NOBSMV_Email_Storage
	 */
	private $storage;

	/**
	 * Initialize the class
	 *
	 * @param NOBSMV_Email_Storage $storage Email storage instance
	 */
	public function __construct( $storage ) {
		$this->storage = $storage;
		add_action( 'template_redirect', [ $this, 'handle_email_view' ] );
		add_action( 'template_redirect', [ $this, 'block_direct_access' ], 1 );
	}

	/**
	 * Handle frontend email viewing
	 */
	public function handle_email_view() {
		// Check if this is an email view request
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.ValidatedSanitizedInput.MissingUnslash -- Nonce not needed for public token-based viewing, sanitization handled by sanitize_text_field below
		if ( ! isset( $_GET['mid'] ) || ! isset( $_SERVER['REQUEST_URI'] ) || strpos( wp_unslash( $_SERVER['REQUEST_URI'] ), '/email/' ) === false ) {
			return;
		}

		// phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.MissingUnslash -- Nonce not needed for public token-based viewing, unslash handled by sanitize_text_field
		$token = isset( $_GET['mid'] ) ? sanitize_text_field( wp_unslash( $_GET['mid'] ) ) : '';

		// Retrieve email by token
		$email = $this->storage->get_email_by_token( $token );

		if ( ! $email ) {
			// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- wp_die handles escaping internally
			wp_die( esc_html__( 'Email not found or has expired.', 'nobs-mail-view-email-fallback-viewer-in-browser' ), 404 );
		}

		$content = $email['message'];

		// Check if content is HTML or plain text
		if ( ! preg_match( '/<html|<body|<div|<p|<br/i', $content ) ) {
			// Plain text email - convert line breaks to HTML
			$content = wpautop( $content );
		}

		// Enqueue email viewer styles
		$this->enqueue_viewer_styles();

		// Wrap content in container
		$output = '<!DOCTYPE html><html><head><meta charset="utf-8">';
		$output .= '<title>' . esc_html( $email['subject'] ) . '</title>';
		// phpcs:ignore WordPress.WP.EnqueuedResources.NonEnqueuedStylesheet -- Direct HTML output bypasses WordPress enqueue system
		$output .= '<link rel="stylesheet" href="' . esc_url( NOBSMV_PLUGIN_URL . 'assets/css/email-viewer.css' ) . '" />';
		$output .= '</head><body>';
		$output .= '<div class="nobsmv-email-container">';
		$output .= $content;
		$output .= '</div>';
		$output .= '</body></html>';

		// Output the email content as HTML
		header( 'Content-Type: text/html; charset=utf-8' );
		// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- Intentionally outputting email HTML content
		echo $output;
		exit;
	}

	/**
	 * Enqueue viewer styles
	 */
	private function enqueue_viewer_styles() {
		wp_enqueue_style(
			'nobsmv-email-viewer',
			NOBSMV_PLUGIN_URL . 'assets/css/email-viewer.css',
			[],
			NOBSMV_VERSION
		);
	}

	/**
	 * Block direct access to email posts via permalink
	 */
	public function block_direct_access() {
		if ( is_singular( 'nobsmv_outgoing_email' ) ) {
			global $wp_query;
			$wp_query->set_404();
			status_header( 404 );
			get_template_part( 404 );
			exit;
		}
	}
}
