<?php
/**
 * Email Storage Handler
 *
 * @package Email_Safe_View_Fallback_Viewer
 */

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class NOBSMV_Email_Storage
 *
 * Handles storage and retrieval of outgoing emails
 */
class NOBSMV_Email_Storage {

	/**
	 * Initialize the class
	 */
	public function __construct() {
		add_action( 'init', [ $this, 'register_post_type' ] );
	}

	/**
	 * Register custom post type for storing outgoing emails
	 */
	public function register_post_type() {
		register_post_type( 'nobsmv_outgoing_email', [
			'labels' => [
				'name'               => __( 'Outgoing Emails', 'nobs-mail-view-email-fallback-viewer-in-browser' ),
				'singular_name'      => __( 'Outgoing Email', 'nobs-mail-view-email-fallback-viewer-in-browser' ),
				'menu_name'          => __( 'Outgoing Emails', 'nobs-mail-view-email-fallback-viewer-in-browser' ),
				'all_items'          => __( 'All Emails', 'nobs-mail-view-email-fallback-viewer-in-browser' ),
				'view_item'          => __( 'View Email', 'nobs-mail-view-email-fallback-viewer-in-browser' ),
				'search_items'       => __( 'Search Emails', 'nobs-mail-view-email-fallback-viewer-in-browser' ),
				'not_found'          => __( 'No emails found', 'nobs-mail-view-email-fallback-viewer-in-browser' ),
				'not_found_in_trash' => __( 'No emails found in trash', 'nobs-mail-view-email-fallback-viewer-in-browser' ),
			],
			'public'              => false,
			'publicly_queryable'  => false,
			'show_ui'             => true,
			'show_in_menu'        => true,
			'show_in_nav_menus'   => false,
			'show_in_admin_bar'   => false,
			'exclude_from_search' => true,
			'has_archive'         => false,
			'rewrite'             => false,
			'hierarchical'        => false,
			'supports'            => [ 'title', 'editor' ],
			'menu_icon'           => 'dashicons-email',
			'capability_type'     => 'post',
			'capabilities'        => [
				'create_posts' => 'do_not_allow',
			],
			'map_meta_cap'        => true,
		] );
	}

	/**
	 * Store outgoing email as custom post
	 *
	 * @param string $recipient_email Recipient email address
	 * @param string $subject Email subject
	 * @param string $message Email body (HTML)
	 * @param array  $headers Email headers
	 * @param array  $attachments Email attachments
	 * @return int|false Post ID on success, false on failure
	 */
	public function store_email( $recipient_email, $subject, $message, $headers = [], $attachments = [] ) {
		// Create post bypassing capabilities
		$post_id = wp_insert_post( [
			'post_type'    => 'nobsmv_outgoing_email',
			'post_title'   => $subject,
			'post_content' => $message,
			'post_status'  => 'publish',
		], false, false );

		if ( ! $post_id || is_wp_error( $post_id ) ) {
			return false;
		}

		// Store recipient email
		update_post_meta( $post_id, '_nobsmv_recipient_email', sanitize_email( $recipient_email ) );

		// Store email headers
		if ( ! empty( $headers ) ) {
			update_post_meta( $post_id, '_nobsmv_email_headers', $headers );
		}

		// Store attachments
		if ( ! empty( $attachments ) ) {
			update_post_meta( $post_id, '_nobsmv_email_attachments', $attachments );
		}

		// Generate and store unique token for browser access (26 characters)
		$token = wp_generate_password( 26, false );
		update_post_meta( $post_id, '_nobsmv_access_token', $token );

		// Store timestamp
		update_post_meta( $post_id, '_nobsmv_sent_timestamp', time() );

		return $post_id;
	}

	/**
	 * Retrieve email by post ID
	 *
	 * @param int $post_id Post ID
	 * @return array|false Email data or false if not found
	 */
	public function get_email( $post_id ) {
		$post = get_post( $post_id );

		if ( ! $post || 'nobsmv_outgoing_email' !== $post->post_type ) {
			return false;
		}

		return [
			'subject'     => $post->post_title,
			'message'     => $post->post_content,
			'recipient'   => get_post_meta( $post_id, '_nobsmv_recipient_email', true ),
			'headers'     => get_post_meta( $post_id, '_nobsmv_email_headers', true ),
			'attachments' => get_post_meta( $post_id, '_nobsmv_email_attachments', true ),
			'token'       => get_post_meta( $post_id, '_nobsmv_access_token', true ),
			'timestamp'   => get_post_meta( $post_id, '_nobsmv_sent_timestamp', true ),
		];
	}

	/**
	 * Retrieve email by access token
	 *
	 * @param string $token Access token
	 * @return array|false Email data or false if not found
	 */
	public function get_email_by_token( $token ) {
		$args = [
			'post_type'      => 'nobsmv_outgoing_email',
			'posts_per_page' => 1,
			// phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query -- Necessary for token-based email lookup
			'meta_query'     => [
				[
					'key'   => '_nobsmv_access_token',
					'value' => sanitize_text_field( $token ),
				],
			],
		];

		$query = new WP_Query( $args );

		if ( ! $query->have_posts() ) {
			return false;
		}

		return $this->get_email( $query->posts[0]->ID );
	}
}
