<?php
/**
 * Plugin Name: Noaiseo Automatic AI Writer — SEO, Styling & Images
 * Plugin URI: https://noaiseo.com/#monthly-plan
 * Description: Automatically generates and publishes SEO-optimized articles with AI, including images and custom styling.
 * Version: 3.1.0
 * Author: Noaiseo
 * Author URI: https://noaiseo.com
 * Text Domain: noaiseo-ai-article-generator
 * License: GPLv2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 */



if ( ! defined( 'ABSPATH' ) ) { exit; }
/* ==== NoAI: Logging kill switch (disable or redirect all logs) ==== */

if ( ! function_exists('noai_log') ) {
    function noai_log( $message ) {
        // Only log when explicitly enabled
        if ( defined('NOAI_DEBUG') && NOAI_DEBUG ) {
            if ( is_array($message) || is_object($message) ) {
                $message = print_r($message, true);
            }
            // Use WordPress safe log
            error_log( '[NoAI] ' . $message );
        }
    }
}






/* =========================================================================
 *  No-AI Service Connection Settings (API URLs & Security Key)
 * ========================================================================= */

if ( ! defined( 'NOAI_REG_API' ) ) {
    define( 'NOAI_REG_API', 'https://api.noaiseo.com/reg/en' );
}
if ( ! defined( 'NOAI_API_BASE' ) ) {
    define( 'NOAI_API_BASE', 'https://api.noaiseo.com/download/no-ai' );
}

/* Publish Settings */
if ( ! defined( 'NOAI_PUBLISH_STATUS' ) ) {
    define( 'NOAI_PUBLISH_STATUS', 'draft' );
}
if ( ! defined( 'NOAI_PUBLISH_DAYS' ) ) {
    define( 'NOAI_PUBLISH_DAYS', '1,3,5' );
}
if ( ! defined( 'NOAI_PUBLISH_HOUR' ) ) {
    define( 'NOAI_PUBLISH_HOUR', 10 );
}
if ( ! defined( 'NOAI_PUBLISH_TZ' ) ) {
    define( 'NOAI_PUBLISH_TZ', 'Asia/Tehran' );
}

/* Beacon Scheduler config */
if ( ! defined( 'NOAI_TICK_KEY' ) ) {
    define( 'NOAI_TICK_KEY', '3b82f6c0f8e14d2aaf7d91bcd56a9e21f4e67890' ); 
}
if ( ! defined( 'NOAI_BEACON_INTERVAL_MS' ) ) {
    define( 'NOAI_BEACON_INTERVAL_MS', 60000 ); 
}

// === WP-Cron: add every_minute interval ===
add_filter('cron_schedules', function ($schedules) {
    if (!isset($schedules['every_minute'])) {
        $schedules['every_minute'] = array(
            'interval' => 60,
            'display'  => ('Every Minute')
        );
    }
    return $schedules;
});
// Ensure beacon cron is scheduled (only when needed)
function noai_ensure_beacon_scheduled() {
    if ( ! wp_next_scheduled('noai_ai_beacon_event') ) {
        wp_schedule_event(time() + 60, 'every_minute', 'noai_ai_beacon_event');
    }
}




/* =========================================================================
 *  Activation defaults
 * ========================================================================= */
function noai_content_seo_activate() {
    if ( get_option('noai_has_settings', false) ) return;

    add_option( 'noai_site_name',     '' );
    add_option( 'noai_site_topic',    '' );
	add_option( 'noai_api_key', '' ); // NEW
    add_option( 'noai_site_color',    '#d40000' );
    add_option( 'noai_bg_color', '#f5f7fa' ); // Default light white
    add_option( 'noai_site_category', 0 );
    add_option( 'noai_site_slug',     '' );
    add_option( 'noai_last_settings_push', 0 );
    add_option( 'noai_last_article_imported', 0 );
    add_option( 'noai_has_settings', false );
}


/* =========================================================================
 *  Admin Menu
 * ========================================================================= */
add_action( 'admin_menu', function () {
    add_menu_page(
        'No-AI Dashboard',
        'No-AI Dashboard',
        'manage_options',
        'no-ai-dashboard',
        'noai_dashboard_page',
        'dashicons-chart-bar',
        20
    );
    add_submenu_page(
        'no-ai-dashboard',
        'No-AI Settings',
        'Settings',
        'manage_options',
        'no-ai-settings',
        'noai_content_seo_settings_page'
    );
} );



/* =========================================================================
 *  Dashboard Page
 * ========================================================================= */
function noai_refresh_articles_remaining() {
    $api_key = get_option('noai_api_key', '');
    if (!$api_key) return;

    $url = rtrim(NOAI_REG_API, '/') . '/validate_key';
    $args = array(
        'headers' => array('Content-Type' => 'application/json'),
        'body'    => wp_json_encode(array('api_key' => $api_key)),
        'timeout' => 15,
        'sslverify' => true,
    );

    $response = wp_remote_post($url, $args);
    if (is_wp_error($response)) {
        
        return;
    }

    $code = wp_remote_retrieve_response_code($response);
    $body = wp_remote_retrieve_body($response);
    $data = json_decode($body, true);

    if ($code == 200 && isset($data['articles_remaining'])) {
        update_option('noai_articles_remaining', intval($data['articles_remaining']));
    }
}

function noai_count_uploaded_articles() {
    $upload_dir = wp_upload_dir();
    $base_dir   = trailingslashit($upload_dir['basedir']) . 'noai-articles';

    if ( !is_dir($base_dir) ) return 0;

    $folders = array_filter(scandir($base_dir), function($f) use ($base_dir) {
        return $f !== '.' && $f !== '..' && is_dir($base_dir . '/' . $f) && strpos($f, 'article_') === 0;
    });

    return count($folders);
}

function noai_get_articles_dates() {
    $upload_dir = wp_upload_dir();
    $base_dir   = trailingslashit($upload_dir['basedir']) . 'noai-articles';

    $dates = [];

    if ( !is_dir($base_dir) ) return $dates;

    $folders = array_filter(scandir($base_dir), function($f) use ($base_dir) {
        return $f !== '.' && $f !== '..' && is_dir($base_dir . '/' . $f) && strpos($f, 'article_') === 0;
    });

    foreach ($folders as $folder) {
        $path = $base_dir . '/' . $folder . '/index.html';
        if ( file_exists($path) ) {
            $dates[] = filemtime($path);
        }
    }

    return $dates;
}

function noai_dashboard_page() {
    noai_refresh_articles_remaining();

    $remaining = intval(get_option('noai_articles_remaining', 0));
    $uploaded  = noai_count_uploaded_articles();
    $total     = $remaining + $uploaded;
    $dates     = noai_get_articles_dates();

    $chart_sets = [
        'week'  => [],
        'month' => [],
        'year'  => [],
    ];

    foreach ($dates as $ts) {
$day   = gmdate("Y-m-d", $ts);
$daynum = gmdate("j", $ts);
$monthName = gmdate("F", $ts);
$year  = gmdate("Y", $ts);


        if ($ts >= strtotime("-7 days")) {
            if (!isset($chart_sets['week'][$day])) $chart_sets['week'][$day] = 0;
            $chart_sets['week'][$day]++;
        }
        if ($ts >= strtotime("-1 month")) {
            if (!isset($chart_sets['month'][$daynum])) $chart_sets['month'][$daynum] = 0;
            $chart_sets['month'][$daynum]++;
        }
        if ($ts >= strtotime("-1 year")) {
            if (!isset($chart_sets['year'][$monthName])) $chart_sets['year'][$monthName] = 0;
            $chart_sets['year'][$monthName]++;
        }
    }

    foreach ($chart_sets as $key => $data) {
        ksort($chart_sets[$key]);
    }
	set_transient('noai_chart_data', $chart_sets, 60);

    ?>

    <div class="wrap noai-dashboard">
      <h1>Hello 👋 Welcome back</h1>

      <div class="stats-cards">
        <div class="glass-card">
          <h2><?php echo esc_html( $remaining ); ?></h2>
          <p>Articles remaining in subscription</p>
          <a href="https://noaiseo.com/panel/#sites" target="_blank" class="button button-primary">
            Renew Subscription
          </a>
        </div>
        <div class="glass-card">
          <h2><?php echo esc_html( $uploaded ); ?></h2>
          <p>Uploaded Articles</p>
        </div>
      </div>

      <div class="chart-section">
        <h2>📈 Articles Statistics</h2>
        <select id="noai-chart-range">
          <option value="week">Week</option>
          <option value="month">Month</option>
          <option value="year">Year</option>
        </select>
        <canvas id="noai-chart" height="120"></canvas>
      </div>
    </div>

    <script>
      document.addEventListener("DOMContentLoaded", function() {
        const ctx = document.getElementById('noai-chart');
        const chartSets = <?php echo wp_json_encode( $chart_sets ); ?>;

        let currentRange = 'week';
        let chart;

        function renderChart(range) {
          const labels = Object.keys(chartSets[range]);
          const data   = Object.values(chartSets[range]);

          if (chart) chart.destroy();

          chart = new Chart(ctx, {
            type: 'line',
            data: {
              labels: labels,
              datasets: [{
                label: 'Articles',
                data: data,
                borderColor: '#3b82f6',
                backgroundColor: 'rgba(59,130,246,0.3)',
                fill: true,
                tension: 0.3
              }]
            },
            options: { 
              plugins: { legend: { display: false } },
              scales: {
                x: {
                  ticks: { color: '#fff' },
                  grid: { color: 'rgba(255,255,255,0.1)', drawBorder: false }
                },
                y: {
                  beginAtZero: true,
                  ticks: { color: '#fff', stepSize: 1 },
                  grid: {
                    drawBorder: false,
                    color: ctx => (ctx.tick.value === 0 ? 'rgba(0,0,0,0)' : 'rgba(255,255,255,0.1)'),
                    lineWidth: ctx => (ctx.tick.value === 0 ? 0 : 1)
                  }
                }
              }
            }
          });
        }

        renderChart(currentRange);
        document.getElementById("noai-chart-range").addEventListener("change", e => {
          currentRange = e.target.value;
          renderChart(currentRange);
        });
      });
    </script>

    <?php
}




/* =========================================================================
 *  Push settings → Flask (blocking)
 * ========================================================================= */
function noai_push_settings( $site_name, $site_topic, $site_color, $site_category, $site_slug ) {

    $payload = array(
        'site_name'        => $site_name,
        'site_topic'       => $site_topic,
        'site_color'       => $site_color,
        'api_key'          => get_option('noai_api_key', ''),
        'site_category'    => $site_category,
        'site_slug'        => $site_slug,
        'publish_mode'     => NOAI_PUBLISH_STATUS,
        'publish_days'     => explode(',', NOAI_PUBLISH_DAYS),
        'publish_hour_tehran' => NOAI_PUBLISH_HOUR,
        'timezone'         => NOAI_PUBLISH_TZ,
        'background_color' => get_option('noai_bg_color', '#f5f7fa'),
    );

    $args = array(
        'headers'   => array(
    'Content-Type' => 'application/json; charset=utf-8',
),

        'body'      => wp_json_encode( $payload, JSON_UNESCAPED_UNICODE ),
        'method'    => 'POST',
        'timeout'   => 60,
        'blocking'  => true,
        'sslverify' => true,
    );

    $url = rtrim( NOAI_REG_API, '/' ) . '/register_site';
    $response = wp_remote_post( $url, $args );

   
    if ( is_wp_error( $response ) ) {
        $err = $response->get_error_message();
        return [
            'ok'      => false,
            'error'   => $err,
            'message' => 'Connection error'
        ];
    }

    $code = wp_remote_retrieve_response_code( $response );
    $body = wp_remote_retrieve_body( $response );
    $data = json_decode( $body, true );

   
    if ( $code < 200 || $code >= 300 ) {
        if ( is_array($data) ) {
            return $data + ['ok' => false];
        }
        return [
            'ok'      => false,
            'error'   => $body ?: "HTTP $code",
            'message' => "HTTP $code"
        ];
    }


    if ( isset( $data['slug'] ) ) {
        update_option( 'noai_site_slug', sanitize_title( $data['slug'] ) );
    }
    update_option( 'noai_last_settings_push', time() );

    return $data ?: [
        'ok'      => false,
        'error'   => 'Invalid JSON',
        'message' => $body
    ];
}




/* =========================================================================
 *  Settings Page UI
 * ========================================================================= */
function noai_content_seo_settings_page() {

    if ( isset( $_POST['noai_save_settings'] ) && check_admin_referer( 'noai_save_options', 'noai_nonce' ) ) {
$api_key       = isset($_POST['noai_api_key']) ? sanitize_text_field( wp_unslash($_POST['noai_api_key']) ) : '';
$site_name     = isset($_POST['noai_site_name']) ? sanitize_text_field( wp_unslash($_POST['noai_site_name']) ) : '';
$site_topic    = isset($_POST['noai_site_topic']) ? sanitize_text_field( wp_unslash($_POST['noai_site_topic']) ) : '';
$site_color    = isset($_POST['noai_site_color']) ? sanitize_hex_color( wp_unslash($_POST['noai_site_color']) ) : '#d40000';
$site_category = isset($_POST['noai_site_category']) ? intval( wp_unslash($_POST['noai_site_category']) ) : 0;
$bg_color      = isset($_POST['noai_bg_color']) ? sanitize_hex_color( wp_unslash($_POST['noai_bg_color']) ) : '#f5f7fa';


        if ( empty( $site_color ) ) {
            $site_color = '#d40000';
        }
        $site_slug = sanitize_title( $site_name );

        update_option( 'noai_site_name',     $site_name );
		update_option( 'noai_api_key',      $api_key );
        update_option( 'noai_site_topic',    $site_topic );
        update_option( 'noai_site_color',    $site_color );

        if ( empty( $bg_color ) ) {
            $bg_color = '#f5f7fa'; // Default
        }
        update_option( 'noai_bg_color', $bg_color );

        update_option( 'noai_site_category', $site_category );
        update_option( 'noai_site_slug',     $site_slug );

     $result = noai_push_settings( $site_name, $site_topic, $site_color, $site_category, $site_slug );

if ( isset($result['ok']) && $result['ok'] ) {
    update_option('noai_has_settings', true);
        
        // 🔹 Reset old timers
delete_option('noai_ai_next_generate_at');
delete_option('noai_ai_import_at');
delete_option('noai_ai_download_at');

$last_num = 0;
$site_slug = get_option('noai_site_slug', '');
if ($site_slug) {
    $url = rtrim(NOAI_API_BASE, '/') . '/' . rawurlencode($site_slug) . '/last';
    $resp = wp_remote_get($url, ['timeout' => 15, 'sslverify' => true]);
    if (!is_wp_error($resp) && wp_remote_retrieve_response_code($resp) === 200) {
        $body = json_decode(wp_remote_retrieve_body($resp), true);
        if (isset($body['last_article'])) {
            $last_num = intval($body['last_article']);
        }
    }
}

update_option('noai_last_article_imported', $last_num);

update_option('noai_ai_download_at', time() + 60*5);

if (function_exists('wp_schedule_single_event')) {
    wp_schedule_single_event(time() + 60*5, 'noai_ai_beacon_event');
}

    // 🔹 Ensure beacon is active
    noai_ensure_beacon_scheduled();
    noai_self_ping();


    echo '<div class="notice notice-success is-dismissible"><p>'
        . esc_html($result['message'] ?? '✅ Settings saved')
        . '</p></div>';
} else {
    $msg = $result['error'] ?? $result['message'] ?? '❌ Unknown error';

    if (is_array($result)) {
        $msg = $msg ?: print_r($result, true);
    } elseif (is_string($result)) {
        $decoded = json_decode($result, true);
        if ($decoded && is_array($decoded)) {
            $msg = $decoded['error'] ?? $decoded['message'] ?? $msg;
        }
    }

    echo '<div class="notice notice-error is-dismissible"><p>'
         . esc_html($msg)
         . '</p></div>';
}

} 


    $has_settings = get_option('noai_has_settings', false);



    $has_settings = get_option('noai_has_settings', false);

    $site_name     = $has_settings ? get_option('noai_site_name', '')     : '';
    $site_topic    = $has_settings ? get_option('noai_site_topic', '')    : '';
    $site_color    = $has_settings ? get_option('noai_site_color', '#d40000') : '#d40000';
    $bg_color      = $has_settings ? get_option('noai_bg_color', '#f5f7fa') : '#f5f7fa';
    $site_category = $has_settings ? get_option('noai_site_category', 0)  : 0;

    $categories = get_categories( array( 'hide_empty' => false ) );
    ?>
<div class="wrap noai-settings">
  <h1>⚙️ No-AI Content Settings</h1>
  <div class="noai-card">
    <form method="post">
      <?php wp_nonce_field( 'noai_save_options', 'noai_nonce' ); ?>
      <div class="grid">
        <div class="row full">
          <label for="noai_site_name">Site Name:</label>
          <input type="text" id="noai_site_name" name="noai_site_name"
                 value="<?php echo esc_attr($site_name); ?>">
        </div>
        <div class="row full">
          <label for="noai_site_topic">Site Topic:</label>
          <input type="text" id="noai_site_topic" name="noai_site_topic"
                 value="<?php echo esc_attr($site_topic); ?>">
          <div class="desc">General topic of the site for article generation.</div>
        </div>
        <div class="row">
          <label for="noai_site_color">Primary Color:</label>
          <input type="color" id="noai_site_color" name="noai_site_color"
                 value="<?php echo esc_attr($site_color); ?>">
          <div class="desc">Used for article headings and highlights.</div>
        </div>
        <div class="row">
          <label for="noai_bg_color">Background Color:</label>
          <input type="color" id="noai_bg_color" name="noai_bg_color"
                 value="<?php echo esc_attr($bg_color); ?>">
          <div class="desc">Background color for articles (change if your site uses dark theme).</div>
        </div>
        <div class="row full">
          <label for="noai_site_category">Article Category:</label>
          <select id="noai_site_category" name="noai_site_category">
            <option value="0">— No Category —</option>
            <?php foreach ($categories as $cat): ?>
<option value="<?php echo esc_attr( $cat->term_id ); ?>" <?php selected( $site_category, $cat->term_id ); ?>>
    <?php echo esc_html( $cat->name ); ?>
</option>

            <?php endforeach; ?>
          </select>
        </div>
        <div class="row full">
          <label for="noai_api_key">API Key:</label>
          <input type="text" id="noai_api_key" name="noai_api_key"
                 value="<?php echo esc_attr(get_option('noai_api_key', '')); ?>"
                 placeholder="Enter the API key you received from No-AI service">
        </div>
      </div>
      <div class="actions">
        <?php submit_button( 'Save Settings', 'primary', 'noai_save_settings', false ); ?>
      </div>
    </form>
  </div>
</div>

    <?php
}

/* =========================================================================
 *  One-time cron to import article 1 five minutes after saving settings
 * ========================================================================= */


function noai_download_article_to_local($article_id) {
    $upload_dir = wp_upload_dir();
    $base_dir   = trailingslashit($upload_dir['basedir']) . 'noai-articles';
    $article_dir = trailingslashit($base_dir) . 'article_' . $article_id;

    if (!wp_mkdir_p($article_dir)) {
        return false;
    }

    $files = [
        'meta.json',
        'index.html',
        'style.css',
        'Poppins-Regular.ttf',
        'images/img_1.webp',
        'images/img_2.webp',
        'images/img_3.webp'
    ];

    $remote_base = noai_build_remote_article_url($article_id);

    foreach ($files as $file) {
        $remote_url = $remote_base . '/' . $file;
        $local_path = trailingslashit($article_dir) . $file;

        // Create subfolders if needed
        $subfolder = dirname($local_path);
        if (!file_exists($subfolder)) {
            wp_mkdir_p($subfolder);
        }

        // --- Only wp_remote_get (no fallback at all) ---
        $response = wp_remote_get($remote_url, [
            'timeout'   => 30,
            'sslverify' => true,
        ]);

        if (is_wp_error($response)) {
            continue;
        }

        $code = wp_remote_retrieve_response_code($response);
        if ($code !== 200) {
            continue;
        }

        $body = wp_remote_retrieve_body($response);
        if (empty($body)) {
            continue;
        }

        // --- Save file ---
        file_put_contents($local_path, $body);
    }

    return $article_dir;
}


function noai_build_remote_article_url($article_num = 1) {
    $site_slug = get_option('noai_site_slug', '');
    if (!$site_slug) {
        $site_slug = sanitize_title(get_bloginfo('name'));
    }

    return NOAI_API_BASE . '/' . rawurlencode($site_slug) . '/articles/article_' . intval($article_num);
}

function noai_copy_template_to_theme() {
    $template_file = plugin_dir_path(__FILE__) . 'template/no-ai-article-template.php';
    $theme_dir     = get_stylesheet_directory(); // Active theme path
    $destination   = trailingslashit($theme_dir) . 'no-ai-article-template.php';

    if (!file_exists($template_file)) {
     
        return;
    }

    if (!copy($template_file, $destination)) {
 
        return;
    }

    
}



// ---------------------------------------------------------------------------
// ✅ Daily cron job to import articles
// ---------------------------------------------------------------------------

function noai_import_articles() {



    $imported_any = false;
        $expected = (int) get_option('noai_last_article_imported', 0) + 1;
    $expected_folder = 'article_' . $expected;


    $upload_dir = wp_upload_dir()['basedir'] . '/noai-articles/';
    if (!is_dir($upload_dir)) return;

    $folders = scandir($upload_dir);


    foreach ($folders as $folder) {
     

        if ($folder === '.' || $folder === '..') continue;

       
        if (get_option("noai_ai_imported_$folder", false)) {
          
            continue;
        }

        $article_path = $upload_dir . $folder;
        if (!is_dir($article_path)) continue;
        if ($folder !== $expected_folder) continue;


        $meta_file = $article_path . '/meta.json';
        $html_file = $article_path . '/index.html';
        $img_main  = $article_path . '/images/img_1.webp';

        if (!file_exists($meta_file) || !file_exists($html_file)) continue;

        $meta_raw = @file_get_contents($meta_file);
$meta = $meta_raw ? json_decode($meta_raw, true) : false;

if (!$meta) {
    update_option('noai_ai_download_at', time() + 60);   
    update_option('noai_ai_import_at', time() + 120);    
  
    continue;
}

$html = file_get_contents($html_file);
$orig_html = $html;


$body_ok = false;
if (class_exists('DOMDocument')) {
    libxml_use_internal_errors(true);
    $doc = new DOMDocument();
    $doc->loadHTML($orig_html, LIBXML_HTML_NOIMPLIED | LIBXML_HTML_NODEFDTD);
    libxml_clear_errors();
    $bodies = $doc->getElementsByTagName('body');
    if ($bodies->length > 0) {
        $inner = '';
        foreach ($bodies->item(0)->childNodes as $child) {
            $inner .= $doc->saveHTML($child);
        }
        if ($inner !== '') {
            $html = $inner;
            $body_ok = true;
        }
    }
}
if (!$body_ok) {
    if (preg_match('/<body\b[^>]*>(.*)<\/body>/is', $orig_html, $mm)) {
        $html = $mm[1];
    }
}


$uploads_url      = wp_upload_dir()['baseurl'];
$article_url_base = $uploads_url . '/noai-articles/' . $folder;

$jsonld_all   = '';
$jsonld_plain = '';
if (preg_match_all('/<script[^>]+type=["\']application\/ld\+json["\'][^>]*>.*?<\/script>/is', $html, $m) && !empty($m[0])) {
    $jsonld_all = implode("\n", $m[0]);
    $html = str_replace($jsonld_all, '', $html);
}

$jsonld_tail = noai_grab_trailing_jsonld($html, $article_url_base);
if ($jsonld_tail) {
    $jsonld_plain = $jsonld_tail;
  
}

$html = str_replace('src="images/', 'src="' . $article_url_base . '/images/', $html);
$html = str_replace("src='images/", "src='" . $article_url_base . "/images/", $html);
$html = str_replace('url("images/', 'url("' . $article_url_base . '/images/', $html);
$html = str_replace("url('images/", "url('" . $article_url_base . "/images/", $html);

$html = preg_replace_callback('/\b(alt|title)\s*=\s*("|\')(.*?)\2/si', function ($m) {
    $val = str_replace('"', '&quot;', $m[3]);
    return $m[1] . '="' . $val . '"';
}, $html);

        $cat_id = intval(get_option('noai_site_category', 0));

        
        $post_data = [
            'post_title'    => $meta['title'],
            'post_name'     => sanitize_title($meta['slug']),
            'post_content'  => $html,
            'post_status'   => 'publish',
            'post_type'     => 'post',
            'post_category' => $cat_id ? [$cat_id] : [],
        ];
        $post_id = wp_insert_post($post_data, true);
       

        if (is_wp_error($post_id) || !$post_id) {
            $msg = is_wp_error($post_id) ? $post_id->get_error_message() : 'insert returned 0';
           
            continue;
        }

        
       // ✅ Force assign template safely (fix for cron context)
if (!is_wp_error($post_id) && $post_id) {
    clean_post_cache($post_id);
    update_post_meta($post_id, '_wp_page_template', 'no-ai-article-template.php');

    // fallback (if cache delay)
    wp_update_post([
        'ID' => $post_id,
        'meta_input' => ['_wp_page_template' => 'no-ai-article-template.php']
    ]);
}


        
       if (!empty($jsonld_all)) {
    update_post_meta($post_id, 'noai_article_jsonld', $jsonld_all);
} elseif (!empty($jsonld_plain)) {
    update_post_meta($post_id, 'noai_article_jsonld', $jsonld_plain);
}


      
        if (file_exists($img_main)) {
            $upload = wp_upload_bits("img_1.webp", null, file_get_contents($img_main));
            if (!$upload['error']) {
                $image_id = wp_insert_attachment([
                    'post_mime_type' => 'image/webp',
                    'post_title'     => sanitize_file_name($upload['file']),
                    'post_content'   => '',
                    'post_status'    => 'inherit'
                ], $upload['file'], $post_id);
                require_once(ABSPATH . 'wp-admin/includes/image.php');
                $attach_data = wp_generate_attachment_metadata($image_id, $upload['file']);
                wp_update_attachment_metadata($image_id, $attach_data);
                set_post_thumbnail($post_id, $image_id);
            }
        }

      
      // --- Map meta.json -> Yoast fields (ONLY YOAST KEYS) ---
$focus = $meta['yoast_focus_keyphrase'] ?? '';
$seo_t = $meta['yoast_seo_title'] ?? '';
$desc  = $meta['yoast_meta_description'] ?? '';
$slug  = $meta['slug'] ?? ''; 

// --- Save to Yoast meta ---
if ($seo_t) {
    update_post_meta($post_id, '_yoast_wpseo_title', $seo_t);
}
if ($desc) {
    update_post_meta($post_id, '_yoast_wpseo_metadesc', $desc);
}
if ($focus) {
    update_post_meta($post_id, '_yoast_wpseo_focuskw', $focus);
}
if ($slug) {
    update_post_meta($post_id, '_yoast_wpseo_slug', $slug);
}
// --- Save to Rank Math meta (for full compatibility) ---
if ($seo_t) {
    update_post_meta($post_id, 'rank_math_title', $seo_t);
}
if ($desc) {
    update_post_meta($post_id, 'rank_math_description', $desc);
}
if ($focus) {
    // Rank Math uses 'rank_math_focus_keyword' for the main keyphrase
    update_post_meta($post_id, 'rank_math_focus_keyword', $focus);
}

// Optional: support keyword synonyms or multiple keywords
if (!empty($meta['keywords'])) {
    // Clean up any comma-separated list
    $keywords_str = is_array($meta['keywords']) ? implode(',', $meta['keywords']) : $meta['keywords'];
    update_post_meta($post_id, 'rank_math_secondary_keywords', $keywords_str);
}

// --- Keywords (only Yoast synonyms, no WordPress tags) ---
if (!empty($meta['keywords'])) {

    update_post_meta($post_id, '_yoast_wpseo_keywordsynonyms', $meta['keywords']);

}


// --- Update post slug (raw from meta.json) ---
if ($slug) {
    wp_update_post([
        'ID'        => $post_id,
        'post_name' => $slug,
    ]);
}

// --- Save folder & update counters ---
update_post_meta($post_id, 'noai_article_folder', $folder);
update_option("noai_ai_imported_$folder", true);

$article_num = intval(str_replace('article_', '', $folder));
update_option('noai_last_article_imported', $article_num);
update_option('noai_ai_next_generate_at', time() + 172800); // 2 روز


$imported_any = true;
}
if (!$imported_any) {
    $next = (int) get_option('noai_last_article_imported', 0) + 1;
    $expected_dir = trailingslashit($upload_dir) . 'article_' . $next;
    $need_redownload = (!is_dir($expected_dir)) ||
                       !file_exists($expected_dir . '/meta.json') ||
                       !file_exists($expected_dir . '/index.html');

    if ($need_redownload) {
        update_option('noai_ai_download_at', time() + 60);
       
    }
}

return $imported_any;
}




 function noai_ai_request_generate_article() {
    $site_slug = get_option('noai_site_slug', '');
    if ( ! $site_slug ) return;

    $url = rtrim(NOAI_REG_API, '/') . '/generate_article/' . rawurlencode($site_slug);
    $response = wp_remote_post($url, array(
        'headers'  => array('Content-Type' => 'application/json'),
        'body'     => wp_json_encode(array(
            'api_key' => get_option('noai_api_key', '')
        )),
        'timeout'  => 30,
        'sslverify'=> true,
    ));

    if ( is_wp_error($response) ) {
        return; // بی‌صدا رد شو؛ لاگ نداریم
    }
}









register_activation_hook( __FILE__, function () {
  
    wp_clear_scheduled_hook('noai_ai_generate_job');
    wp_clear_scheduled_hook('noai_ai_fetch_job');
    wp_clear_scheduled_hook('noai_ai_beacon_event');

    
    noai_content_seo_activate();

    $template_file = plugin_dir_path(__FILE__) . 'template/no-ai-article-template.php';
    $theme_dir     = get_stylesheet_directory();
    $destination   = trailingslashit($theme_dir) . 'no-ai-article-template.php';

    if (file_exists($template_file)) {
       
        copy($template_file, $destination);
    } else {
    
        if (defined('WP_DEBUG') && WP_DEBUG) {
           
        }
    }

   
    add_option('noai_has_settings', false);
    noai_ensure_beacon_scheduled();
});




/* =========================================================================
 *  No-AI Beacon Scheduler (no WP-Cron, no panel cron)
 * ========================================================================= */

function noai_lock($key, $ttl = 20) {
    if ( get_transient($key) ) return false;
    set_transient($key, 1, $ttl); return true;
}
function noai_unlock($key) { delete_transient($key); }
// ===== Run queue once (shared by tick + cron + daemon) =====
function noai_process_queue($source = 'BEACON') {
    $now = time();

    if ( ! noai_lock('noai_tick_lock', 20) ) {
        return 'busy';
    }

    try {
        // 1) GENERATE
        $t = (int) get_option('noai_ai_next_generate_at', 0);
        if ( $t && $now >= $t ) {
            update_option('noai_ai_next_generate_at', 0);
            noai_ai_request_generate_article();
            update_option('noai_ai_download_at', $now + 3 * 60);
            return 'generated';
        }

        // 2) DOWNLOAD
        $t = (int) get_option('noai_ai_download_at', 0);
        if ( $t && $now >= $t ) {
            $next = (int) get_option('noai_last_article_imported', 0) + 1;
            $dir  = noai_download_article_to_local($next);
            if ( $dir ) {
                $meta = trailingslashit($dir) . 'meta.json';
                $html = trailingslashit($dir) . 'index.html';
                if ( file_exists($meta) && file_exists($html) ) {
                    delete_option('noai_ai_download_at');

                    $ok = noai_import_articles();
                    if ( $ok ) {
                        delete_option('noai_ai_import_at');
                    } else {
                        update_option('noai_ai_import_at', $now + 10);
                    }
                } else {
                    update_option('noai_ai_download_at', $now + 60);
                }
            } else {
                update_option('noai_ai_download_at', $now + 60);
            }
            return 'downloaded';
        }

        // 3) IMPORT
        $t = (int) get_option('noai_ai_import_at', 0);
        if ( $t && $now >= $t ) {
            $ok = noai_import_articles();
            if ( $ok ) {
                delete_option('noai_ai_import_at');
                update_option('noai_ai_next_generate_at', $now + 172800);
                return 'imported';
            } else {
                update_option('noai_ai_import_at', $now + 10);
                return 'retry';
            }
        }

        return 'noop';
    } catch ( \Throwable $e ) {
        return 'error';
    } finally {
        noai_unlock('noai_tick_lock');
    }
}


// --- Self-ping helpers (fire & forget)
function noai_tick_url() {
    return rest_url('noai/v1/tick') . '?key=' . rawurlencode(NOAI_TICK_KEY);
}
function noai_self_ping() {
    $url = noai_tick_url();
    $args = array(
        'method'    => 'POST',
        'timeout'   => 3,   
        'blocking'  => true, 
        'sslverify' => true,
        'headers'   => array(
            'X-NOAI-Key'    => NOAI_TICK_KEY,
            'Cache-Control' => 'no-cache',
        ),
    );
    wp_remote_post($url, $args);
}
// === WP-Cron job: ping tick endpoint every minute ===
add_action('noai_ai_beacon_event', function () {
   
  
    noai_process_queue('CRON');
   
 
});



// ============================================================================
// ✅ REST API Endpoint (noai/v1/tick) — Secure + Review-compliant
// ============================================================================
add_action('rest_api_init', function () {
    register_rest_route('noai/v1', '/tick', [
        'methods'             => ['POST', 'GET'],
        'callback'            => 'noai_rest_tick_handler',
        'permission_callback' => 'noai_rest_permission_check',
        'args'                => [
            'key' => [
                'sanitize_callback' => 'sanitize_text_field',
                'validate_callback' => function($param) {
                    return is_string($param) && strlen($param) >= 10;
                }
            ]
        ],
    ]);
});

/**
 * 🔒 Permission check for NoAI tick endpoint
 */
function noai_rest_permission_check(\WP_REST_Request $req) {
    $key_header = sanitize_text_field($req->get_header('x-noai-key'));
    $key_param  = sanitize_text_field($req->get_param('key'));
    $provided   = $key_header ?: $key_param;

    if ( empty($provided) ) {
        return new \WP_Error(
            'noai_missing_key',
            'API key missing.',
            ['status' => 401]
        );
    }

    if ( $provided !== NOAI_TICK_KEY ) {
        return new \WP_Error(
            'noai_invalid_key',
            'Invalid authentication key.',
            ['status' => 403]
        );
    }

    return true;
}

/**
 * 🛰️ Process tick request securely
 */
function noai_rest_tick_handler(\WP_REST_Request $req) {
    $result = noai_process_queue('TICK');
    return new \WP_REST_Response(
        ['status' => 'ok', 'action' => $result],
        200
    );
}




// --- [NoAI] JSON-LD output disabled in public release ---
function noai_print_jsonld_tag() {
    // JSON-LD intentionally disabled in public version
    // This prevents output of schema or structured data in <head>
    return;
}

// Ensure no other instance tries to hook it again
remove_action('wp_head', 'noai_print_jsonld_tag', 5);
add_action('wp_head', 'noai_print_jsonld_tag', 5);


function noai_normalize_smart_quotes($s) {
    return str_replace(
        ["\xE2\x80\x9C","\xE2\x80\x9D","\xE2\x80\x98","\xE2\x80\x99"],
        ['"','"',"'", "'"],
        $s
    );
}

function noai_grab_trailing_jsonld(&$html, $article_url_base) {
  
    $tail = substr($html, -20000);
    $norm = noai_normalize_smart_quotes($tail);

    $pos = mb_strripos($norm, '@context');
    if ($pos === false) return '';

   
    $start = -1;
    for ($i = $pos; $i >= 0; $i--) {
        $c = $norm[$i];
        if ($c === '{' || $c === '[') { $start = $i; break; }
    }
    if ($start === -1) return '';

    
    $candidate = trim(substr($norm, $start));

    
    if (preg_match('/([\[{][\s\S]*[\]}])\s*$/u', $candidate, $m)) {
        $candidate = $m[1];
    }

    $decoded = json_decode($candidate, true);
    if (json_last_error() !== JSON_ERROR_NONE || !$decoded) {
       
        return '';
    }

   
    $rewrite = function (&$node) use (&$rewrite, $article_url_base) {
        if (is_array($node)) {
            foreach ($node as $k => &$v) {
                if (is_string($v)) {
                    
                    if (strpos($v, 'images/') === 0) {
                        $v = rtrim($article_url_base, '/') . '/' . ltrim($v, '/');
                    }
                } elseif (is_array($v)) {
                    $rewrite($v);
                }
            }
        }
    };
    $rewrite($decoded);

    $clean_json = json_encode($decoded, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);

    $global_start = strlen($html) - strlen($tail) + $start;
    $html = substr($html, 0, $global_start);

    return $clean_json; 
}
// ✅ Load admin CSS and JS only in No-AI pages
// ✅ Load admin CSS and JS only in No-AI pages
function noai_admin_assets($hook) {
    
    if ( ! in_array( $hook, ['toplevel_page_no-ai-dashboard', 'no-ai-dashboard_page_no-ai-settings'], true ) ) {
        return;
    }

    // ✅ Admin CSS
    wp_enqueue_style(
        'noai-admin',
        plugins_url('assets/css/admin.css', __FILE__),
        [],
        '3.1.0'
    );


    if ($hook === 'toplevel_page_no-ai-dashboard') {
        // enqueue Chart.js
        wp_enqueue_script(
            'noai-chart',
            plugins_url('assets/js/chart.js', __FILE__),
            [],
            '4.4.0',
            true
        );

        // get chart data
        $chart_sets = get_transient('noai_chart_data');

        $chart_js = '
        document.addEventListener("DOMContentLoaded", function() {
          const ctx = document.getElementById("noai-chart");
          const chartSets = ' . wp_json_encode($chart_sets) . ';
          if (!ctx || typeof Chart === "undefined") return;

          let currentRange = "week";
          let chart;

          function renderChart(range) {
            const labels = Object.keys(chartSets[range]);
            const data   = Object.values(chartSets[range]);
            if (chart) chart.destroy();
            chart = new Chart(ctx, {
              type: "line",
              data: {
                labels: labels,
                datasets: [{
                  label: "Articles",
                  data: data,
                  borderColor: "#3b82f6",
                  backgroundColor: "rgba(59,130,246,0.3)",
                  fill: true,
                  tension: 0.3
                }]
              },
              options: { 
                plugins: { legend: { display: false } },
                scales: {
                  x: {
                    ticks: { color: "#fff" },
                    grid: { color: "rgba(255,255,255,0.1)", drawBorder: false }
                  },
                  y: {
                    beginAtZero: true,
                    ticks: { color: "#fff", stepSize: 1 },
                    grid: {
                      drawBorder: false,
                      color: ctx => (ctx.tick.value === 0 ? "rgba(0,0,0,0)" : "rgba(255,255,255,0.1)"),
                      lineWidth: ctx => (ctx.tick.value === 0 ? 0 : 1)
                    }
                  }
                }
              }
            });
          }

          renderChart(currentRange);
          document.getElementById("noai-chart-range").addEventListener("change", e => {
            currentRange = e.target.value;
            renderChart(currentRange);
          });
        });
        ';
        wp_add_inline_script('noai-chart', $chart_js, 'after');
    }
}
add_action('admin_enqueue_scripts', 'noai_admin_assets');
