<?php
/**
 * Plugin Name: Nigeria Cities for WooCommerce
 * Plugin URI: https://jacksonjonah.com.ng/nigeria-cities-for-woocommerce/
 * Description: Adds Nigerian states and their LGAs as "states" to WooCommerce shipping zones.
 * Version: 1.1.0
 * Author: Jackson Jonah
 * Author URI: https://jacksonjonah.com.ng/about-me/
 * License: GPL2
 * Text Domain: nigeria-cities-for-woocommerce
 * Requires at least: 5.8
 * Requires PHP: 7.2
 * Requires Plugins: woocommerce
 */

defined('ABSPATH') || exit; // Prevent direct access

/**
 * Load Nigerian States and LGAs from local JSON file and add them to WooCommerce as states.
 *
 * @param array $states Existing WooCommerce states.
 * @return array Modified WooCommerce states with Nigerian LGAs added.
 */
function ncfws_add_nigeria_lgas_to_woocommerce($states)
{
    // Path to the local JSON file
    $json_file_path = plugin_dir_path(__FILE__) . 'nigeria-states-lgas.json';

    // Check if the file exists
    if (!file_exists($json_file_path)) {
        wp_die(esc_html__('JSON file not found. Please ensure the file exists in the plugin directory.', 'nigeria-cities-for-woocommerce'));
    }

    // Read the file content
    $json_content = file_get_contents($json_file_path);

    // Decode the JSON data
    $nigeria_lgas = json_decode($json_content, true);

    if (json_last_error() !== JSON_ERROR_NONE) {
        wp_die(esc_html__('Invalid JSON file format. Please check the plugin settings.', 'nigeria-cities-for-woocommerce'));
    }

    // Add Nigerian states and LGAs to WooCommerce
    foreach ($nigeria_lgas as $state => $lgas) {
        // Add state as a parent
        $states['NG'][$state] = sprintf(
            /* translators: %1$s is a Nigerian state name */
            esc_html__('%1$s State', 'nigeria-cities-for-woocommerce'),
            esc_html($state) // Escape state name
        );

        // Loop through LGAs (cities) under each state
        foreach ($lgas as $lga) {
            $cities_key = $state . '-' . $lga; // Creating unique key for each city
            $states['NG'][$cities_key] = sprintf(
                /* translators: %1$s is an LGA (local government area) name, %2$s is a Nigerian state name */
                esc_html__('%1$s, %2$s State', 'nigeria-cities-for-woocommerce'),
                esc_html($lga), // Escape LGA name
                esc_html($state) // Escape state name
            );
        }
    }

    return $states;
}
add_filter('woocommerce_states', 'ncfws_add_nigeria_lgas_to_woocommerce');

/**
 * Verify nonce input for security.
 */
function ncfws_verify_nonce()
{
    if (isset($_GET['ncfws_nonce'])) {
        $nonce = sanitize_text_field(wp_unslash($_GET['ncfws_nonce']));
        if (!wp_verify_nonce($nonce, 'ncfws_action')) {
            wp_die(esc_html__('Nonce verification failed.', 'nigeria-cities-for-woocommerce'));
        }
    }
}
add_action('init', 'ncfws_verify_nonce');