<?php
/**
 * Plugin Name: Nice Price Editor
 * Description: Admin price editor for WooCommerce products.
 * Version: 1.0.0
 * License: GPLv2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: nice-price-editor
 * Domain Path: /languages
 */

if (!defined('ABSPATH')) {
    exit;
}

if (!defined('NICEPRED_PLUGIN_FILE')) {
    define('NICEPRED_PLUGIN_FILE', __FILE__);
}

function nicepred_render_template($template_name, $vars = array()) {
    $template_path = plugin_dir_path(__FILE__) . 'templates/' . $template_name;
    if (!file_exists($template_path)) {
        return;
    }

    if (!empty($vars)) {
        extract($vars, EXTR_SKIP);
    }

    include $template_path;
}

function nicepred_format_price_display($price) {
    if ($price === '' || $price === null) {
        return '';
    }

    return number_format(
        (float) $price,
        wc_get_price_decimals(),
        wc_get_price_decimal_separator(),
        ' '
    );
}

function nicepred_register_admin_menu() {
    $hook = add_menu_page(
        'Nice Price Editor',
        'Nice Price Editor',
        'manage_woocommerce',
        'nice-price-editor',
        'nicepred_render_admin_page',
        'dashicons-money-alt',
        56
    );

    add_action('admin_enqueue_scripts', function ($current_hook) use ($hook) {
        if ($current_hook !== $hook) {
            return;
        }

        wp_enqueue_style(
            'nice-price-editor',
            plugins_url('assets/nice-price-editor.css', __FILE__),
            array(),
            '1.0.0'
        );

        wp_enqueue_script(
            'nice-price-editor',
            plugins_url('assets/nice-price-editor.js', __FILE__),
            array('jquery'),
            '1.0.0',
            true
        );

        wp_localize_script('nice-price-editor', 'nicepredData', array(
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('nicepred_update_price'),
        ));
    });
}
add_action('admin_menu', 'nicepred_register_admin_menu');

function nicepred_render_price_table($rows, $show_attributes, $attribute_columns = array()) {
    nicepred_render_template('price-table.php', array(
        'rows' => $rows,
        'show_attributes' => $show_attributes,
        'attribute_columns' => $attribute_columns,
    ));
}

function nicepred_render_admin_page() {
    if (!class_exists('WooCommerce')) {
        ?>
        <div class="notice notice-error">
            <p><?php echo esc_html__('WooCommerce is not active.', 'nice-price-editor'); ?></p>
        </div>
        <?php
        return;
    }

    $page = isset($_GET['page']) ? sanitize_text_field(wp_unslash($_GET['page'])) : '';
    $filter_nonce = isset($_GET['nicepred_filter_nonce']) ? sanitize_text_field(wp_unslash($_GET['nicepred_filter_nonce'])) : '';
    $filter_nonce_value = wp_create_nonce('nicepred_filter_products');
    $default_per_page = 20;
    $per_page_options = array(10, 20, 50, 100);
    if (!empty($filter_nonce) && wp_verify_nonce($filter_nonce, 'nicepred_filter_products')) {
        $paged = isset($_GET['paged']) ? absint($_GET['paged']) : 1;
        $selected_category = isset($_GET['category']) ? sanitize_text_field(wp_unslash($_GET['category'])) : '';
        $per_page = isset($_GET['per_page']) ? absint($_GET['per_page']) : $default_per_page;
    } else {
        $paged = 1;
        $selected_category = '';
        $per_page = $default_per_page;
    }

    if (!in_array($per_page, $per_page_options, true)) {
        $per_page = $default_per_page;
    }

    $categories = get_terms(array('taxonomy' => 'product_cat'));
    nicepred_render_template('category-filter.php', array(
        'categories' => $categories,
        'selected_category' => $selected_category,
        'page' => $page,
        'paged' => $paged,
        'per_page' => $per_page,
        'per_page_options' => $per_page_options,
    ));

    $args = array(
        'post_type' => 'product',
        'posts_per_page' => $per_page,
        'paged' => $paged,
    );

    if (!empty($selected_category)) {
        $args['tax_query'] = array(
            array(
                'taxonomy' => 'product_cat',
                'field' => 'slug',
                'terms' => $selected_category,
            ),
        );
    }

    $loop = new WP_Query($args);

    while ($loop->have_posts()) :
        $loop->the_post();
        global $product;
        $is_variable_type = $product->is_type('variable');
        $table_label = $is_variable_type
            ? __('Variable product (variations)', 'nice-price-editor')
            : __('Simple product', 'nice-price-editor');
        $rows = array();
        $show_attributes = false;
        $attribute_columns = array();
        if ($is_variable_type) {
            foreach ($product->get_variation_attributes() as $key => $value) {
                $attribute_columns[$key] = wc_attribute_label($key);
            }
            $show_attributes = !empty($attribute_columns);
            $available_variations = $product->get_available_variations();
            foreach ($available_variations as $variation) {
                $variation_id = $variation['variation_id'];
                $variation_obj = new WC_Product_Variation($variation_id);
                $attributes = array();
                foreach ($attribute_columns as $key => $label) {
                    $meta = get_post_meta($variation_id, 'attribute_' . sanitize_title($key), true);
                    $term = get_term_by('slug', $meta, sanitize_title($key));
                    $term_name = $term ? $term->name : $meta;
                    $attributes[$key] = array(
                        'label' => $label,
                        'value' => $term_name,
                    );
                }
                $rows[] = array(
                    'id' => $variation_obj->get_id(),
                    'sku' => $variation_obj->get_sku(),
                    'attributes' => $attributes,
                    'price_display' => nicepred_format_price_display($variation_obj->get_regular_price()),
                );
            }
        } else {
            $rows[] = array(
                'id' => $product->get_id(),
                'sku' => $product->get_sku(),
                'attributes' => array(),
                'price_display' => nicepred_format_price_display($product->get_regular_price()),
            );
        }

        nicepred_render_template('product-card.php', array(
            'product' => $product,
            'table_label' => $table_label,
            'rows' => $rows,
            'show_attributes' => $show_attributes,
            'attribute_columns' => $attribute_columns,
        ));
    endwhile;

    nicepred_render_template('pagination.php', array(
        'paged' => $paged,
        'max_num_pages' => $loop->max_num_pages,
        'page' => $page,
        'selected_category' => $selected_category,
        'filter_nonce' => $filter_nonce_value,
        'per_page' => $per_page,
    ));

    wp_reset_postdata();
}

function nicepred_update_price() {
    if (!current_user_can('manage_woocommerce')) {
        wp_send_json_error(array('message' => __('Insufficient permissions.', 'nice-price-editor')), 403);
    }

    check_ajax_referer('nicepred_update_price', 'nonce');

    $variation_id = isset($_POST['variationID']) ? absint($_POST['variationID']) : 0;
    $new_price_raw = isset($_POST['newPrice']) ? sanitize_text_field(wp_unslash($_POST['newPrice'])) : '';
    $new_price_normalized = str_replace(',', '.', $new_price_raw);
    $new_price = wc_format_decimal($new_price_normalized, wc_get_price_decimals());

    if (!$variation_id || $new_price === '') {
        wp_send_json_error(array('message' => __('Missing price or ID.', 'nice-price-editor')), 400);
    }

    $product = wc_get_product($variation_id);
    if (!$product) {
        wp_send_json_error(array('message' => __('Product not found.', 'nice-price-editor')), 404);
    }

    $product->set_regular_price($new_price);
    $product->set_price($new_price);
    $product->save();

    wp_send_json_success(array('newPrice' => nicepred_format_price_display($product->get_regular_price())));
}
add_action('wp_ajax_nicepred_update_price', 'nicepred_update_price');
