<?php

namespace NHRRob\Secure;

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Vulnerability Checker Class
 */
class Vulnerability
{

    /**
     * API Base URL
     */
    const API_BASE_URL = 'https://www.wpvulnerability.net/';

    /**
     * Cache key
     */
    const CACHE_KEY = 'nhrrob_secure_vulnerability_results';

    /**
     * Initialize the class
     */
    public function __construct()
    {
        add_action('nhrrob_secure_vulnerability_scan_cron', [$this, 'run_scan']);
    }

    /**
     * Get scan results (cached)
     *
     * @return array
     */
    public function get_results()
    {
        $results = get_transient(self::CACHE_KEY);

        if (false === $results) {
            return $this->run_scan();
        }

        return $results;
    }

    /**
     * Run the vulnerability scan
     *
     * @return array
     */
    public function run_scan()
    {
        $results = [
            'core' => $this->check_core(),
            'plugins' => $this->check_plugins(),
            'themes' => $this->check_themes(),
            'last_scan' => time(),
        ];

        set_transient(self::CACHE_KEY, $results, DAY_IN_SECONDS);

        // If vulnerabilities found, send email notification
        if ($this->has_vulnerabilities($results)) {
            $this->send_notification($results);
        }

        return $results;
    }

    /**
     * Check if any vulnerabilities were found
     *
     * @param array $results
     * @return bool
     */
    private function has_vulnerabilities($results)
    {
        if (!empty($results['core']))
            return true;

        foreach ($results['plugins'] as $plugin) {
            if (!empty($plugin['vulnerabilities']))
                return true;
        }

        foreach ($results['themes'] as $theme) {
            if (!empty($theme['vulnerabilities']))
                return true;
        }

        return false;
    }

    /**
     * Check WP Core
     */
    private function check_core()
    {
        global $wp_version;
        $response = wp_remote_get(self::API_BASE_URL . 'core/' . $wp_version);

        if (is_wp_error($response))
            return [];

        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);

        return !empty($data['data']['vulnerability']) ? $data['data']['vulnerability'] : [];
    }

    /**
     * Check Plugins
     */
    private function check_plugins()
    {
        if (!function_exists('get_plugins')) {
            require_once ABSPATH . 'wp-admin/includes/plugin.php';
        }

        $plugins = get_plugins();
        $vulnerable_plugins = [];

        foreach ($plugins as $plugin_path => $plugin_data) {
            $slug = dirname($plugin_path);
            if ('.' === $slug) {
                $slug = str_replace('.php', '', $plugin_path);
            }

            $response = wp_remote_get(self::API_BASE_URL . 'plugin/' . $slug);
            if (is_wp_error($response))
                continue;

            $body = wp_remote_retrieve_body($response);
            $data = json_decode($body, true);

            $vulnerabilities = ! empty( $data['data']['vulnerability'] ) ? $data['data']['vulnerability'] : [];

            if (!empty($vulnerabilities)) {
                $installed_version = $plugin_data['Version'];
                $matched_vulnerabilities = [];

                foreach ($vulnerabilities as $vuln) {
                    $is_vulnerable = false;

                    if (!empty($vuln['operator'])) {
                        $op = $vuln['operator'];
                        $min_ver = !empty($op['min_version']) ? $op['min_version'] : null;
                        $min_op  = !empty($op['min_operator']) ? $op['min_operator'] : null;
                        $max_ver = !empty($op['max_version']) ? $op['max_version'] : null;
                        $max_op  = !empty($op['max_operator']) ? $op['max_operator'] : null;
                        $unfixed = !empty($op['unfixed']) ? $op['unfixed'] : false;

                        // Check min version constraint
                        $min_check = true;
                        if ($min_ver && $min_op) {
                            $min_check = version_compare($installed_version, $min_ver, $min_op);
                        }

                        // Check max version constraint
                        $max_check = true;
                        if ($max_ver && $max_op) {
                            $max_check = version_compare($installed_version, $max_ver, $max_op);
                        }

                        if ($min_check && $max_check) {
                            $is_vulnerable = true;
                        }
                        
                        // If it's unfixed and we are within the range (or no range specified implies all?), 
                        // usually max_version is set even for unfixed to indicate "affected <= x.y.z (unfixed)".
                        // If unfixed is true, and we matched the version criteria, we are definitely vulnerable.
                    } else {
                        // Fallback: if no operator info, assume vulnerable if listed
                        $is_vulnerable = true;
                    }

                    if ($is_vulnerable) {
                        $matched_vulnerabilities[] = $vuln;
                    }
                }

                if (!empty($matched_vulnerabilities)) {
                    $vulnerable_plugins[] = [
                        'name' => $plugin_data['Name'],
                        'version' => $installed_version,
                        'vulnerabilities' => $matched_vulnerabilities
                    ];
                }
            }
        }

        return $vulnerable_plugins;
    }

    /**
     * Check Themes
     */
    private function check_themes()
    {
        $themes = wp_get_themes();
        $vulnerable_themes = [];

        foreach ($themes as $slug => $theme_data) {
            $response = wp_remote_get(self::API_BASE_URL . 'theme/' . $slug);
            if (is_wp_error($response))
                continue;

            $body = wp_remote_retrieve_body($response);
            $data = json_decode($body, true);

            $vulnerabilities = ! empty( $data['data']['vulnerability'] ) ? $data['data']['vulnerability'] : [];

            if (!empty($vulnerabilities)) {
                $installed_version = $theme_data->get('Version');
                $matched_vulnerabilities = [];

                foreach ($vulnerabilities as $vuln) {
                    $is_vulnerable = false;

                    if (!empty($vuln['operator'])) {
                        $op = $vuln['operator'];
                        $min_ver = !empty($op['min_version']) ? $op['min_version'] : null;
                        $min_op  = !empty($op['min_operator']) ? $op['min_operator'] : null;
                        $max_ver = !empty($op['max_version']) ? $op['max_version'] : null;
                        $max_op  = !empty($op['max_operator']) ? $op['max_operator'] : null;
                        $unfixed = !empty($op['unfixed']) ? $op['unfixed'] : false;

                        // Check min version constraint
                        $min_check = true;
                        if ($min_ver && $min_op) {
                            $min_check = version_compare($installed_version, $min_ver, $min_op);
                        }

                        // Check max version constraint
                        $max_check = true;
                        if ($max_ver && $max_op) {
                            $max_check = version_compare($installed_version, $max_ver, $max_op);
                        }

                        if ($min_check && $max_check) {
                            $is_vulnerable = true;
                        }
                    } else {
                        // Fallback: if no operator info, assume vulnerable if listed
                        $is_vulnerable = true;
                    }

                    if ($is_vulnerable) {
                        $matched_vulnerabilities[] = $vuln;
                    }
                }

                if (!empty($matched_vulnerabilities)) {
                    $vulnerable_themes[] = [
                        'name' => $theme_data->get('Name'),
                        'version' => $installed_version,
                        'vulnerabilities' => $matched_vulnerabilities
                    ];
                }
            }
        }

        return $vulnerable_themes;
    }

    /**
     * Send email notification
     */
    private function send_notification($results)
    {
        $to = get_option('admin_email');
        $subject = sprintf('[%s] Security Alert: Vulnerabilities Detected', get_bloginfo('name'));

        $message = "Vulnerabilities have been detected on your site.\n\n";

        if (!empty($results['core'])) {
            $message .= "WordPress Core:\n";
            foreach ($results['core'] as $v) {
                $message .= "- " . $v['name'] . "\n";
            }
            $message .= "\n";
        }

        if (!empty($results['plugins'])) {
            $message .= "Plugins:\n";
            foreach ($results['plugins'] as $plugin) {
                $message .= "- " . $plugin['name'] . " (" . $plugin['version'] . ")\n";
                foreach ($plugin['vulnerabilities'] as $v) {
                    $message .= "  * " . $v['name'] . "\n";
                }
            }
            $message .= "\n";
        }

        if (!empty($results['themes'])) {
            $message .= "Themes:\n";
            foreach ($results['themes'] as $theme) {
                $message .= "- " . $theme['name'] . " (" . $theme['version'] . ")\n";
                foreach ($theme['vulnerabilities'] as $v) {
                    $message .= "  * " . $v['name'] . "\n";
                }
            }
            $message .= "\n";
        }

        $message .= "\nPlease log in to your dashboard to take action: " . admin_url('tools.php?page=nhrrob-secure-settings');

        wp_mail($to, $subject, $message);
    }
}
