import { Card, CardBody, Button, Spinner, Notice, Icon } from '@wordpress/components';
import { __ } from '@wordpress/i18n';
import { useState, useEffect } from '@wordpress/element';
import apiFetch from '@wordpress/api-fetch';
import { decodeEntities } from '@wordpress/html-entities';

const VulnerabilityChecker = () => {
    const [results, setResults] = useState(null);
    const [loading, setLoading] = useState(true);
    const [scanning, setScanning] = useState(false);
    const [error, setError] = useState(null);

    useEffect(() => {
        fetchStatus();
    }, []);

    const fetchStatus = async () => {
        try {
            const data = await apiFetch({ path: '/nhrrob-secure/v1/vulnerability/status' });
            setResults(data);
            setLoading(false);
        } catch (err) {
            setError(__('Failed to fetch vulnerability status', 'nhrrob-secure'));
            setLoading(false);
        }
    };

    const handleScan = async () => {
        setScanning(true);
        setError(null);
        try {
            const data = await apiFetch({ path: '/nhrrob-secure/v1/vulnerability/scan', method: 'POST' });
            setResults(data);
        } catch (err) {
            setError(__('Failed to run vulnerability scan', 'nhrrob-secure'));
        } finally {
            setScanning(false);
        }
    };

    if (loading) {
        return (
            <Card className="nhrrob-secure-card">
                <CardBody>
                    <Spinner />
                </CardBody>
            </Card>
        );
    }

    const hasVulnerabilities = results && (
        results.core.length > 0 ||
        results.plugins.length > 0 ||
        results.themes.length > 0
    );

    const formatDate = (timestamp) => {
        if (!timestamp) return __('Never', 'nhrrob-secure');
        return new Date(timestamp * 1000).toLocaleString();
    };

    return (
        <Card className="nhrrob-secure-card nhrrob-secure-vulnerability-card">
            <CardBody>
                <div className="nhrrob-secure-card-header-flex">
                    <h2 className="nhrrob-secure-card-title">
                        {__('Vulnerability Checker', 'nhrrob-secure')}
                    </h2>
                    <Button
                        variant="primary"
                        onClick={handleScan}
                        isBusy={scanning}
                        disabled={scanning}
                        icon="update"
                        iconPosition="right"
                    >
                        {scanning ? __('Scanning...', 'nhrrob-secure') : __('Scan Now', 'nhrrob-secure')}
                    </Button>
                </div>

                <p className="nhrrob-secure-last-scan">
                    <strong>{__('Last Scan:', 'nhrrob-secure')}</strong> {formatDate(results.last_scan)}
                </p>

                {error && <Notice status="error" isDismissible={false}>{error}</Notice>}

                {!hasVulnerabilities ? (
                    <div className="nhrrob-secure-status-success">
                        <span className="dashicons dashicons-yes-alt"></span>
                        {__('No known vulnerabilities detected.', 'nhrrob-secure')}
                    </div>
                ) : (
                    <div className="nhrrob-secure-vulnerability-list">
                        <Notice status="warning" isDismissible={false}>
                            {__('Vulnerabilities detected! Please review and update the items below.', 'nhrrob-secure')}
                        </Notice>

                        {results.core.length > 0 && (
                            <div className="vulnerability-section">
                                <h3>{__('WordPress Core', 'nhrrob-secure')}</h3>
                                <ul>
                                    {results.core.map((v, i) => (
                                        <li key={i}>{decodeEntities(v.name)}</li>
                                    ))}
                                </ul>
                            </div>
                        )}

                        {results.plugins.length > 0 && (
                            <div className="vulnerability-section">
                                <h3>{__('Plugins', 'nhrrob-secure')}</h3>
                                {results.plugins.map((plugin, i) => (
                                    <div key={i} className="vulnerability-item">
                                        <strong>
                                            <a href="plugins.php" style={{ textDecoration: 'none', color: 'inherit' }}>
                                                {decodeEntities(plugin.name)}
                                            </a> 
                                            <span style={{ fontWeight: 'normal', color: '#666' }}> ({plugin.version})</span>
                                        </strong>
                                        <ul>
                                            {plugin.vulnerabilities.map((v, j) => (
                                                <li key={j}>{decodeEntities(v.name)}</li>
                                            ))}
                                        </ul>
                                    </div>
                                ))}
                            </div>
                        )}

                        {results.themes.length > 0 && (
                            <div className="vulnerability-section">
                                <h3>{__('Themes', 'nhrrob-secure')}</h3>
                                {results.themes.map((theme, i) => (
                                    <div key={i} className="vulnerability-item">
                                        <strong>{decodeEntities(theme.name)} ({theme.version})</strong>
                                        <ul>
                                            {theme.vulnerabilities.map((v, j) => (
                                                <li key={j}>{decodeEntities(v.name)}</li>
                                            ))}
                                        </ul>
                                    </div>
                                ))}
                            </div>
                        )}
                    </div>
                )}
            </CardBody>
        </Card>
    );
};

export default VulnerabilityChecker;
