import { registerBlockType } from '@wordpress/blocks';
import {
    useBlockProps,
    InspectorControls,
    PanelColorSettings
} from '@wordpress/block-editor';
import {
    PanelBody,
    TextControl,
    SelectControl,
    Spinner,
    FontSizePicker
} from '@wordpress/components';
import { useState, useEffect, useRef } from '@wordpress/element';
import apiFetch from '@wordpress/api-fetch';
import metadata from './block.json';

function debounce(func, delay) {
    let timeoutId;
    return (...args) => {
        clearTimeout(timeoutId);
        timeoutId = setTimeout(() => func(...args), delay);
    };
}

registerBlockType(metadata.name, {
    edit: EditComponent,
    save: () => null, // Use dynamic rendering on PHP side
});

function EditComponent({ attributes, setAttributes }) {
    const blockProps = useBlockProps();
    const [tempUsername, setTempUsername] = useState(attributes.username);
    const [previewContent, setPreviewContent] = useState('');
    const [isLoading, setIsLoading] = useState(false);

    // Create debounced function with useRef to maintain reference
    const updateUsernameDebounced = useRef(
        debounce((username) => {
            setAttributes({ username });
        }, 500)
    ).current;

    // Separate effect for setting initial values
    useEffect(() => {
        if (!attributes.username) {
            setAttributes({ 
                username: nhrccCoreContributionsCommonObj?.nhrccSettings?.username || '',
                preset: nhrccCoreContributionsCommonObj?.nhrccSettings?.preset || 'default'
            });
        }
    }, []); // Empty dependency array - runs once

    useEffect(() => {
        if (!attributes.username) {
            setPreviewContent('');
            return;
        }

        setIsLoading(true);

        fetchPreview(attributes)
            .then(setPreviewContent)
            .catch((error) => {
                console.error('Failed to load preview:', error);
                setPreviewContent(`Error: ${error.message}`);
            })
            .finally(() => setIsLoading(false));
    }, [
        attributes.username,
        attributes.preset,
        attributes.backgroundColor,
        attributes.textColor,
        attributes.linkColor,
        attributes.borderColor,
        attributes.borderRadius,
        attributes.padding,
        attributes.margin,
        attributes.fontSize,
        attributes.fontWeight,
        attributes.titleColor,
        attributes.titleBackgroundColor,
        attributes.titleFontSize,
        attributes.titleFontWeight,
        attributes.accentColor,
        attributes.metaColor,
        attributes.paginationColor,
        JSON.stringify(attributes.style)
    ]);

    return (
        <div {...blockProps}>
            <InspectorControls>
                <PanelBody title="Core Contributions Settings">
                    <TextControl
                        label="WordPress.org Username"
                        value={tempUsername}
                        onChange={(username) => {
                            setTempUsername(username);
                            updateUsernameDebounced(username);
                        }}
                        help="Enter your WordPress.org username to display contributions"
                    />
                    <SelectControl
                        label="Design Style"
                        value={attributes.preset}
                        options={PRESET_OPTIONS}
                        onChange={(preset) => setAttributes({ preset })}
                    />
                </PanelBody>
            </InspectorControls>
            <InspectorControls group="styles">
                <PanelColorSettings
                    title="Title Colors"
                    initialOpen={false}
                    colorSettings={[
                        {
                            value: attributes.titleColor,
                            onChange: (titleColor) => setAttributes({ titleColor }),
                            label: 'Title Color',
                        },
                        {
                            value: attributes.titleBackgroundColor,
                            onChange: (titleBackgroundColor) => setAttributes({ titleBackgroundColor }),
                            label: 'Title Background',
                        },
                    ]}
                />
                <PanelBody title="Title Typography" initialOpen={false}>
                    <FontSizePicker
                        value={attributes.titleFontSize}
                        onChange={(titleFontSize) => setAttributes({ titleFontSize })}
                    />
                    <SelectControl
                        label="Font Weight"
                        value={attributes.titleFontWeight}
                        options={[
                            { label: 'Default', value: '' },
                            { label: 'Normal', value: '400' },
                            { label: 'Medium', value: '500' },
                            { label: 'Semibold', value: '600' },
                            { label: 'Bold', value: '700' },
                            { label: 'Extra Bold', value: '800' },
                        ]}
                        onChange={(titleFontWeight) => setAttributes({ titleFontWeight })}
                    />
                </PanelBody>
                <PanelColorSettings
                    title="Item & Accent Colors"
                    initialOpen={false}
                    colorSettings={[
                        {
                            value: attributes.accentColor,
                            onChange: (accentColor) => setAttributes({ accentColor }),
                            label: 'Accent Color (Icons/Highlight)',
                        },
                        {
                            value: attributes.metaColor,
                            onChange: (metaColor) => setAttributes({ metaColor }),
                            label: 'Meta Text Color',
                        },
                    ]}
                />
                <PanelColorSettings
                    title="Pagination Colors"
                    initialOpen={false}
                    colorSettings={[
                        {
                            value: attributes.paginationColor,
                            onChange: (paginationColor) => setAttributes({ paginationColor }),
                            label: 'Pagination Color',
                        },
                    ]}
                />
            </InspectorControls>
            <PreviewContent 
                isLoading={isLoading}
                username={attributes.username}
                content={previewContent}
            />
        </div>
    );
}

// Constants
const PRESET_OPTIONS = [
    { label: 'Default', value: 'default' },
    { label: 'Minimal', value: 'minimal' },
    { label: 'Modern', value: 'modern' },
];

// API and Helper Functions
async function fetchPreview(attributes) {
    try {
        const response = await apiFetch({
            path: '/nhrcc-core-contributions/v1/core-contributions/render',
            method: 'POST',
            data: attributes,
        });
        return response.content || '';
    } catch (error) {
        console.error('Failed to fetch preview:', error);
        throw new Error('Failed to load preview');
    }
}

// Presentational Components
function PreviewContent({ isLoading, username, content }) {
    if (isLoading) {
        return <Spinner />;
    }
    
    if (!username) {
        return (
            <div className="components-placeholder">
                <p>Please set a WordPress.org username to preview the contributions.</p>
            </div>
        );
    }
    
    return (
        <div 
            className="nhr-core-contributions-preview"
            dangerouslySetInnerHTML={{ __html: content }}
        />
    );
}