/**
 * FAQ Accordion functionality for NextBrill Autopost
 */
(function() {
    'use strict';
    
    function initFAQAccordion() {
        // Find all FAQ accordions on the page
        const accordions = document.querySelectorAll('.nbapf-faq-accordion');
        
        accordions.forEach(function(accordion) {
            const questions = accordion.querySelectorAll('.nbapf-faq-question');
            
            questions.forEach(function(questionBtn) {
                // Skip if already initialized
                if (questionBtn.hasAttribute('data-initialized')) {
                    return;
                }
                
                // Mark as initialized
                questionBtn.setAttribute('data-initialized', 'true');
                
                // Get the FAQ ID from data attribute
                const faqId = questionBtn.getAttribute('data-faq-id');
                if (!faqId) return;
                
                // Get the answer element and icon
                const answer = accordion.querySelector('#nbapf-faq-answer-' + faqId);
                const icon = accordion.querySelector('#nbapf-faq-icon-' + faqId);
                
                if (!answer || !icon) return;
                
                // Add click event listener
                questionBtn.addEventListener('click', function(e) {
                    e.preventDefault();
                    toggleFAQ(answer, icon);
                });
            });
        });
    }
    
    function toggleFAQ(answer, icon) {
        const isOpen = answer.style.display === 'block' || answer.classList.contains('nbapf-faq-open');
        
        if (!isOpen) {
            // Open
            answer.style.display = 'block';
            answer.classList.add('nbapf-faq-open');
            icon.textContent = '−';
            icon.classList.add('nbapf-faq-open');
        } else {
            // Close
            answer.style.display = 'none';
            answer.classList.remove('nbapf-faq-open');
            icon.textContent = '+';
            icon.classList.remove('nbapf-faq-open');
        }
    }
    
    // Initialize when DOM is ready
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', initFAQAccordion);
    } else {
        initFAQAccordion();
    }
    
    // Also initialize for dynamically loaded content (AJAX, etc.)
    if (typeof jQuery !== 'undefined') {
        jQuery(document).ready(function($) {
            initFAQAccordion();
            
            // Re-initialize on AJAX content load
            $(document).on('DOMNodeInserted', function(e) {
                if ($(e.target).find('.nbapf-faq-accordion').length > 0) {
                    setTimeout(initFAQAccordion, 100);
                }
            });
        });
    }
})();

