<?php
if (!defined('ABSPATH')) exit;

class NBAPF_Scheduler {
    public function __construct() {
        add_action('nbapf_process_queue', array($this, 'process_queue'));
        
        // Add manual process button
        add_action('admin_post_nbapf_process_now', array($this, 'manual_process'));
        
        // Ensure cron is scheduled with the correct interval
        $this->ensure_cron_scheduled();
        
        // Check queue on every page load (frontend and admin) for reliable processing
        // This ensures posts are published at their scheduled time even if cron is delayed
        add_action('init', array($this, 'maybe_process_queue_on_load'), 999);
    }
    
    /**
     * Ensure the cron is scheduled with the correct interval (every minute)
     */
    private function ensure_cron_scheduled() {
        $next_scheduled = wp_next_scheduled('nbapf_process_queue');
        
        if (!$next_scheduled) {
            // No cron scheduled, schedule it
            wp_schedule_event(time(), 'nbapf_every_minute', 'nbapf_process_queue');
        } else {
            // Check if it's using the old hourly schedule
            $cron_array = _get_cron_array();
            if ($cron_array) {
                foreach ($cron_array as $timestamp => $cron) {
                    if (isset($cron['nbapf_process_queue'])) {
                        foreach ($cron['nbapf_process_queue'] as $hook => $details) {
                            // If it's scheduled with 'hourly', reschedule it
                            if (isset($details['schedule']) && $details['schedule'] === 'hourly') {
                                wp_unschedule_event($timestamp, 'nbapf_process_queue');
                                wp_schedule_event(time(), 'nbapf_every_minute', 'nbapf_process_queue');
                                break 2;
                            }
                        }
                    }
                }
            }
        }
    }
    
    /**
     * Check and process queue on every page load
     * This ensures items are processed at their scheduled time even if WordPress cron is delayed
     * Works for all interval types: minutes, hours, days, weeks, months
     */
    public function maybe_process_queue_on_load() {
        // Throttle: only check once per minute to avoid performance issues
        // Use a shorter throttle for better responsiveness
        $last_check = get_transient('nbapf_queue_check');
        $throttle_seconds = 30; // Check every 30 seconds minimum
        
        if ($last_check && (time() - $last_check) < $throttle_seconds) {
            return;
        }
        
        set_transient('nbapf_queue_check', time(), 120); // 2 minutes
        
        // Check if there are any pending items that are due
        global $wpdb;
        $table = $wpdb->prefix . 'nbapf_queue';
        $table_escaped = esc_sql($table);
        
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared,WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Table name is safe, custom table operation, real-time data needed
        $pending_items = $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM `{$table_escaped}` 
            WHERE status = 'pending' 
            AND scheduled_time <= %s
            LIMIT 1",
            current_time('mysql')
        ));
        
        // If there are items due, process them
        if ($pending_items > 0) {
            // Process in background to avoid blocking page load
            $this->process_queue();
        }
    }
    
    /**
     * Manual process trigger
     */
    public function manual_process() {
        check_admin_referer('nbapf-process-now');
        $this->process_queue();
        wp_safe_redirect(add_query_arg('message', 'queue_processed', admin_url('admin.php?page=nbapf-queue')));
        exit;
    }
    
    public function process_queue() {
        global $wpdb;
        $table = $wpdb->prefix . 'nbapf_queue';
        
        // Get all pending items that are due (scheduled_time has passed)
        // Process up to 5 items per run to avoid timeouts
        $table_escaped = esc_sql($table);
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared,WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Table name is safe, custom table operation, real-time data needed
        $items = $wpdb->get_results($wpdb->prepare(
            "SELECT * FROM `{$table_escaped}` 
            WHERE status = 'pending' 
            AND scheduled_time <= %s 
            ORDER BY scheduled_time ASC 
            LIMIT 5",
            current_time('mysql')
        ));
        
        if (empty($items)) {
            return; // No items to process
        }
        
        if (defined('WP_DEBUG') && WP_DEBUG) {
            // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Conditional logging for debugging
            error_log('NBAPF: Processing ' . count($items) . ' queue items');
        }
        
        foreach ($items as $item) {
            $this->process_item($item);
        }
    }
    
    private function process_item($item) {
        global $wpdb;
        
        if (defined('WP_DEBUG') && WP_DEBUG) {
            // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Conditional logging for debugging
            error_log('NBAPF: Processing item #' . $item->id . ' - ' . $item->topic_title);
        }
        
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table operation, necessary for plugin functionality
        $wpdb->update($wpdb->prefix . 'nbapf_queue', array('status' => 'processing'), array('id' => $item->id), array('%s'), array('%d'));
        
        $generator = new NBAPF_Generator();
        $result = $generator->generate_post($item);
        
        if ($result['success']) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Conditional logging for debugging
                error_log('NBAPF: Successfully created post #' . $result['post_id']);
            }
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table operation, necessary for plugin functionality
            $wpdb->update($wpdb->prefix . 'nbapf_queue', array('status' => 'completed', 'post_id' => $result['post_id']), array('id' => $item->id), array('%s', '%d'), array('%d'));
        } else {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Conditional logging for debugging
                error_log('NBAPF: Failed to create post - ' . ($result['message'] ?? 'Unknown error'));
            }
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table operation, necessary for plugin functionality
            $wpdb->update($wpdb->prefix . 'nbapf_queue', array('status' => 'failed', 'error_message' => $result['message'] ?? 'Unknown error'), array('id' => $item->id), array('%s', '%s'), array('%d'));
        }
    }
}

