<?php
if (!defined('ABSPATH')) exit;

class NBAPF_Generator {
    public function generate_post($item) {
        $api_key = get_option('nbapf_openai_api_key');
        if (empty($api_key)) return array('success' => false, 'message' => 'OpenAI API key not configured');
        
        $content = $this->generate_content($item, $api_key);
        if (empty($content)) return array('success' => false, 'message' => 'Failed to generate content');
        
        // Clean up image placeholders from content - images are only used as featured images
        $content = $this->clean_image_placeholders($content);
        
        // Ensure all external links have rel="nofollow"
        $content = $this->add_nofollow_to_external_links($content);
        
        // Generate meta description for SEO (short, no quotes)
        $meta_description = $this->generate_meta_description($item, $api_key);
        
        // Create SEO-optimized title (includes focus keyword near beginning and a number)
        $seo_title = $this->create_seo_title($item);
        
        // Create SEO-optimized slug with focus keyword (under 94 characters)
        $slug = $this->create_seo_slug($item);
        
        // Generate FAQs if requested
        $faq_content = '';
        if (!empty($item->include_faq) && $item->include_faq == 1) {
            $faq_content = $this->generate_faqs($item, $api_key);
        }
        
        // Combine content and FAQs
        $full_content = $content;
        if (!empty($faq_content)) {
            $full_content .= "\n\n" . $faq_content;
        }
        
        // Create WordPress post
        $post_id = wp_insert_post(array(
            'post_title' => $item->topic_title,
            'post_content' => $full_content,
            'post_status' => 'publish',
            'post_name' => $slug,
            'post_category' => $item->category ? array($item->category) : array(),
            'tags_input' => $item->tags
        ));
        
        if (is_wp_error($post_id)) {
            return array('success' => false, 'message' => $post_id->get_error_message());
        }
        

        // Generate and set featured image using DALL-E
        $this->generate_featured_image($post_id, $item, $api_key);
        
        // Set Yoast SEO meta data
        $this->set_yoast_meta($post_id, $item, $meta_description, $slug, $seo_title);
        
        // Set Rank Math SEO meta data
        $this->set_rankmath_meta($post_id, $item, $meta_description, $seo_title);
        
        // Add FAQ schema if FAQs were generated
        if (!empty($faq_content)) {
            $this->add_faq_schema($post_id, $item, $api_key);
        }
        
        return array('success' => true, 'post_id' => $post_id);
    }
    
    /**
     * Clean up image placeholders from generated content
     * Images are only used as featured images, not in content
     */
    private function clean_image_placeholders($content) {
        // Remove image placeholders in format [IMAGE: ...]
        // Pattern matches [IMAGE: ...] with any content inside
        $content = preg_replace('/\[IMAGE:[^\]]+\]/i', '', $content);
        
        // Also remove any variations like [IMAGE: ... - Alt text: ...]
        $content = preg_replace('/\[IMAGE:[^\]]*Alt text:[^\]]+\]/i', '', $content);
        
        // Remove any remaining image-related placeholders
        $content = preg_replace('/\[IMAGE[^\]]*\]/i', '', $content);
        
        // Remove any HTML img tags that might have been generated
        $content = preg_replace('/<img[^>]*>/i', '', $content);
        
        // Remove any image-related HTML comments
        $content = preg_replace('/<!--[^>]*IMAGE[^>]*-->/i', '', $content);
        
        // Clean up extra blank lines that might result from removal
        $content = preg_replace('/\n\s*\n\s*\n/', "\n\n", $content);
        
        return trim($content);
    }
    
    /**
     * Add rel="nofollow" to all external links in content
     * Ensures all external links have nofollow attribute for SEO
     */
    private function add_nofollow_to_external_links($content) {
        // Get the site's domain for comparison
        $site_url = home_url();
        $parsed_url = wp_parse_url($site_url);
        $site_domain = isset($parsed_url['host']) ? $parsed_url['host'] : '';
        
        // Pattern to match all anchor tags
        $pattern = '/<a\s+([^>]*)>/i';
        
        $content = preg_replace_callback($pattern, function($matches) use ($site_domain) {
            $attributes = $matches[1];
            
            // Extract href attribute
            if (preg_match('/href=["\']([^"\']+)["\']/i', $attributes, $href_match)) {
                $href = $href_match[1];
                
                // Check if it's an external link
                $is_external = false;
                
                // Absolute URL check
                if (preg_match('/^https?:\/\//i', $href)) {
                    $parsed_link = wp_parse_url($href);
                    $link_domain = isset($parsed_link['host']) ? $parsed_link['host'] : '';
                    // If domain exists and is different from site domain, it's external
                    if ($link_domain && strtolower($link_domain) !== strtolower($site_domain)) {
                        $is_external = true;
                    }
                } elseif (!preg_match('/^#|^\/|^mailto:|^tel:/i', $href)) {
                    // Relative links that don't start with /, #, mailto:, or tel: are treated as external
                    $is_external = true;
                }
                
                if ($is_external) {
                    // Check if rel attribute already exists
                    $rel_attrs = array();
                    if (preg_match('/rel=["\']([^"\']*)["\']/i', $attributes, $rel_match)) {
                        $existing_rel = trim($rel_match[1]);
                        $rel_attrs = array_filter(array_map('trim', explode(' ', $existing_rel)));
                    }
                    
                    // Ensure nofollow is present
                    if (!in_array('nofollow', $rel_attrs)) {
                        $rel_attrs[] = 'nofollow';
                    }
                    
                    // Ensure noopener is present if target="_blank" exists
                    if (preg_match('/target=["\']_blank["\']/i', $attributes) && !in_array('noopener', $rel_attrs)) {
                        $rel_attrs[] = 'noopener';
                    }
                    
                    // Rebuild rel attribute
                    $new_rel = implode(' ', array_unique($rel_attrs));
                    
                    // Replace or add rel attribute
                    if (preg_match('/rel=["\'][^"\']*["\']/i', $attributes)) {
                        $attributes = preg_replace('/rel=["\'][^"\']*["\']/i', 'rel="' . $new_rel . '"', $attributes);
                    } else {
                        $attributes .= ' rel="' . $new_rel . '"';
                    }
                }
            }
            
            return '<a ' . $attributes . '>';
        }, $content);
        
        return $content;
    }
    
    /**
     * Generate FAQs in accordion format (6 FAQs)
     */
    private function generate_faqs($item, $api_key) {
        $model = get_option('nbapf_openai_model', 'gpt-4');
        
        $prompt = "Generate 4-6 frequently asked questions (FAQs) about '{$item->topic_title}' with the focus keyword '{$item->focus_keyword}'. 

REQUIREMENTS:
- Generate exactly 4-6 FAQs (preferably 6, but at least 4)
- Each FAQ should be relevant to the topic
- Include the focus keyword '{$item->focus_keyword}' naturally in some questions
- Answers should be concise (2-3 sentences each)
- Format as JSON array with 'question' and 'answer' fields

Return ONLY valid JSON in this exact format:
[
  {\"question\": \"Question 1 here?\", \"answer\": \"Answer 1 here.\"},
  {\"question\": \"Question 2 here?\", \"answer\": \"Answer 2 here.\"},
  ...
]";

        $response = wp_remote_post('https://api.openai.com/v1/chat/completions', array(
            'headers' => array('Authorization' => 'Bearer ' . $api_key, 'Content-Type' => 'application/json'),
            'body' => json_encode(array(
                'model' => $model,
                'messages' => array(
                    array('role' => 'system', 'content' => 'You are an expert content writer. Generate FAQs in valid JSON format only.'),
                    array('role' => 'user', 'content' => $prompt)
                ),
                'max_tokens' => 1000,
                'temperature' => 0.7
            )),
            'timeout' => 60
        ));
        
        if (is_wp_error($response)) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Conditional logging for debugging
                error_log('NBAPF: FAQ generation failed - ' . $response->get_error_message());
            }
            return '';
        }
        
        $body = json_decode(wp_remote_retrieve_body($response), true);
        $faq_json = $body['choices'][0]['message']['content'] ?? '';
        
        // Clean JSON (remove markdown code blocks if present)
        $faq_json = preg_replace('/```json\s*/', '', $faq_json);
        $faq_json = preg_replace('/```\s*/', '', $faq_json);
        $faq_json = trim($faq_json);
        
        $faqs = json_decode($faq_json, true);
        
        if (empty($faqs) || !is_array($faqs)) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Conditional logging for debugging
                error_log('NBAPF: Failed to parse FAQ JSON');
            }
            return '';
        }
        
        // It will generate 6 FAQs
        $faqs = array_slice($faqs, 0, 6);
        
        // Generate accordion HTML (no inline JS or CSS - handled by separate files)
        $accordion_html = '<div class="nbapf-faq-accordion">';
        $accordion_html .= '<h2>Frequently Asked Questions</h2>';
        
        foreach ($faqs as $index => $faq) {
            $faq_id = 'faq-' . $index;
            $accordion_html .= '<div class="nbapf-faq-item">';
            $accordion_html .= '<button class="nbapf-faq-question" data-faq-id="' . esc_attr($index) . '" type="button">';
            $accordion_html .= '<span>' . esc_html($faq['question']) . '</span>';
            $accordion_html .= '<span class="nbapf-faq-icon" id="nbapf-faq-icon-' . esc_attr($index) . '">+</span>';
            $accordion_html .= '</button>';
            $accordion_html .= '<div class="nbapf-faq-answer" id="nbapf-faq-answer-' . esc_attr($index) . '">';
            $accordion_html .= '<p>' . wp_kses_post($faq['answer']) . '</p>';
            $accordion_html .= '</div>';
            $accordion_html .= '</div>';
        }
        
        $accordion_html .= '</div>';
        
        return $accordion_html;
    }
    
    /**
     * Add FAQ schema markup for SEO
     */
    private function add_faq_schema($post_id, $item, $api_key) {
        // This will be handled by Yoast SEO if installed, but we can add custom schema if needed
        // For now, Yoast SEO handles FAQ schema automatically when it detects FAQ content
        if (defined('WP_DEBUG') && WP_DEBUG) {
            // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Conditional logging for debugging
            error_log('NBAPF: FAQ content added to post #' . $post_id . ' - Yoast SEO will handle schema');
        }
    }
    
    /**
     * Create SEO-optimized title (includes focus keyword near beginning and a number)
     * Rank Math requires: Focus keyword near beginning + number in title
     */
    private function create_seo_title($item) {
        $focus = $item->focus_keyword;
        $max_length = 60; // Rank Math allows up to 60 chars for better SEO
        
        // Generate a number (use year, or a relevant number like "5", "10", "7", etc.)
        $numbers = array('5', '7', '10', '2024', '2025');
        $number = $numbers[array_rand($numbers)];
        
        // Try different formats to ensure focus keyword is near beginning
        $formats = array(
            $focus . ' ' . $number . ' ' . 'Tips Guide',
            $number . ' ' . $focus . ' ' . 'Strategies',
            $focus . ': ' . $number . ' ' . 'Ways',
            'Top ' . $number . ' ' . $focus . ' ' . 'Tips',
            $focus . ' - ' . $number . ' ' . 'Best Practices'
        );
        
        // Find the best format that fits within max_length
        $best_title = '';
        foreach ($formats as $format) {
            if (strlen($format) <= $max_length) {
                $best_title = $format;
                break;
            }
        }
        
        // If no format fits, create a simple one
        if (empty($best_title)) {
            // Try: "Focus Keyword 2024 Guide"
            $simple = $focus . ' ' . $number . ' Guide';
            if (strlen($simple) <= $max_length) {
                $best_title = $simple;
            } else {
                // Last resort: just focus keyword + number
                $best_title = substr($focus . ' ' . $number, 0, $max_length);
            }
        }
        
        return $best_title;
    }
    
    /**
     * Create SEO-optimized slug with focus keyword (under 94 characters for Rank Math)
     */
    private function create_seo_slug($item) {
        $focus = strtolower(trim($item->focus_keyword));
        $title = strtolower(trim($item->topic_title));
        
        // Create slug starting with focus keyword
        $focus_slug = sanitize_title($focus);
        $title_slug = sanitize_title($title);
        
        // If focus keyword is already in title, use title as base
        if (strpos($title_slug, $focus_slug) !== false) {
            $slug = $title_slug;
        } else {
            // Combine: focus-keyword-topic-title
            $slug = $focus_slug . '-' . $title_slug;
        }
        
        // Ensure slug is under 94 characters (Rank Math requirement)
        $max_slug_length = 94;
        if (strlen($slug) > $max_slug_length) {
            // Truncate but keep focus keyword
            $focus_length = strlen($focus_slug);
            if ($focus_length < $max_slug_length) {
                $remaining = $max_slug_length - $focus_length - 1; // -1 for hyphen
                $title_part = substr($title_slug, 0, $remaining);
                $slug = $focus_slug . '-' . $title_part;
            } else {
                // Focus keyword itself is too long, just truncate it
                $slug = substr($focus_slug, 0, $max_slug_length);
            }
        }
        
        // Remove trailing hyphens
        $slug = rtrim($slug, '-');
        
        return $slug;
    }
    
    /**
     * Generate featured image using DALL-E
     */
    private function generate_featured_image($post_id, $item, $api_key) {
        $use_dalle = get_option('nbapf_use_dalle', false);
        
        if (!$use_dalle) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Conditional logging for debugging
                error_log('NBAPF: DALL-E disabled, skipping featured image generation');
            }
            return false;
        }
        
        // Use featured_image_keyword if provided, otherwise use focus_keyword
        $image_prompt = !empty($item->featured_image_keyword) 
            ? $item->featured_image_keyword 
            : $item->focus_keyword;
        
        // Create detailed prompt for DALL-E
        $prompt = "Professional blog header image about {$image_prompt}. Clean, modern, high-quality, suitable for business blog. No text or words in the image.";
        
        if (defined('WP_DEBUG') && WP_DEBUG) {
            // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Conditional logging for debugging
            error_log('NBAPF: Generating DALL-E image with prompt: ' . $prompt);
        }
        
        // Call DALL-E API
        $response = wp_remote_post('https://api.openai.com/v1/images/generations', array(
            'headers' => array(
                'Authorization' => 'Bearer ' . $api_key,
                'Content-Type' => 'application/json'
            ),
            'body' => json_encode(array(
                'model' => 'dall-e-3',
                'prompt' => $prompt,
                'n' => 1,
                'size' => '1792x1024', // Wide format for blog header
                'quality' => 'standard'
            )),
            'timeout' => 60
        ));
        
        if (is_wp_error($response)) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Conditional logging for debugging
                error_log('NBAPF: DALL-E request failed - ' . $response->get_error_message());
            }
            return false;
        }
        
        $body = json_decode(wp_remote_retrieve_body($response), true);
        
        if (empty($body['data'][0]['url'])) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Conditional logging for debugging
                error_log('NBAPF: DALL-E response missing image URL');
            }
            return false;
        }
        
        $image_url = $body['data'][0]['url'];
        
        // Download image
        $image_data = file_get_contents($image_url);
        if (!$image_data) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Conditional logging for debugging
                error_log('NBAPF: Failed to download DALL-E image');
            }
            return false;
        }
        
        // Upload to WordPress
        $upload = wp_upload_bits(
            sanitize_file_name($item->focus_keyword) . '-' . time() . '.png',
            null,
            $image_data
        );
        
        if ($upload['error']) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Conditional logging for debugging
                error_log('NBAPF: Failed to upload image - ' . $upload['error']);
            }
            return false;
        }
        
        // Create attachment
        $attachment = array(
            'post_mime_type' => 'image/png',
            'post_title' => $item->focus_keyword,
            'post_content' => '',
            'post_status' => 'inherit'
        );
        
        $attach_id = wp_insert_attachment($attachment, $upload['file'], $post_id);
        
        if (is_wp_error($attach_id)) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Conditional logging for debugging
                error_log('NBAPF: Failed to create attachment');
            }
            return false;
        }
        
        // Generate metadata
        require_once(ABSPATH . 'wp-admin/includes/image.php');
        $attach_data = wp_generate_attachment_metadata($attach_id, $upload['file']);
        wp_update_attachment_metadata($attach_id, $attach_data);
        
        // Set alt text with focus keyword
        update_post_meta($attach_id, '_wp_attachment_image_alt', $item->focus_keyword);
        
        // Set as featured image
        set_post_thumbnail($post_id, $attach_id);
        
        if (defined('WP_DEBUG') && WP_DEBUG) {
            // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Conditional logging for debugging
            error_log('NBAPF: Featured image set successfully for post #' . $post_id);
        }
        return true;
    }
    
    /**
     * Set Yoast SEO meta data
     */
    private function set_yoast_meta($post_id, $item, $meta_description, $slug, $seo_title) {
        // Focus keyphrase - use focus keyword
        update_post_meta($post_id, '_yoast_wpseo_focuskw', $item->focus_keyword);
        
        // Meta description - generated by AI (short, no quotes)
        update_post_meta($post_id, '_yoast_wpseo_metadesc', $meta_description);
        
        // SEO Title - optimized (includes focus keyword near beginning and number)
        update_post_meta($post_id, '_yoast_wpseo_title', $seo_title);
        
        // Canonical URL
        update_post_meta($post_id, '_yoast_wpseo_canonical', get_permalink($post_id));
        
        // Cornerstone content
        update_post_meta($post_id, '_yoast_wpseo_is_cornerstone', '0');
        
        if (defined('WP_DEBUG') && WP_DEBUG) {
            // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Conditional logging for debugging
            error_log('NBAPF: Set Yoast SEO meta for post #' . $post_id . ' - Title: ' . $seo_title . ' (' . strlen($seo_title) . ' chars)');
        }
    }
    
    /**
     * Set Rank Math SEO meta data
     */
    private function set_rankmath_meta($post_id, $item, $meta_description, $seo_title) {
        // Focus keyword - Rank Math uses 'rank_math_focus_keyword'
        update_post_meta($post_id, 'rank_math_focus_keyword', $item->focus_keyword);
        
        // SEO Title - Rank Math uses 'rank_math_title'
        update_post_meta($post_id, 'rank_math_title', $seo_title);
        
        // Meta Description - Rank Math uses 'rank_math_description'
        update_post_meta($post_id, 'rank_math_description', $meta_description);
        
        // Canonical URL - Rank Math uses 'rank_math_canonical_url'
        update_post_meta($post_id, 'rank_math_canonical_url', get_permalink($post_id));
        
        // Set primary focus keyword (for Rank Math analysis)
        update_post_meta($post_id, 'rank_math_primary_focus_keyword', $item->focus_keyword);
        
        if (defined('WP_DEBUG') && WP_DEBUG) {
            // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Conditional logging for debugging
            error_log('NBAPF: Set Rank Math SEO meta for post #' . $post_id . ' - Focus Keyword: ' . $item->focus_keyword);
        }
    }
    
    /**
     * Generate meta description using OpenAI (under 150 chars, no quotes, includes focus keyword)
     * Rank Math requires: Focus keyword in meta description
     */
    private function generate_meta_description($item, $api_key) {
        $model = get_option('nbapf_openai_model', 'gpt-4');
        
        $prompt = "Write a compelling meta description for a blog post about '{$item->topic_title}'. CRITICAL: MUST include the focus keyword '{$item->focus_keyword}' in the meta description. Must be under 150 characters total. Do NOT use quotation marks. Make it click-worthy and actionable. Start with or include the focus keyword naturally.";
        
        $response = wp_remote_post('https://api.openai.com/v1/chat/completions', array(
            'headers' => array('Authorization' => 'Bearer ' . $api_key, 'Content-Type' => 'application/json'),
            'body' => json_encode(array(
                'model' => $model,
                'messages' => array(
                    array('role' => 'system', 'content' => 'You are an SEO expert. Write meta descriptions under 150 characters with NO quotation marks.'),
                    array('role' => 'user', 'content' => $prompt)
                ),
                'max_tokens' => 100,
                'temperature' => 0.7
            )),
            'timeout' => 30
        ));
        
        if (is_wp_error($response)) return '';
        
        $body = json_decode(wp_remote_retrieve_body($response), true);
        $description = $body['choices'][0]['message']['content'] ?? '';
        
        // Remove any quotation marks (including smart quotes)
        $quotes = array('"', "'", '`', chr(0xE2) . chr(0x80) . chr(0x9C), chr(0xE2) . chr(0x80) . chr(0x9D), chr(0xE2) . chr(0x80) . chr(0x98), chr(0xE2) . chr(0x80) . chr(0x99));
        $description = str_replace($quotes, '', $description);
        
        // Ensure it's within 150 characters
        if (strlen($description) > 150) {
            $description = substr($description, 0, 147) . '...';
        }
        
        return trim($description);
    }
    
    private function generate_content($item, $api_key) {
        $model = get_option('nbapf_openai_model', 'gpt-4');
        // Ensure minimum 600 words for Rank Math SEO (user requirement)
        $word_count = max(600, $item->desired_word_count ?? 1500);
        
        // Parse internal links
        $internal_links = '';
        if (!empty($item->internal_links)) {
            $links_array = explode("\n", $item->internal_links);
            $internal_links = "\n\nInternal links to naturally include (use descriptive anchor text):\n";
            foreach ($links_array as $link) {
                if (!empty(trim($link))) {
                    $internal_links .= "- " . trim($link) . "\n";
                }
            }
        }
        
        // Parse external links
        $external_links = '';
        if (!empty($item->external_links)) {
            $links_array = explode("\n", $item->external_links);
            $external_links = "\n\nExternal links to include (use descriptive anchor text):\n";
            foreach ($links_array as $link) {
                if (!empty(trim($link))) {
                    $external_links .= "- " . trim($link) . "\n";
                }
            }
        } else {
            $external_links = "\n\nIMPORTANT: You MUST include 2-3 authoritative external links automatically. Link to: Wikipedia, .gov sites, .edu sites, major industry publications, research studies, or reputable organizations. Use descriptive anchor text with relevant keywords.";
        }
        
        // Enhanced prompt with all SEO requirements for both Yoast and Rank Math
        $prompt = "Write a comprehensive, SEO-optimized blog post about '{$item->topic_title}'.

CRITICAL SEO REQUIREMENTS (Yoast + Rank Math):

1. FOCUS KEYWORD: '{$item->focus_keyword}'
   - CRITICAL: Use this EXACT phrase at the BEGINNING of the first paragraph (within first 50 words) - ONCE
   - Target keyword density: ~1% (approximately 6-10 times for {$word_count} words)
   - Use it in at least 2-3 H2 headings (Rank Math requires focus keyword in subheadings)
   - Use it in at least 1-2 H3 headings
   - Use it naturally throughout the content body (6-10 times total)
   - Use keyword SYNONYMS and related terms in other headings and content

2. SECONDARY KEYWORDS: {$item->secondary_keywords}
   - Use these 2-3 times EACH naturally throughout the content
   - Use synonyms and related terms frequently

3. HEADINGS STRUCTURE (CRITICAL for Rank Math):
   - Use focus keyword in at least 2-3 H2 headings (Rank Math requirement)
   - Use focus keyword in at least 1-2 H3 headings
   - Minimum 4-5 H2 sections, each with 1-2 H3 subsections
   - Use semantic variations and LSI keywords in remaining headings
   - Example: If focus keyword is 'digital marketing', use it in 2-3 H2s, synonyms in others

4. EXTERNAL LINKS (CRITICAL):
   {$external_links}
   - Format: <a href=\"https://example.com\" target=\"_blank\" rel=\"nofollow noopener\">Descriptive Anchor Text</a>
   - CRITICAL: ALL external links MUST have rel=\"nofollow noopener\" attribute
   - Use descriptive anchor text with relevant keywords

5. INTERNAL LINKS (CRITICAL):
   {$internal_links}
   - Naturally weave these internal links into the content
   - Use descriptive anchor text that hints at the linked page's content
   - Spread them throughout different sections

6. CONTENT STRUCTURE:
   - Target word count: {$word_count} words MINIMUM (Rank Math requires at least 600 words)
   - FIRST PARAGRAPH MUST START with focus keyword '{$item->focus_keyword}' within first 50 words
   - Write in short paragraphs (2-3 sentences max)
   - Use bullet points and numbered lists for readability
   - Include a strong conclusion with a call-to-action
   - CRITICAL: DO NOT include any images, image placeholders, [IMAGE: ...] tags, or <img> tags in the content
   - Images are handled separately as featured images only - do not reference images in the content

7. FORMAT IN HTML:
   - <h2 style=\"font-size: 28px;\"> for main sections (use focus keyword in 2-3 H2s)
   - <h3 style=\"font-size: 24px;\"> for subsections (use focus keyword in 1-2 H3s)
   - <p> for paragraphs
   - <ul><li> for unordered lists
   - <ol><li> for numbered lists
   - <strong> for emphasis on keywords (sparingly)
   - <a href=\"...\"> for links (both internal and outbound)
   - CRITICAL: All H2 must have max font-size: 28px, all H3 must have max font-size: 24px (use inline styles)
   - DO NOT use <img> tags or any image-related HTML in the content

8. KEYPHRASE DENSITY (Rank Math requirement):
   - Focus keyword: ~1% density (approximately 6-10 times for {$word_count} words)
   - Use synonyms, related terms, and semantic variations throughout
   - Natural keyword distribution - don't force the focus keyword
   - Ensure focus keyword appears in: first paragraph, 2-3 H2 headings, 1-2 H3 headings, and body content

9. CONTENT QUALITY:
   - Make it engaging, informative, and optimized for search engines
   - Provide real value to readers
   - Avoid keyword stuffing but ensure proper keyword density (~1%)
   - Write naturally while meeting all SEO requirements

Make it engaging, informative, and optimized for both Yoast SEO and Rank Math SEO while providing real value to readers!";
        
        $response = wp_remote_post('https://api.openai.com/v1/chat/completions', array(
            'headers' => array('Authorization' => 'Bearer ' . $api_key, 'Content-Type' => 'application/json'),
            'body' => json_encode(array(
                'model' => $model,
                'messages' => array(
                    array('role' => 'system', 'content' => 'You are an expert SEO content writer who creates high-ranking blog posts with proper internal linking, outbound links to authoritative sources, and keyword optimization. Always follow SEO best practices for on-page optimization.'),
                    array('role' => 'user', 'content' => $prompt)
                ),
                'max_tokens' => 4000,
                'temperature' => 0.7
            )),
            'timeout' => 120
        ));
        
        if (is_wp_error($response)) return '';
        $body = json_decode(wp_remote_retrieve_body($response), true);
        return $body['choices'][0]['message']['content'] ?? '';
    }
}

