<?php
/**
 * Admin interface for NextBrill Autopost
 */

if (!defined('ABSPATH')) {
    exit;
}

class NBAPF_Admin {
    
    public function __construct() {
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_assets'));
        
        // Form handlers
        add_action('admin_post_nbapf_save_settings', array($this, 'save_settings'));
        add_action('admin_post_nbapf_add_to_queue', array($this, 'add_to_queue'));
        add_action('admin_post_nbapf_delete_queue_item', array($this, 'delete_queue_item'));
    }
    
    /**
     * Add admin menu - Uses 'access_autopost' capability which is granted to ALL logged-in users
     */
    public function add_admin_menu() {
        add_menu_page(
            __('NextBrill Autopost', 'nextbrill-autopost'),
            __('Autopost', 'nextbrill-autopost'),
            'access_autopost', // Custom capability granted to all logged-in users
            'nextbrill-autopost',
            array($this, 'render_dashboard'),
            'dashicons-edit',
            30
        );
        
        add_submenu_page(
            'nextbrill-autopost',
            __('Dashboard', 'nextbrill-autopost'),
            __('Dashboard', 'nextbrill-autopost'),
            'access_autopost',
            'nextbrill-autopost',
            array($this, 'render_dashboard')
        );
        
        add_submenu_page(
            'nextbrill-autopost',
            __('Add to Queue', 'nextbrill-autopost'),
            __('Add to Queue', 'nextbrill-autopost'),
            'access_autopost',
            'nbapf-add-queue',
            array($this, 'render_add_queue')
        );
        
        add_submenu_page(
            'nextbrill-autopost',
            __('Queue', 'nextbrill-autopost'),
            __('Queue', 'nextbrill-autopost'),
            'access_autopost',
            'nbapf-queue',
            array($this, 'render_queue')
        );
        
        add_submenu_page(
            'nextbrill-autopost',
            __('Settings', 'nextbrill-autopost'),
            __('Settings', 'nextbrill-autopost'),
            'access_autopost',
            'nbapf-settings',
            array($this, 'render_settings')
        );
    }
    
    /**
     * Enqueue admin assets
     */
    public function enqueue_admin_assets($hook) {
        if (strpos($hook, 'nextbrill-autopost') === false && strpos($hook, 'nbapf-') === false) {
            return;
        }

        wp_enqueue_style('nbapf-admin-css', NBAPF_PLUGIN_URL . 'admin/css/admin.css', array(), NBAPF_VERSION);
        wp_enqueue_script('nbapf-admin-js', NBAPF_PLUGIN_URL . 'admin/js/admin.js', array('jquery'), NBAPF_VERSION, true);
        wp_localize_script('nbapf-admin-js', 'nbapfData', array(
            'proUrl' => 'https://nextbrill.com/product/nextbrill-autopost/'
        ));
        // Queue page: delete confirmation (no inline onclick)
        if (strpos($hook, 'nbapf-queue') !== false) {
            wp_add_inline_script('nbapf-admin-js', "jQuery(document).on('click', '.nbapf-delete-queue-item', function(e){ if (!confirm(this.getAttribute('data-confirm') || 'Delete this item?')) e.preventDefault(); });");
        }
    }
    
    /**
     * Render functions
     */
    public function render_dashboard() {
        $queue = new NBAPF_Queue();
        $stats = $queue->get_statistics();
    
        
        include NBAPF_PLUGIN_DIR . 'admin/views/dashboard.php';
    }
    
    public function render_add_queue() {
        $queue = new NBAPF_Queue();
        $can_add_post = ! $queue->has_pending_or_processing();
        include NBAPF_PLUGIN_DIR . 'admin/views/add-queue.php';
    }
    
    public function render_queue() {
        $queue = new NBAPF_Queue();
        $items = $queue->get_all_items();
        include NBAPF_PLUGIN_DIR . 'admin/views/queue.php';
    }
    
    public function render_settings() {
        include NBAPF_PLUGIN_DIR . 'admin/views/settings.php';
    }
    
    /**
     * Form handlers
     */
    public function save_settings() {
        check_admin_referer('nbapf-save-settings');
        
        update_option('nbapf_openai_api_key', sanitize_text_field(wp_unslash($_POST['openai_api_key'] ?? '')));
        update_option('nbapf_openai_model', sanitize_text_field(wp_unslash($_POST['openai_model'] ?? 'gpt-4')));
        update_option('nbapf_use_dalle', isset($_POST['use_dalle']) ? true : false);
        
        wp_safe_redirect(add_query_arg('message', 'settings_saved', admin_url('admin.php?page=nbapf-settings')));
        exit;
    }
    
    public function add_to_queue() {
        check_admin_referer('nbapf-add-to-queue');
        
        $queue = new NBAPF_Queue();
        if ($queue->has_pending_or_processing()) {
            wp_safe_redirect(add_query_arg('message', 'complete_current_first', admin_url('admin.php?page=nbapf-add-queue')));
            exit;
        }
        
        $topic_title = sanitize_text_field(wp_unslash($_POST['topic_title'] ?? ''));
        $focus_keyword = sanitize_text_field(wp_unslash($_POST['focus_keyword'] ?? ''));
        if (empty($topic_title) || empty($focus_keyword)) {
            wp_safe_redirect(add_query_arg('message', 'add_failed', admin_url('admin.php?page=nbapf-add-queue')));
            exit;
        }
        
        $data = array(
            'topic_title' => $topic_title,
            'focus_keyword' => $focus_keyword,
            'secondary_keywords' => sanitize_textarea_field(wp_unslash($_POST['secondary_keywords'] ?? '')),
            'internal_links' => sanitize_textarea_field(wp_unslash($_POST['internal_links'] ?? '')),
            'external_links' => sanitize_textarea_field(wp_unslash($_POST['external_links'] ?? '')),
            'include_faq' => isset($_POST['include_faq']) && $_POST['include_faq'] === '1' ? 1 : 0,
            'category' => intval($_POST['category'] ?? 0),
            'tags' => sanitize_text_field(wp_unslash($_POST['tags'] ?? '')),
            'featured_image_keyword' => '',
            'desired_word_count' => intval($_POST['desired_word_count'] ?? 1500)
        );
        
        if ($queue->add_to_queue($data)) {
            wp_safe_redirect(add_query_arg('message', 'added_to_queue', admin_url('admin.php?page=nbapf-add-queue')));
        } else {
            wp_safe_redirect(add_query_arg('message', 'add_failed', admin_url('admin.php?page=nbapf-add-queue')));
        }
        exit;
    }
    
    public function delete_queue_item() {
        check_admin_referer('nbapf-delete-queue-item');
        
        $id = intval($_GET['id'] ?? 0);
        
        $queue = new NBAPF_Queue();
        $queue->delete_item($id);
        
        wp_safe_redirect(add_query_arg('message', 'item_deleted', admin_url('admin.php?page=nbapf-queue')));
        exit;
    }
}

