<?php
/**
 * Plugin Name: Next Exit Chatbot
 * Description: Easily install Next Exit on your WordPress site.
 * Version: 1.0.1
 * Author:            Next Exit, LLC
 * Author URI:        https://www.nextexit.ai
 * License:           GNU General Public License v3.0
 * License URI:       https://www.gnu.org/licenses/gpl-3.0.html

 * @package NextExitChatbot
 **/

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

// Register the block.
add_action(
	'init',
	function () {
		register_block_type(
			'nextexit/widget',
			array(
				'editor_script'   => 'nextexit-widget-block',
				'editor_style'    => 'nextexit-widget-editor-style',
				'render_callback' => 'nextexit_widget_render_callback',
				'attributes'      => array(
					'firmId'      => array(
						'type'    => 'string',
						'default' => '',
					),
					'containerId' => array(
						'type'    => 'string',
						'default' => 'nextexit-widget-' . uniqid(),
					),
				),
			)
		);
	}
);

// Enqueue block editor assets.
add_action(
	'enqueue_block_editor_assets',
	function () {
		wp_enqueue_script(
			'nextexit-widget-block',
			plugin_dir_url( __FILE__ ) . 'block.js',
			array( 'wp-blocks', 'wp-element', 'wp-editor', 'wp-components' ),
			'1.0.0',
			false
		);
		wp_enqueue_style(
			'nextexit-widget-editor-style',
			plugin_dir_url( __FILE__ ) . 'editor-style.css',
			array(),
			'1.0.0'
		);
	}
);

add_action(
	'wp_enqueue_scripts',
	function () {
		wp_enqueue_script(
			'nextexit-widget-script',
			'https://storage.googleapis.com/www.nextexit.ai/nextexit-widget.umd.cjs',
			array(),
			'1.0.0',
			true
		);
	}
);

// Global array to collect widget configurations.
global $nextexit_widgets;
$nextexit_widgets = array();

/**
 * Render callback for the Next Exit Chatbot block.
 *
 * @param array $attributes Block attributes.
 * @return string HTML markup to render the chatbot.
 */
function nextexit_widget_render_callback( $attributes ) {
	global $nextexit_widgets;

	$firm_id      = $attributes['firmId'] ?? '';
	$container_id = $attributes['containerId'] ?? 'nextexit-widget-' . uniqid();

	if ( empty( $firm_id ) ) {
		return '<div class="nextexit-widget-placeholder" style="padding: 20px; border: 2px dashed #ccc; text-align: center; color: #666;">
            <p>⚠️ Next Exit</p>
            <p>Please configure your Firm ID in the block settings.</p>
        </div>';
	}

	$login_url = 'https://front.nextexit.ai/api/login';

	// Collect widget configuration.
	$nextexit_widgets[] = array(
		'container' => '#' . sanitize_text_field( $container_id ), // Ensure container ID is sanitized.
		'firmId'    => sanitize_text_field( $firm_id ), // Ensure firmId is sanitized.
		'loginUrl'  => $login_url, // Hardcoded and trusted, does need escaping.
	);

	ob_start();
	?>
	<div id="<?php echo esc_attr( $container_id ); ?>" class="nextexit-widget-container">
		<!-- The Next Exit chatbot will be rendered here -->
	</div>
	<?php
	return ob_get_clean();
}

// Output initialization script in footer.
add_action(
	'wp_footer',
	function () {
		global $nextexit_widgets;

		if ( ! empty( $nextexit_widgets ) ) {
			wp_add_inline_script(
				'nextexit-widget-script',
				sprintf(
					'window.addEventListener("DOMContentLoaded", function() {
                console.log("doing stuff");
                if (typeof NextExitWidget !== "undefined") {
                    console.log("and more...");
                    %s
                }
            });',
					implode(
						"\n",
						array_map(
							function ( $widget ) {
								return sprintf(
									'NextExitWidget.init(%s);',
									wp_json_encode( $widget ) // Escape late: json_encode() ensures safe JS output of sanitized widget config.
								);
							},
							$nextexit_widgets
						)
					)
				)
			);
		}
	}
);

// Add settings page for global configuration.
add_action(
	'admin_menu',
	function () {
		add_options_page(
			'Next Exit Chatbot',
			'Next Exit Chatbot',
			'manage_options',
			'nextexit-widget',
			'nextexit_widget_settings_page'
		);
	}
);

// Register settings.
add_action(
	'admin_init',
	function () {
		register_setting(
			'nextexit_widget_options',
			'nextexit_widget_default_firm_id',
			array(
				'type'              => 'string',
				'sanitize_callback' => 'sanitize_text_field',
				'default'           => '',
			)
		);
	}
);

/**
 * Render the admin settings page for the Next Exit Chatbot plugin.
 */
function nextexit_widget_settings_page() {
	?>
	<div class="wrap">
		<h1>Next Exit Chatbot Configuration</h1>
		<form method="post" action="options.php">
			<?php settings_fields( 'nextexit_widget_options' ); ?>
			<table class="form-table">
				<tr valign="top">
					<th scope="row">Default Firm ID</th>
					<td>
						<input type="text" name="nextexit_widget_default_firm_id" value="<?php echo esc_attr( get_option( 'nextexit_widget_default_firm_id' ) ); ?>" style="width: 400px;" />
						<p class="description">Default Firm ID to use when creating new widget blocks. Users can still override this in individual blocks.</p>
					</td>
				</tr>
			</table>
			<?php submit_button(); ?>
		</form>
		
		<div style="margin-top: 30px; padding: 20px; background: #f9f9f9; border-left: 4px solid #0073aa;">
			<h3>How to Use the Next Exit Chatbot Block:</h3>
			<ol>
				<li>Edit any page or post using the block editor</li>
				<li>Click the "+" button to add a new block</li>
				<li>Search for "Next Exit Chatbot" and select it</li>
				<li>Enter your Firm ID in the block settings (right sidebar)</li>
				<li>Publish or update your page</li>
			</ol>
			<p><strong>Note:</strong> The chatbot will only appear on the frontend if a Firm ID is configured.</p>
		</div>
	</div>
	<?php
}
