const { registerBlockType } = wp.blocks;
const { createElement: el } = wp.element;
const { InspectorControls, useBlockProps } = wp.blockEditor;
const { PanelBody, TextControl, Notice } = wp.components;
const { __ } = wp.i18n;

registerBlockType('nextexit/widget', {
    title: __('Next Exit'),
    description: __('Install Next Exit on your page.'),
    category: 'widgets',
    icon: 'admin-generic',
    keywords: ['next', 'exit', 'widget', 'ai', 'chat'],
    supports: {
        html: false,
        align: ['wide', 'full'],
    },
    attributes: {
        firmId: {
            type: 'string',
            default: '',
        },
        containerId: {
            type: 'string',
            default: 'nextexit-widget-' + Math.random().toString(36).substr(2, 9),
        },
    },
    example: {},
    
    edit: function(props) {
        const { attributes, setAttributes, isSelected } = props;
        const { firmId, containerId } = attributes;
        const blockProps = useBlockProps();

        // Inserter preview: styled card that mimics the real block
        if (!isSelected) {
            return el('div', {
                style: {
                    width: '100%',
                    height: '100%',
                    minHeight: '144px',
                    display: 'flex',
                    flexDirection: 'column',
                    justifyContent: 'center',
                    alignItems: 'center',
                    background: '#f9f9f9',
                    border: '2px dashed #ccc',
                    borderRadius: '8px',
                    boxSizing: 'border-box',
                    padding: '16px',
                }
            }, [
                el('div', {
                    style: {
                        fontSize: '32px',
                        marginBottom: '8px',
                        color: '#0073aa',
                    }
                }, '🤖'),
                el('div', {
                    style: {
                        fontWeight: 'bold',
                        fontSize: '16px',
                        color: '#222',
                        marginBottom: '4px',
                        textAlign: 'center',
                    }
                }, __('Next Exit Chatbot')),
                el('div', {
                    style: {
                        fontSize: '12px',
                        color: '#666',
                        textAlign: 'center',
                        lineHeight: '1.4',
                    }
                }, __('AI-powered chat for your website')),
            ]);
        }
        
        // Get default firm ID from WordPress settings
        const defaultFirmId = wp.data.select('core').getEntityRecord('root', 'site')?.nextexit_widget_default_firm_id || '';
        
        const handleFirmIdChange = (value) => {
            setAttributes({ firmId: value });
        };
        
        return el('div', blockProps, [
            el(InspectorControls, { key: 'inspector' }, [
                el(PanelBody, { 
                    title: __('Next Exit Settings'),
                    initialOpen: true 
                }, [
                    el(TextControl, {
                        label: __('Firm ID'),
                        value: firmId || defaultFirmId,
                        onChange: handleFirmIdChange,
                        help: __('Enter your Firm ID from the Next Exit portal.'),
                        placeholder: 'e.g., cc72789f-c016-43b2-b600-498724b476dd',
                    }),
                    el(Notice, {
                        status: 'info',
                        isDismissible: false,
                    }, __('The Next Exit chatbot will appear once your Firm ID is configured.')),
                ]),
            ]),
            
            // Editor preview
            el('div', {
                key: 'preview',
                style: {
                    padding: '20px',
                    border: '2px dashed #ccc',
                    borderRadius: '4px',
                    textAlign: 'center',
                    backgroundColor: '#f9f9f9',
                    minHeight: '100px',
                    display: 'flex',
                    flexDirection: 'column',
                    justifyContent: 'center',
                    alignItems: 'center',
                }
            }, [
                el('div', {
                    style: {
                        fontSize: '24px',
                        marginBottom: '10px',
                        color: '#0073aa',
                    }
                }, '🤖'),
                el('h3', {
                    style: {
                        margin: '0 0 10px 0',
                        color: '#333',
                    }
                }, __('Next Exit Chatbot')),
                el('p', {
                    style: {
                        margin: '0',
                        color: '#666',
                        fontSize: '14px',
                    }
                }, firmId ? 
                    __('The chatbot will be displayed here with Firm ID: ') + firmId :
                    __('Configure Firm ID in block settings to display the chatbot.')
                ),
                el('div', {
                    style: {
                        marginTop: '15px',
                        padding: '8px 12px',
                        backgroundColor: '#fff',
                        border: '1px solid #ddd',
                        borderRadius: '3px',
                        fontSize: '12px',
                        color: '#666',
                        fontFamily: 'monospace',
                    }
                }, 'Container ID: ' + containerId),
            ]),
        ]);
    },
    
    save: function() {
        // This block uses a render callback, so we don't save any content
        return null;
    },
});
