<?php
/**
 * Plugin Name: NExS App Embed
 * Plugin URI: http://www.nexs.com/wordpress-plugin
 * Description: Embed spreadsheet-driven calculators, smart forms and more in your WordPress document with ease.
 * Version: 1.0.0
 * Author: nexs
 * Author URI: http://www.nexs.com
 * License: GPLv2
 */

//
// Shortcodes:
//	[nexs_app id="appId"]
//	[nexs_output cell="cellRef"]
//

// Make sure we don't expose any info if called directly
if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

if ( !function_exists( 'add_action' ) ) {
	echo 'Hi there!  I\'m just a plugin, not much I can do when called directly.';
	exit;
}

define( 'NEXS_VERSION', '0.1' );
define( 'NEXS_MINIMUM_WP_VERSION', '4.0' );
define( 'NEXS__PLUGIN_DIR', plugin_dir_path( __FILE__ ) );

//
// nexs_settings.php creates the options panel
//
require_once("nexs_settings.php");

//
// Activation Hook - set default options
//
function nexs_activate() {
	$options = array(
		'host' => 'platform.nexs.com',
		'static' => 'static.nexs.com',
		'https' => 1,
	);
	add_option( 'nexs_settings', $options );
}
register_activation_hook( __FILE__, 'nexs_activate' );

//
// Deactivation Hook - delete options
//
function nexs_deactivate() {
	delete_option( 'nexs_settings' );
}
register_deactivation_hook( __FILE__, 'nexs_deactivate' );


/*
 * [nexs_app]
 *
 * What it does:
 *
 *   [nexs_app id="appId"] inserts a responsive <iframe> containing the specified NExS app.
 *
 * Required attributes:
 *
 *   id - NExS app ID
 *
 * Optional attributes:
 *
 *   host - the app's host server (default: platform.nexs.com)
 *   static - the static content server (default: nexs-launch-static.s3.amazonaws.com)
 */

function nexs_app_func( $atts ) {

	$options = get_option( 'nexs_settings' );

	$a = shortcode_atts( array(
		'id' => '',
		'host' => $options['host'],
		'static' => $options['static'],
	), $atts );

	if ($a['id'] == '') {
		return "<strong>Error! A valid NExS app id is required.</strong>";
	}

	return "<script class='nexs' " .
			"src='//{$a['static']}/js/nexs_embed.js' " .
			"data-nexs-app-url='//{$a['host']}/app/{$a['id']}'" .
			"></script>";
}

add_shortcode( 'nexs_app', 'nexs_app_func' );


/*
 * [nexs_output]
 *
 * What it does:
 *
 *   [nexs_output cell="cellRef"] inserts a <span> containing the value of the referenced cell.
 *
 * Required attributes:
 *
 *   cell - a cell reference; e.g., "B17" (case insensitive)
 *
 * Optional attributes:
 *
 *   app_index - the index of the app being referenced (default 0)
 *       NOTE: only relevant if multiple apps are embedded in the page
 *   view_index - the index of the view being referenced (default 0)
 *       NOTE: only relevant if the referenced app has multiple views
 *   raw - whether or not to display numeric values as raw numbers (default false)
 *       Example: if cell B6 contains $6,497.95, the "raw" value would be 6497.95
 */

function nexs_output_func( $atts ) {

	$a = shortcode_atts( array(
		'cell' => '',
		'app_index' => 0,
		'view_index' => 0,
		'raw' => false,
	), $atts );

	$type = $a['raw'] ? 'raw' : 'formatted';

	return "<span class='nexs__output_{$type}' " .
			"data-nexs-cell-addr=".
			"'{$a['app_index']}:{$a['view_index']}:{$a['cell']}'" .
			"></span>";
}

add_shortcode( 'nexs_output', 'nexs_output_func' );


/*
 * [nexs_input]
 *
 * What it does:
 *
 *   [nexs_input cell="cellRef"] inserts an <input> which connects to the referenced cell.
 *
 * Required attributes:
 *
 *   cell - a cell reference; e.g., "B17" (case insensitive)
 *
 * Optional attributes:
 *
 *   type - the input type (default "text")
 *       Values for "type":
 *          "text" - <input type="text">
 *          "checkbox" - <input type="checkbox">
 *          "radio" - <input type="radio">
 *          "select" - <select>
 *   values - a delimited list of values for "radio" and "select" input types
 *       default: "Option 1, Option 2"
 *   delimiter - the delimiter character for the "values" list (default ",")
 *   app_index - the index of the app being referenced (default 0)
 *       NOTE: only relevant if multiple apps are embedded in the page
 *   view_index - the index of the view being referenced (default 0)
 *       NOTE: only relevant if the referenced app has multiple views
 */

function nexs_input_func( $atts ) {

	static $rg_counter = 0;

	$a = shortcode_atts( array(
		'cell' => '',
		'type' => 'text',
		'options' => "Option 1,Option 2",
		'delimiter' => ',',
		'app_index' => 0,
		'view_index' => 0,
		'inline_radio' => 1,
		'check_label' => '',
	), $atts );

	switch ($a['type']) {
		case 'text':
			$html = "<input class='nexs__input'" .
				" data-nexs-cell-addr=" .
				"'{$a['app_index']}:{$a['view_index']}:{$a['cell']}'" .
				"/>";
			break;
		case 'checkbox':
			$html = "<label><input type='checkbox' class='nexs__input'" .
				" data-nexs-cell-addr=" .
				"'{$a['app_index']}:{$a['view_index']}:{$a['cell']}'" .
				" onchange='this.value = this.checked ? 1:0'" .
				"/> {$a['check_label']} </label>";
			break;
		case 'radio':
			$options = explode($a['delimiter'], $a['options']);
			$rg_counter++;	// radio group counter for "name" attribute
			$html = "";
			for ($i = 0; $i < count($options); $i++) {
				$option = trim($options[$i]);
				$html .= "<label><input type='radio' class='nexs__input'" .
					" name='radio{$rg_counter}'" .
					" data-nexs-cell-addr=" .
					"'{$a['app_index']}:{$a['view_index']}:{$a['cell']}'" .
					" value='{$option}'/> {$option} </label>";
				if ($a['inline_radio'] == 0 || $a['inline_radio'] == 'false') {
					$html .= "<br/>";
				}
			}
			break;
		case 'select':
			$options = explode($a['delimiter'], $a['options']);
			$html = "<select class='nexs__input'" .
				" data-nexs-cell-addr=" .
				"'{$a['app_index']}:{$a['view_index']}:{$a['cell']}'>";
			for ($i = 0; $i < count($options); $i++) {
				$option = trim($options[$i]);
				$html .= "<option>{$option}</option>";
			}
			$html .= "</select>";
			break;
	}

	return $html;
}

add_shortcode( 'nexs_input', 'nexs_input_func' );

/*****************************************************************************/

/*
 * [nexs_viewselect]
 *
 * What it does:
 *
 *   [nexs_viewselect] inserts an <input> which sets the visible view
 *
 * Required attributes:
 *
 *   none
 *
 * Optional attributes:
 *
 *   app_index - the index of the app being referenced (default 0)
 *       NOTE: only relevant if multiple apps are embedded in the page
 */

function nexs_viewselect_func( $atts ) {

	static $rg_counter = 0;

	$a = shortcode_atts( array(
		'app_index' => 0,
	), $atts );

	$html = "<select class='nexs__viewselect' data-nexs-app='{$a['app_index']}'></select>";

	return $html;
}

add_shortcode( 'nexs_viewselect', 'nexs_viewselect_func' );
?>
