<?php
/**
 * Uninstall script for NexlifyDesk
 * 
 * It handles cleanup based on user settings.
 */

// Exit if accessed directly or not in uninstall context
if (!defined('WP_UNINSTALL_PLUGIN')) {
    exit;
}

// Prevent execution if not in proper uninstall context
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Main uninstall cleanup function
 * Removes options, transients, roles, capabilities, custom tables, and uploaded files based on user settings.
 */
function nexlifydesk_execute_uninstall() {
    global $wpdb;

    $settings = get_option('nexlifydesk_settings', array());
    $keep_data = isset($settings['keep_data_on_uninstall']) ? (bool)$settings['keep_data_on_uninstall'] : true;

    wp_clear_scheduled_hook('nexlifydesk_sla_check');
    wp_clear_scheduled_hook('nexlifydesk_auto_close_tickets');
    wp_clear_scheduled_hook('nexlifydesk_check_orphaned_tickets');
    delete_transient('nexlifydesk_sla_tickets');
    delete_transient('nexlifydesk_resolved_tickets');
    delete_transient('nexlifydesk_google_oauth_state');
    
    wp_cache_flush();

    if ($keep_data) {
        delete_option('nexlifydesk_db_version');
        return;
    }

    $options_to_remove = array(
        'nexlifydesk_settings',
        'nexlifydesk_email_templates',
        'nexlifydesk_last_ticket_number',
        'nexlifydesk_db_version',
        'nexlifydesk_agent_positions',
        'nexlifydesk_rate_limits',
        'nexlifydesk_imap_settings',
        'nexlifydesk_aws_processed_emails',
    );
    
    foreach ($options_to_remove as $option) {
        delete_option($option);
        if (is_multisite()) {
            delete_network_option(null, $option);
        }
    }

    $table_names = array(
        $wpdb->prefix . 'nexlifydesk_tickets',
        $wpdb->prefix . 'nexlifydesk_replies',
        $wpdb->prefix . 'nexlifydesk_attachments',
        $wpdb->prefix . 'nexlifydesk_categories'
    );

    foreach ($table_names as $table_name) {
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.DirectDatabaseQuery.SchemaChange -- Plugin uninstall requires direct table deletion
        $wpdb->query("DROP TABLE IF EXISTS " . esc_sql($table_name));
    }

    remove_role('nexlifydesk_agent');
    remove_role('nexlifydesk_supervisor');

    $admin_role = get_role('administrator');
    if ($admin_role) {
        $capabilities = array(
            'nexlifydesk_manage_tickets',
            'nexlifydesk_view_all_tickets',
            'nexlifydesk_assign_tickets',
            'nexlifydesk_manage_categories',
            'nexlifydesk_view_reports'
        );
        
        foreach ($capabilities as $cap) {
            $admin_role->remove_cap($cap);
        }
    }

    $upload_dir = wp_upload_dir();
    $plugin_upload_dir = trailingslashit($upload_dir['basedir']) . 'nexlifydesk/';
    
    if (is_dir($plugin_upload_dir)) {
        nexlifydesk_recursive_delete_directory($plugin_upload_dir);
    }

    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct deletion is required for uninstall cleanup
    $wpdb->query($wpdb->prepare(
        "DELETE FROM {$wpdb->usermeta} WHERE meta_key LIKE %s",
        'nexlifydesk_%'
    ));
    wp_cache_flush();

    $cache_keys_patterns = array(
        'nexlifydesk_ticket_',
        'nexlifydesk_user_tickets_',
        'nexlifydesk_assigned_tickets_',
        'nexlifydesk_ticket_replies_',
        'nexlifydesk_sla_tickets_',
        'nexlifydesk_resolved_tickets_',
        'nexlifydesk_email_',
    );
    
    global $wp_object_cache;
    if (isset($wp_object_cache->cache)) {
        foreach ($wp_object_cache->cache as $group => $group_cache) {
            if (is_array($group_cache)) {
                foreach (array_keys($group_cache) as $key) {
                    foreach ($cache_keys_patterns as $pattern) {
                        if (strpos($key, $pattern) === 0) {
                            wp_cache_delete($key, $group);
                        }
                    }
                }
            }
        }
    }
    
    // Flush cache groups
    wp_cache_flush_group('nexlifydesk_tickets_grid');

    // Clear any remaining caches
    wp_cache_flush();
}

/**
 * Recursively delete a directory and its contents
 * 
 * @param string $dir Directory path to delete
 * @return bool Success status
 */
function nexlifydesk_recursive_delete_directory($dir) {
    if (!is_dir($dir)) {
        return false;
    }

    $files = array_diff(scandir($dir), array('.', '..'));
    
    foreach ($files as $file) {
        $path = $dir . DIRECTORY_SEPARATOR . $file;
        
        if (is_dir($path)) {
            nexlifydesk_recursive_delete_directory($path);
        } else {
            wp_delete_file($path);
        }
    }
    
    global $wp_filesystem;
    if (empty($wp_filesystem)) {
        require_once(ABSPATH . '/wp-admin/includes/file.php');
        WP_Filesystem();
    }
    return $wp_filesystem->rmdir($dir, true);
}

/** Execute the uninstall process */
nexlifydesk_execute_uninstall();
