<?php
if (!defined('ABSPATH')) {
    exit;
}
/**
 * Template for displaying a single support ticket.
 *
 * This template is used to display the details of a specific support ticket,
 * including the conversation history, ticket details, and reply form.
 *
 * @package NexlifyDesk
 */
if (isset($_SERVER['REQUEST_METHOD']) && $_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'nexlifydesk_ticket_action')) {
        wp_die(esc_html__('Security check failed', 'nexlifydesk'));
    }

    if (isset($_POST['ticket_id'])) {
        $_GET['ticket_id'] = sanitize_text_field(wp_unslash($_POST['ticket_id']));
    }
}

if (!is_user_logged_in()) {
    ?>
    <div class="nexlifydesk-frontend-container">
        <p><?php esc_html_e('Please log in to view your support tickets.', 'nexlifydesk'); ?></p>
        <p>
            <a href="<?php echo esc_url(wp_login_url(get_permalink())); ?>" class="btn btn-primary"><?php esc_html_e('Log In', 'nexlifydesk'); ?></a>
            <?php if (get_option('users_can_register')) : ?>
                <a href="<?php echo esc_url(wp_registration_url()); ?>" class="btn btn-secondary"><?php esc_html_e('Register', 'nexlifydesk'); ?></a>
            <?php endif; ?>
        </p>
    </div>
    <?php
    return;
}

$ticket_id_param = isset($_GET['ticket_id']) ? sanitize_text_field(wp_unslash($_GET['ticket_id'])) : '';
$ticket = NexlifyDesk_Tickets::get_ticket_by_ticket_id($ticket_id_param);

if (!$ticket && is_numeric($ticket_id_param)) {
    $ticket = NexlifyDesk_Tickets::get_ticket(absint($ticket_id_param));
}

$current_user = wp_get_current_user();
$is_ticket_owner = ($ticket && (int)$ticket->user_id === (int)$current_user->ID);

if (!$is_ticket_owner && $ticket && (int)$ticket->user_id === 0) {
    $customer_email = get_post_meta($ticket->id, 'customer_email', true);
    $is_ticket_owner = ($customer_email && $customer_email === $current_user->user_email);
}

$can_view_ticket = $is_ticket_owner;

if (!$ticket || !$can_view_ticket) {
    ?>
    <div class="nexlifydesk-frontend-container">
        <p><?php esc_html_e('You are not authorized to view this ticket or the ticket does not exist.', 'nexlifydesk'); ?></p>
        <p class="back-link"><a href="<?php echo esc_url(get_permalink(get_option('nexlifydesk_settings')['ticket_page_id'])); ?>">
                <?php esc_html_e('← Back to All Tickets', 'nexlifydesk'); ?>
            </a></p>
    </div>
    <?php
    return;
}


$user = get_userdata($ticket->user_id);
$category = NexlifyDesk_Tickets::get_category($ticket->category_id);
$assigned_agent = $ticket->assigned_to ? get_userdata($ticket->assigned_to) : null;
$replies = NexlifyDesk_Tickets::get_replies($ticket->id);
$initial_attachments = NexlifyDesk_Tickets::get_attachments($ticket->id);

$customer_details = nexlifydesk_extract_customer_details($ticket->message);
$customer_name = $user ? $user->display_name : ($customer_details['name'] ?: 'Guest');
$customer_email = $user ? $user->user_email : ($customer_details['email'] ?: 'N/A');
$clean_initial_message = $customer_details['message'] ?: $ticket->message;
$user_initials = $user ? strtoupper(substr($user->first_name, 0, 1) . substr($user->last_name, 0, 1)) : 'G';
if (empty(trim($user_initials))) {
    $user_initials = $user ? strtoupper(substr($user->display_name, 0, 2)) : strtoupper(substr($customer_name, 0, 2));
}

?>
<div class="nexlifydesk-frontend-container">
    <?php
    if (isset($_GET['ticket_submitted'])) {
        echo '<div class="nexlifydesk-message success">';
        echo '<p>' . esc_html__('Your ticket has been submitted successfully!', 'nexlifydesk') . '</p>';
        echo '</div>';
    } elseif (isset($_GET['duplicate_detected'])) {
        echo '<div class="nexlifydesk-message info">';
        echo '<p>' . esc_html__('We found a similar ticket from you. Your message has been added to this existing ticket below.', 'nexlifydesk') . '</p>';
        echo '</div>';
    }
    ?>
    
    <div class="header">
        <div class="ticket-info">
            <div>
                <h1 class="ticket-title"><?php echo esc_html($ticket->subject); ?></h1>
                <div class="ticket-meta">
                    <span class="ticket-id">Ticket #<?php echo esc_html($ticket->ticket_id); ?></span>
                    <span class="status <?php echo esc_attr($ticket->status); ?>"><?php echo esc_html(ucfirst($ticket->status)); ?></span>
                    <span class="priority <?php echo esc_attr($ticket->priority); ?>"><?php echo esc_html(ucfirst($ticket->priority)); ?> Priority</span>
                </div>
            </div>
            <div class="user-avatar" title="<?php echo esc_attr($customer_name); ?>"><?php echo esc_html($user_initials); ?></div>
        </div>
    </div>

    <div class="main-content">
        <div class="ticket-details">
            <div class="section-title"><?php esc_html_e('Description', 'nexlifydesk'); ?></div>
            <div class="description">
                <?php echo nl2br(esc_html($clean_initial_message)); ?>
            </div>

            <?php if (!empty($initial_attachments)) : ?>
                <div class="attachments">
                    <?php foreach ($initial_attachments as $attachment) : ?>
                        <div class="attachment">
                            <span class="attachment-icon">📄</span>
                            <a href="<?php echo esc_url($attachment->file_path); ?>" target="_blank" rel="noopener"><?php echo esc_html($attachment->file_name); ?></a>
                        </div>
                    <?php endforeach; ?>
                </div>
            <?php endif; ?>

            <div class="section-title"><?php esc_html_e('Conversation', 'nexlifydesk'); ?></div>
            <div class="conversation" id="nexlifydesk-replies-list">
                <?php
                if (!empty($replies)) {
                    foreach ($replies as $reply) {
                        $reply_user = get_userdata($reply->user_id);
                        $reply->attachments = NexlifyDesk_Tickets::get_attachments($ticket->id, $reply->id);
                        include NEXLIFYDESK_PLUGIN_DIR . 'templates/frontend/partials/single-reply.php';
                    }
                }
                ?>
            </div>

            <?php if ($ticket->status !== 'closed') : ?>
                <div class="nexlifydesk-reply-form">
                    <form id="nexlifydesk-reply-form" method="post" enctype="multipart/form-data">
                        <div class="nexlifydesk-form-messages"></div>
                        <div class="form-group">
                            <label for="reply-message"><?php esc_html_e('Your Reply', 'nexlifydesk'); ?></label>
                            <textarea id="reply-message" name="message" rows="5" required></textarea>
                        </div>
                        <div class="form-group">
                            <label for="reply-attachments"><?php esc_html_e('Attachments', 'nexlifydesk'); ?></label>
                            <input type="file" name="attachments[]" id="reply-attachments" multiple>
                            <span class="file-info" style="margin-left:10px;"></span>
                            <div class="nexlifydesk-file-error" style="color:red;margin-top:5px;font-size:16px;line-height:1.4;"></div>
                            <p class="description">
                                <?php
                                $settings = get_option('nexlifydesk_settings', array());
                                $max_size = isset($settings['max_file_size']) ? (int)$settings['max_file_size'] : 2;
                                $allowed_types = isset($settings['allowed_file_types']) ? $settings['allowed_file_types'] : 'jpg,jpeg,png,pdf';
                                /* translators: 1: Maximum file size in MB, 2: Allowed file types (comma separated) */
                                printf(esc_html__('Max file size: %1$sMB. Allowed types: %2$s', 'nexlifydesk'), esc_html($max_size), esc_html($allowed_types));
                                ?>
                            </p>
                        </div>
                        <input type="hidden" name="action" value="nexlifydesk_add_reply">
                        <input type="hidden" name="ticket_id" value="<?php echo esc_attr($ticket->id); ?>">
                        <?php wp_nonce_field('nexlifydesk-ajax-nonce', 'nonce'); ?>
                        <div class="actions">
                            <button type="submit" class="btn btn-primary"><?php esc_html_e('Reply to Ticket', 'nexlifydesk'); ?></button>
                        </div>
                    </form>
                </div>
            <?php else : ?>
                <div class="nexlifydesk-closed-message">
                    <p><?php esc_html_e('This ticket is closed. If you need further assistance, please open a new ticket.', 'nexlifydesk'); ?></p>
                    <?php
                    $ticket_form_page_id = get_option('nexlifydesk_settings')['ticket_form_page_id'] ?? 0;
                    if ($ticket_form_page_id) : ?>
                        <a href="<?php echo esc_url(get_permalink($ticket_form_page_id)); ?>" class="btn btn-primary">
                            <?php esc_html_e('Create New Ticket', 'nexlifydesk'); ?>
                        </a>
                    <?php endif; ?>
                </div>
            <?php endif; ?>
        </div>

        <div class="sidebar">
            <div class="section-title"><?php esc_html_e('Ticket Details', 'nexlifydesk'); ?></div>
            <div class="sidebar-item">
                <div class="sidebar-label"><?php esc_html_e('Created', 'nexlifydesk'); ?></div>
                <div class="sidebar-value"><?php echo esc_html(date_i18n(get_option('date_format') . ' \a\t ' . get_option('time_format'), strtotime($ticket->created_at))); ?></div>
            </div>
            <div class="sidebar-item">
                <div class="sidebar-label"><?php esc_html_e('Last Updated', 'nexlifydesk'); ?></div>
                <div class="sidebar-value"><?php echo esc_html(human_time_diff(strtotime($ticket->updated_at), current_time('timestamp'))) . ' ' . esc_html__('ago', 'nexlifydesk'); ?></div>
            </div>
            <div class="sidebar-item">
                <div class="sidebar-label"><?php esc_html_e('Customer', 'nexlifydesk'); ?></div>
                <div class="sidebar-value"><?php echo esc_html($user ? $user->display_name : 'Guest'); ?></div>
            </div>
            <div class="sidebar-item">
                <div class="sidebar-label"><?php esc_html_e('Email', 'nexlifydesk'); ?></div>
                <div class="sidebar-value"><?php echo esc_html($user ? $user->user_email : 'N/A'); ?></div>
            </div>
            <div class="sidebar-item">
                <div class="sidebar-label"><?php esc_html_e('Category', 'nexlifydesk'); ?></div>
                <div class="sidebar-value"><?php echo esc_html($category ? $category->name : 'N/A'); ?></div>
            </div>
            <div class="sidebar-item">
                <div class="sidebar-label"><?php esc_html_e('Assigned Agent', 'nexlifydesk'); ?></div>
                <div class="sidebar-value"><?php echo esc_html($assigned_agent ? $assigned_agent->display_name : 'Unassigned'); ?></div>
            </div>
            <div class="actions">
                 <a href="<?php echo esc_url(get_permalink(get_option('nexlifydesk_settings')['ticket_page_id'])); ?>" class="btn btn-secondary"><?php esc_html_e('← All Tickets', 'nexlifydesk'); ?></a>
            </div>
        </div>
    </div>
</div>