<?php
/**
 * Plugin Name: NexlifyDesk
 * Description: A modern, user-friendly support ticket system for WordPress with ticket submission, threaded replies, file attachments, agent assignment, and customizable.
 * Version: 1.0.5
 * Author URI: https://nexlifylabs.com
 * Supported Versions: 6.2+
 * Tested up to: 6.2 < 6.8.
 * Author: NexlifyLabs
 * License: GPL-2.0+
 * Text Domain: nexlifydesk
 * Domain Path: /languages
 */

if (!defined('ABSPATH')) {
    exit;
}

if ( ! function_exists( 'nexlifydesk' ) ) {
    function nexlifydesk() {
        global $nexlifydesk;

        if ( ! isset( $nexlifydesk ) ) {
            require_once dirname( __FILE__ ) . '/vendor/freemius/start.php';
            $nexlifydesk = fs_dynamic_init( array(
                'id'                  => '19551',
                'slug'                => 'nexlifydesk',
                'type'                => 'plugin',
                'public_key'          => 'pk_cbbd298a238d22db2fed1cc83307e',
                'is_premium'          => false,
                'has_addons'          => false,
                'has_paid_plans'      => false,
                'menu'                => array(
                    'slug'           => 'nexlifydesk_tickets',
                    'account'        => false,
                    'contact'        => false,
                ),
            ) );
        }

        return $nexlifydesk;
    }

    nexlifydesk();
    
    do_action( 'nexlifydesk_loaded' );
    
    nexlifydesk()->add_action('after_uninstall', 'nexlifydesk_freemius_uninstall_cleanup');
}

define('NEXLIFYDESK_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('NEXLIFYDESK_PLUGIN_URL', plugin_dir_url(__FILE__));
define('NEXLIFYDESK_VERSION', '1.0.5');
define('NEXLIFYDESK_TABLE_PREFIX', 'nexlifydesk_');
define('NEXLIFYDESK_CAP_VIEW_ALL_TICKETS', 'nexlifydesk_view_all_tickets');
define('NEXLIFYDESK_CAP_ASSIGN_TICKETS', 'nexlifydesk_assign_tickets');
define('NEXLIFYDESK_CAP_MANAGE_CATEGORIES', 'nexlifydesk_manage_categories');
define('NEXLIFYDESK_CAP_VIEW_REPORTS', 'nexlifydesk_view_reports');

require_once NEXLIFYDESK_PLUGIN_DIR . 'includes/class-nexlifydesk-database.php';
require_once NEXLIFYDESK_PLUGIN_DIR . 'includes/class-nexlifydesk-tickets.php';
require_once NEXLIFYDESK_PLUGIN_DIR . 'includes/class-nexlifydesk-users.php';
require_once NEXLIFYDESK_PLUGIN_DIR . 'includes/class-nexlifydesk-ajax.php';
require_once NEXLIFYDESK_PLUGIN_DIR . 'includes/class-nexlifydesk-shortcodes.php';
require_once NEXLIFYDESK_PLUGIN_DIR . 'includes/class-nexlifydesk-admin.php';
require_once NEXLIFYDESK_PLUGIN_DIR . 'includes/nexlifydesk-functions.php';
require_once NEXLIFYDESK_PLUGIN_DIR . 'includes/class-nexlifydesk-reports.php';
require_once NEXLIFYDESK_PLUGIN_DIR . 'includes/class-support.php';
require_once NEXLIFYDESK_PLUGIN_DIR . 'includes/class-nexlifydesk-rate-limiter.php';
require_once NEXLIFYDESK_PLUGIN_DIR . 'email-source/nexlifydesk-email-pipe.php';
require_once NEXLIFYDESK_PLUGIN_DIR . 'includes/helpers.php';

// Initialize the plugin
function nexlifydesk_init() {
    
    NexlifyDesk_Database::init();
    
    // Run database upgrades if needed
    NexlifyDesk_Database::upgrade_database();
    
    NexlifyDesk_Database::check_and_run_migrations();
    NexlifyDesk_Tickets::init();
    NexlifyDesk_Users::init();
    NexlifyDesk_Ajax::init();
    NexlifyDesk_Shortcodes::init();
    
    if (is_admin()) {
        NexlifyDesk_Admin::init();
    }
}
add_action('plugins_loaded', 'nexlifydesk_init');
add_filter('plugin_action_links_' . plugin_basename(__FILE__), 'nexlifydesk_plugin_action_links');
add_filter('plugin_row_meta', 'nexlifydesk_plugin_row_meta', 10, 2);

function nexlifydesk_plugin_action_links($links) {
    $donate_link = '<a href="https://paypal.me/devteejay" target="_blank" style="color: #e74c3c; font-weight: bold;">❤️ ' . __('Donate', 'nexlifydesk') . '</a>';
    array_unshift($links, $donate_link);
    return $links;
}

function nexlifydesk_plugin_row_meta($links, $file) {
    if (plugin_basename(__FILE__) === $file) {
        $row_meta = array(
            'support' => '<a href="' . esc_url('https://nexlifylabs.com/support') . '" target="_blank">' . __('Support', 'nexlifydesk') . '</a>',
            'docs' => '<a href="' . esc_url('https://nexlifylabs.com/nexlifydesk-documentation/') . '" target="_blank">' . __('Documentation', 'nexlifydesk') . '</a>',
            'donate' => '<a href="https://paypal.me/devteejay" target="_blank" style="color: #e74c3c; font-weight: bold;">❤️ ' . __('Support Development', 'nexlifydesk') . '</a>',
        );
        return array_merge($links, $row_meta);
    }
    return $links;
}

add_action('admin_notices', 'nexlifydesk_show_template_update_notice');
add_action('wp_ajax_nexlifydesk_update_email_templates', 'nexlifydesk_ajax_update_email_templates');
add_action('wp_ajax_nexlifydesk_dismiss_template_notice', 'nexlifydesk_ajax_dismiss_template_notice');

function nexlifydesk_show_template_update_notice() {
    if (!current_user_can('manage_options')) {
        return;
    }
    
    if (get_option('nexlifydesk_template_notice_dismissed', false)) {
        return;
    }
    
    $screen = get_current_screen();
    if (!$screen || strpos($screen->id, 'nexlifydesk') === false) {
        return;
    }
    
    $existing_templates = get_option('nexlifydesk_email_templates', array());
    $needs_update = false;
    
    if (!empty($existing_templates)) {
        foreach (['new_ticket', 'new_reply', 'status_changed', 'sla_breach'] as $template_type) {
            if (isset($existing_templates[$template_type]) && 
                !empty($existing_templates[$template_type]) &&
                strpos($existing_templates[$template_type], '{user_name}') !== false) {
                $needs_update = true;
                break;
            }
        }
    }
    
    if ($needs_update) {
        ?>
        <div class="notice notice-warning is-dismissible" id="nexlifydesk-template-update-notice">
            <p>
                <strong><?php esc_html_e('NexlifyDesk Email Templates Update Available', 'nexlifydesk'); ?></strong><br>
                <?php esc_html_e('Your email templates are using the old format. Click below to update them to use the new template files with better styling and functionality.', 'nexlifydesk'); ?>
            </p>
            <p>
                <button type="button" class="button button-primary" onclick="nexlifydesk_update_templates()">
                    <?php esc_html_e('Update Email Templates', 'nexlifydesk'); ?>
                </button>
                <button type="button" class="button" onclick="nexlifydesk_dismiss_notice()">
                    <?php esc_html_e('Dismiss (Keep Current Templates)', 'nexlifydesk'); ?>
                </button>
            </p>
        </div>
        <?php
    }
}

function nexlifydesk_ajax_update_email_templates() {
    // Verify nonce and permissions
    if (!current_user_can('manage_options') || 
        !isset($_POST['nonce']) || 
        !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'nexlifydesk_update_templates')) {
        wp_die('Unauthorized');
    }
    
    $templates = array(
        'new_ticket' => '',
        'new_reply' => '',
        'status_changed' => '',
        'sla_breach' => '',
        'email_auto_response' => get_option('nexlifydesk_email_templates', array())['email_auto_response'] ?? '',
    );
    
    update_option('nexlifydesk_email_templates', $templates);
    
    wp_send_json_success(array('message' => __('Email templates updated successfully!', 'nexlifydesk')));
}

function nexlifydesk_ajax_dismiss_template_notice() {
    // Verify nonce and permissions
    if (!current_user_can('manage_options') || 
        !isset($_POST['nonce']) || 
        !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'nexlifydesk_dismiss_notice')) {
        wp_die('Unauthorized');
    }
    
    update_option('nexlifydesk_template_notice_dismissed', true);
    
    wp_send_json_success(array('message' => __('Notice dismissed successfully!', 'nexlifydesk')));
}

// Activation function
function nexlifydesk_activate() {
    // Activate database tables
    NexlifyDesk_Database::activate();

    // Activate user roles and capabilities
    NexlifyDesk_Users::activate();

    $default_settings = get_option('nexlifydesk_settings', array());
    if (!isset($default_settings['keep_data_on_uninstall'])) {
        $default_settings['keep_data_on_uninstall'] = 1;
        update_option('nexlifydesk_settings', $default_settings);
    }

    // Load email templates from template files on activation
    nexlifydesk_load_default_email_templates();
}

/**
 * Load email templates from template files and populate the admin interface
 */
function nexlifydesk_load_default_email_templates() {
    $existing_templates = get_option('nexlifydesk_email_templates', array());
    
    $needs_update = empty($existing_templates) || 
                   (isset($existing_templates['new_reply']) && 
                    strpos($existing_templates['new_reply'], '{user_name}') !== false);
    
    if ($needs_update) {
        $templates = array(
            'new_ticket' => '',
            'new_reply' => '',
            'status_changed' => '',
            'sla_breach' => '',
            'email_auto_response' => '<p>Thank you for contacting us. We have received your support request and have assigned it ticket ID <strong>#{ticket_id}</strong>.</p>

<p><strong>Subject:</strong> {subject}</p>

<p>Our support team will review your request and get back to you as soon as possible. You can reference this ticket ID in any future correspondence.</p>

<p>Best regards,<br>
{site_name} Support Team<br>
<a href="{site_url}">{site_url}</a></p>',
        );
        
        update_option('nexlifydesk_email_templates', $templates);
    }
}

// Deactivation function
function nexlifydesk_deactivate() {

    // Remove scheduled events
    wp_clear_scheduled_hook('nexlifydesk_sla_check');
    wp_clear_scheduled_hook('nexlifydesk_auto_close_tickets');

    // Note: Email templates are now handled by template files in templates/emails/ directory
}

register_activation_hook(__FILE__, 'nexlifydesk_activate'); 
register_deactivation_hook(__FILE__, 'nexlifydesk_deactivate');

register_activation_hook(__FILE__, function() {
    add_option('nexlifydesk_last_ticket_number', 0);
});

add_action('nexlifydesk_sla_check', 'nexlifydesk_check_sla');

function nexlifydesk_check_sla() {
    global $wpdb;
    $settings = get_option('nexlifydesk_settings');
    $sla_hours = isset($settings['sla_response_time']) ? intval($settings['sla_response_time']) : 0;

    $cache_key = 'nexlifydesk_sla_tickets_' . md5($sla_hours . gmdate('Y-m-d-H'));
    $tickets = wp_cache_get($cache_key);
    
    if (false === $tickets) {
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Custom table requires direct query
        $tickets = $wpdb->get_results(
            $wpdb->prepare(
                "SELECT * FROM {$wpdb->prefix}nexlifydesk_tickets WHERE status IN ('open', 'pending') AND created_at < %s",
                gmdate('Y-m-d H:i:s', strtotime("-$sla_hours hours"))
            )
        );
        
        wp_cache_set($cache_key, $tickets, '', 1800);
    }
    
    foreach ($tickets as $ticket) {
        NexlifyDesk_Tickets::send_notification($ticket, 'sla_breach');
    }
}

add_action('wp', function() {
    if (!wp_next_scheduled('nexlifydesk_sla_check')) {
        wp_schedule_event(time(), 'hourly', 'nexlifydesk_sla_check');
    }
});

function nexlifydesk_auto_close_resolved_tickets() {
    global $wpdb;
    
    $cache_key = 'nexlifydesk_resolved_tickets_' . gmdate('Y-m-d-H');
    $resolved_tickets = wp_cache_get($cache_key);
    
    if (false === $resolved_tickets) {
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Custom table requires direct query
        $resolved_tickets = $wpdb->get_results(
            $wpdb->prepare(
                "SELECT id, ticket_id, user_id, subject, status, updated_at FROM {$wpdb->prefix}nexlifydesk_tickets 
                WHERE status = 'resolved' 
                AND updated_at < %s
                LIMIT 50",
                gmdate('Y-m-d H:i:s', strtotime('-48 hours'))
            )
        );
        
        wp_cache_set($cache_key, $resolved_tickets, '', 1800);
    }
    
    if (!empty($resolved_tickets) && is_array($resolved_tickets)) {
        foreach ($resolved_tickets as $ticket) {
            $update_result = NexlifyDesk_Tickets::update_ticket_status($ticket->id, 'closed');
            
            if ($update_result) {
                $system_note = array(
                    'ticket_id' => $ticket->id,
                    'user_id' => 0,
                    'message' => __('This ticket was automatically closed after being resolved for 48 hours with no activity.', 'nexlifydesk'),
                    'is_admin_reply' => 1
                );
                NexlifyDesk_Tickets::add_reply($system_note);
                
                wp_cache_delete('nexlifydesk_ticket_' . $ticket->id);
                wp_cache_delete('nexlifydesk_user_tickets_' . $ticket->user_id);
            }
        }
        
        wp_cache_delete($cache_key);
    }
}

add_action('nexlifydesk_auto_close_tickets', 'nexlifydesk_auto_close_resolved_tickets');

add_action('wp', function() {
    if (!wp_next_scheduled('nexlifydesk_auto_close_tickets')) {
        wp_schedule_event(time(), 'hourly', 'nexlifydesk_auto_close_tickets');
    }
});

add_action('wp_enqueue_scripts', function() {
    global $post;
    
    $should_enqueue = false;
    
    if (is_a($post, 'WP_Post')) {
        if (has_shortcode($post->post_content, 'nexlifydesk_ticket_form') || 
            has_shortcode($post->post_content, 'nexlifydesk_ticket_list')) {
            $should_enqueue = true;
        }
    }
    
    if (!$should_enqueue && (is_page() || is_single())) {
        $should_enqueue = true;
    }
    
    if ($should_enqueue) {
        wp_enqueue_script('jquery');
        // Enqueue CSS
        wp_enqueue_style(
            'nexlifydesk-frontend',
            NEXLIFYDESK_PLUGIN_URL . 'assets/css/nexlifydesk.css',
            array(),
            NEXLIFYDESK_VERSION
        );
        // Enqueue JavaScript with jQuery dependency
        wp_enqueue_script(
            'nexlifydesk-frontend',
            NEXLIFYDESK_PLUGIN_URL . 'assets/js/nexlifydesk.js',
            array('jquery'),
            NEXLIFYDESK_VERSION,
            true
        );
        $settings = get_option('nexlifydesk_settings', array());
        $max_file_size = isset($settings['max_file_size']) ? $settings['max_file_size'] : 2;
        $allowed_file_types = isset($settings['allowed_file_types']) ? $settings['allowed_file_types'] : 'jpg,jpeg,png,gif,pdf,doc,docx,txt';
        
        wp_localize_script('nexlifydesk-frontend', 'nexlifydesk_vars', array(
            'ajaxurl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('nexlifydesk-ajax-nonce'),
            'uploading_text' => __('Uploading...', 'nexlifydesk'),
            'max_file_size' => $max_file_size,
            'allowed_file_types' => $allowed_file_types,
            'submitting_text' => __('Submitting...', 'nexlifydesk'),
            'submit_ticket_text' => __('Submit Ticket', 'nexlifydesk'),
            'error_occurred_text' => __('An error occurred. Please try again.', 'nexlifydesk'),
            'adding_text' => __('Adding...', 'nexlifydesk'),
            'add_category_text' => __('Add Category', 'nexlifydesk'),
            'add_reply_text' => __('Add Reply', 'nexlifydesk'),
            'add_note_text' => __('Add Note', 'nexlifydesk'),
            'ticket_form_url' => get_permalink(get_option('nexlifydesk_settings')['ticket_form_page_id'] ?? 0),
            'create_new_ticket_text' => __('Create New Ticket', 'nexlifydesk'),
            'ticket_closed_text' => __('This ticket is closed. Please create a new ticket for further assistance.', 'nexlifydesk'),
            'status_updated_text' => __('Status updated successfully!', 'nexlifydesk'),
            'ticket_page_url' => nexlifydesk_get_ticket_page_url(),
            ));
    }
}, 20);

add_action('admin_init', function() {
    $current_user = wp_get_current_user();
});

add_action('admin_menu', function() {
    if (current_user_can('nexlifydesk_manage_tickets') && !current_user_can('manage_options')) {
        remove_menu_page('edit.php');
        remove_menu_page('upload.php');
        remove_menu_page('edit-comments.php');
    }
}, 999);

/**
 * This is called by Freemius when the plugin is uninstalled through their system
 */
function nexlifydesk_freemius_uninstall_cleanup() {
    $uninstall_file = plugin_dir_path(__FILE__) . 'uninstall.php';
    if (file_exists($uninstall_file)) {
        if (!defined('WP_UNINSTALL_PLUGIN')) {
            define('WP_UNINSTALL_PLUGIN', true);
        }
        include_once $uninstall_file;
    }
}

register_uninstall_hook(__FILE__, 'nexlifydesk_freemius_uninstall_cleanup');

add_action('wp_ajax_nexlifydesk_update_status', 'nexlifydesk_update_status_callback');
function nexlifydesk_update_status_callback() {
    check_ajax_referer('nexlifydesk-ajax-nonce', 'nonce');

    $ticket_id = isset($_POST['ticket_id']) ? intval(wp_unslash($_POST['ticket_id'])) : 0;
    $status = isset($_POST['status']) ? sanitize_text_field(wp_unslash($_POST['status'])) : '';

    if (!$ticket_id || !$status) {
        wp_send_json_error(__('Invalid request.', 'nexlifydesk'));
    }

    $allowed_statuses = ['open', 'pending', 'in_progress', 'resolved', 'closed'];
    if (!in_array($status, $allowed_statuses, true)) {
        wp_send_json_error(__('Invalid status.', 'nexlifydesk'));
    }

    $result = NexlifyDesk_Tickets::update_ticket_status($ticket_id, $status);

    if ($result) {
        wp_send_json_success(__('Status updated successfully!', 'nexlifydesk'));
    } else {
        wp_send_json_error(__('Failed to update status.', 'nexlifydesk'));
    }
}

add_action('nexlifydesk_check_orphaned_tickets', 'nexlifydesk_check_orphaned_tickets');

function nexlifydesk_check_orphaned_tickets() {
    NexlifyDesk_Tickets::handle_orphaned_tickets();
}

add_action('wp', function() {
    if (!wp_next_scheduled('nexlifydesk_check_orphaned_tickets')) {
        wp_schedule_event(time(), 'daily', 'nexlifydesk_check_orphaned_tickets');
    }
});

add_action('wp_ajax_nexlifydesk_purge_data', function() {
    if (!current_user_can('manage_options')) {
        wp_send_json_error(__('You do not have permission to perform this action.', 'nexlifydesk'));
    }
    check_ajax_referer('nexlifydesk-ajax-nonce', '_ajax_nonce');
    global $wpdb;
    $purged = array();
    
    $tickets_table = $wpdb->prefix . 'nexlifydesk_tickets';
    $replies_table = $wpdb->prefix . 'nexlifydesk_replies';
    $attachments_table = $wpdb->prefix . 'nexlifydesk_attachments';
    $one_year_ago = gmdate('Y-m-d H:i:s', strtotime('-1 year'));
    
    // phpcs:disable WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table name is constructed from $wpdb->prefix and is safe
    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table maintenance requires direct queries without caching
    $old_ticket_ids = $wpdb->get_col(
        $wpdb->prepare(
            "SELECT id FROM {$tickets_table} WHERE created_at < %s",
            $one_year_ago
        )
    );
    if (!empty($old_ticket_ids)) {
        $ids_placeholder = implode(',', array_map('absint', $old_ticket_ids));
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table maintenance requires direct queries without caching
        $wpdb->query("DELETE FROM $attachments_table WHERE ticket_id IN ($ids_placeholder)");
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table maintenance requires direct queries without caching
        $wpdb->query("DELETE FROM $replies_table WHERE ticket_id IN ($ids_placeholder)");
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table maintenance requires direct queries without caching
        $wpdb->query("DELETE FROM $tickets_table WHERE id IN ($ids_placeholder)");
        // translators: %d: Number of tickets purged.
        $purged[] = sprintf(__('Purged %d tickets older than 1 year.', 'nexlifydesk'), count($old_ticket_ids));
    } else {
        $purged[] = __('No tickets older than 1 year found.', 'nexlifydesk');
    }
    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table maintenance requires direct queries without caching
    $wpdb->query("DELETE FROM $attachments_table WHERE ticket_id NOT IN (SELECT id FROM $tickets_table)");
    $purged[] = __('Purged orphaned attachments.', 'nexlifydesk');
    wp_send_json_success(implode('<br>', $purged));

    // phpcs:enable WordPress.DB.PreparedSQL.InterpolatedNotPrepared
});

add_action('nexlifydesk_fetch_emails_event', 'nexlifydesk_fetch_emails');

function nexlifydesk_get_fetch_interval() {
    $settings = get_option('nexlifydesk_imap_settings', []);
    $interval = isset($settings['fetch_interval']) ? intval($settings['fetch_interval']) : 5;
    
    // Ensure valid interval
    if (!in_array($interval, [2, 5, 10])) {
        $interval = 5; // Default to 5 minutes
    }
    
    return $interval;
}

function nexlifydesk_reschedule_email_fetch() {
    $interval = nexlifydesk_get_fetch_interval();
    $schedule_name = "nexlifydesk_{$interval}_minutes";
    
    wp_clear_scheduled_hook('nexlifydesk_fetch_emails_event');
    
    if (!wp_next_scheduled('nexlifydesk_fetch_emails_event')) {
        wp_schedule_event(time(), $schedule_name, 'nexlifydesk_fetch_emails_event');
    }
}

register_activation_hook(__FILE__, function() {
    nexlifydesk_reschedule_email_fetch();
});

add_action('update_option_nexlifydesk_imap_settings', function($old_value, $value, $option) {
    $old_interval = isset($old_value['fetch_interval']) ? intval($old_value['fetch_interval']) : 5;
    $new_interval = isset($value['fetch_interval']) ? intval($value['fetch_interval']) : 5;
    
    if ($old_interval !== $new_interval) {
        nexlifydesk_reschedule_email_fetch();
    }
}, 10, 3);

add_action('init', function() {
    if (!wp_next_scheduled('nexlifydesk_fetch_emails_event')) {
        nexlifydesk_reschedule_email_fetch();
    }
});

add_filter('cron_schedules', function($schedules) {
    $schedules['nexlifydesk_2_minutes'] = array(
        'interval' => 120, // 2 minutes
        'display'  => __('Every 2 Minutes (NexlifyDesk)', 'nexlifydesk')
    );
    $schedules['nexlifydesk_5_minutes'] = array(
        'interval' => 300, // 5 minutes
        'display'  => __('Every 5 Minutes (NexlifyDesk)', 'nexlifydesk')
    );
    $schedules['nexlifydesk_10_minutes'] = array(
        'interval' => 600, // 10 minutes
        'display'  => __('Every 10 Minutes (NexlifyDesk)', 'nexlifydesk')
    );
    return $schedules;
});

add_action('wp_ajax_nexlifydesk_fetch_emails_now', function() {
    check_ajax_referer('nexlifydesk_fetch_emails_now');
    
    if (!current_user_can('nexlifydesk_manage_tickets') && !current_user_can('manage_options')) {
        wp_send_json_error(__('You do not have permission.', 'nexlifydesk'));
    }
    
    if (function_exists('nexlifydesk_fetch_emails')) {
        nexlifydesk_fetch_emails();
        wp_send_json_success();
    } else {
        wp_send_json_error(__('Fetch function not found.', 'nexlifydesk'));
    }
});

/**
 * Gets tickets for the initial grid view.
 *
 * @param array $args
 * @return array
 */
function nexlifydesk_get_tickets_for_grid($args = []) {
    if (!class_exists('NexlifyDesk_Tickets')) {
        return [];
    }
    return NexlifyDesk_Tickets::get_tickets_for_grid($args);
}

/**
 * Renders a single ticket card.
 *
 * @param object $ticket
 */
function nexlifydesk_render_ticket_card($ticket) {
    if (!is_object($ticket)) {
        return;
    }
    $assignee_name = !empty($ticket->assigned_to_display_name) ? $ticket->assigned_to_display_name : __('Unassigned', 'nexlifydesk');
    $assignee_initials = strtoupper(substr($assignee_name, 0, 2));
    $status_class = 'status-' . str_replace('_', '-', $ticket->status);
    $priority_class = 'priority-' . $ticket->priority;
    $ticket_url = admin_url('admin.php?page=nexlifydesk_tickets&ticket_id=' . $ticket->id);
    ?>
    <div class="ticket-card" onclick="window.location.href='<?php echo esc_url($ticket_url); ?>';" data-id="<?php echo esc_attr($ticket->id); ?>">
        <div class="ticket-header">
            <div class="ticket-id">#<?php echo esc_html($ticket->ticket_id); ?></div>
        </div>
        <div class="ticket-title"><?php echo esc_html($ticket->subject); ?></div>
        <div class="ticket-description"><?php echo esc_html(wp_trim_words($ticket->message, 15)); ?></div>
        <div class="ticket-meta">
            <span class="status-badge <?php echo esc_attr($status_class); ?>"><?php echo esc_html(ucwords(str_replace('_', ' ', $ticket->status))); ?></span>
            <span class="priority-badge <?php echo esc_attr($priority_class); ?>"><?php echo esc_html(ucfirst($ticket->priority)); ?></span>
        </div>
        <div class="ticket-footer">
            <div class="assignee">
                <div class="avatar"><?php echo esc_html($assignee_initials); ?></div>
                <span><?php echo esc_html($assignee_name); ?></span>
            </div>
            <div><?php echo esc_html(human_time_diff(strtotime($ticket->created_at))) . ' ' . esc_html__('ago', 'nexlifydesk'); ?></div>
        </div>
    </div>
    <?php
}