<?php
if (!defined('ABSPATH')) {
    exit;
}

// Load necessary interfaces and classes for text analysis
require_once __DIR__ . '/textanalysis/Interfaces/IDistance.php';
require_once __DIR__ . '/textanalysis/Interfaces/ISimilarity.php';
require_once __DIR__ . '/textanalysis/Interfaces/ITokenTransformation.php';
require_once __DIR__ . '/textanalysis/Interfaces/IStemmer.php';
require_once __DIR__ . '/textanalysis/Interfaces/IExtractStrategy.php';
require_once __DIR__ . '/textanalysis/Documents/DocumentAbstract.php';
require_once __DIR__ . '/textanalysis/Documents/TokensDocument.php';
require_once __DIR__ . '/textanalysis/Tokenizers/TokenizerAbstract.php';
require_once __DIR__ . '/textanalysis/Tokenizers/WhitespaceTokenizer.php';
require_once __DIR__ . '/textanalysis/Tokenizers/GeneralTokenizer.php';
require_once __DIR__ . '/textanalysis/Comparisons/CosineSimilarityComparison.php';

/**
 * NexlifyDesk Email Filtering & Spam Protection Features
 * 
 * This file contains functions for filtering emails and preventing spam from creating tickets.
 * 
 * Features include:
 * - Admin email blocking: Prevents emails from the admin email address (like notification emails)
 * - Notification subject filtering: Blocks emails with [Admin] or [Agent] in the subject
 * - Blocked email addresses: Custom list of email addresses to ignore
 * - Domain blocking: Block entire domains (useful for known spam domains)
 * - URL filtering: Blocks emails with too many links or suspicious URLs
 * - Keyword filtering: Blocks emails containing specified spam keywords
 * 
 * All settings can be configured in WP Admin > Settings > NexlifyDesk > Email Piping
 */

/**
 * Translate ticket status to localized strings
 */
function nexlifydesk_translate_status($status) {
    $statuses = array(
        'open'     => __('Open', 'nexlifydesk'),
        'pending'  => __('Pending', 'nexlifydesk'),
        'resolved' => __('Resolved', 'nexlifydesk'),
        'closed'   => __('Closed', 'nexlifydesk')
    );
    
    return isset($statuses[$status]) ? $statuses[$status] : ucfirst($status);
}

/**
 * Check if an email should be blocked based on spam protection settings
 *
 * @param string $email_address The sender's email address
 * @param string $subject The email subject
 * @param string $message The email message content
 * @param array $settings The IMAP settings with spam protection options
 * @return bool True if email should be blocked, false otherwise
 */
function nexlifydesk_should_block_email($email_address, $subject, $message, $settings) {
    // Check if admin email blocking is enabled
    if (!empty($settings['block_admin_emails']) && $settings['block_admin_emails']) {
        $admin_email = get_option('admin_email');
        if (strtolower($email_address) === strtolower($admin_email)) {
            return true;
        }
    }

    if (!empty($settings['block_notification_subjects']) && $settings['block_notification_subjects']) {
        if (preg_match('/\[(Admin|Agent)\]/i', $subject)) {
            return true;
        }
    }

    if (!empty($settings['blocked_emails'])) {
        $blocked_emails = array_map('trim', array_filter(explode("\n", $settings['blocked_emails'])));
        foreach ($blocked_emails as $blocked_email) {
            if (strtolower($email_address) === strtolower($blocked_email)) {
                return true;
            }
        }
    }

    if (!empty($settings['blocked_domains'])) {
        $blocked_domains = array_map('trim', array_filter(explode("\n", $settings['blocked_domains'])));
        $email_domain = substr(strrchr($email_address, '@'), 1);
        foreach ($blocked_domains as $blocked_domain) {
            if (strtolower($email_domain) === strtolower($blocked_domain)) {
                return true;
            }
        }
    }

    if (!empty($settings['blocked_keywords'])) {
        $blocked_keywords = array_map('trim', array_filter(explode("\n", $settings['blocked_keywords'])));
        $full_content = $subject . ' ' . $message;
        foreach ($blocked_keywords as $keyword) {
            if (stripos($full_content, $keyword) !== false) {
                return true;
            }
        }
    }

    if (!empty($settings['spam_url_filtering']) && $settings['spam_url_filtering']) {
        $max_links = isset($settings['max_links_per_email']) ? (int)$settings['max_links_per_email'] : 3;
        
        $url_pattern = '/https?:\/\/[^\s<>"{}|\\^`\[\]]+/i';
        preg_match_all($url_pattern, $message, $matches);
        $url_count = count($matches[0]);
        
        if ($url_count > $max_links) {
            return true;
        }

        $suspicious_patterns = [
            '/bit\.ly|tinyurl|t\.co|goo\.gl|short\.link/i', 
            '/\.(tk|ml|ga|cf)(\s|\/|$)/i', 
            '/[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}/i',
        ];
        
        foreach ($suspicious_patterns as $pattern) {
            if (preg_match($pattern, $message)) {
                return true;
            }
        }
    }

    if (!empty($settings['block_marketing_emails']) && $settings['block_marketing_emails']) {
        if (function_exists('nexlifydesk_is_marketing_email') && 
            nexlifydesk_is_marketing_email($email_address, $subject, $message, $settings)) {
            return true;
        }
    }

    return false;
}

/**
 * Get the ticket page URL from plugin settings
 * @return string The ticket page URL or empty string if not found
 */
function nexlifydesk_get_ticket_page_url() {
    $settings = get_option('nexlifydesk_settings', array());
    if (isset($settings['ticket_page_id']) && $settings['ticket_page_id']) {
        return get_permalink($settings['ticket_page_id']);
    }
    return '';
}

/**
 * Extract order numbers from subject or message using flexible patterns
 * @param string $text
 * @return array Array of detected order numbers (as strings)
 */
function nexlifydesk_extract_order_numbers($text) {
    $order_numbers = array();
    if (empty($text)) return $order_numbers;
    $patterns = array(
        '/order\s*(number)?\s*#?\s*(\d{4,})/i',
        '/#(\d{4,})/', 
        '/\b(\d{4,})\b/' 
    );
    foreach ($patterns as $pattern) {
        if (preg_match_all($pattern, $text, $matches)) {
            foreach ($matches[2] ?? $matches[1] ?? array() as $num) {
                if ($num && !in_array($num, $order_numbers)) {
                    $order_numbers[] = $num;
                }
            }
        }
    }
    return $order_numbers;
}

/**
 * Apply keyword mapping to normalize semantic tokens
 * @param array $tokens Array of tokens to normalize
 * @param array $map Keyword mapping array
 * @return array Normalized tokens
 */
function nexlifydesk_apply_keyword_map($tokens, $map) {
    foreach ($tokens as &$token) {
        if (isset($map[$token])) {
            $token = $map[$token];
        }
    }
    return $tokens;
}

/**
 * Get common English stopwords for text processing
 * @return array Array of stopwords
 */
function nexlifydesk_get_stopwords() {
    return array('the','and','is','it','to','of','in','on','for','with','a','an','as','at','by','from','this','that','can','be','has','have','was','were','but','or','not','so','do','does','did','will','would','should','could','may','might','must','you','your','i','me','my','we','us','our','he','she','him','her','they','them','their');
}

/**
 * 
 * @param array $ticket_data (user_id, email, subject, message, source)
 * @return object|null Existing ticket if found, null otherwise
 */
function nexlifydesk_find_duplicate_ticket($ticket_data) {
    global $wpdb;
    $user_id = isset($ticket_data['user_id']) ? absint($ticket_data['user_id']) : 0;
    $email = isset($ticket_data['email']) ? sanitize_email($ticket_data['email']) : '';
    $subject = isset($ticket_data['subject']) ? sanitize_text_field($ticket_data['subject']) : '';
    $message = isset($ticket_data['message']) ? sanitize_text_field($ticket_data['message']) : '';
    $table_name = $wpdb->prefix . 'nexlifydesk_tickets';
    
    // A. REGISTERED USER LOGIC
    if ($user_id > 0) {
        $cache_key = 'nexlifydesk_user_has_tickets_' . $user_id;
        $user_has_tickets = wp_cache_get($cache_key);
        
        if (false === $user_has_tickets) {
            // phpcs:disable WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table name is safe.
            $user_ticket_count = $wpdb->get_var( $wpdb->prepare(// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Custom table name is safe and required direct query.
                "SELECT COUNT(*) FROM `{$table_name}` WHERE user_id = %d AND status IN ('open','pending','in_progress')",
                $user_id
            ) );
            // phpcs:enable WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table name is safe.
            $user_has_tickets = (int)$user_ticket_count > 0;
            wp_cache_set($cache_key, $user_has_tickets, '', 300); // Cache for 5 minutes
        }
        
        if (!$user_has_tickets) {
            return null;
        }
        
        $order_numbers = array();
        if (!empty($subject)) {
            $order_numbers = array_merge($order_numbers, nexlifydesk_extract_order_numbers($subject));
        }
        if (!empty($message)) {
            $order_numbers = array_merge($order_numbers, nexlifydesk_extract_order_numbers($message));
        }
        $order_numbers = array_unique($order_numbers);
        
        if (!empty($order_numbers)) {
            $order_regex = implode('|', array_map('preg_quote', $order_numbers));
            $cache_key = 'nexlifydesk_user_order_match_' . md5($user_id . $order_regex);
            $existing_ticket = wp_cache_get($cache_key);

            if (false === $existing_ticket) {
                // phpcs:disable WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table name is safe.
                $existing_ticket = $wpdb->get_row( $wpdb->prepare(// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Custom table name is safe and required direct query.
                    "SELECT * FROM `{$table_name}` WHERE user_id = %d AND (subject REGEXP %s OR message REGEXP %s) AND status IN ('open','pending','in_progress') ORDER BY created_at DESC LIMIT 1",
                    $user_id, $order_regex, $order_regex
                ) );
                // phpcs:enable WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table name is safe.
                wp_cache_set($cache_key, $existing_ticket, '', 60);
            }
            if ($existing_ticket) return $existing_ticket;
        }
        
        if (!empty($subject)) {
            $cache_key = 'nexlifydesk_user_subject_match_' . md5($user_id . $subject);
            $existing_ticket = wp_cache_get($cache_key);

            if (false === $existing_ticket) {
                // phpcs:disable WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table name is safe.
                $existing_ticket = $wpdb->get_row( $wpdb->prepare(// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Custom table name is safe and required direct query.
                    "SELECT * FROM `{$table_name}` WHERE user_id = %d AND subject = %s AND status IN ('open','pending','in_progress') ORDER BY created_at DESC LIMIT 1",
                    $user_id, $subject
                ) );
                // phpcs:enable WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table name is safe.
                wp_cache_set($cache_key, $existing_ticket, '', 60);
            }
            if ($existing_ticket) return $existing_ticket;
        }
        
        if (class_exists('TextAnalysis\\Comparisons\\CosineSimilarityComparison') && class_exists('TextAnalysis\\Tokenizers\\GeneralTokenizer')) {
            $similarity_threshold = 0.12; // Optimized threshold for better matching
            $cache_key = 'nexlifydesk_user_tickets_semantic_' . $user_id;
            $user_tickets = wp_cache_get($cache_key);

            if (false === $user_tickets) {
                // phpcs:disable WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table name is safe.
                $user_tickets = $wpdb->get_results( $wpdb->prepare( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Custom table name is safe and required direct query.
                    "SELECT * FROM `{$table_name}` WHERE user_id = %d AND status IN ('open','pending','in_progress') ORDER BY created_at DESC LIMIT 10",
                    $user_id
                ) );
                // phpcs:enable WordPress.DB.PreparedSQL.InterpolatedNotPrepated -- Table name is safe.
                wp_cache_set($cache_key, $user_tickets, '', 60);
            }
            
            if (!empty($user_tickets)) {
                $keyword_map = array(
                    'profile' => 'account', 'dashboard' => 'account', 'user' => 'account', 'panel' => 'account',
                    'restoring' => 'unlock', 'restored' => 'unlock', 'recovered' => 'unlock', 'reset' => 'unlock',
                    'unlock' => 'unlock', 'activate' => 'unlock', 'recover' => 'unlock', 'recovery' => 'unlock',
                    'lockedout' => 'locked', 'locked' => 'locked', 'blocked' => 'locked', 'out' => 'locked',
                    'credentials' => 'login', 'login' => 'login', 'signin' => 'login', 'sign' => 'login', 'access' => 'login',
                    'password' => 'password', 'passcode' => 'password', 'pwd' => 'password',
                    'help' => 'help', 'support' => 'help', 'assistance' => 'help', 'team' => 'help',
                    'issue' => 'problem', 'problem' => 'problem', 'error' => 'problem', 'failed' => 'problem',
                    'unable' => 'problem', 'cant' => 'problem', 'cannot' => 'problem', 'rejecting' => 'problem', 'not' => 'problem', 'anymore' => 'problem'
                );
                
                $tokenizer = new TextAnalysis\Tokenizers\GeneralTokenizer();
                $stopwords = nexlifydesk_get_stopwords();
                $incoming_text = $subject . ' ' . $message;
                $incoming_tokens = array_map(function($t) {
                    return strtolower(preg_replace('/[^a-z0-9]/i', '', $t));
                }, $tokenizer->tokenize($incoming_text));
                $incoming_tokens = array_diff($incoming_tokens, $stopwords);
                $incoming_tokens = nexlifydesk_apply_keyword_map($incoming_tokens, $keyword_map);
                
                foreach ($user_tickets as $ticket) {
                    $existing_text = $ticket->subject . ' ' . $ticket->message;
                    $existing_tokens = array_map(function($t) {
                        return strtolower(preg_replace('/[^a-z0-9]/i', '', $t));
                    }, $tokenizer->tokenize($existing_text));
                    $existing_tokens = array_diff($existing_tokens, $stopwords);
                    $existing_tokens = nexlifydesk_apply_keyword_map($existing_tokens, $keyword_map);
                    
                    $similarity = (new TextAnalysis\Comparisons\CosineSimilarityComparison())->similarity(
                        $incoming_tokens,
                        $existing_tokens
                    );
                    if ($similarity >= $similarity_threshold) {
                        return $ticket; 
                    }
                }
            }
        }
        
        return null;
    }

    // B. UNREGISTERED USER LOGIC
    if ($user_id == 0 && !empty($email)) {
        $cache_key = 'nexlifydesk_email_consolidation_' . md5($email);
        $existing_ticket = wp_cache_get($cache_key);

        if (false === $existing_ticket) {
            // phpcs:disable WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table name is safe.
            $existing_ticket = $wpdb->get_row( $wpdb->prepare(// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Custom table name is safe and required direct query.
                "SELECT t.* FROM `{$table_name}` t 
                 LEFT JOIN `{$wpdb->prefix}postmeta` pm ON pm.post_id = t.id AND pm.meta_key = 'customer_email' 
                 WHERE t.user_id = 0 AND pm.meta_value = %s AND t.status IN ('open','pending','in_progress','resolved')
                 ORDER BY t.created_at DESC LIMIT 1",
                $email
            ) );
            // phpcs:enable WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table name is safe.
            wp_cache_set($cache_key, $existing_ticket, '', 60);
        }
        
        return $existing_ticket; 
    }
    
    return null;
}

/**
 * Extract keywords from text for similarity comparison
 * @param string $text Input text to process
 * @return array Array of unique keywords
 */
function nexlifydesk_extract_keywords($text) {
    $text = strtolower($text);
    $text = preg_replace('/[^a-z0-9 ]/', ' ', $text);
    $words = array_filter(explode(' ', $text));
    $stopwords = nexlifydesk_get_stopwords();
    $keywords = array_diff($words, $stopwords);
    return array_unique($keywords);
}

/**
 * Calculate similarity between two messages using keyword intersection
 * @param string $msg1 First message
 * @param string $msg2 Second message  
 * @return float Similarity score between 0 and 1
 */
function nexlifydesk_message_similarity($msg1, $msg2) {
    $kw1 = nexlifydesk_extract_keywords($msg1);
    $kw2 = nexlifydesk_extract_keywords($msg2);
    if (empty($kw1) || empty($kw2)) return 0;
    $intersection = array_intersect($kw1, $kw2);
    $union = array_unique(array_merge($kw1, $kw2));
    return count($intersection) / count($union);
}