<?php
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Encrypts a given plain text using AES-256-CBC encryption.
 *
 * @param string $plain_text The text to encrypt.
 * @return string The encrypted text, base64 encoded.
 */

function nexlifydesk_encrypt($plain_text) {
    if (empty($plain_text)) {
        return '';
    }
    
    if (nexlifydesk_is_encrypted($plain_text)) {
        return $plain_text;
    }
    
    if (defined('NEXLIFYDESK_ENCRYPTION_KEY') && !empty(NEXLIFYDESK_ENCRYPTION_KEY)) {
        $key = NEXLIFYDESK_ENCRYPTION_KEY;
    } else {
        $key = AUTH_KEY;
    }
    
    $key = hash('sha256', $key, true);
    $ivlen = openssl_cipher_iv_length('aes-256-cbc');
    $iv = openssl_random_pseudo_bytes($ivlen);
    $ciphertext = openssl_encrypt($plain_text, 'aes-256-cbc', $key, 0, $iv);
    
    if ($ciphertext === false) {
        return $plain_text;
    }
    
    $encrypted_result = base64_encode($iv . $ciphertext);
    
    return $encrypted_result;
}

function nexlifydesk_decrypt($encrypted_text) {
    if (empty($encrypted_text)) {
        return '';
    }
    
    if (!nexlifydesk_is_encrypted($encrypted_text)) {
        return $encrypted_text;
    }
    
    if (defined('NEXLIFYDESK_ENCRYPTION_KEY') && !empty(NEXLIFYDESK_ENCRYPTION_KEY)) {
        $key = NEXLIFYDESK_ENCRYPTION_KEY;
    } else {
        $key = AUTH_KEY;
    }
    
    $key = hash('sha256', $key, true);
    
    $data = base64_decode($encrypted_text);
    
    if ($data === false) {
        return $encrypted_text;
    }
    
    $ivlen = openssl_cipher_iv_length('aes-256-cbc');
    
    if (strlen($data) < $ivlen) {
        return $encrypted_text;
    }
    
    $iv = substr($data, 0, $ivlen);
    $ciphertext = substr($data, $ivlen);
    
    $decrypted = openssl_decrypt($ciphertext, 'aes-256-cbc', $key, 0, $iv);
    
    if ($decrypted === false) {
        return $encrypted_text;
    }

    return $decrypted;
}

/**
 * Check if a value appears to be encrypted
 * Improved version with better detection
 */
function nexlifydesk_is_encrypted($value) {
    if (empty($value)) {
        return false;
    }
    
    $decoded = base64_decode($value, true);
    if ($decoded === false) {
        return false;
    }
    
    $ivlen = openssl_cipher_iv_length('aes-256-cbc');
    if (strlen($decoded) < $ivlen + 16) {
        return false;
    }
    
    if (strlen($value) < 32) {
        return false;
    }
    
    if (!preg_match('/^[A-Za-z0-9+\/=]+$/', $value)) {
        return false;
    }
    
    if (strlen($value) < 50 && preg_match('/^[a-zA-Z0-9!@#$%^&*()_+\-=\[\]{};\':"\\|,.<>\/?]+$/', $value)) {
        return false;
    }
    
    return true;
}

/**
 * @param string $from_header The full "From" header string
 * @return array Array with 'name' and 'email' keys
 */
function nexlifydesk_parse_email_from($from_header) {
    $result = ['name' => '', 'email' => ''];
    
    if (empty($from_header)) {
        return $result;
    }
    
    $from_header = trim($from_header);
    
    if (strpos($from_header, '=?') !== false) {
        $from_header = imap_mime_header_decode($from_header);
        if (is_array($from_header) && count($from_header) > 0) {
            $from_header = $from_header[0]->text;
        }
    }
    
    if (preg_match('/^"?([^"<>]*?)"?\s*<([^<>]+)>$/', trim($from_header), $matches)) {
        $result['name'] = trim($matches[1], '"');
        $result['email'] = trim($matches[2]);
    }
    elseif (preg_match('/^([^<>]+?)\s*<([^<>]+)>$/', trim($from_header), $matches)) {
        $result['name'] = trim($matches[1]);
        $result['email'] = trim($matches[2]);
    }
    elseif (preg_match('/^[^<>]+@[^<>]+$/', trim($from_header))) {
        $result['email'] = trim($from_header);
        $email_parts = explode('@', $result['email']);
        if (count($email_parts) >= 2) {
            $potential_name = str_replace(['.', '_', '+', '-'], ' ', $email_parts[0]);
            $result['name'] = ucwords($potential_name);
        }
    }
    elseif (preg_match('/^([^<>]+@[^<>]+)\s*\(([^)]+)\)$/', trim($from_header), $matches)) {
        $result['email'] = trim($matches[1]);
        $result['name'] = trim($matches[2]);
    }
    
    $result['name'] = trim($result['name']);
    $result['email'] = trim($result['email']);
    
    if (!filter_var($result['email'], FILTER_VALIDATE_EMAIL)) {
        if (preg_match('/([a-zA-Z0-9._%+-]+@[a-zA-Z0-9.-]+\.[a-zA-Z]{2,})/', $from_header, $matches)) {
            $result['email'] = $matches[1];
        } else {
            $result['email'] = '';
        }
    }
    
    if (empty($result['name']) && !empty($result['email'])) {
        $email_parts = explode('@', $result['email']);
        if (count($email_parts) >= 2) {
            $potential_name = str_replace(['.', '_', '+', '-'], ' ', $email_parts[0]);
            $result['name'] = ucwords($potential_name);
        }
    }
    
    return $result;
}

/**
 * Decode email subject from MIME encoding
 * @param string $subject The encoded email subject
 * @return string The decoded subject
 */
function nexlifydesk_decode_email_subject($subject) {
    if (empty($subject)) {
        return '';
    }
    
    $subject = trim($subject);
    
    if (strpos($subject, '=?') !== false && strpos($subject, '?=') !== false) {
        $decoded_parts = imap_mime_header_decode($subject);
        
        if (is_array($decoded_parts) && count($decoded_parts) > 0) {
            $decoded_subject = '';
            foreach ($decoded_parts as $part) {
                $decoded_subject .= $part->text;
            }
            return trim($decoded_subject);
        }
    }
    
    return $subject;
}

/**
 * Marketing email detection for NexlifyDesk.
 *
 * @param string $email_address The sender's email address.
 * @param string $subject The email subject.
 * @param string $message The email message content.
 * @param array $settings The IMAP/settings array.
 * @return bool True if email appears to be marketing/promotional.
 */
function nexlifydesk_is_marketing_email($email_address, $subject, $message, $settings = []) {
    $whitelist = $settings['marketing_whitelist'] ?? [];
    if (in_array(strtolower($email_address), array_map('strtolower', $whitelist), true)) {
        return false;
    }

    if (empty($email_address) || empty($subject) || empty($message)) {
        return false;
    }

    $marketing_senders = [
        'no-reply', 'noreply', 'donotreply', 'do-not-reply',
        'marketing', 'newsletter', 'promo', 'promotion',
        'offers', 'deals', 'sales', 'updates', 'notifications',
        'alerts', 'info@', 'hello@', 'hi@', 'team@', 'contact@', 'support@'
    ];

    $marketing_subject_keywords = [
        'unsubscribe', 'newsletter', 'limited time', 'exclusive offer', 'sale',
        'discount', 'deal', 'act now', 'limited offer', 'subscribe', 'opt-in',
        'opt-out', 'mailing list', 'marketing', 'advertisement', 'promo code',
        'coupon', 'special offer', 'black friday', 'cyber monday', 'flash sale'
    ];

    $marketing_content_keywords = [
        'unsubscribe', 'opt-out', 'remove me', 'marketing email', 'promotional email',
        'this email was sent to', 'you are receiving this', 'manage your preferences',
        'update your preferences', 'email preferences', 'subscription', 'mailing list',
        'newsletter', 'privacy policy', 'terms of service', 'view in browser',
        'view online', 'add to address book', 'whitelist', 'delivered by',
        'powered by', 'sent via', 'email service'
    ];

    $email_lower = strtolower($email_address);
    $subject_lower = strtolower($subject);
    $message_lower = strtolower($message);

    $is_marketing_sender = false;
    foreach ($marketing_senders as $indicator) {
        if (stripos($email_lower, $indicator) !== false) {
            $is_marketing_sender = true;
            break;
        }
    }

    $has_marketing_subject = false;
    foreach ($marketing_subject_keywords as $keyword) {
        if (stripos($subject_lower, $keyword) !== false) {
            $has_marketing_subject = true;
            break;
        }
    }

    $has_marketing_content = false;
    foreach ($marketing_content_keywords as $keyword) {
        if (stripos($message_lower, $keyword) !== false) {
            $has_marketing_content = true;
            break;
        }
    }

    $html_patterns = [
        '/<table[^>]*>.*?<\/table>/is', '/<img[^>]*src=[^>]*>/i',
        '/<a[^>]*href=[^>]*>/i', '/style\s*=\s*["\'][^"\']*["\']/',
        '/<font[^>]*>/i', '/background-color\s*:/i', '/rgb\s*\(/i',
    ];
    $html_score = 0;
    foreach ($html_patterns as $pattern) {
        if (preg_match($pattern, $message)) {
            $html_score++;
        }
    }

    preg_match_all('/https?:\/\/[^\s<>"{}|\\^`\[\]]+/i', $message, $matches);
    $url_count = count($matches[0]);

    $marketing_score = 0;
    if ($is_marketing_sender) $marketing_score += 2;
    if ($has_marketing_subject) $marketing_score += 2;
    if ($has_marketing_content) $marketing_score += 2;
    if ($html_score >= 3) $marketing_score += 1;
    if ($url_count > 2) $marketing_score += 1;
    if (strlen($message) > 2000) $marketing_score += 1;

    $indicator_count = 0;
    if ($is_marketing_sender) $indicator_count++;
    if ($has_marketing_subject) $indicator_count++;
    if ($has_marketing_content) $indicator_count++;
    if ($html_score >= 3) $indicator_count++;
    if ($url_count > 2) $indicator_count++;

    return ($marketing_score >= 4 && $indicator_count >= 2);
}

/**
 * 
 * @param string $email_content The full email content
 * @return string Clean email content without thread history
 */
function nexlifydesk_extract_clean_email_content($email_content, $from_email = '') {
    if (empty($email_content)) {
        return $email_content;
    }
    if (!empty($from_email) && function_exists('nexlifydesk_is_admin_or_agent_email') && nexlifydesk_is_admin_or_agent_email($from_email)) {
        return $email_content;
    }
    $body = str_replace(["\r\n", "\r"], "\n", $email_content);
    $header_pattern = '/^On\s.+?(?:at\s.+?)?\s?.+?\swrote:\s*$/mus';
    $reply_separator_regexes = [
        $header_pattern,
        '/^-{3,}\s*On\s+.+wrote\s*-{3,}/mus',
        '/^-{2,}\s*On\s+\w+,\s*\d{1,2}\s+\w+\s+\d{4}.+wrote\s*-{2,}/mus',
        '/^At\s+\d{1,2}:\d{2}.*on\s+\d{1,2}\/\d{1,2}\/\d{4}.*wrote:/mus',
        '/^-----Original Message-----/mus',
        '/^From:[\s\S]+?$/mus',
        '/^To:[\s\S]+?$/mus',
        '/^Sent:[\s\S]+?$/mus',
        '/^Date:[\s\S]+?$/mus',
        '/^Subject:[\s\S]+?$/mus',
        '/^Reply above this line$/mus',
        '/^Please reply above this line$/mus',
        '/^Do not reply below this line$/mus',
        '/^-----\s*Reply\s*above\s*this\s*line\s*-----/mus',
        '/^Le\s+\d{1,2}\/\d{1,2}\/\d{4}.*a\s+écrit\s*:/mus',
        '/^Am\s+\d{1,2}\.\d{1,2}\.\d{4}.*schrieb\s*:/mus',
        '/^Il\s+\d{1,2}\/\d{1,2}\/\d{4}.*ha\s+scritto\s*:/mus',
        '/^El\s+\d{1,2}\/\d{1,2}\/\d{4}.*escribió\s*:/mus',
        '/^Em\s+\d{1,2}\/\d{1,2}\/\d{4}.*escreveu\s*:/mus',
        '/=+$/mus',
        '/-{10,}/u',
        '/_{10,}/u',
    ];
    $separator_pos = false;
    foreach ($reply_separator_regexes as $regex) {
        if (preg_match($regex, $body, $match, PREG_OFFSET_CAPTURE)) {
            $separator_pos = $match[0][1];
            break;
        }
    }
    if ($separator_pos !== false) {
        $body = substr($body, 0, $separator_pos);
    }
    $body = preg_replace('/^\s*(>|).*/m', '', $body);
    $body = preg_replace('/\n\s*-\s*\n+/u', "\n- ", $body);
    $body = preg_replace('/\n{2,}-/u', "\n-", $body);
    $body = preg_replace('/-\s*\n{2,}/u', "-\n", $body); 
    $body = preg_replace('/-\s*\n+\s*-/', "-\n-", $body);
    $body = preg_replace('/\n{3,}/u', "\n\n", $body);
    $body = preg_replace('/[ \t]+$/m', '', $body);
    $clean_content = trim($body);
    return $clean_content;
}

function nexlifydesk_extract_customer_details($message) {
    if (empty($message)) {
        return ['name' => '', 'email' => '', 'message' => ''];
    }
    
    $decoded_message = nexlifydesk_decode_email_content($message);
    $clean_message = nexlifydesk_extract_clean_email_content($decoded_message);
    
    $name = '';
    $email = '';
    $actual_message = $clean_message;
    
    if (preg_match('/\[Customer Details\]\s*(.*?)\s*\[(?:Message|Reply)\]\s*(.*)/s', $clean_message, $matches)) {
        $customer_section = trim($matches[1]);
        $actual_message = trim($matches[2]);
        
        if (preg_match('/Name:\s*([^\n\r]+)/i', $customer_section, $name_matches)) {
            $name = trim($name_matches[1]);
        }
        if (preg_match('/Email:\s*([^\n\r]+)/i', $customer_section, $email_matches)) {
            $email = trim($email_matches[1]);
        }
    } else {
        if (preg_match('/From:\s*([^<\n]+)\s*<([^>]+)>/', $clean_message, $matches)) {
            $name = trim($matches[1]);
            $email = trim($matches[2]);
        } elseif (preg_match('/Name:\s*([^\n]+)/', $clean_message, $matches)) {
            $name = trim($matches[1]);
        }
        
        if (empty($email) && preg_match('/([a-zA-Z0-9._%+-]+@[a-zA-Z0-9.-]+\.[a-zA-Z]{2,})/', $clean_message, $matches)) {
            $email = $matches[1];
        }
    }
    
    if (empty($name) && !empty($email)) {
        $email_parts = explode('@', $email);
        if (count($email_parts) >= 2) {
            $potential_name = str_replace(['.', '_', '+', '-'], ' ', $email_parts[0]);
            $name = ucwords($potential_name);
        }
    }
    
    return [
        'name' => $name,
        'email' => $email,
        'message' => $actual_message 
    ];
}

/**
 * Enhanced SSL detection for NexlifyDesk
 * Centralized function to check if SSL is properly enabled
 */
function nexlifydesk_check_ssl_enabled() {
    $host = isset($_SERVER['HTTP_HOST']) ? sanitize_text_field(wp_unslash($_SERVER['HTTP_HOST'])) : '';
    $trusted_ssl_domains = [
        'staging.nexlifylabs.com',
        'nexlifylabs.com',
        'www.nexlifylabs.com'
    ];
    
    foreach ($trusted_ssl_domains as $domain) {
        if (strpos($host, $domain) !== false) {
            return true;
        }
    }
    
    $home_url_scheme = wp_parse_url(home_url(), PHP_URL_SCHEME);
    if ($home_url_scheme === 'https') {
        return true;
    }
    
    $site_url_scheme = wp_parse_url(site_url(), PHP_URL_SCHEME);
    if ($site_url_scheme === 'https') {
        return true;
    }
    
    if (is_ssl()) {
        return true;
    }
    
    if (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on') {
        return true;
    }
    
    if (isset($_SERVER['HTTP_X_FORWARDED_PROTO']) && $_SERVER['HTTP_X_FORWARDED_PROTO'] === 'https') {
        return true;
    }
    
    if (isset($_SERVER['HTTP_X_FORWARDED_SSL']) && $_SERVER['HTTP_X_FORWARDED_SSL'] === 'on') {
        return true;
    }
    
    if (isset($_SERVER['REQUEST_SCHEME']) && $_SERVER['REQUEST_SCHEME'] === 'https') {
        return true;
    }
    
    if (defined('FORCE_SSL_ADMIN') && FORCE_SSL_ADMIN) {
        return true;
    }
    
    $home_url = home_url();
    if (strpos($home_url, 'https://') === 0) {
        return true;
    }
    
    return false;
}

/**
 * Generate a secure encryption key for NexlifyDesk
 * This function creates a cryptographically secure 256-bit key
 * 
 * @return string Base64 encoded encryption key
 */
function nexlifydesk_generate_encryption_key() {
    // Generate 32 random bytes (256 bits) for AES-256
    $key_bytes = openssl_random_pseudo_bytes(32);
    
    // Convert to base64 for easy storage in wp-config.php
    $key_base64 = base64_encode($key_bytes);
    
    return $key_base64;
}

/**
 * Display encryption key generation instructions for admin
 * This function shows how to generate and add the key to wp-config.php
 */
function nexlifydesk_show_encryption_key_instructions() {
    if (!current_user_can('manage_options')) {
        return;
    }
    
    $sample_key = nexlifydesk_generate_encryption_key();
    
    echo '<div class="nexlifydesk-encryption-key-instructions">';
    echo '<h3>' . esc_html__('Enhanced Security: Custom Encryption Key', 'nexlifydesk') . '</h3>';
    echo '<p>' . esc_html__('For enhanced security, you can define a custom encryption key in your wp-config.php file.', 'nexlifydesk') . '</p>';
    
    echo '<div class="nexlifydesk-key-generation">';
    echo '<h4>' . esc_html__('Step 1: Generate Your Encryption Key', 'nexlifydesk') . '</h4>';
    echo '<p>' . esc_html__('Click the button below to generate a secure encryption key:', 'nexlifydesk') . '</p>';
    echo '<button type="button" id="generate-encryption-key" class="button button-primary">' . esc_html__('Generate New Key', 'nexlifydesk') . '</button>';
    echo '<div id="generated-key-display" style="display:none; margin-top:10px;">';
    echo '<p><strong>' . esc_html__('Your generated encryption key:', 'nexlifydesk') . '</strong></p>';
    echo '<textarea id="encryption-key-output" readonly style="width:100%; height:60px; font-family:monospace; font-size:12px;"></textarea>';
    echo '<button type="button" id="copy-encryption-key" class="button" style="margin-top:5px;">' . esc_html__('Copy to Clipboard', 'nexlifydesk') . '</button>';
    echo '</div>';
    echo '</div>';
    
    echo '<div class="nexlifydesk-key-installation">';
    echo '<h4>' . esc_html__('Step 2: Add to wp-config.php', 'nexlifydesk') . '</h4>';
    echo '<p>' . esc_html__('Add the following line to your wp-config.php file (before the "stop editing" line):', 'nexlifydesk') . '</p>';
    echo '<pre style="background:#f4f4f4; padding:10px; font-family:monospace; font-size:12px; border:1px solid #ddd;">';
    echo "define('NEXLIFYDESK_ENCRYPTION_KEY', 'YOUR_GENERATED_KEY_HERE');\n";
    echo '</pre>';
    echo '<p><strong>' . esc_html__('Security Note:', 'nexlifydesk') . '</strong> ' . esc_html__('Keep this key secret and make sure to backup your wp-config.php file. If you lose this key, you will need to re-enter all encrypted settings.', 'nexlifydesk') . '</p>';
    echo '</div>';
    
    echo '<div class="nexlifydesk-key-status">';
    echo '<h4>' . esc_html__('Current Status', 'nexlifydesk') . '</h4>';
    if (defined('NEXLIFYDESK_ENCRYPTION_KEY')) {
        echo '<p style="color:green;"><span class="dashicons dashicons-yes"></span> ' . esc_html__('Custom encryption key is active', 'nexlifydesk') . '</p>';
        echo '<p><small>' . esc_html__('Key length:', 'nexlifydesk') . ' ' . esc_html(strlen(NEXLIFYDESK_ENCRYPTION_KEY)) . ' ' . esc_html__('characters', 'nexlifydesk') . '</small></p>';
    } else {
        echo '<p style="color:orange;"><span class="dashicons dashicons-warning"></span> ' . esc_html__('Using default WordPress AUTH_KEY for encryption', 'nexlifydesk') . '</p>';
        echo '<p><small>' . esc_html__('This is functional but less secure. Consider adding a custom key.', 'nexlifydesk') . '</small></p>';
    }
    echo '</div>';
    
    echo '</div>';
    
    ?>
    <?php
}

add_action('wp_ajax_nexlifydesk_generate_encryption_key', 'nexlifydesk_ajax_generate_encryption_key');

function nexlifydesk_ajax_generate_encryption_key() {
    if (
        !current_user_can('manage_options') ||
        !isset($_POST['nonce']) ||
        !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'nexlifydesk_generate_key')
    ) {
        wp_send_json_error('Unauthorized');
        return;
    }
    
    try {
        $key = nexlifydesk_generate_encryption_key();
        wp_send_json_success(array('key' => $key));
    } catch (Exception $e) {
        wp_send_json_error('Failed to generate key: ' . $e->getMessage());
    }
}

/**
 * Check server configuration for potential 413 errors
 * 
 * @return array Array of server configuration warnings
 */
function nexlifydesk_check_server_limits() {
    $warnings = array();
    
    $post_max_size = ini_get('post_max_size');
    $upload_max_filesize = ini_get('upload_max_filesize');
    $max_file_uploads = ini_get('max_file_uploads');
    $max_execution_time = ini_get('max_execution_time');
    $memory_limit = ini_get('memory_limit');
    $post_max_bytes = wp_convert_hr_to_bytes($post_max_size);
    $upload_max_bytes = wp_convert_hr_to_bytes($upload_max_filesize);
    $memory_limit_bytes = wp_convert_hr_to_bytes($memory_limit);
    
    if ($post_max_bytes < 10 * 1024 * 1024) { 
        $warnings[] = sprintf(
            // translators: %s: Current post_max_size value (e.g., "2M", "8M").
            __('PHP post_max_size is set to %s. Consider increasing to at least 10M to prevent 413 errors.', 'nexlifydesk'),
            $post_max_size
        );
    }
    
    if ($upload_max_bytes < 5 * 1024 * 1024) {
        $warnings[] = sprintf(
            // translators: %s: Current upload_max_filesize value (e.g., "2M", "8M").
            __('PHP upload_max_filesize is set to %s. Consider increasing to at least 5M.', 'nexlifydesk'),
            $upload_max_filesize
        );
    }
    
    if ($max_file_uploads < 5) {
        $warnings[] = sprintf(
            // translators: %d: Current max_file_uploads value.
            __('PHP max_file_uploads is set to %d. Consider increasing to at least 5.', 'nexlifydesk'),
            $max_file_uploads
        );
    }
    
    if ($max_execution_time < 60 && $max_execution_time > 0) {
        $warnings[] = sprintf(
            // translators: %d: Current max_execution_time value in seconds.
            __('PHP max_execution_time is set to %d seconds. Consider increasing to at least 60 for large uploads.', 'nexlifydesk'),
            $max_execution_time
        );
    }

    if ($memory_limit_bytes < 64 * 1024 * 1024) {
        $warnings[] = sprintf(
            // translators: %s: Current memory_limit value (e.g., "32M", "64M").
            __('PHP memory_limit is set to %s. Consider increasing to at least 64M.', 'nexlifydesk'),
            $memory_limit
        );
    }
    
    return $warnings;
}

/**
 * Display admin notice for server configuration issues
 */
function nexlifydesk_admin_server_warnings() {
    if (!current_user_can('manage_options')) {
        return;
    }
    
    $warnings = nexlifydesk_check_server_limits();
    
    if (!empty($warnings)) {
        echo '<div class="notice notice-warning is-dismissible">';
        echo '<p><strong>' . esc_html__('NexlifyDesk Server Configuration Warnings:', 'nexlifydesk') . '</strong></p>';
        echo '<ul>';
        foreach ($warnings as $warning) {
            echo '<li>' . esc_html($warning) . '</li>';
        }
        echo '</ul>';
        echo '<p>' . esc_html__('These settings may cause "413 Request Entity Too Large" errors during ticket submission with file uploads.', 'nexlifydesk') . '</p>';
        echo '</div>';
    }
}

add_action('admin_notices', 'nexlifydesk_admin_server_warnings');

/**
 * @param string $content The email content to decode
 * @return string The decoded content
 */
function nexlifydesk_decode_email_content($content) {
    if (empty($content)) {
        return $content;
    }
    
    if (preg_match('/=[0-9A-F]{2}/', $content)) {
        $decoded = quoted_printable_decode($content);
        if ($decoded !== false) {
            $content = $decoded;
        }
    }
    
    if (preg_match('/^[A-Za-z0-9+\/]*={0,2}$/', trim($content)) && strlen($content) > 20) {
        $decoded = base64_decode($content, true);
        if ($decoded !== false && mb_check_encoding($decoded, 'UTF-8')) {
            $content = $decoded;
        }
    }
    
    if (function_exists('mb_convert_encoding')) {
        $encoding = mb_detect_encoding($content, ['UTF-8', 'ISO-8859-1', 'Windows-1252', 'ASCII'], true);
        if ($encoding && $encoding !== 'UTF-8') {
            $content = mb_convert_encoding($content, 'UTF-8', $encoding);
        }
    }
    
    $content = html_entity_decode($content, ENT_QUOTES | ENT_HTML5, 'UTF-8');
    
    $content = preg_replace('/[\x00-\x08\x0B\x0C\x0E-\x1F\x7F]/', '', $content);
    
    return $content;
}

/**
 * 
 * @param string $email_content The raw email content
 * @param string $from_email The sender's email address
 * @return string The cleaned and decoded email content
 */
function nexlifydesk_decode_and_extract_email_content($email_content, $from_email = '') {
    if (empty($email_content)) {
        return $email_content;
    }
    
    $decoded_content = nexlifydesk_decode_email_content($email_content);
    $clean_content = nexlifydesk_extract_clean_email_content($decoded_content, $from_email);
    
    return $clean_content;
}