<?php

if (!defined('ABSPATH')) {
    exit;
}

if (!function_exists('nexlifydesk_extract_customer_details')) {
    require_once __DIR__ . '/helpers.php';
}

$start = microtime(true);

class NexlifyDesk_Tickets {
    public static function init() {
        add_action('init', array(__CLASS__, 'register_post_types'));
    }
    
    public static function register_post_types() {
        // Register custom post type for ticket attachments if needed, required for future enhancements.
    }
    
    public static function create_ticket($data) {
        global $wpdb;

        $defaults = array(
            'user_id' => get_current_user_id(),
            'subject' => '',
            'message' => '',
            'category_id' => 1,
            'priority' => 'medium',
            'status' => 'open',
            'source' => 'web',
            'attachments' => array()
        );

        $data = wp_parse_args($data, $defaults);

        if (empty($data['subject']) || empty($data['message'])) {
            return new WP_Error('missing_fields', __('Subject and message are required.', 'nexlifydesk'));
        }

        $user_id = $data['user_id'];
        $email_address = '';
        
        if ($user_id > 0) {
            $user = get_userdata($user_id);
            $email_address = $user ? $user->user_email : '';
        } else {
            if (isset($data['email'])) {
                $email_address = $data['email'];
            } else {
                if (function_exists('nexlifydesk_extract_customer_details')) {
                    $customer_details = nexlifydesk_extract_customer_details($data['message']);
                    $email_address = $customer_details['email'] ?: 'unknown@example.com';
                } else {
                    $email_address = 'unknown@example.com';
                }
            }
        }
        
        $skip_rate_limit = false;
        if ($user_id > 0) {
            $user = get_userdata($user_id);
            if ($user && (in_array('administrator', $user->roles) || in_array('nexlifydesk_agent', $user->roles))) {
                $skip_rate_limit = true;
            }
        }
        
        if (!$skip_rate_limit && class_exists('NexlifyDesk_Rate_Limiter')) {
            if (NexlifyDesk_Rate_Limiter::is_rate_limited($user_id, $email_address)) {
                $error_message = NexlifyDesk_Rate_Limiter::get_rate_limit_error_message($user_id, $email_address);
                return new WP_Error('rate_limit_exceeded', $error_message);
            }
        }

        $data['email'] = $email_address;

        if (!empty($data['source']) && $data['source'] === 'email' && $user_id == 0) {
            usleep(100000); // 100ms delay
        }

        $existing_ticket = self::check_for_duplicate_ticket($data);
        if ($existing_ticket) {
            $reply_data = array(
                'ticket_id' => $existing_ticket->id,
                'message' => $data['message'],
                'user_id' => $data['user_id'],
                'is_admin_reply' => 0,
                'attachments' => isset($data['attachments']) ? $data['attachments'] : array()
            );
            
            $reply_id = self::add_reply($reply_data);
            
            if (!is_wp_error($reply_id)) {
                $current_time = current_time('mysql');
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Custom table requires direct query
                $wpdb->update(
                    NexlifyDesk_Database::get_table('tickets'),
                    array('updated_at' => $current_time),
                    array('id' => $existing_ticket->id),
                    array('%s'),
                    array('%d')
                );
                
                $existing_ticket->is_duplicate = true;
                $existing_ticket->new_reply_id = $reply_id;
                return $existing_ticket;
            }
        }

        $ticket_id = NexlifyDesk_Database::generate_ticket_id();
        $current_time = current_time('mysql');
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Custom table requires direct query
        $result = $wpdb->insert(
            NexlifyDesk_Database::get_table('tickets'),
            array(
                'ticket_id' => $ticket_id,
                'user_id' => $data['user_id'],
                'category_id' => $data['category_id'],
                'subject' => sanitize_text_field($data['subject']),
                'message' => wp_kses_post($data['message']),
                'priority' => sanitize_text_field($data['priority']),
                'status' => sanitize_text_field($data['status']),
                'source' => sanitize_text_field($data['source']),
                'created_at' => $current_time,
                'updated_at' => $current_time
            ),
            array('%s', '%d', '%d', '%s', '%s', '%s', '%s', '%s', '%s', '%s')
        );

        if (!$result) {
            return new WP_Error('db_error', __('Could not create ticket.', 'nexlifydesk'));
        }

        $new_ticket_id = $wpdb->insert_id;
        
        if (!empty($data['email'])) {
            update_post_meta($new_ticket_id, 'customer_email', $data['email']);
        }
        
        if (!empty($data['attachments'])) {
            self::handle_attachments($data['attachments'], $new_ticket_id, null, $data['user_id']);
        }
        
        $ticket = self::get_ticket($new_ticket_id);

        if (!$skip_rate_limit && class_exists('NexlifyDesk_Rate_Limiter')) {
            NexlifyDesk_Rate_Limiter::record_ticket_creation($user_id, $email_address);
        }

        register_shutdown_function(function() use ($ticket, $data) {
            if (!empty($data['source']) && $data['source'] === 'email') {
                NexlifyDesk_Tickets::send_email_channel_notification($ticket, 'new_ticket');
            } else {
                NexlifyDesk_Tickets::send_notification($ticket, 'new_ticket');
            }
        });


        $settings = get_option('nexlifydesk_settings');
        if (!empty($settings['auto_assign'])) {
            $assigned_user_id = self::get_best_assignee();
            
            if ($assigned_user_id) {
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Custom table requires direct query
                $wpdb->update(
                    NexlifyDesk_Database::get_table('tickets'),
                    array('assigned_to' => $assigned_user_id),
                    array('id' => $new_ticket_id),
                    array('%d'),
                    array('%d')
                );
                $ticket->assigned_to = $assigned_user_id;
                
                wp_cache_delete('nexlifydesk_ticket_' . intval($new_ticket_id));
                
                $assigned_user = get_userdata($assigned_user_id);
                $user_role = in_array('nexlifydesk_agent', $assigned_user->roles) ? 'Agent' : 'Administrator';
            } 
        }

        wp_cache_flush_group('nexlifydesk_tickets_grid');

        return $ticket;
    }
    
    public static function get_ticket($id) {
        global $wpdb;

        $cache_key = 'nexlifydesk_ticket_' . intval($id);
        $ticket = wp_cache_get($cache_key);

        if (false === $ticket) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Custom table requires direct query
            $ticket = $wpdb->get_row(
                $wpdb->prepare(
                    "SELECT * FROM {$wpdb->prefix}nexlifydesk_tickets WHERE id = %d",
                    $id
                )
            );

            wp_cache_set($cache_key, $ticket, '', 300);
        }

        if (!$ticket) {
            return null;
        }

        $ticket->attachments = self::get_attachments($ticket->id);
        $ticket->replies = self::get_replies($ticket->id);
        $ticket->category = self::get_category($ticket->category_id);

        return $ticket;
    }
    
    public static function get_user_tickets($user_id, $status = null) {
        global $wpdb;
    
        $cache_key = 'nexlifydesk_user_tickets_' . intval($user_id) . '_' . (string)$status;
        $tickets = wp_cache_get($cache_key);
    
        if ($tickets === false) {
            $table_name = NexlifyDesk_Database::get_table('tickets');
            $query = "SELECT * FROM `{$wpdb->prefix}nexlifydesk_tickets` WHERE user_id = %d";
            $params = array($user_id);
    
            if ($status) {
                $query .= " AND status = %s";
                $params[] = $status;
            }
    
            $query .= " ORDER BY created_at DESC";
    
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Table name is safe and controlled
            $tickets = $wpdb->get_results(
                // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared -- Table name is safe and controlled
                $wpdb->prepare($query, ...$params)
            );
    
            foreach ($tickets as $ticket) {
                $ticket->category = self::get_category($ticket->category_id);
            }
    
            wp_cache_set($cache_key, $tickets, '', 300);
        }
    
        return $tickets;
    }

    /**
     * Get tickets assigned to a specific user.
     *
     * @param int $user_id The user ID.
     * @param string|null $status Optional. Filter by status.
     * @return array Array of ticket objects.
     */
    public static function get_assigned_tickets($user_id, $status = null) {
        global $wpdb;

        $cache_key = 'nexlifydesk_assigned_tickets_' . intval($user_id) . '_' . (string)$status;
        $tickets = wp_cache_get($cache_key);

        if ($tickets === false) {
            $query = "SELECT * FROM `{$wpdb->prefix}nexlifydesk_tickets` WHERE assigned_to = %d";
            $params = array($user_id);

            if ($status) {
                $query .= " AND status = %s";
                $params[] = $status;
            }

            $query .= " ORDER BY created_at DESC";

            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Table name is safe and controlled
            $tickets = $wpdb->get_results(
                // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared
                $wpdb->prepare($query, ...$params)
            );

            foreach ($tickets as $ticket) {
                $ticket->category = self::get_category($ticket->category_id);
            }

            wp_cache_set($cache_key, $tickets, '', 300);
        }

        return $tickets;
    }
    
    public static function add_reply($data) {
        global $wpdb;
        
        $defaults = array(
            'ticket_id' => 0,
            'user_id' => get_current_user_id(),
            'message' => '',
            'is_admin_reply' => current_user_can('manage_options'),
            'is_internal_note' => false,
            'source' => 'web' 
        );
        
        $data = wp_parse_args($data, $defaults);
        
        if (empty($data['message'])) {
            return new WP_Error('missing_message', __('Message is required.', 'nexlifydesk'));
        }
        
        if ($data['source'] === 'web' && $data['user_id'] > 0) {
            $user = get_userdata($data['user_id']);
            if ($user && (in_array('administrator', $user->roles) || in_array('nexlifydesk_agent', $user->roles))) {
                // Admin/agent reply from web interface
            }
        }
        
        $current_time = current_time('mysql');
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Table name is safe and controlled
        $result = $wpdb->insert(
            NexlifyDesk_Database::get_table('replies'),
            array(
                'ticket_id' => $data['ticket_id'],
                'user_id' => $data['user_id'],
                'message' => wp_kses_post($data['message']),
                'created_at' => $current_time,
                'is_admin_reply' => $data['is_admin_reply'],
                'is_internal_note' => $data['is_internal_note'] ? 1 : 0
            ),
            array('%d', '%d', '%s', '%s', '%d', '%d')
        );
        
        if (!$result) {
            return new WP_Error('db_error', __('Could not add reply.', 'nexlifydesk'));
        }
        
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Table name is safe and controlled
        $wpdb->update(
        NexlifyDesk_Database::get_table('tickets'),
        array('updated_at' => $current_time),
        array('id' => $data['ticket_id']),
        array('%s'),
        array('%d')
    );

    wp_cache_delete('nexlifydesk_ticket_' . intval($data['ticket_id']));
    wp_cache_delete('nexlifydesk_user_tickets_' . intval($data['user_id']));
    wp_cache_delete('nexlifydesk_ticket_replies_' . intval($data['ticket_id']) . '_all');
    wp_cache_delete('nexlifydesk_ticket_replies_' . intval($data['ticket_id']) . '_public');
        
        $reply_id = $wpdb->insert_id;
        
        if (!empty($data['source']) && $data['source'] === 'email' && class_exists('NexlifyDesk_Rate_Limiter')) {
            $email_address = '';
            if ($data['user_id'] > 0) {
                $user = get_userdata($data['user_id']);
                $email_address = $user ? $user->user_email : '';
            } else {
                $ticket = self::get_ticket($data['ticket_id']);
                if ($ticket) {
                    $email_address = get_post_meta($ticket->id, 'customer_email', true);
                }
            }
            
            if (!empty($email_address)) {
                NexlifyDesk_Rate_Limiter::record_email_activity($data['user_id'], $email_address);
            }
        }
        
        if (!empty($data['attachments'])) {
            self::handle_attachments($data['attachments'], $data['ticket_id'], $reply_id, $data['user_id']);
        }
        
        $ticket = self::get_ticket($data['ticket_id']);
        
        if (!$data['is_internal_note']) {
            register_shutdown_function(function() use ($ticket, $reply_id, $data) {
                if (!empty($data['source']) && $data['source'] === 'email') {
                    NexlifyDesk_Tickets::send_email_channel_notification($ticket, 'new_reply', $reply_id);
                } else {
                    NexlifyDesk_Tickets::send_notification($ticket, 'new_reply', $reply_id);
                }
            });
        }

        self::clear_all_ticket_caches();
        
        $ticket = self::get_ticket($data['ticket_id']);
        if ($ticket && !$data['is_internal_note']) {
            $users_to_mark_unread = array();
            
            if ($ticket->assigned_to && $ticket->assigned_to != $data['user_id']) {
                $users_to_mark_unread[] = $ticket->assigned_to;
            }
            
            $admin_users = get_users(array('role' => 'administrator'));
            foreach ($admin_users as $admin_user) {
                if ($admin_user->ID != $data['user_id'] && !in_array($admin_user->ID, $users_to_mark_unread)) {
                    $users_to_mark_unread[] = $admin_user->ID;
                }
            }
            
            if (class_exists('NexlifyDesk_Users')) {
                $all_agents = get_users(array('role' => 'nexlifydesk_agent'));
                foreach ($all_agents as $agent) {
                    if ($agent->ID != $data['user_id'] && 
                        !in_array($agent->ID, $users_to_mark_unread) &&
                        NexlifyDesk_Users::agent_can('nexlifydesk_view_all_tickets', $agent->ID)) {
                        $users_to_mark_unread[] = $agent->ID;
                    }
                }
            }
            
            if (!empty($users_to_mark_unread)) {
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Custom table requires direct query
                $wpdb->query(
                    $wpdb->prepare(
                        "DELETE FROM {$wpdb->prefix}nexlifydesk_ticket_reads 
                         WHERE ticket_id = %d AND user_id IN (" . implode(',', array_fill(0, count($users_to_mark_unread), '%d')) . ")",
                        $data['ticket_id'],
                        ...$users_to_mark_unread
                    )
                );
            }
        }

        return $reply_id;
    }
    
    public static function get_replies($ticket_id, $include_internal = null) {
        global $wpdb;
        
        if ($include_internal === null) {
            $include_internal = current_user_can('manage_options') || current_user_can('nexlifydesk_agent');
        }
        
        $cache_key = 'nexlifydesk_ticket_replies_' . intval($ticket_id) . '_' . ($include_internal ? 'all' : 'public');
        $replies = wp_cache_get($cache_key);
        
        if (false === $replies) {
            $query = "SELECT r.*, u.display_name, u.user_email 
                FROM `{$wpdb->prefix}nexlifydesk_replies` r 
                LEFT JOIN {$wpdb->users} u ON r.user_id = u.ID 
                WHERE r.ticket_id = %d";
            $query_params = array($ticket_id);

            if (!$include_internal) {
                $query .= " AND (r.is_internal_note IS NULL OR r.is_internal_note = 0)";
            }

            $query .= " ORDER BY r.created_at ASC";

            // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery -- Custom table query with safe table names
            $replies = $wpdb->get_results($wpdb->prepare($query, ...$query_params));
            
            foreach ($replies as $reply) {
                $reply->attachments = self::get_attachments($ticket_id, $reply->id);
            }
            
            wp_cache_set($cache_key, $replies, '', 300);
        }
        
        return $replies;
    }
    
    public static function handle_attachments($attachments, $ticket_id, $reply_id = null, $user_id = null) {
        if (empty($attachments)) return;
        
        if (!$user_id) {
            $user_id = get_current_user_id();
        }
        
        $upload_dir = wp_upload_dir();
        $nexlifydesk_dir = $upload_dir['basedir'] . '/nexlifydesk';
        
        if (!file_exists($nexlifydesk_dir)) {
            wp_mkdir_p($nexlifydesk_dir);
        }
        
        $settings = get_option('nexlifydesk_settings', array());
        $max_size = isset($settings['max_file_size']) ? (int)$settings['max_file_size'] * 1024 * 1024 : 2 * 1024 * 1024;
        $allowed_types = isset($settings['allowed_file_types']) ? explode(',', $settings['allowed_file_types']) : array('jpg', 'jpeg', 'png', 'pdf');
        
        if (isset($attachments['name']) && is_array($attachments['name'])) {
            $file_count = count($attachments['name']);
            
            for ($i = 0; $i < $file_count; $i++) {
                if (empty($attachments['name'][$i]) || $attachments['error'][$i] !== 0) {
                    continue;
                }
                
                $file_data = array(
                    'name' => $attachments['name'][$i],
                    'type' => $attachments['type'][$i],
                    'tmp_name' => $attachments['tmp_name'][$i],
                    'error' => $attachments['error'][$i],
                    'size' => $attachments['size'][$i]
                );
                
                $result = self::process_single_attachment($file_data, $ticket_id, $reply_id, $user_id, $max_size, $allowed_types);
            }
        } else {
            foreach ($attachments as $attachment) {
                if (is_array($attachment) && isset($attachment['name'])) {
                    $result = self::process_single_attachment($attachment, $ticket_id, $reply_id, $user_id, $max_size, $allowed_types);
                }
            }
        }
    }

    private static function process_single_attachment($attachment, $ticket_id, $reply_id, $user_id, $max_size, $allowed_types) {
        
        if ($attachment['size'] > $max_size) {
            return false;
        }
        
        $file_ext = strtolower(pathinfo($attachment['name'], PATHINFO_EXTENSION));
        if (!in_array($file_ext, $allowed_types, true)) {
            return false;
        }
        
        if (isset($attachment['tmp_name']) && is_uploaded_file($attachment['tmp_name'])) {
            $finfo = finfo_open(FILEINFO_MIME_TYPE);
            $actual_mime = finfo_file($finfo, $attachment['tmp_name']);
            finfo_close($finfo);
            
            $valid_mime_map = array(
                'jpg' => 'image/jpeg',
                'jpeg' => 'image/jpeg', 
                'png' => 'image/png',
                'pdf' => 'application/pdf',
                'doc' => 'application/msword',
                'docx' => 'application/vnd.openxmlformats-officedocument.wordprocessingml.document'
            );
            
            if (!isset($valid_mime_map[$file_ext]) || $actual_mime !== $valid_mime_map[$file_ext]) {
                return false;
            }
        }

        // Use WordPress file handling
        $file_data = array(
            'name'     => $attachment['name'],
            'type'     => $attachment['type'],
            'tmp_name' => $attachment['tmp_name'],
            'error'    => $attachment['error'],
            'size'     => $attachment['size']
        );

        $upload_result = wp_handle_upload($file_data, array('test_form' => false));
        
        if (!isset($upload_result['error']) && isset($upload_result['url'])) {
            global $wpdb;
            
            $table_name = NexlifyDesk_Database::get_table('attachments');
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Custom table requires direct query
            $result = $wpdb->insert(
                $table_name,
                array(
                    'ticket_id'  => $ticket_id,
                    'reply_id'   => $reply_id,
                    'user_id'    => $user_id,
                    'file_name'  => $attachment['name'],
                    'file_path'  => $upload_result['url'],
                    'file_type'  => $attachment['type'],
                    'file_size'  => $attachment['size'],
                    'created_at' => current_time('mysql')
                ),
                array('%d', '%d', '%d', '%s', '%s', '%s', '%d', '%s')
            );
            
            return $result !== false;
        }
        
        return false;
    }

    public static function get_attachments($ticket_id, $reply_id = null) {
        global $wpdb;
        
        $cache_key = 'nexlifydesk_attachments_' . intval($ticket_id) . '_' . (is_null($reply_id) ? 'null' : intval($reply_id));
        $results = wp_cache_get($cache_key);
        
        if (false === $results) {
            $query = "SELECT * FROM `{$wpdb->prefix}nexlifydesk_attachments` WHERE ticket_id = %d";
            $params = array($ticket_id);
            
            if ($reply_id) {
                $query .= " AND reply_id = %d";
                $params[] = $reply_id;
            } else {
                $query .= " AND reply_id IS NULL";
            }
            
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Table name is safe and controlled
            $results = $wpdb->get_results(
                // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared
                $wpdb->prepare($query, ...$params)
            );
            
            wp_cache_set($cache_key, $results, '', 300);
        }
        
        return $results;
    }
    
    public static function get_categories() {
        global $wpdb;
        
        $cache_key = 'nexlifydesk_categories';
        $categories = wp_cache_get($cache_key);
        
        if (false === $categories) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Custom table requires direct query
            $categories = $wpdb->get_results(
                "SELECT * FROM {$wpdb->prefix}nexlifydesk_categories WHERE is_active = 1 ORDER BY name ASC"
            );
            
            wp_cache_set($cache_key, $categories, '', 3600);
        }
        
        return $categories;
    }
    
    public static function get_category($category_id) {
        global $wpdb;
        
        $cache_key = 'nexlifydesk_category_' . intval($category_id);
        $category = wp_cache_get($cache_key);
        
        if (false === $category) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Custom table requires direct query
            $category = $wpdb->get_row(
                $wpdb->prepare(
                    "SELECT * FROM {$wpdb->prefix}nexlifydesk_categories WHERE id = %d AND is_active = 1",
                    $category_id
                )
            );
            
            // Cache the result for 1 hour (3600 seconds) since categories don't change often
            wp_cache_set($cache_key, $category, '', 3600);
        }
        
        return $category;
    }
    
    public static function update_ticket_status($ticket_id, $status) {
        global $wpdb;
        
        $allowed_statuses = array('open', 'pending', 'in_progress', 'resolved', 'closed');
        if (!in_array($status, $allowed_statuses)) {
            return false;
        }
        
        $ticket_before = self::get_ticket($ticket_id);
        if (!$ticket_before) {
            return false;
        }
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Table name is safe and controlled
        $result = $wpdb->update(
            NexlifyDesk_Database::get_table('tickets'),
            array(
                'status' => $status,
                'updated_at' => current_time('mysql')
            ),
            array('id' => $ticket_id),
            array('%s', '%s'),
            array('%d')
        );
        
        if ($result) {
            // Clear all ticket-related caches to ensure fresh data
            self::clear_all_ticket_caches();
            
            // Invalidate specific ticket cache
            wp_cache_delete('nexlifydesk_ticket_' . intval($ticket_id));
            
            // Invalidate users ticket caches (for all statuses)
            wp_cache_delete('nexlifydesk_user_tickets_' . intval($ticket_before->user_id) . '_');
            wp_cache_delete('nexlifydesk_user_tickets_' . intval($ticket_before->user_id) . '_open');
            wp_cache_delete('nexlifydesk_user_tickets_' . intval($ticket_before->user_id) . '_in_progress');
            wp_cache_delete('nexlifydesk_user_tickets_' . intval($ticket_before->user_id) . '_pending');
            wp_cache_delete('nexlifydesk_user_tickets_' . intval($ticket_before->user_id) . '_resolved');
            wp_cache_delete('nexlifydesk_user_tickets_' . intval($ticket_before->user_id) . '_closed');
            
            // If ticket is assigned, invalidate agent's ticket caches
            if (!empty($ticket_before->assigned_to)) {
                wp_cache_delete('nexlifydesk_assigned_tickets_' . intval($ticket_before->assigned_to) . '_');
                wp_cache_delete('nexlifydesk_assigned_tickets_' . intval($ticket_before->assigned_to) . '_open');
                wp_cache_delete('nexlifydesk_assigned_tickets_' . intval($ticket_before->assigned_to) . '_in_progress');
                wp_cache_delete('nexlifydesk_assigned_tickets_' . intval($ticket_before->assigned_to) . '_pending');
                wp_cache_delete('nexlifydesk_assigned_tickets_' . intval($ticket_before->assigned_to) . '_resolved');
                wp_cache_delete('nexlifydesk_assigned_tickets_' . intval($ticket_before->assigned_to) . '_closed');
                
                $was_closed = in_array($ticket_before->status, array('closed', 'resolved'));
                $is_reopened = !in_array($status, array('closed', 'resolved')) && $was_closed;
                
                if ($is_reopened) {
                    self::mark_ticket_unread_for_assignee($ticket_id, $ticket_before->assigned_to, 'status_change');
                }
            }
            
            $ticket = self::get_ticket($ticket_id);

            $settings = get_option('nexlifydesk_settings');
                if (!empty($settings['status_change_notification'])) {
                    register_shutdown_function(function() use ($ticket) {
                        NexlifyDesk_Tickets::send_notification($ticket, 'status_changed');
                    });
                }

            return true;
        }
        
        return false;
    }
    
    public static function send_notification($ticket, $type, $reply_id = null) {
        $settings = get_option('nexlifydesk_settings');
        if (empty($settings['email_notifications'])) return;

        $user = get_userdata($ticket->user_id);
        $admin_email = get_option('admin_email');
        $admin_notifications_enabled = !empty($settings['admin_email_notifications']);
        $customer_details = nexlifydesk_extract_customer_details($ticket->message);
        $customer_name = $user ? $user->display_name : ($customer_details['name'] ?: 'Guest');
        $customer_email = $user ? $user->user_email : ($customer_details['email'] ?: null);
        
        $headers = self::get_email_headers($ticket);

        $ticket_url = add_query_arg(
            array('ticket_id' => $ticket->ticket_id),
            NexlifyDesk_Admin::get_ticket_page_url()
        );
        $ticket_admin_url = add_query_arg(
            array(
                'page' => 'nexlifydesk_tickets',
                'ticket_id' => $ticket->id,
            ),
            admin_url('admin.php')
        );

        $emailed = array();

        switch ($type) {
            case 'new_ticket':
                // translators: 1: Ticket ID, 2: Ticket subject.
                        $subject = sprintf(__('[#%1$s] %2$s', 'nexlifydesk'), $ticket->ticket_id, $ticket->subject);
                $message = self::get_email_template('new_ticket', $ticket);
                $message = str_replace(
                    array('{ticket_url}', '{ticket_admin_url}'),
                    array(esc_url($ticket_url), esc_url($ticket_admin_url)),
                    $message
                );

                if ($customer_email && !in_array($customer_email, $emailed)) {
                    wp_mail($customer_email, $subject, $message, $headers);
                    $emailed[] = $customer_email;
                }

                if (!empty($ticket->assigned_to)) {
                    $agent = get_userdata($ticket->assigned_to);
                    if ($agent && !in_array($agent->user_email, $emailed)) {
                        wp_mail($agent->user_email, '[Agent] ' . $subject, $message, $headers);
                        $emailed[] = $agent->user_email;
                    }
                }

                if ($admin_notifications_enabled && !in_array($admin_email, $emailed)) {
                    $should_notify_admin = false;
                    
                    if (empty($ticket->assigned_to)) {
                        $should_notify_admin = true;
                    } else {
                        $assigned_user = get_userdata($ticket->assigned_to);
                        if ($assigned_user && in_array('administrator', $assigned_user->roles)) {
                            $should_notify_admin = true;
                        }
                    }
                    
                    if ($should_notify_admin) {
                        wp_mail($admin_email, '[Admin] ' . $subject, $message, $headers);
                        $emailed[] = $admin_email;
                    }
                }
                break;

            case 'new_reply':
                if ($reply_id) {
                    global $wpdb;
                    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Table name is safe and controlled
                    $reply_cache_key = 'nexlifydesk_reply_is_internal_note_' . intval($reply_id);
                    $reply = wp_cache_get($reply_cache_key);

                    if (false === $reply) {
                        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Custom table requires direct query
                        $reply = $wpdb->get_row($wpdb->prepare(
                            "SELECT user_id, is_internal_note FROM `{$wpdb->prefix}nexlifydesk_replies` WHERE id = %d",
                            $reply_id
                        ));
                        wp_cache_set($reply_cache_key, $reply, '', 300);
                    }
                    
                    if ($reply && $reply->is_internal_note) {
                        return;
                    }
                }
                
                // translators: 1: Ticket ID, 2: Ticket subject.
                $subject = sprintf(__('[#%1$s] %2$s', 'nexlifydesk'), $ticket->ticket_id, $ticket->subject);
                $message = self::get_email_template('new_reply', $ticket, $reply_id);
                $message = str_replace(
                    array('{ticket_url}', '{ticket_admin_url}'),
                    array(esc_url($ticket_url), esc_url($ticket_admin_url)),
                    $message
                );

                $reply_user_id = $reply ? $reply->user_id : get_current_user_id();
                $reply_user = get_userdata($reply_user_id);
                $is_agent_reply = $reply_user && (in_array('administrator', $reply_user->roles) || in_array('nexlifydesk_agent', $reply_user->roles));

                if ($is_agent_reply) {
                    if ($customer_email && !in_array($customer_email, $emailed)) {
                        wp_mail($customer_email, $subject, $message, $headers);
                        $emailed[] = $customer_email;
                    }
                } else {
                    if (!empty($ticket->assigned_to)) {
                        $agent = get_userdata($ticket->assigned_to);
                        if ($agent && !in_array($agent->user_email, $emailed)) {
                            wp_mail($agent->user_email, '[Agent] ' . $subject, $message, $headers);
                            $emailed[] = $agent->user_email;
                        }
                    } else if ($admin_notifications_enabled) {
                        if (!in_array($admin_email, $emailed)) {
                            wp_mail($admin_email, '[Admin] ' . $subject, $message, $headers);
                            $emailed[] = $admin_email;
                        }
                    }
                }
                break;

            case 'status_changed':
                // translators: 1: Ticket ID, 2: Ticket subject.
                $subject = sprintf(__('[#%1$s] %2$s - Status Changed', 'nexlifydesk'), $ticket->ticket_id, $ticket->subject);
                $message = self::get_email_template('status_changed', $ticket);
                $message = str_replace(
                    array('{ticket_url}', '{ticket_admin_url}'),
                    array(esc_url($ticket_url), esc_url($ticket_admin_url)),
                    $message
                );
                
                if ($customer_email && !in_array($customer_email, $emailed)) {
                    wp_mail($customer_email, $subject, $message, $headers);
                    $emailed[] = $customer_email;
                }
                
                if (!empty($ticket->assigned_to)) {
                    $agent = get_userdata($ticket->assigned_to);
                    if ($agent && !in_array($agent->user_email, $emailed)) {
                        wp_mail($agent->user_email, '[Agent] ' . $subject, $message, $headers);
                        $emailed[] = $agent->user_email;
                    }
                }
                
                if ($admin_notifications_enabled && !in_array($admin_email, $emailed)) {
                    $should_notify_admin = false;
                    
                    if (empty($ticket->assigned_to)) {
                        $should_notify_admin = true;
                    } else {
                        $assigned_user = get_userdata($ticket->assigned_to);
                        if ($assigned_user && in_array('administrator', $assigned_user->roles)) {
                            $should_notify_admin = true;
                        }
                    }
                    
                    if ($should_notify_admin) {
                        wp_mail($admin_email, '[Admin] ' . $subject, $message, $headers);
                        $emailed[] = $admin_email;
                    }
                }
                break;

            case 'sla_breach':
                // translators: 1: Ticket ID, 2: Ticket subject.  
                $subject = sprintf(__('[#%1$s] SLA Breach: %2$s', 'nexlifydesk'), $ticket->ticket_id, $ticket->subject);
                $message = self::get_email_template('sla_breach', $ticket);
                $message = str_replace(
                    array('{ticket_url}', '{ticket_admin_url}'),
                    array(esc_url($ticket_url), esc_url($ticket_admin_url)),
                    $message
                );
                
                if (!in_array($admin_email, $emailed)) {
                    wp_mail($admin_email, '[Admin] ' . $subject, $message, $headers);
                    $emailed[] = $admin_email;
                }
                
                if (!empty($ticket->assigned_to)) {
                    $agent = get_userdata($ticket->assigned_to);
                    if ($agent && !in_array($agent->user_email, $emailed)) {
                        wp_mail($agent->user_email, '[Agent] ' . $subject, $message, $headers);
                        $emailed[] = $agent->user_email;
                    }
                }
                break;
        }
    }

    private static function get_email_template($template, $ticket, $reply_id = null) {
        $templates = get_option('nexlifydesk_email_templates', array());
        $template_content = isset($templates[$template]) ? $templates[$template] : '';

        if (empty($template_content)) {
            ob_start();
            include NEXLIFYDESK_PLUGIN_DIR . 'templates/emails/' . $template . '.php';
            $template_content = ob_get_clean();
            
            if (empty($template_content)) {
                $template_content = self::get_fallback_email_template($template, $ticket, $reply_id);
            }
            
            return $template_content;
        }

        $user = get_userdata($ticket->user_id);
        
        $customer_details = nexlifydesk_extract_customer_details($ticket->message);
        $customer_name = $user ? $user->display_name : ($customer_details['name'] ?: 'Guest');
        $customer_email = $user ? $user->user_email : ($customer_details['email'] ?: '');
        $clean_message = $customer_details['message'] ?: $ticket->message;
        
        $placeholders = array(
            '{ticket_id}'   => esc_html($ticket->ticket_id),
            '{subject}'     => esc_html($ticket->subject),
            '{message}'     => wpautop(wp_kses_post($clean_message)),
            '{user_name}'   => esc_html($customer_name),
            '{user_email}'  => esc_html($customer_email),
            '{status}'      => esc_html(ucfirst($ticket->status)),
            '{priority}'    => esc_html(ucfirst($ticket->priority)),
            '{category}'    => esc_html(($ticket->category_id ? (NexlifyDesk_Tickets::get_category($ticket->category_id)->name ?? '') : '')),
            '{created_at}'  => esc_html(gmdate(get_option('date_format') . ' ' . get_option('time_format'), strtotime($ticket->created_at))),
            '{updated_at}'  => esc_html(gmdate(get_option('date_format') . ' ' . get_option('time_format'), strtotime($ticket->updated_at))),
            '{ticket_url}' => esc_url(
                add_query_arg(
                    array('ticket_id' => $ticket->ticket_id),
                    NexlifyDesk_Admin::get_ticket_page_url()
                )
            ),
            '{ticket_admin_url}' => esc_url(
                add_query_arg(
                    array(
                        'page' => 'nexlifydesk_tickets',
                        'ticket_id' => $ticket->ticket_id,
                    ),
                    admin_url('admin.php')
                )
            ),
        );

        if ($template === 'new_reply' && $reply_id) {
            $reply = null;
            if (method_exists('NexlifyDesk_Tickets', 'get_reply')) {
                $reply = NexlifyDesk_Tickets::get_reply($reply_id);
            }
            if (!$reply) {
                $replies = NexlifyDesk_Tickets::get_replies($ticket->id);
                $reply = end($replies);
            }
            if ($reply) {
                $reply_customer_details = nexlifydesk_extract_customer_details($reply->message);
                $clean_reply_message = $reply_customer_details['message'] ?: $reply->message;
                
                $clean_reply_message = wp_kses_post($clean_reply_message);
                $clean_reply_message = wpautop($clean_reply_message);
                
                $placeholders['{reply_message}'] = $clean_reply_message;
                $reply_user = get_userdata($reply->user_id);
                $reply_customer_name = $reply_user ? $reply_user->display_name : ($reply_customer_details['name'] ?: 'Guest');
                $placeholders['{reply_user_name}'] = esc_html($reply_customer_name);
            }
        }

        $content = strtr($template_content, $placeholders);

        return $content;
    }

    /**
     * Get fallback email template if no custom template is set and file template is empty
     */
    private static function get_fallback_email_template($template, $ticket, $reply_id = null) {
        $user = get_userdata($ticket->user_id);
        $customer_details = nexlifydesk_extract_customer_details($ticket->message);
        $customer_name = $user ? $user->display_name : ($customer_details['name'] ?: 'Guest');
        
        switch ($template) {
            case 'new_ticket':
                return '<p>Hello ' . esc_html($customer_name) . ',</p><p>Your support ticket #' . esc_html($ticket->ticket_id) . ' has been created.</p><p>Subject: ' . esc_html($ticket->subject) . '</p><p>We will get back to you soon.</p>';
            case 'new_reply':
                return '<p>Hello ' . esc_html($customer_name) . ',</p><p>You have a new reply on ticket #' . esc_html($ticket->ticket_id) . '.</p>';
            case 'status_changed':
                return '<p>Hello ' . esc_html($customer_name) . ',</p><p>The status of your ticket #' . esc_html($ticket->ticket_id) . ' has been updated to: ' . esc_html(ucfirst($ticket->status)) . '.</p>';
            case 'sla_breach':
                return '<p>Attention: Ticket #' . esc_html($ticket->ticket_id) . ' has breached its SLA.</p>';
            default:
                return '<p>Email notification for ticket #' . esc_html($ticket->ticket_id) . '</p>';
        }
    }

    /**
     * Get email headers for the ticket
     * @since 1.0.1
     * @param object $ticket The ticket object
     * @return array Email headers array
     */
    private static function get_email_headers($ticket) {
        $admin_email = get_option('admin_email');
        $headers = array(
            'Content-Type: text/html; charset=UTF-8',
            'From: ' . get_bloginfo('name') . ' <' . $admin_email . '>'
        );

        $smtp_plugins_active = (
            is_plugin_active('wp-mail-smtp/wp_mail_smtp.php') ||
            is_plugin_active('easy-wp-smtp/easy-wp-smtp.php') ||
            is_plugin_active('gmail-smtp/main.php') ||
            is_plugin_active('wp-smtp/wp-smtp.php') ||
            function_exists('wp_mail_smtp') ||
            class_exists('WPMailSMTP') ||
            class_exists('Easy_WP_SMTP')
        );

        if (!$smtp_plugins_active) {
            $domain = wp_parse_url(home_url(), PHP_URL_HOST);
            $headers[] = 'Message-ID: <ticket-' . $ticket->ticket_id . '-' . time() . '@' . $domain . '>';
            $headers[] = 'In-Reply-To: <ticket-' . $ticket->ticket_id . '@' . $domain . '>';
            $headers[] = 'References: <ticket-' . $ticket->ticket_id . '@' . $domain . '>';
        }

        return $headers;
    }

    /**
     * Send email channel notification for new tickets or replies
     * @param object $ticket The ticket object
     * @param string $type The notification type ('new_ticket' or 'new_reply')
     * @param int $reply_id Optional reply ID for reply notifications
     */
    public static function send_email_channel_notification($ticket, $type, $reply_id = null) {
        $settings = get_option('nexlifydesk_settings');
        if (empty($settings['email_notifications'])) return;

        $admin_email = get_option('admin_email');
        $admin_notifications_enabled = !empty($settings['admin_email_notifications']);
        
        $customer_email = get_post_meta($ticket->id, 'customer_email', true);
        if (empty($customer_email)) {
            $customer_details = nexlifydesk_extract_customer_details($ticket->message);
            $customer_email = $customer_details['email'] ?: null;
        }
        
        if (empty($customer_email)) {
            return;
        }

        $headers = self::get_email_headers($ticket);

        switch ($type) {
            case 'new_ticket':
                self::send_auto_response($ticket, $customer_email, $headers);
                
                self::notify_admin_about_email_ticket($ticket, $admin_email, $headers, $admin_notifications_enabled);
                break;

            case 'new_reply':
                self::send_raw_email_reply($ticket, $reply_id, $customer_email, $headers);
                break;
        }
    }

    /**
     * Send auto-response to customer for new email tickets
     */
    private static function send_auto_response($ticket, $customer_email, $headers) {
        $templates = get_option('nexlifydesk_email_templates', array());
        
        $auto_response_message = isset($templates['email_auto_response']) && !empty($templates['email_auto_response'])
            ? $templates['email_auto_response'] 
            : self::get_default_auto_response();

        $customer_details = nexlifydesk_extract_customer_details($ticket->message);
        $customer_name = $customer_details['name'] ?: 'Customer';
        
        $placeholders = array(
            '{customer_name}' => esc_html($customer_name),
            '{ticket_id}' => esc_html($ticket->ticket_id),
            '{subject}' => esc_html($ticket->subject),
            '{site_name}' => esc_html(get_bloginfo('name')),
            '{site_url}' => esc_url(home_url()),
            '{admin_email}' => esc_html(get_option('admin_email'))
        );

        $message = strtr($auto_response_message, $placeholders);
        
        // translators: 1: Ticket ID, 2: Ticket subject.
        $subject = sprintf(__('[#%1$s] %2$s', 'nexlifydesk'), $ticket->ticket_id, $ticket->subject);

        wp_mail($customer_email, $subject, $message, $headers);
    }

    /**
     * Send raw email reply to customer (as if agent is replying directly)
     */
    private static function send_raw_email_reply($ticket, $reply_id, $customer_email, $headers) {
        global $wpdb;
        
        if (!$reply_id) return;

        $reply_cache_key = 'nexlifydesk_reply_' . intval($reply_id);
        $reply = wp_cache_get($reply_cache_key);
        if (false === $reply) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Custom table requires direct query
            $reply = $wpdb->get_row($wpdb->prepare(
                "SELECT r.*, u.display_name, u.user_email 
                 FROM `{$wpdb->prefix}nexlifydesk_replies` r 
                 LEFT JOIN {$wpdb->users} u ON r.user_id = u.ID 
                 WHERE r.id = %d",
                $reply_id
            ));
            wp_cache_set($reply_cache_key, $reply, '', 300);
        }

        if (!$reply) return;

        $reply_user = get_userdata($reply->user_id);
        $is_agent_reply = $reply_user && (
            in_array('administrator', $reply_user->roles) || 
            in_array('nexlifydesk_agent', $reply_user->roles)
        );

        if (!$is_agent_reply) {
            $settings = get_option('nexlifydesk_settings');
            $admin_notifications_enabled = !empty($settings['admin_email_notifications']);
            self::notify_admin_about_email_reply($ticket, $reply, $headers, $admin_notifications_enabled);
            return;
        }

        $clean_message = $reply->message;
        $clean_message = preg_replace('/\[Customer Details\].*?\[Message\]/s', '', $clean_message);
        $clean_message = preg_replace('/\[Customer Details\].*?\[Reply\]/s', '', $clean_message);
        $clean_message = trim($clean_message);
        
        if (strpos($clean_message, '<') !== false) {
            $clean_message = wp_strip_all_tags($clean_message);
            $clean_message = html_entity_decode($clean_message, ENT_QUOTES, 'UTF-8');
        }

        $from_name = $reply_user->display_name ?: get_bloginfo('name');
        $from_email = $reply_user->user_email ?: get_option('admin_email');
        $reply_headers = self::get_email_headers($ticket);
        $reply_headers[1] = 'From: ' . $from_name . ' <' . $from_email . '>';
        
        // translators: 1: Ticket ID, 2: Ticket subject.
        $subject = sprintf(__('[#%1$s] %2$s', 'nexlifydesk'), $ticket->ticket_id, $ticket->subject);

        wp_mail($customer_email, $subject, $clean_message, $reply_headers);
    }

    /**
     * Notify admin/agents about new email ticket
     */
    private static function notify_admin_about_email_ticket($ticket, $admin_email, $headers, $admin_notifications_enabled = true) {
        $customer_details = nexlifydesk_extract_customer_details($ticket->message);
        $customer_name = $customer_details['name'] ?: 'Unknown Customer';
        $customer_email = $customer_details['email'] ?: 'Unknown Email';
        $clean_message = $customer_details['message'] ?: $ticket->message;

        $is_html = false;
        if (is_array($headers)) {
            foreach ($headers as $header) {
                if (stripos($header, 'Content-Type: text/html') !== false) {
                    $is_html = true;
                    break;
                }
            }
        } else {
            $is_html = (stripos($headers, 'Content-Type: text/html') !== false);
        }

        $formatted_message = $clean_message;
        if ($is_html) {
            $formatted_message = nl2br(esc_html($formatted_message));
        } else {
            $formatted_message = str_replace(array("\r\n", "\r"), "\n", $formatted_message);
        }

        if ($is_html) {
            $admin_message = sprintf(
                "<strong>New support ticket received via email:</strong><br><br>" .
                "<strong>Ticket ID:</strong> %s<br>" .
                "<strong>From:</strong> %s &lt;%s&gt;<br>" .
                "<strong>Subject:</strong> %s<br>" .
                "<strong>Priority:</strong> %s<br><br>" .
                "<strong>Message:</strong><br>%s<br><br>" .
                "<a href=\"%s\">View ticket</a>",
                esc_html($ticket->ticket_id),
                esc_html($customer_name),
                esc_html($customer_email),
                esc_html($ticket->subject),
                esc_html(ucfirst($ticket->priority)),
                $formatted_message,
                esc_url(add_query_arg(
                    array(
                        'page' => 'nexlifydesk_tickets',
                        'ticket_id' => $ticket->id,
                    ),
                    admin_url('admin.php')
                ))
            );
        } else {
            $admin_message = sprintf(
                "New support ticket received via email:\n\n" .
                "Ticket ID: %s\n" .
                "From: %s <%s>\n" .
                "Subject: %s\n" .
                "Priority: %s\n\n" .
                "Message:\n%s\n\n" .
                "View ticket: %s",
                $ticket->ticket_id,
                $customer_name,
                $customer_email,
                $ticket->subject,
                ucfirst($ticket->priority),
                $formatted_message,
                add_query_arg(
                    array(
                        'page' => 'nexlifydesk_tickets',
                        'ticket_id' => $ticket->id,
                    ),
                    admin_url('admin.php')
                )
            );
        }

        // translators: 1: Ticket ID, 2: Ticket subject.
        $subject = sprintf(__('[New Email Ticket] [#%1$s] %2$s', 'nexlifydesk'), $ticket->ticket_id, $ticket->subject);

        if ($admin_notifications_enabled) {
            $should_notify_admin = false;
            
            if (empty($ticket->assigned_to)) {
                $should_notify_admin = true;
            } else {
                $assigned_user = get_userdata($ticket->assigned_to);
                if ($assigned_user && in_array('administrator', $assigned_user->roles)) {
                    $should_notify_admin = true;
                }
            }
            
            if ($should_notify_admin) {
                wp_mail($admin_email, $subject, $admin_message, $headers);
            }
        }

        if (!empty($ticket->assigned_to)) {
            $agent = get_userdata($ticket->assigned_to);
            if ($agent && $agent->user_email !== $admin_email) {
                wp_mail($agent->user_email, '[Agent] ' . $subject, $admin_message, $headers);
            }
        }
    }

    /**
     * Notify admin/agents about customer email reply
     */
    private static function notify_admin_about_email_reply($ticket, $reply, $headers, $admin_notifications_enabled = true) {
        $admin_email = get_option('admin_email');
        
        $customer_details = nexlifydesk_extract_customer_details($reply->message);
        $customer_name = $customer_details['name'] ?: 'Unknown Customer';
        $customer_email = $customer_details['email'] ?: 'Unknown Email';
        $clean_message = $customer_details['message'] ?: $reply->message;

        $is_html = false;
        if (is_array($headers)) {
            foreach ($headers as $header) {
                if (stripos($header, 'Content-Type: text/html') !== false) {
                    $is_html = true;
                    break;
                }
            }
        } else {
            $is_html = (stripos($headers, 'Content-Type: text/html') !== false);
        }

        $formatted_message = $clean_message;
        if ($is_html) {
            $formatted_message = nl2br(esc_html($formatted_message));
        } else {
            $formatted_message = str_replace(array("\r\n", "\r"), "\n", $formatted_message);
        }

        if ($is_html) {
            $admin_message = sprintf(
                "<strong>Customer replied to ticket via email:</strong><br><br>" .
                "<strong>Ticket ID:</strong> %s<br>" .
                "<strong>From:</strong> %s &lt;%s&gt;<br>" .
                "<strong>Subject:</strong> %s<br><br>" .
                "<strong>Reply:</strong><br>%s<br><br>" .
                "<a href=\"%s\">View ticket</a>",
                esc_html($ticket->ticket_id),
                esc_html($customer_name),
                esc_html($customer_email),
                esc_html($ticket->subject),
                $formatted_message,
                esc_url(add_query_arg(
                    array(
                        'page' => 'nexlifydesk_tickets',
                        'ticket_id' => $ticket->id,
                    ),
                    admin_url('admin.php')
                ))
            );
        } else {
            $admin_message = sprintf(
                "Customer replied to ticket via email:\n\n" .
                "Ticket ID: %s\n" .
                "From: %s <%s>\n" .
                "Subject: %s\n\n" .
                "Reply:\n%s\n\n" .
                "View ticket: %s",
                $ticket->ticket_id,
                $customer_name,
                $customer_email,
                $ticket->subject,
                $formatted_message,
                add_query_arg(
                    array(
                        'page' => 'nexlifydesk_tickets',
                        'ticket_id' => $ticket->id,
                    ),
                    admin_url('admin.php')
                )
            );
        }

        // translators: 1: Ticket ID, 2: Ticket subject.
        $subject = sprintf(__('[Customer Reply] [#%1$s] %2$s', 'nexlifydesk'), $ticket->ticket_id, $ticket->subject);

        if ($admin_notifications_enabled) {
            $should_notify_admin = false;
            
            if (empty($ticket->assigned_to)) {
                $should_notify_admin = true;
            } else {
                $assigned_user = get_userdata($ticket->assigned_to);
                if ($assigned_user && in_array('administrator', $assigned_user->roles)) {
                    $should_notify_admin = true;
                }
            }
            
            if ($should_notify_admin) {
                wp_mail($admin_email, $subject, $admin_message, $headers);
            }
        }

        if (!empty($ticket->assigned_to)) {
            $agent = get_userdata($ticket->assigned_to);
            if ($agent && $agent->user_email !== $admin_email) {
                wp_mail($agent->user_email, '[Agent] ' . $subject, $admin_message, $headers);
            }
        }
    }

    /**
     * Get default auto-response message with HTML formatting
     */
    private static function get_default_auto_response() {
        return '<p>Hello {customer_name},</p>' . "\n\n" .
               '<p>Thank you for contacting us. We have received your support request and have assigned it ticket ID <strong>#{ticket_id}</strong>.</p>' . "\n\n" .
               '<p><strong>Subject:</strong> {subject}</p>' . "\n\n" .
               '<p>Our support team will review your request and get back to you as soon as possible. You can reference this ticket ID in any future correspondence.</p>' . "\n\n" .
               '<p>Best regards,<br>' . "\n" .
               '{site_name} Support Team<br>' . "\n" .
               '<a href="{site_url}">{site_url}</a></p>';
    }

    /**
     * Get plugin image URL
     */
    public static function get_image_url($image_name) {
        return NEXLIFYDESK_PLUGIN_URL . 'assets/images/' . $image_name;
    }

    /**
     * Display status with icon
     */
    public static function status_with_icon($status) {
        $icon_url = self::get_image_url('status/' . $status . '.png');
        return '<span class="nexlifydesk-status status-' . esc_attr($status) . '">' . 
                // phpcs:ignore PluginCheck.CodeAnalysis.ImageFunctions.NonEnqueuedImage -- Plugin asset image, not media library image
               '<img src="' . esc_url($icon_url) . '" alt="' . esc_attr($status) . '" width="16" height="16"> ' . 
               esc_html(ucfirst($status)) . '</span>';
    }

    /**
     * Display priority with icon
     */
    public static function priority_with_icon($priority) {
        $icon_url = self::get_image_url('priority/' . $priority . '.png');
        return '<span class="nexlifydesk-priority priority-' . esc_attr($priority) . '">' . 
        // phpcs:ignore PluginCheck.CodeAnalysis.ImageFunctions.NonEnqueuedImage -- Plugin asset image, not media library image
               '<img src="' . esc_url($icon_url) . '" alt="' . esc_attr($priority) . '" width="16" height="16"> ' . 
               esc_html(ucfirst($priority)) . '</span>';
    }

    /**
     * Get the best user to assign a ticket to
     * Priority: Available agents -> Admin (only if no agents available)
     */
    public static function get_best_assignee() {
        global $wpdb;
        $settings = get_option('nexlifydesk_settings', array());
        $auto_assign_to_admin = !empty($settings['auto_assign_to_admin']);

        $agents = get_users(array(
            'role' => 'nexlifydesk_agent',
            'fields' => array('ID', 'display_name'),
            'orderby' => 'registered',
            'order' => 'ASC'
        ));

        if (!empty($agents)) {
            $least_busy_agent = null;
            $min_tickets = null;
            foreach ($agents as $agent) {
                $user = get_userdata($agent->ID);
                if (!$user || !in_array('nexlifydesk_agent', $user->roles)) {
                    continue;
                }
                $cache_key = 'nexlifydesk_agent_ticket_count_' . intval($agent->ID);
                $count = wp_cache_get($cache_key);
                if ($count === false) {
                    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Custom table requires direct query
                    $count = $wpdb->get_var(
                        $wpdb->prepare(
                            "SELECT COUNT(*) FROM {$wpdb->prefix}nexlifydesk_tickets WHERE assigned_to = %d AND status IN ('open', 'pending')",
                            $agent->ID
                        )
                    );
                    wp_cache_set($cache_key, $count, '', 300);
                }
                if ($min_tickets === null || $count < $min_tickets) {
                    $min_tickets = $count;
                    $least_busy_agent = $agent->ID;
                }
            }
            if ($least_busy_agent) {
                return $least_busy_agent;
            }
        }

        // If no agent, assign to admin only if setting is enabled
        if ($auto_assign_to_admin) {
            $admins = get_users(array('role' => 'administrator', 'fields' => array('ID')));
            $least_busy_admin = null;
            $min_tickets = null;
            foreach ($admins as $admin) {
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Custom table requires direct query
                $count = $wpdb->get_var(
                    $wpdb->prepare(
                        "SELECT COUNT(*) FROM {$wpdb->prefix}nexlifydesk_tickets WHERE assigned_to = %d AND status IN ('open', 'pending')",
                        $admin->ID
                    )
                );
                if ($min_tickets === null || $count < $min_tickets) {
                    $min_tickets = $count;
                    $least_busy_admin = $admin->ID;
                }
            }
            if ($least_busy_admin) {
                return $least_busy_admin;
            }
        }

        // If not allowed to assign to admin, return null (keep unassigned)
        return null;
    }

    /**
     * Reassign orphaned tickets when agents are deleted or deactivated
     */
    public static function handle_orphaned_tickets($user_id = null) {
        global $wpdb;
        $reassigned_count = 0;

        if ($user_id) {
            $cache_key = 'nexlifydesk_orphaned_tickets_user_' . intval($user_id);
            $tickets = wp_cache_get($cache_key);

            if (false === $tickets) {
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Custom table requires direct query
                $tickets = $wpdb->get_results(
                    $wpdb->prepare(
                        "SELECT id, assigned_to FROM {$wpdb->prefix}nexlifydesk_tickets WHERE status IN ('open', 'pending') AND assigned_to = %d",
                        $user_id
                    )
                );
                wp_cache_set($cache_key, $tickets, '', 300);
            }
        } else {
            $cache_key = 'nexlifydesk_orphaned_tickets_all';
            $tickets = wp_cache_get($cache_key);

            if (false === $tickets) {
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Custom table requires direct query
                $tickets = $wpdb->get_results(
                    "SELECT id, assigned_to FROM {$wpdb->prefix}nexlifydesk_tickets WHERE status IN ('open', 'pending') AND assigned_to IS NOT NULL"
                );
                wp_cache_set($cache_key, $tickets, '', 300);
            }
        }

        foreach ($tickets as $ticket) {
            $assigned_user = get_userdata($ticket->assigned_to);

            $needs_reassignment = false;

            if (!$assigned_user) {
                $needs_reassignment = true;
            } elseif (!in_array('nexlifydesk_agent', $assigned_user->roles) && !in_array('administrator', $assigned_user->roles)) {
                $needs_reassignment = true;
            }

            if ($needs_reassignment) {
                $new_assignee = self::get_best_assignee();

                if (!$new_assignee) {
                    $admins = get_users(array('role' => 'administrator', 'fields' => array('ID')));
                    $least_busy_admin = null;
                    $min_tickets = null;
                    foreach ($admins as $admin) {
                        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Custom table requires direct query
                        $count = $wpdb->get_var(
                            $wpdb->prepare(
                                "SELECT COUNT(*) FROM {$wpdb->prefix}nexlifydesk_tickets WHERE assigned_to = %d AND status IN ('open', 'pending')",
                                $admin->ID
                            )
                        );
                        if ($min_tickets === null || $count < $min_tickets) {
                            $min_tickets = $count;
                            $least_busy_admin = $admin->ID;
                        }
                    }
                    $new_assignee = $least_busy_admin;
                }

                if ($new_assignee && $new_assignee != $ticket->assigned_to) {
                    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Custom table requires direct query
                    $wpdb->update(
                        NexlifyDesk_Database::get_table('tickets'),
                        array('assigned_to' => $new_assignee, 'updated_at' => current_time('mysql')),
                        array('id' => $ticket->id),
                        array('%d', '%s'),
                        array('%d')
                    );

                    $new_user = get_userdata($new_assignee);
                    $assignee_type = in_array('nexlifydesk_agent', $new_user->roles) ? 'agent' : 'administrator';

                    $system_note = array(
                        'ticket_id' => $ticket->id,
                        'user_id' => 0,
                        'message' => sprintf(
                            // translators: 1: New assignee's display name, 2: Assignee type (agent or administrator).
                            __('Ticket automatically reassigned to %1$s (%2$s) due to previous agent being unavailable.', 'nexlifydesk'),
                            $new_user->display_name,
                            $assignee_type
                        ),
                        'is_admin_reply' => 1
                    );
                    self::add_reply($system_note);

                    wp_cache_delete('nexlifydesk_ticket_' . $ticket->id);
                    $reassigned_count++;
                }
            }
        }

        return $reassigned_count;
    }
    
    /**
     * Mark ticket as unread for specific user(s) when reassigned or status changes
     * 
     * @param int $ticket_id Ticket ID
     * @param int $new_assignee_id New assignee user ID
     * @param string $reason Reason for marking unread (reassignment, status_change, customer_reply)
     */
    public static function mark_ticket_unread_for_assignee($ticket_id, $new_assignee_id, $reason = 'reassignment') {
        global $wpdb;
        
        if (!$ticket_id || !$new_assignee_id) {
            return;
        }
        
        // Remove read status for the new assignee so they see it as unread
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Custom table requires direct query
        $wpdb->delete(
            NexlifyDesk_Database::get_table('ticket_reads'),
            array(
                'ticket_id' => $ticket_id,
                'user_id' => $new_assignee_id
            ),
            array('%d', '%d')
        );
        
        // Clear cache
        wp_cache_delete('nexlifydesk_ticket_reads_' . $ticket_id);
        wp_cache_flush_group('nexlifydesk_tickets_grid');
    }
    
    /**
     * Mark ticket as read for current user with smart role-based logic
     * 
     * @param int $ticket_id Ticket ID
     * @param int $user_id User ID (optional, defaults to current user)
     */
    public static function mark_ticket_read($ticket_id, $user_id = null) {
        global $wpdb;
        
        if (!$ticket_id) {
            return;
        }
        
        $user_id = $user_id ?: get_current_user_id();
        if (!$user_id) {
            return;
        }
        
        // Get ticket details
        $ticket = self::get_ticket($ticket_id);
        if (!$ticket) {
            return;
        }
        
        $current_user = get_userdata($user_id);
        if (!$current_user) {
            return;
        }
        
        $users_to_mark_read = array($user_id);
        
        $is_agent = in_array('nexlifydesk_agent', $current_user->roles) && !in_array('administrator', $current_user->roles);
        
        if ($is_agent) {
            
            $admin_users = get_users(array('role' => 'administrator'));
            foreach ($admin_users as $admin_user) {
                if (!in_array($admin_user->ID, $users_to_mark_read)) {
                    $users_to_mark_read[] = $admin_user->ID;
                }
            }
            
            if (class_exists('NexlifyDesk_Users')) {
                $all_agents = get_users(array('role' => 'nexlifydesk_agent'));
                foreach ($all_agents as $agent) {
                    if ($agent->ID != $user_id && 
                        !in_array($agent->ID, $users_to_mark_read) &&
                        NexlifyDesk_Users::agent_can('nexlifydesk_view_all_tickets', $agent->ID)) {
                        $users_to_mark_read[] = $agent->ID;
                    }
                }
            }
        }
        
        foreach ($users_to_mark_read as $mark_user_id) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Custom table requires direct query
            $wpdb->replace(
                NexlifyDesk_Database::get_table('ticket_reads'),
                array(
                    'ticket_id' => $ticket_id,
                    'user_id' => $mark_user_id,
                    'read_at' => current_time('mysql')
                ),
                array('%d', '%d', '%s')
            );
        }
        
        // Clear cache
        wp_cache_delete('nexlifydesk_ticket_reads_' . $ticket_id);
        self::clear_all_ticket_caches();
    }
    
    /**
     * Check if user is a supervisor (can view all tickets)
     * 
     * @param int $user_id User ID (optional, defaults to current user)
     * @return bool True if user is supervisor
     */
    public static function is_supervisor($user_id = null) {
        $user_id = $user_id ?: get_current_user_id();
        if (!$user_id) {
            return false;
        }
        
        if (user_can($user_id, 'manage_options')) {
            return true;
        }
        
        if (class_exists('NexlifyDesk_Users')) {
            return NexlifyDesk_Users::agent_can('nexlifydesk_view_all_tickets', $user_id);
        }
        
        return false;
    }

    /**
     * Check if there are any active agents available
     * 
     * @return bool True if agents are available, false otherwise
     */
    public static function has_available_agents() {
        $agents = get_users(array(
            'role' => 'nexlifydesk_agent',
            'fields' => array('ID'),
            'number' => 1
        ));
        
        foreach ($agents as $agent) {
            $user = get_userdata($agent->ID);
            if ($user && in_array('nexlifydesk_agent', $user->roles)) {
                return true;
            }
        }
        
        return false;
    }

    public static function get_ticket_by_ticket_id($ticket_id) {
        global $wpdb;
        $cache_key = 'nexlifydesk_ticket_by_ticket_id_' . sanitize_key($ticket_id);
        $ticket = wp_cache_get($cache_key);

        if (false === $ticket) {
            $table_name = NexlifyDesk_Database::get_table('tickets');

            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Custom table requires direct query
            $ticket = $wpdb->get_row(
                $wpdb->prepare(
                    "SELECT * FROM {$table_name} WHERE ticket_id = %s LIMIT 1",
                    $ticket_id
                )
            );

            if (!$ticket && is_numeric($ticket_id)) {
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Custom table requires direct query
                $ticket = $wpdb->get_row(
                    $wpdb->prepare(
                        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared $table_name is generated by (NexlifyDesk_Database::get_table('tickets')), not user input
                        "SELECT * FROM {$table_name} WHERE ticket_id = %s LIMIT 1", 
                        'T' . $ticket_id
                    )
                );
            }

            if (!$ticket && is_numeric($ticket_id)) {
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Custom table requires direct query
                $ticket = $wpdb->get_row(
                    $wpdb->prepare(
                        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared $table_name is generated by (NexlifyDesk_Database::get_table('tickets')), not user input
                        "SELECT * FROM {$table_name} WHERE id = %d LIMIT 1", 
                        absint($ticket_id)
                    )
                );
            }

            if (!$ticket) {
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Custom table requires direct query
                $existing_ids = $wpdb->get_col("SELECT ticket_id FROM {$table_name} LIMIT 10");
            }

            if ($ticket) {
                $ticket->attachments = self::get_attachments($ticket->id);
                $ticket->replies = self::get_replies($ticket->id);
                $ticket->category = self::get_category($ticket->category_id);
            }

            wp_cache_set($cache_key, $ticket, '', 300);
        }

        return $ticket;
    }
    
    /**
     * @param array $data Ticket data to check for duplicates
     * @return object|false Returns existing ticket if duplicate found, false otherwise
     */
    public static function check_for_duplicate_ticket($data) {
        $settings = get_option('nexlifydesk_settings', array());
        $check_duplicates = isset($settings['check_duplicates']) ? $settings['check_duplicates'] : false;
        
        if (!$check_duplicates) {
            return false;
        }
        
        if (!function_exists('nexlifydesk_find_duplicate_ticket')) {
            require_once dirname(__FILE__) . '/nexlifydesk-functions.php';
        }
        
        if (!function_exists('nexlifydesk_find_duplicate_ticket')) {
            return false;
        }
        
        return nexlifydesk_find_duplicate_ticket($data);
    }

    /**
     * Get tickets for the admin grid view.
     *
     * @param array $args
     * @return array
     */
    public static function get_tickets_for_grid($args = []) {
        global $wpdb;

        $defaults = [
            'status' => 'all',
            'priority' => 'all',
            'search' => '',
            'per_page' => 20,
            'offset' => 0,
        ];
        $args = wp_parse_args($args, $defaults);

        $where_clauses = ['1=1'];
        $query_params = [];
        
        $current_user = wp_get_current_user();
        $is_agent = in_array('nexlifydesk_agent', $current_user->roles) && !in_array('administrator', $current_user->roles);
        
        if ($is_agent) {
            $can_view_all = NexlifyDesk_Users::agent_can('nexlifydesk_view_all_tickets', $current_user->ID);
            
            if (!$can_view_all) {
                $where_clauses[] = 't.assigned_to = %d';
                $query_params[] = $current_user->ID;
            }
        }

        if ($args['status'] !== 'all') {
            $where_clauses[] = 't.status = %s';
            $query_params[] = $args['status'];
        }
        if ($args['priority'] !== 'all') {
            $where_clauses[] = 't.priority = %s';
            $query_params[] = $args['priority'];
        }
        if (!empty($args['search'])) {
            $search_term = '%' . $wpdb->esc_like($args['search']) . '%';
            $where_clauses[] = '(t.subject LIKE %s OR t.message LIKE %s OR u.display_name LIKE %s OR u.user_email LIKE %s OR t.ticket_id LIKE %s)';
            $query_params[] = $search_term;
            $query_params[] = $search_term;
            $query_params[] = $search_term;
            $query_params[] = $search_term;
            $query_params[] = $search_term;
        }

        $where_sql = implode(' AND ', $where_clauses);

        $query_params[] = (int) $args['per_page'];
        $query_params[] = (int) $args['offset'];

        $current_user_id = get_current_user_id();
        
        $agent_view_all = false;
        if ($is_agent) {
            $agent_view_all = NexlifyDesk_Users::agent_can('nexlifydesk_view_all_tickets', $current_user_id);
        }
        
        $is_supervisor = $agent_view_all || current_user_can('manage_options') ? 1 : 0;
        
        $cache_key = 'nexlifydesk_tickets_grid_v7_' . md5(serialize($args) . $current_user_id . ($is_agent ? '_agent_' . ($agent_view_all ? 'all' : 'assigned') : '_admin'));
        $results = wp_cache_get($cache_key, 'nexlifydesk_tickets_grid');
        if ($results === false) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Custom table requires direct query
            $results = $wpdb->get_results(
                $wpdb->prepare(
                    "SELECT t.*, 
                           u.display_name as user_name, 
                           u.user_email as user_email, 
                           a.display_name as assigned_to_display_name,
                           COALESCE(MAX(r.created_at), t.created_at) as last_reply_time,
                           COALESCE(MAX(r.created_at), t.created_at) as last_updated_time,
                           CASE 
                               WHEN NOT EXISTS (
                                   SELECT 1 FROM {$wpdb->prefix}nexlifydesk_ticket_reads tr 
                                   WHERE tr.ticket_id = t.id AND tr.user_id = %d
                               ) THEN 1
                               WHEN EXISTS (
                                   SELECT 1 FROM {$wpdb->prefix}nexlifydesk_replies r_new 
                                   WHERE r_new.ticket_id = t.id 
                                   AND r_new.created_at > COALESCE(
                                       (SELECT read_at FROM {$wpdb->prefix}nexlifydesk_ticket_reads tr_read 
                                        WHERE tr_read.ticket_id = t.id AND tr_read.user_id = %d), 
                                       '1970-01-01'
                                   )
                                   AND r_new.is_internal_note = 0
                               ) THEN 1
                               WHEN t.created_at > COALESCE(
                                   (SELECT read_at FROM {$wpdb->prefix}nexlifydesk_ticket_reads tr_created 
                                    WHERE tr_created.ticket_id = t.id AND tr_created.user_id = %d), 
                                   '1970-01-01'
                               ) THEN 1
                               ELSE 0
                           END as is_unread
                    FROM {$wpdb->prefix}nexlifydesk_tickets t
                    LEFT JOIN {$wpdb->users} u ON t.user_id = u.ID
                    LEFT JOIN {$wpdb->users} a ON t.assigned_to = a.ID
                    LEFT JOIN {$wpdb->prefix}nexlifydesk_replies r ON t.id = r.ticket_id
                    WHERE {$where_sql}
                    GROUP BY t.id
                    ORDER BY is_unread DESC, last_reply_time DESC
                    LIMIT %d OFFSET %d",
                    $current_user_id,
                    $current_user_id,
                    $current_user_id,
                    ...$query_params
                )
            );
            
            if (!empty($results)) {
                foreach ($results as $ticket) {
                    if (!$ticket->user_id || empty($ticket->user_name)) {
                        if (function_exists('nexlifydesk_extract_customer_details')) {
                            $customer_details = nexlifydesk_extract_customer_details($ticket->message);
                            $ticket->user_name = $customer_details['name'] ?: 'Guest';
                            $ticket->user_email = $customer_details['email'] ?: 'N/A';
                        } else {
                            $ticket->user_name = 'Guest';
                            $ticket->user_email = 'N/A';
                        }
                    }
                }
            }
            
            wp_cache_set($cache_key, $results, 'nexlifydesk_tickets_grid', 30);
        }
        
        return $results;
    }

    /**
     * Update ticket priority
     *
     * @param int $ticket_id Ticket ID
     * @param string $priority New priority
     * @return bool Success
     */
    public static function update_ticket_priority($ticket_id, $priority) {
        global $wpdb;
        
        $allowed_priorities = array('low', 'medium', 'high', 'urgent');
        if (!in_array($priority, $allowed_priorities)) {
            return false;
        }
        
        $ticket_before = self::get_ticket($ticket_id);
        if (!$ticket_before) {
            return false;
        }
        
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Table name is safe and controlled
        $result = $wpdb->update(
            NexlifyDesk_Database::get_table('tickets'),
            array(
                'priority' => $priority,
                'updated_at' => current_time('mysql')
            ),
            array('id' => $ticket_id),
            array('%s', '%s'),
            array('%d')
        );
        
        if ($result !== false) {
            // Clear all ticket-related caches to ensure fresh data
            self::clear_all_ticket_caches();
            
            // Clear specific ticket cache
            wp_cache_delete('nexlifydesk_ticket_' . $ticket_id);
            
            return true;
        }
        
        return false;
    }

    /**
     * Assign ticket to an agent
     *
     * @param int $ticket_id Ticket ID
     * @param int $agent_id Agent user ID (0 for unassigned)
     * @return bool Success
     */
    public static function assign_ticket($ticket_id, $agent_id) {
        global $wpdb;
        
        $ticket_before = self::get_ticket($ticket_id);
        if (!$ticket_before) {
            return false;
        }
        
        if ($agent_id > 0) {
            $agent = get_userdata($agent_id);
            if (!$agent) {
                return false;
            }
            
            if (!user_can($agent_id, 'nexlifydesk_manage_tickets') && !user_can($agent_id, 'administrator')) {
                return false;
            }
        }
        
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Table name is safe and controlled
        $result = $wpdb->update(
            NexlifyDesk_Database::get_table('tickets'),
            array(
                'assigned_to' => $agent_id === 0 ? null : $agent_id,
                'updated_at' => current_time('mysql')
            ),
            array('id' => $ticket_id),
            array('%d', '%s'),
            array('%d')
        );
        
        if ($result !== false) {
            self::clear_all_ticket_caches();
            
            wp_cache_delete('nexlifydesk_ticket_' . $ticket_id);
            
            if ($agent_id > 0 && $ticket_before->assigned_to != $agent_id) {
                self::mark_ticket_unread_for_assignee($ticket_id, $agent_id, 'reassignment');
            }
            
            return true;
        }
        
        return false;
    }
    
    /**
     *
     * @param int $last_refresh_timestamp Unix timestamp of last refresh (unused for now)
     * @return array Array of ticket objects with unread status
     */
    public static function get_tickets_for_refresh($last_refresh_timestamp = 0) {
        return self::get_tickets_for_grid(['per_page' => 50]);
    }
    
    /**
     *
     * @param int $ticket_id Ticket ID
     * @param int $user_id User ID
     * @return bool Success
     */
    public static function mark_ticket_as_read($ticket_id, $user_id) {
        global $wpdb;
        
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Custom table requires direct query
        $result = $wpdb->replace(
            $wpdb->prefix . 'nexlifydesk_ticket_reads',
            array(
                'ticket_id' => $ticket_id,
                'user_id' => $user_id,
                'read_at' => current_time('mysql')
            ),
            array('%d', '%d', '%s')
        );
        
        if ($result) {
            // Clear relevant caches
            wp_cache_delete('nexlifydesk_ticket_' . $ticket_id);
            return true;
        }
        
        return false;
    }
    
    /**
     * Delete a ticket permanently from the database
     */
    public static function delete_ticket($ticket_id) {
        global $wpdb;
        
        $ticket = self::get_ticket($ticket_id);
        if (!$ticket) {
            return false;
        }
        
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Custom table requires direct query
        $wpdb->query('START TRANSACTION');
        
        try {
            $attachments = self::get_attachments($ticket_id);
            if (!empty($attachments)) {
                foreach ($attachments as $attachment) {
                    $file_path = wp_upload_dir()['basedir'] . '/nexlifydesk/' . $attachment->file_name;
                    if (file_exists($file_path)) {
                        wp_delete_file($file_path);
                    }
                }
            }
            
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Custom table requires direct query
            $attachments_deleted = $wpdb->delete(
                $wpdb->prefix . 'nexlifydesk_attachments',
                array('ticket_id' => $ticket_id),
                array('%d')
            );
            
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Custom table requires direct query
            $replies_deleted = $wpdb->delete(
                $wpdb->prefix . 'nexlifydesk_replies',
                array('ticket_id' => $ticket_id),
                array('%d')
            );
            
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Custom table requires direct query
            $ticket_deleted = $wpdb->delete(
                $wpdb->prefix . 'nexlifydesk_tickets',
                array('id' => $ticket_id),
                array('%d')
            );
            
            if ($ticket_deleted === false) {
                throw new Exception('Failed to delete ticket from database');
            }
            
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Custom table requires direct query
            $wpdb->query('COMMIT');
            
            wp_cache_delete('nexlifydesk_ticket_' . $ticket_id);
            wp_cache_delete('nexlifydesk_ticket_replies_' . $ticket_id);
            wp_cache_delete('nexlifydesk_attachments_' . $ticket_id);
            
            do_action('nexlifydesk_ticket_deleted', $ticket_id, $ticket);
            
            return true;
            
        } catch (Exception $e) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Custom table requires direct query
            $wpdb->query('ROLLBACK');
            return false;
        }
    }

    /**
     * Clear all ticket-related caches to ensure fresh data
     */
    public static function clear_all_ticket_caches() {
        // Clear ticket list caches for different contexts
        wp_cache_delete('nexlifydesk_tickets_grid');
        wp_cache_delete('nexlifydesk_tickets_admin_grid');
        wp_cache_delete('nexlifydesk_tickets_agent_grid');
        wp_cache_delete('nexlifydesk_tickets_count');
        wp_cache_delete('nexlifydesk_unread_count');
        
        // Clear general caches that might contain ticket data
        wp_cache_flush_group('nexlifydesk');
        
        // Trigger action for other plugins/components to clear their caches
        do_action('nexlifydesk_clear_ticket_caches');
    }

}