<?php
if (!defined('ABSPATH')) {
    exit;
}

class NexlifyDesk_Reports {
    
    public static function get_dashboard_stats() {
        global $wpdb;
        
        $current_user = wp_get_current_user();
        $is_agent = in_array('nexlifydesk_agent', $current_user->roles) && !in_array('administrator', $current_user->roles);
        
        // Check if agent can view all tickets through their position
        $agent_can_view_all = false;
        if ($is_agent) {
            $agent_can_view_all = NexlifyDesk_Users::agent_can('nexlifydesk_view_all_tickets', $current_user->ID);
        }
        
        $cache_key = 'nexlifydesk_dashboard_stats_' . gmdate('Y-m-d-H') . ($is_agent ? '_agent_' . $current_user->ID . ($agent_can_view_all ? '_all' : '_assigned') : '_admin');
        $stats = wp_cache_get($cache_key);
        
        if (false === $stats) {
            $stats = array();
            
            // Build WHERE clause for agents who can't view all tickets
            $where_clause = '';
            $params = array();
            if ($is_agent && !$agent_can_view_all) {
                $where_clause = ' WHERE assigned_to = %d';
                $params[] = $current_user->ID;
            }
            
            // phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery
            // Total tickets
            if ($is_agent && !$agent_can_view_all) {
                $stats['total_tickets'] = $wpdb->get_var(
                    $wpdb->prepare("SELECT COUNT(*) FROM %i WHERE assigned_to = %d", 
                    $wpdb->prefix . 'nexlifydesk_tickets', $current_user->ID)
                );
            } else {
                $stats['total_tickets'] = $wpdb->get_var(
                    $wpdb->prepare("SELECT COUNT(*) FROM %i", $wpdb->prefix . 'nexlifydesk_tickets')
                );
            }
            
            // Active tickets (open + pending + in_progress)
            if ($is_agent && !$agent_can_view_all) {
                $stats['active_tickets'] = $wpdb->get_var(
                    $wpdb->prepare("SELECT COUNT(*) FROM %i WHERE status IN ('open', 'pending', 'in_progress') AND assigned_to = %d", 
                    $wpdb->prefix . 'nexlifydesk_tickets', $current_user->ID)
                );
            } else {
                $stats['active_tickets'] = $wpdb->get_var(
                    $wpdb->prepare("SELECT COUNT(*) FROM %i WHERE status IN ('open', 'pending', 'in_progress')", 
                    $wpdb->prefix . 'nexlifydesk_tickets')
                );
            }
            
            // Closed tickets
            if ($is_agent && !$agent_can_view_all) {
                $stats['closed_tickets'] = $wpdb->get_var(
                    $wpdb->prepare("SELECT COUNT(*) FROM %i WHERE status = 'closed' AND assigned_to = %d", 
                    $wpdb->prefix . 'nexlifydesk_tickets', $current_user->ID)
                );
            } else {
                $stats['closed_tickets'] = $wpdb->get_var(
                    $wpdb->prepare("SELECT COUNT(*) FROM %i WHERE status = 'closed'", 
                    $wpdb->prefix . 'nexlifydesk_tickets')
                );
            }
            
            // Status breakdown
            if ($is_agent && !$agent_can_view_all) {
                $status_results = $wpdb->get_results(
                    $wpdb->prepare("SELECT status, COUNT(*) as count FROM %i WHERE assigned_to = %d GROUP BY status", 
                    $wpdb->prefix . 'nexlifydesk_tickets', $current_user->ID)
                );
            } else {
                $status_results = $wpdb->get_results(
                    $wpdb->prepare("SELECT status, COUNT(*) as count FROM %i GROUP BY status", 
                    $wpdb->prefix . 'nexlifydesk_tickets')
                );
            }
            
            $stats['status_breakdown'] = array();
            foreach ($status_results as $row) {
                $stats['status_breakdown'][$row->status] = (int)$row->count;
            }
            
            // Priority breakdown
            if ($is_agent && !$agent_can_view_all) {
                $priority_results = $wpdb->get_results(
                    $wpdb->prepare("SELECT priority, COUNT(*) as count FROM %i WHERE assigned_to = %d GROUP BY priority", 
                    $wpdb->prefix . 'nexlifydesk_tickets', $current_user->ID)
                );
            } else {
                $priority_results = $wpdb->get_results(
                    $wpdb->prepare("SELECT priority, COUNT(*) as count FROM %i GROUP BY priority", 
                    $wpdb->prefix . 'nexlifydesk_tickets')
                );
            }
            // phpcs:enable
            
            $stats['priority_breakdown'] = array();
            foreach ($priority_results as $row) {
                $stats['priority_breakdown'][$row->priority] = (int)$row->count;
            }
            
            // Average response time
            $stats['avg_response_time'] = self::calculate_avg_response_time(($is_agent && !$agent_can_view_all) ? $current_user->ID : null);
            
            // Monthly data for chart
            $stats['monthly_data'] = self::get_monthly_ticket_data(($is_agent && !$agent_can_view_all) ? $current_user->ID : null);
            
            // Agent performance
            $stats['agent_performance'] = self::get_agent_performance(($is_agent && !$agent_can_view_all) ? $current_user->ID : null);
            
            // Recent activity
            $stats['recent_activity'] = self::get_recent_activity(($is_agent && !$agent_can_view_all) ? $current_user->ID : null);
            
            wp_cache_set($cache_key, $stats, 'nexlifydesk', HOUR_IN_SECONDS);
        }
        
        return $stats;
    }
    
    private static function calculate_avg_response_time($agent_id = null) {
        global $wpdb;
        
        $cache_key = 'nexlifydesk_avg_response_time' . ($agent_id ? '_agent_' . $agent_id : '');
        $avg_response = wp_cache_get($cache_key);
        
        if (false === $avg_response) {
            // phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
            if ($agent_id) {
                $avg_seconds = $wpdb->get_var(
                    $wpdb->prepare(
                        "SELECT AVG(TIMESTAMPDIFF(SECOND, t.created_at, r.created_at)) 
                        FROM %i t 
                        INNER JOIN %i r ON t.id = r.ticket_id 
                        WHERE r.is_admin_reply = 1 
                        AND t.assigned_to = %d
                        AND r.id = (SELECT MIN(id) FROM %i WHERE ticket_id = t.id AND is_admin_reply = 1)",
                        $wpdb->prefix . 'nexlifydesk_tickets',
                        $wpdb->prefix . 'nexlifydesk_replies',
                        $agent_id,
                        $wpdb->prefix . 'nexlifydesk_replies'
                    )
                );
            } else {
                $avg_seconds = $wpdb->get_var(
                    $wpdb->prepare(
                        "SELECT AVG(TIMESTAMPDIFF(SECOND, t.created_at, r.created_at)) 
                        FROM %i t 
                        INNER JOIN %i r ON t.id = r.ticket_id 
                        WHERE r.is_admin_reply = 1 
                        AND r.id = (SELECT MIN(id) FROM %i WHERE ticket_id = t.id AND is_admin_reply = 1)",
                        $wpdb->prefix . 'nexlifydesk_tickets',
                        $wpdb->prefix . 'nexlifydesk_replies',
                        $wpdb->prefix . 'nexlifydesk_replies'
                    )
                );
            }
            // phpcs:enable
            if ($avg_seconds) {
                $minutes = round($avg_seconds / 60, 1);
                $avg_response = $minutes . 'm';
            } else {
                $avg_response = 'N/A';
            }
            
            wp_cache_set($cache_key, $avg_response, 'nexlifydesk', 2 * HOUR_IN_SECONDS);
        }
        
        return $avg_response;
    }
    
    private static function get_monthly_ticket_data($agent_id = null) {
        global $wpdb;
        
        $data = array();
        
        for ($i = 29; $i >= 0; $i--) {
            $date = gmdate('Y-m-d', strtotime("-$i days"));
            $cache_key = 'nexlifydesk_tickets_count_' . $date . ($agent_id ? '_agent_' . $agent_id : '');
            $count = wp_cache_get($cache_key, 'nexlifydesk');
            if (false === $count) {
                // phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery
                if ($agent_id) {
                    $count = $wpdb->get_var(
                        $wpdb->prepare(
                            "SELECT COUNT(*) FROM {$wpdb->prefix}nexlifydesk_tickets WHERE DATE(created_at) = %s AND assigned_to = %d",
                            $date, $agent_id
                        )
                    );
                } else {
                    $count = $wpdb->get_var(
                        $wpdb->prepare(
                            "SELECT COUNT(*) FROM {$wpdb->prefix}nexlifydesk_tickets WHERE DATE(created_at) = %s",
                            $date
                        )
                    );
                }
                // phpcs:enable
                wp_cache_set($cache_key, $count, 'nexlifydesk', HOUR_IN_SECONDS);
            }
            
            $data[] = array(
                'date' => $date,
                'tickets' => (int)$count
            );
        }
        
        return $data;
    }
    
    private static function get_agent_performance($agent_id = null) {
        global $wpdb;
        
        $agents = get_users(array(
            'role__in' => array('administrator', 'nexlifydesk_agent'),
            'fields' => array('ID', 'display_name')
        ));
        
        $performance = array();
        
        foreach ($agents as $agent) {
            $assigned_cache_key = 'nexlifydesk_agent_assigned_' . $agent->ID;
            $assigned = wp_cache_get($assigned_cache_key, 'nexlifydesk');
            if (false === $assigned) {
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
                $assigned = $wpdb->get_var(
                    $wpdb->prepare(
                        "SELECT COUNT(*) FROM {$wpdb->prefix}nexlifydesk_tickets WHERE assigned_to = %d",
                        $agent->ID
                    )
                );
                wp_cache_set($assigned_cache_key, $assigned, 'nexlifydesk', HOUR_IN_SECONDS);
            }
            
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
            $closed = $wpdb->get_var(
                $wpdb->prepare(
                    "SELECT COUNT(*) FROM {$wpdb->prefix}nexlifydesk_tickets WHERE assigned_to = %d AND status = 'closed'",
                    $agent->ID
                )
            );
            
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
            $replied_tickets = $wpdb->get_var(
                $wpdb->prepare(
                    "SELECT COUNT(DISTINCT t.id)
                    FROM {$wpdb->prefix}nexlifydesk_tickets t
                    INNER JOIN {$wpdb->prefix}nexlifydesk_replies r ON t.id = r.ticket_id
                    WHERE t.assigned_to = %d AND r.user_id = %d AND r.is_admin_reply = 1",
                    $agent->ID,
                    $agent->ID
                )
            );

            $response_rate = $assigned > 0 ? round(($replied_tickets / $assigned) * 100) : 0;
            
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
            $avg_response = $wpdb->get_var(
                $wpdb->prepare(
                    "SELECT AVG(TIMESTAMPDIFF(SECOND, t.created_at, r.created_at)) 
                     FROM {$wpdb->prefix}nexlifydesk_tickets t 
                     INNER JOIN {$wpdb->prefix}nexlifydesk_replies r ON t.id = r.ticket_id 
                     WHERE r.user_id = %d AND r.is_admin_reply = 1",
                    $agent->ID
                )
            );
            
            $avg_response_formatted = $avg_response ? round($avg_response / 60, 1) . 'm' : 'N/A';
            
            $performance[] = array(
                'name' => $agent->display_name,
                'assigned' => $assigned,
                'closed' => $closed,
                'response_rate' => $response_rate,
                'avg_response_time' => $avg_response_formatted
            );
        }
        
        return $performance;
    }
    
    private static function get_recent_activity($agent_id = null) {
        global $wpdb;
        
        $activities = array();
        
        // Recent tickets
        $recent_tickets_cache_key = 'nexlifydesk_recent_tickets';
        $recent_tickets = wp_cache_get($recent_tickets_cache_key, 'nexlifydesk');
        if (false === $recent_tickets) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
            $recent_tickets = $wpdb->get_results(
                "SELECT id, ticket_id, subject, status, created_at 
                 FROM {$wpdb->prefix}nexlifydesk_tickets 
                 ORDER BY created_at DESC 
                 LIMIT 5"
            );
            wp_cache_set($recent_tickets_cache_key, $recent_tickets, 'nexlifydesk', HOUR_IN_SECONDS);
        }
        
        foreach ($recent_tickets as $ticket) {
            $activities[] = array(
                'type' => 'new',
                // translators: 1: Ticket ID (e.g., T1005), 2: Ticket subject
                'message' => sprintf(__('New ticket #%1$s: %2$s', 'nexlifydesk'), $ticket->ticket_id, $ticket->subject),
                'time' => human_time_diff(strtotime($ticket->created_at), current_time('timestamp')) . ' ago'
            );
        }

        
        // Recent replies
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
        $recent_replies = $wpdb->get_results(
            "SELECT r.*, t.ticket_id, t.subject 
             FROM {$wpdb->prefix}nexlifydesk_replies r 
             INNER JOIN {$wpdb->prefix}nexlifydesk_tickets t ON r.ticket_id = t.id 
             WHERE r.is_admin_reply = 1 
             ORDER BY r.created_at DESC 
             LIMIT 3"
        );
        
        foreach ($recent_replies as $reply) {
            $user = get_userdata($reply->user_id);
            $display_name = $user ? $user->display_name : __('Unknown User', 'nexlifydesk');
            $activities[] = array(
                'type' => 'reply',
                // translators: 1: User display name, 2: Ticket ID (e.g., T1005)
                'message' => sprintf(__('%1$s replied to ticket #%2$s', 'nexlifydesk'), $display_name, $reply->ticket_id),
                'time' => human_time_diff(strtotime($reply->created_at), current_time('timestamp')) . ' ago'
            );
        }
        
        // Sort by time
        usort($activities, function($a, $b) {
            return strcmp($b['time'], $a['time']);
        });
        
        return array_slice($activities, 0, 8);
    }
}