<?php
if (!defined('ABSPATH')) {
    exit;
}

/**
 * NexlifyDesk Rate Limiter
 * 
 * Prevents spam by limiting email activity to 5 emails per 30 minutes
 * Works for both manual web submissions and email piping (tickets and replies)
 */
class NexlifyDesk_Rate_Limiter {
    
    /**
     * Maximum emails allowed per time window
     */
    const MAX_TICKETS_PER_WINDOW = 5;
    
    /**
     * Time window in seconds (30 minutes)
     */
    const TIME_WINDOW_SECONDS = 1800;
    
    /**
     * Check if a user/email has exceeded the rate limit
     *
     * @param int $user_id WordPress user ID (0 for non-registered users)
     * @param string $email_address Email address
     * @return bool True if rate limit exceeded, false otherwise
     */
    public static function is_rate_limited($user_id, $email_address) {
        $identifier = self::get_rate_limit_identifier($user_id, $email_address);
        $cache_key = 'nexlifydesk_rate_limit_' . $identifier;
        
        $ticket_count = get_transient($cache_key);
        
        if ($ticket_count === false) {
            return false;
        }
        
        if ($ticket_count >= self::MAX_TICKETS_PER_WINDOW) {
            return true;
        }
        
        return false;
    }
    
    /**
     * Record a ticket creation for rate limiting
     *
     * @param int $user_id WordPress user ID (0 for non-registered users)
     * @param string $email_address Email address
     * @return bool True on success, false on failure
     */
    public static function record_ticket_creation($user_id, $email_address) {
        return self::record_email_activity($user_id, $email_address);
    }
    
    /**
     * @param int $user_id WordPress user ID (0 for non-registered users)
     * @param string $email_address Email address
     * @return bool True on success, false on failure
     */
    public static function record_email_activity($user_id, $email_address) {
        $identifier = self::get_rate_limit_identifier($user_id, $email_address);
        $cache_key = 'nexlifydesk_rate_limit_' . $identifier;
        
        $current_count = get_transient($cache_key);
        
        if ($current_count === false) {
            $new_count = 1;
        } else {
            $new_count = intval($current_count) + 1;
        }
        
        $result = set_transient($cache_key, $new_count, self::TIME_WINDOW_SECONDS);
        
        return $result;
    }
    
    /**
     * @param int $user_id WordPress user ID (0 for non-registered users)
     * @param string $email_address Email address
     * @return int Number of tickets remaining
     */
    public static function get_remaining_tickets($user_id, $email_address) {
        $identifier = self::get_rate_limit_identifier($user_id, $email_address);
        $cache_key = 'nexlifydesk_rate_limit_' . $identifier;
        
        $ticket_count = get_transient($cache_key);
        
        if ($ticket_count === false) {
            return self::MAX_TICKETS_PER_WINDOW;
        }
        
        return max(0, self::MAX_TICKETS_PER_WINDOW - intval($ticket_count));
    }
    
    /**
     * @param int $user_id WordPress user ID (0 for non-registered users)
     * @param string $email_address Email address
     * @return int Seconds until reset, 0 if not rate limited
     */
    public static function get_time_until_reset($user_id, $email_address) {
        $identifier = self::get_rate_limit_identifier($user_id, $email_address);
        $cache_key = 'nexlifydesk_rate_limit_' . $identifier;
        
        $reset_time_key = $cache_key . '_reset_time';
        $reset_time = get_transient($reset_time_key);
        
        if ($reset_time === false) {
            $reset_time = time() + self::TIME_WINDOW_SECONDS;
            set_transient($reset_time_key, $reset_time, self::TIME_WINDOW_SECONDS);
        }
        
        $remaining_time = max(0, $reset_time - time());
        
        return $remaining_time;
    }
    
    /**
     * @param int $user_id WordPress user ID (0 for non-registered users)
     * @param string $email_address Email address
     * @return bool True on success, false on failure
     */
    public static function clear_rate_limit($user_id, $email_address) {
        $identifier = self::get_rate_limit_identifier($user_id, $email_address);
        $cache_key = 'nexlifydesk_rate_limit_' . $identifier;
        $reset_time_key = $cache_key . '_reset_time';
        
        $result1 = delete_transient($cache_key);
        $result2 = delete_transient($reset_time_key);
        
        return $result1 || $result2;
    }
    
    /**
     * @param int $user_id WordPress user ID (0 for non-registered users)
     * @param string $email_address Email address
     * @return array Rate limit information
     */
    public static function get_rate_limit_status($user_id, $email_address) {
        $identifier = self::get_rate_limit_identifier($user_id, $email_address);
        $cache_key = 'nexlifydesk_rate_limit_' . $identifier;
        
        $current_count = get_transient($cache_key);
        $current_count = $current_count === false ? 0 : intval($current_count);
        
        $remaining = self::get_remaining_tickets($user_id, $email_address);
        $is_limited = self::is_rate_limited($user_id, $email_address);
        $time_until_reset = self::get_time_until_reset($user_id, $email_address);
        
        return array(
            'current_count' => $current_count,
            'max_allowed' => self::MAX_TICKETS_PER_WINDOW,
            'remaining' => $remaining,
            'is_limited' => $is_limited,
            'time_until_reset' => $time_until_reset,
            'window_seconds' => self::TIME_WINDOW_SECONDS
        );
    }
    
    /**
     * @param int $user_id WordPress user ID (0 for non-registered users)
     * @param string $email_address Email address
     * @return string Unique identifier
     */
    private static function get_rate_limit_identifier($user_id, $email_address) {
        if ($user_id > 0) {
            return 'user_' . $user_id;
        } else {
            return 'email_' . md5(strtolower(trim($email_address)));
        }
    }
    
    /**
     * @param int $seconds Seconds remaining
     * @return string Human readable time
     */
    public static function format_time_remaining($seconds) {
        if ($seconds <= 0) {
            return __('Now', 'nexlifydesk');
        }
        
        $minutes = floor($seconds / 60);
        $seconds = $seconds % 60;
        
        if ($minutes > 0) {
            if ($seconds > 0) {
                return sprintf(
                    /* translators: 1: minutes, 2: seconds */
                    __('%1$d minutes and %2$d seconds', 'nexlifydesk'),
                    $minutes,
                    $seconds
                );
            } else {
                return sprintf(
                    /* translators: %d: minutes */
                    _n('%d minute', '%d minutes', $minutes, 'nexlifydesk'),
                    $minutes
                );
            }
        } else {
            return sprintf(
                /* translators: %d: seconds */
                _n('%d second', '%d seconds', $seconds, 'nexlifydesk'),
                $seconds
            );
        }
    }
    
    /**
     * @param int $user_id WordPress user ID (0 for non-registered users)
     * @param string $email_address Email address
     * @return string Error message
     */
    public static function get_rate_limit_error_message($user_id, $email_address) {
        $status = self::get_rate_limit_status($user_id, $email_address);
        $time_remaining = self::format_time_remaining($status['time_until_reset']);
        
        return sprintf(
            /* translators: 1: max tickets allowed, 2: time window in minutes, 3: time remaining */
            __('Rate limit exceeded. You can only send %1$d emails per %2$d minutes. Please try again in %3$s.', 'nexlifydesk'),
            $status['max_allowed'],
            $status['window_seconds'] / 60,
            $time_remaining
        );
    }
    
    /**
     * Clean up expired rate limit data
     * This is automatically handled by WordPress transients, but can be called manually
     */
    public static function cleanup_expired_data() {
        global $wpdb;
        
        // WordPress handles transient cleanup automatically, but we can help by cleaning up our specific keys
        $prefix = '_transient_nexlifydesk_rate_limit_';
        
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom maintenance task, no caching needed
        $expired_transients = $wpdb->get_results(
            $wpdb->prepare(
                "SELECT option_name FROM {$wpdb->options} 
                WHERE option_name LIKE %s 
                AND option_value < %d",
                $prefix . '%',
                time()
            )
        );
        
        $cleaned_count = 0;
        foreach ($expired_transients as $transient) {
            $transient_name = str_replace('_transient_', '', $transient->option_name);
            if (delete_transient($transient_name)) {
                $cleaned_count++;
            }
        }
        
        return $cleaned_count;
    }
}
