<?php
if (!defined('ABSPATH')) {
    exit;
}

class NexlifyDesk_Database {
    private static $tables = array();
    
    public static function init() {
        global $wpdb;
        self::$tables = array(
            'tickets' => $wpdb->prefix . NEXLIFYDESK_TABLE_PREFIX . 'tickets',
            'replies' => $wpdb->prefix . NEXLIFYDESK_TABLE_PREFIX . 'replies',
            'attachments' => $wpdb->prefix . NEXLIFYDESK_TABLE_PREFIX . 'attachments',
            'categories' => $wpdb->prefix . NEXLIFYDESK_TABLE_PREFIX . 'categories',
            'ticket_reads' => $wpdb->prefix . NEXLIFYDESK_TABLE_PREFIX . 'ticket_reads'
        );
    }   

    public static function activate() {
        global $wpdb;

        self::$tables = array(
            'tickets' => $wpdb->prefix . NEXLIFYDESK_TABLE_PREFIX . 'tickets',
            'replies' => $wpdb->prefix . NEXLIFYDESK_TABLE_PREFIX . 'replies',
            'attachments' => $wpdb->prefix . NEXLIFYDESK_TABLE_PREFIX . 'attachments',
            'categories' => $wpdb->prefix . NEXLIFYDESK_TABLE_PREFIX . 'categories',
            'ticket_reads' => $wpdb->prefix . NEXLIFYDESK_TABLE_PREFIX . 'ticket_reads'
        );

        $charset_collate = $wpdb->get_charset_collate();
        
        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        
        // Tickets table
        $sql = "CREATE TABLE " . self::$tables['tickets'] . " (
            id bigint(20) NOT NULL AUTO_INCREMENT,
            ticket_id varchar(20) NOT NULL,
            user_id bigint(20) NOT NULL,
            category_id bigint(20) NOT NULL,
            subject varchar(255) NOT NULL,
            message longtext NOT NULL,
            status varchar(20) NOT NULL DEFAULT 'open',
            priority varchar(20) NOT NULL DEFAULT 'medium',
            assigned_to bigint(20) DEFAULT NULL,
            source varchar(20) NOT NULL DEFAULT 'web',
            created_at datetime NOT NULL,
            updated_at datetime NOT NULL,
            PRIMARY KEY  (id),
            UNIQUE KEY ticket_id (ticket_id)
        ) $charset_collate;";
        
        // Replies table
        $sql .= "CREATE TABLE " . self::$tables['replies'] . " (
            id bigint(20) NOT NULL AUTO_INCREMENT,
            ticket_id bigint(20) NOT NULL,
            user_id bigint(20) NOT NULL,
            message longtext NOT NULL,
            created_at datetime NOT NULL,
            is_admin_reply tinyint(1) NOT NULL DEFAULT 0,
            is_internal_note tinyint(1) NOT NULL DEFAULT 0,
            PRIMARY KEY  (id),
            KEY ticket_id (ticket_id)
        ) $charset_collate;";
        
        // Attachments table
        $sql .= "CREATE TABLE " . self::$tables['attachments'] . " (
            id bigint(20) NOT NULL AUTO_INCREMENT,
            ticket_id bigint(20) DEFAULT NULL,
            reply_id bigint(20) DEFAULT NULL,
            user_id bigint(20) NOT NULL,
            file_name varchar(255) NOT NULL,
            file_path varchar(255) NOT NULL,
            file_type varchar(100) NOT NULL,
            file_size int(11) NOT NULL,
            created_at datetime NOT NULL,
            PRIMARY KEY  (id),
            KEY ticket_id (ticket_id),
            KEY reply_id (reply_id)
        ) $charset_collate;";
        
        // Categories table
        $sql .= "CREATE TABLE " . self::$tables['categories'] . " (
            id bigint(20) NOT NULL AUTO_INCREMENT,
            name varchar(100) NOT NULL,
            slug varchar(100) NOT NULL,
            description text DEFAULT NULL,
            is_active tinyint(1) NOT NULL DEFAULT 1,
            PRIMARY KEY  (id),
            UNIQUE KEY slug (slug)
        ) $charset_collate;";
        
        // Ticket reads table
        $sql .= "CREATE TABLE " . self::$tables['ticket_reads'] . " (
            id bigint(20) NOT NULL AUTO_INCREMENT,
            ticket_id bigint(20) NOT NULL,
            user_id bigint(20) NOT NULL,
            read_at datetime NOT NULL,
            PRIMARY KEY  (id),
            UNIQUE KEY ticket_user (ticket_id, user_id),
            KEY ticket_id (ticket_id),
            KEY user_id (user_id)
        ) $charset_collate;";
        
        $result = dbDelta($sql);

        $tables_to_check = array('tickets', 'replies', 'attachments', 'categories', 'ticket_reads');
        foreach ($tables_to_check as $table_key) {
            $table_name = self::$tables[$table_key];
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery -- Table existence check during activation, no caching needed
            $table_exists = $wpdb->get_var($wpdb->prepare("SHOW TABLES LIKE %s", $table_name)) == $table_name;
        }

        $categories_table = self::$tables['categories'];
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery -- Table existence check during activation, no caching needed
        if ($wpdb->get_var($wpdb->prepare("SHOW TABLES LIKE %s", $categories_table)) === $categories_table) {
            self::insert_default_categories();
        }

        add_option('nexlifydesk_version', NEXLIFYDESK_VERSION);
        add_option('nexlifydesk_settings', array(
            'email_notifications' => 1,
            'admin_email_notifications' => 1,
            'default_priority' => 'medium',
            'auto_assign' => 0,
            'allowed_file_types' => 'jpg,jpeg,png,pdf',
            'max_file_size' => 2, // 2MB
            'ticket_form_page_id' => 0
        ));
    }
    
    private static function insert_default_categories() {
        global $wpdb;
        
        $categories = array(
            'General Support',
            'Billing',
            'Technical Issues',
            'Feature Requests'
        );
        
        foreach ($categories as $category) {
            $slug = sanitize_title($category);
            
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery -- Category existence check during activation, no caching needed
            $exists = $wpdb->get_var(
                $wpdb->prepare(
                    "SELECT id FROM `{$wpdb->prefix}nexlifydesk_categories` WHERE slug = %s",
                    $slug
                )
            );
            
            if (!$exists) {
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Direct insert during activation, no caching needed
                $result = $wpdb->insert(
                    self::$tables['categories'],
                    array(
                        'name' => $category,
                        'slug' => $slug,
                        'description' => ''
                    ),
                    array('%s', '%s', '%s')
                );
            }
        }
    }
    
    public static function deactivate() {
        // Optional: Clean up on deactivation, required.
    }
    
    public static function get_table($name) {
        return isset(self::$tables[$name]) ? self::$tables[$name] : null;
    }
    
    public static function generate_ticket_id() {
        $settings = get_option('nexlifydesk_settings');
        $prefix = isset($settings['ticket_id_prefix']) ? $settings['ticket_id_prefix'] : 'T';
        $start = isset($settings['ticket_id_start']) ? (int)$settings['ticket_id_start'] : 1001;

        $last_number = (int)get_option('nexlifydesk_last_ticket_number', 0);

        if ($last_number < $start - 1) {
            $last_number = $start - 1;
        }

        $next_number = $last_number + 1;

        update_option('nexlifydesk_last_ticket_number', $next_number);

        return $prefix . $next_number;
    }
    
    public static function check_and_run_migrations() {
        $current_version = get_option('nexlifydesk_db_version', '1.0.5');
        $plugin_version = NEXLIFYDESK_VERSION;
        
        if (version_compare($current_version, '1.0.5', '<')) {
            self::migrate_to_1_0_1();
            update_option('nexlifydesk_db_version', '1.0.5');
        }
        
        if (version_compare($current_version, '1.0.5', '<')) {
            self::migrate_to_1_0_2();
            update_option('nexlifydesk_db_version', '1.0.5');
        }
        
        $security_migration_done = get_option('nexlifydesk_password_encryption_migration', false);
        if (!$security_migration_done) {
            self::migrate_encrypt_passwords();
            update_option('nexlifydesk_password_encryption_migration', true);
        }
        
        if ($current_version !== $plugin_version) {
            update_option('nexlifydesk_db_version', $plugin_version);
        }
    }
    
    private static function migrate_to_1_0_1() {
        global $wpdb;
        
        if (empty(self::$tables)) {
            self::init();
        }

        $table_name = self::$tables['replies'];
        $table_name_escaped = esc_sql($table_name);
        $cache_key = 'is_internal_note_column_exists_' . md5($table_name);

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Migration check during plugin initialization
        $column_exists = wp_cache_get($cache_key, 'migration_checks');
        if (false === $column_exists) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Migration check during plugin initialization, table name is safe
            $column_exists = $wpdb->get_results(
                $wpdb->prepare(
                    "SHOW COLUMNS FROM `{$table_name_escaped}` LIKE %s", // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table name is escaped and safe for direct interpolation
                    'is_internal_note'
                )
            );
            wp_cache_set($cache_key, $column_exists, 'migration_checks', DAY_IN_SECONDS);
        }

        if (empty($column_exists)) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.SchemaChange,WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Database migration during plugin update, table name is safe
            $wpdb->query("ALTER TABLE {$table_name} ADD COLUMN is_internal_note tinyint(1) NOT NULL DEFAULT 0");

            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Database migration during plugin update, table name is safe
            $wpdb->query("UPDATE {$table_name} SET is_internal_note = 1 WHERE message LIKE '[INTERNAL NOTE]%'");

            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Database migration during plugin update, table name is safe and escaped
            $wpdb->query(
                $wpdb->prepare(
                    // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table name is escaped and safe for direct interpolation
                    "UPDATE `$table_name_escaped` SET message = SUBSTRING(message, 17) WHERE is_internal_note = 1 AND message LIKE %s", 
                    '[INTERNAL NOTE] %'
                )
            );
        }
    }
    
    private static function migrate_to_1_0_2() {
        global $wpdb;
        
        if (empty(self::$tables)) {
            self::init();
        }
        
        $table_name = self::$tables['ticket_reads'];
        
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Migration check
        $table_exists = $wpdb->get_var($wpdb->prepare("SHOW TABLES LIKE %s", $table_name)) === $table_name;
        
        if (!$table_exists) {
            $charset_collate = $wpdb->get_charset_collate();
            
            $sql = "CREATE TABLE {$table_name} (
                id bigint(20) NOT NULL AUTO_INCREMENT,
                ticket_id bigint(20) NOT NULL,
                user_id bigint(20) NOT NULL,
                read_at datetime NOT NULL,
                PRIMARY KEY  (id),
                UNIQUE KEY ticket_user (ticket_id, user_id),
                KEY ticket_id (ticket_id),
                KEY user_id (user_id)
            ) {$charset_collate};";
            
            require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
            dbDelta($sql);
        }
    }

    public static function upgrade_database() {
        global $wpdb;
        
        $current_version = get_option('nexlifydesk_version', '1.0.5');
        
        if ($current_version === '1.0.5' && !get_option('nexlifydesk_db_version')) {
            return;
        }
        
        if (empty(self::$tables)) {
            self::init();
        }
        
        $tickets_table = self::$tables['tickets'];
        
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Table existence check during upgrade, no caching needed
        $table_exists = $wpdb->get_var($wpdb->prepare("SHOW TABLES LIKE %s", $tickets_table)) === $tickets_table;
        
        if (!$table_exists) {
            return;
        }
        
        if (version_compare($current_version, NEXLIFYDESK_VERSION, '<')) {
            $tickets_table_escaped = esc_sql($tickets_table);
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table name is escaped and safe
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Schema check during upgrade, no caching needed
            $column_exists = $wpdb->get_results(
                $wpdb->prepare(
                    // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table name is escaped and safe
                    "SHOW COLUMNS FROM `{$tickets_table_escaped}` LIKE %s",
                    'source'
                )
            );
            
            if (empty($column_exists)) {
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.DirectDatabaseQuery.SchemaChange,WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Schema change during plugin upgrade, table name is safe and not user input, no caching needed
                $wpdb->query("ALTER TABLE {$tickets_table} ADD COLUMN source varchar(20) NOT NULL DEFAULT 'web' AFTER assigned_to");
            }
            
            // Update version
            update_option('nexlifydesk_version', NEXLIFYDESK_VERSION);
        }
    }
    
    /**
     * Security migration: Encrypt any plain text passwords in IMAP settings
     */
    private static function migrate_encrypt_passwords() {
        $imap_settings = get_option('nexlifydesk_imap_settings', array());
        
        if (empty($imap_settings)) {
            return;
        }
        
        $updated = false;
        
        if (isset($imap_settings['aws']) && is_array($imap_settings['aws'])) {
            foreach ($imap_settings['aws'] as $key => $aws_config) {
                if (isset($aws_config['password']) && !empty($aws_config['password'])) {
                    if (!nexlifydesk_is_encrypted($aws_config['password'])) {
                        $imap_settings['aws'][$key]['password'] = nexlifydesk_encrypt($aws_config['password']);
                        $updated = true;
                    }
                }
            }
        }
        
        $providers = array('google', 'outlook', 'generic');
        foreach ($providers as $provider) {
            if (isset($imap_settings[$provider]) && is_array($imap_settings[$provider])) {
                foreach ($imap_settings[$provider] as $key => $config) {
                    if (isset($config['password']) && !empty($config['password'])) {
                        if (!nexlifydesk_is_encrypted($config['password'])) {
                            $imap_settings[$provider][$key]['password'] = nexlifydesk_encrypt($config['password']);
                            $updated = true;
                        }
                    }
                }
            }
        }
        
        if ($updated) {
            update_option('nexlifydesk_imap_settings', $imap_settings);
        }
    }
}