<?php
if (!defined('ABSPATH')) {
    exit;
}

class NexlifyDesk_Ajax {
    public static function init() {
        add_action('wp_ajax_nexlifydesk_submit_ticket', array(__CLASS__, 'submit_ticket'));
        add_action('wp_ajax_nopriv_nexlifydesk_submit_ticket', array(__CLASS__, 'submit_ticket'));
        add_action('wp_ajax_nexlifydesk_add_reply', array(__CLASS__, 'add_reply'));
        add_action('wp_ajax_nexlifydesk_add_note', array(__CLASS__, 'add_note'));
        add_action('wp_ajax_nexlifydesk_update_status', array(__CLASS__, 'update_status'));
        add_action('wp_ajax_nexlifydesk_upload_attachment', array(__CLASS__, 'upload_attachment'));
        add_action('wp_ajax_nexlifydesk_reassign_orphaned_tickets', array(__CLASS__, 'reassign_orphaned_tickets_ajax'));
        add_action('wp_ajax_nexlifydesk_admin_get_tickets', array(__CLASS__, 'admin_get_tickets'));
        add_action('wp_ajax_nexlifydesk_admin_assign_ticket', array(__CLASS__, 'admin_assign_ticket'));
        add_action('wp_ajax_nexlifydesk_delete_category', array(__CLASS__, 'delete_category'));
        add_action('wp_ajax_nexlifydesk_add_category', array(__CLASS__, 'add_category'));
        add_action('wp_ajax_nexlifydesk_test_aws_connection', array(__CLASS__, 'test_aws_connection'));
        add_action('wp_ajax_nexlifydesk_aws_diagnostics', array(__CLASS__, 'aws_diagnostics'));
        add_action('wp_ajax_nexlifydesk_manual_fetch_emails', array(__CLASS__, 'manual_fetch_emails'));
        add_action('wp_ajax_nexlifydesk_bulk_action', array(__CLASS__, 'handle_bulk_action'));
        add_action('wp_ajax_nexlifydesk_refresh_ticket_list', array(__CLASS__, 'refresh_ticket_list'));
        add_action('wp_ajax_nexlifydesk_mark_ticket_read', array(__CLASS__, 'mark_ticket_read'));
        add_action('wp_ajax_nexlifydesk_test_custom_connection', array(__CLASS__, 'test_custom_connection'));
        add_action('wp_ajax_nexlifydesk_test_google_connection', array(__CLASS__, 'test_google_connection'));
        add_action('wp_ajax_nexlifydesk_manual_fetch_google_emails', array(__CLASS__, 'manual_fetch_google_emails'));
    }
    
    public static function submit_ticket() {

        check_ajax_referer('nexlifydesk-ajax-nonce', 'nonce');
        
        if (!is_user_logged_in()) {
            wp_send_json_error(__('You must be logged in to submit a ticket.', 'nexlifydesk'));
        }
        
        if (isset($_FILES['attachments']) && isset($_FILES['attachments']['error']) && is_array($_FILES['attachments']['error'])) {
            $attachment_errors = array_map('intval', $_FILES['attachments']['error']); // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Sanitized via array_map with intval
            
            foreach ($attachment_errors as $error) {
                if ($error === UPLOAD_ERR_FORM_SIZE || $error === UPLOAD_ERR_INI_SIZE) {
                    wp_send_json_error(__('One or more files are too large. Please reduce file sizes and try again.', 'nexlifydesk'));
                }
                if ($error === UPLOAD_ERR_PARTIAL) {
                    wp_send_json_error(__('File upload was interrupted. Please try again.', 'nexlifydesk'));
                }
            }
        }
        
        $max_post_size = self::get_max_post_size();
        $content_length = isset($_SERVER['CONTENT_LENGTH']) ? (int) $_SERVER['CONTENT_LENGTH'] : 0;
        
        if ($content_length > $max_post_size) {
            wp_send_json_error(sprintf(
                /* translators: %s: Maximum allowed file size */
                __('Request too large. Maximum allowed size is %s. Please reduce file sizes or number of files.', 'nexlifydesk'),
                size_format($max_post_size)
            ));
        }
        
        if (empty($_POST['subject']) || empty($_POST['message'])) {
            wp_send_json_error(esc_html__('Subject and message are required.', 'nexlifydesk'));
        }

        $data = array(
            'subject' => sanitize_text_field(wp_unslash($_POST['subject'])),
            'message' => wp_kses_post(wp_unslash($_POST['message'])),
            'category_id' => isset($_POST['category_id']) ? absint(wp_unslash($_POST['category_id'])) : 0,
            'priority' => isset($_POST['priority']) ? sanitize_text_field(wp_unslash($_POST['priority'])) : 'medium',
            'source' => 'web'
        );
        
        $attachments = array();

        if (!empty($_FILES['attachments']) && isset($_FILES['attachments']['name'])) {
            $settings = get_option('nexlifydesk_settings', array());
            $max_size = isset($settings['max_file_size']) ? (int)$settings['max_file_size'] * 1024 * 1024 : 2 * 1024 * 1024;
            $allowed_types = isset($settings['allowed_file_types']) ? explode(',', $settings['allowed_file_types']) : array('jpg', 'jpeg', 'png', 'pdf');
            
            $file_names = array_map('sanitize_file_name', wp_unslash($_FILES['attachments']['name']));
            
            foreach ($file_names as $key => $sanitized_name) {
                if (!isset(
                    $_FILES['attachments']['type'][$key],
                    $_FILES['attachments']['tmp_name'][$key],
                    $_FILES['attachments']['error'][$key],
                    $_FILES['attachments']['size'][$key]
                )) {
                    continue;
                }

                if ($_FILES['attachments']['error'][$key] === UPLOAD_ERR_OK) {
                    $file_size = absint($_FILES['attachments']['size'][$key]);
                    $file_ext = strtolower(pathinfo($sanitized_name, PATHINFO_EXTENSION));
                    
                    if ($file_size > $max_size) {
                        /* translators: 1: File name, 2: Maximum file size in MB */
                        wp_send_json_error(sprintf(__('File "%1$s" is too large. Maximum size is %2$dMB.', 'nexlifydesk'), esc_html($sanitized_name), $max_size / 1024 / 1024));
                    }
                    
                    if (!in_array($file_ext, $allowed_types, true)) {
                        /* translators: 1: File extension, 2: Allowed file types */
                        wp_send_json_error(sprintf(__('File type "%1$s" is not allowed. Allowed: %2$s', 'nexlifydesk'), esc_html($file_ext), esc_html(implode(', ', $allowed_types))));
                    }
                    
                    $attachments[] = [
                        'name' => $sanitized_name,
                        'type' => sanitize_mime_type(wp_unslash($_FILES['attachments']['type'][$key])),
                        // No sanitization needed for tmp_name
                        'tmp_name' => $_FILES['attachments']['tmp_name'][$key], // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
                        'error' => absint($_FILES['attachments']['error'][$key]),
                        'size' => $file_size
                    ];
                }
            }
        }
        
        $data['attachments'] = $attachments;
        
        $ticket = NexlifyDesk_Tickets::create_ticket($data);
        
        if (is_wp_error($ticket)) {
            wp_send_json_error($ticket->get_error_message());
        } else {
            $settings = get_option('nexlifydesk_settings', array());
            $ticket_page_ids = isset($settings['ticket_page_id']) ? $settings['ticket_page_id'] : array();
            
            if (!is_array($ticket_page_ids)) {
                $ticket_page_ids = $ticket_page_ids ? array($ticket_page_ids) : array();
            }
            
            $ticket_page_url = '';
            if (!empty($ticket_page_ids)) {
                $ticket_page_id = (int)$ticket_page_ids[0];
                $ticket_page_url = get_permalink($ticket_page_id);
            }
            
            if (!$ticket_page_url || !wp_http_validate_url($ticket_page_url)) {
                $ticket_page_url = isset($_POST['current_url']) ? esc_url_raw(wp_unslash($_POST['current_url'])) : home_url();
            }
            
            if (!$ticket_page_url || !wp_http_validate_url($ticket_page_url)) {
                $ticket_page_url = home_url();
            }
            
            if (isset($ticket->is_duplicate) && $ticket->is_duplicate) {
                $redirect_url = add_query_arg(array(
                    'ticket_id' => $ticket->ticket_id,
                    'duplicate_detected' => 1
                ), $ticket_page_url);
                
                wp_send_json_success(array(
                    'message' => __('We found a similar ticket from you. Your message has been added to the existing ticket.', 'nexlifydesk'),
                    'redirect' => $redirect_url,
                    'is_duplicate' => true,
                    'existing_ticket_id' => $ticket->ticket_id
                ));
            } else {
                $redirect_url = add_query_arg(array(
                    'ticket_id' => $ticket->ticket_id,
                    'ticket_number' => $ticket->ticket_id,
                    'ticket_submitted' => 1
                ), $ticket_page_url);
                
                wp_send_json_success(array(
                    'message' => __('Ticket submitted successfully!', 'nexlifydesk'),
                    'redirect' => $redirect_url,
                    'ticket_id' => $ticket->ticket_id,
                    'ticket_number' => $ticket->ticket_id
                ));
            }
        }
    }
    
    public static function add_reply() {
        check_ajax_referer('nexlifydesk-ajax-nonce', 'nonce');

        if (!is_user_logged_in()) {
            wp_send_json_error(__('You must be logged in to reply.', 'nexlifydesk'));
        }

        if (empty($_POST['ticket_id']) || empty($_POST['message'])) {
            wp_send_json_error(__('Missing required fields.', 'nexlifydesk'));
        }

        $ticket_id_param = sanitize_text_field(wp_unslash($_POST['ticket_id']));
        
        $ticket = NexlifyDesk_Tickets::get_ticket_by_ticket_id($ticket_id_param);
        
        if (!$ticket && is_numeric($ticket_id_param)) {
            $ticket = NexlifyDesk_Tickets::get_ticket(absint($ticket_id_param));
        }
        
        if (!$ticket) {
            wp_send_json_error(__('Ticket not found.', 'nexlifydesk'));
            return;
        }

        $current_user = wp_get_current_user();
        
        $can_reply = false;
        
        if (current_user_can('manage_options')) {
            $can_reply = true;
        }
        elseif (current_user_can('nexlifydesk_manage_tickets')) {
            $is_agent = in_array('nexlifydesk_agent', $current_user->roles) && !in_array('administrator', $current_user->roles);
            if ($is_agent) {
                $can_view_all = NexlifyDesk_Users::agent_can('nexlifydesk_view_all_tickets', $current_user->ID);
                if ($can_view_all || (int)$ticket->assigned_to === (int)$current_user->ID) {
                    $can_reply = true;
                }
            } else {
                $can_reply = true;
            }
        }
        
        if (!$can_reply && (int)$ticket->user_id === (int)$current_user->ID && $ticket->user_id > 0) {
            $can_reply = true;
        }
        
        if (!$can_reply && (int)$ticket->user_id === 0) {
            $customer_email = get_post_meta($ticket->id, 'customer_email', true);
            if ($customer_email && $customer_email === $current_user->user_email) {
                $can_reply = true;
            }
        }
        
        if (!$can_reply) {
            wp_send_json_error(__('You are not authorized to reply to this ticket.', 'nexlifydesk'));
            return;
        }

        $data = array(
            'ticket_id' => $ticket->id,
            'message' => wp_kses_post(wp_unslash($_POST['message']))
        );
        
        if ($ticket && $ticket->status === 'closed') {
            wp_send_json_error(__('This ticket is closed. Please create a new ticket for further assistance.', 'nexlifydesk'));
        }
        
        if ($ticket && $ticket->status === 'resolved' && !current_user_can('nexlifydesk_manage_tickets')) {
            NexlifyDesk_Tickets::update_ticket_status($ticket->id, 'open');
        }

        $attachments = array();
        if (!empty($_FILES['attachments']) && 
            isset($_FILES['attachments']['name'], $_FILES['attachments']['error']) &&
            is_array($_FILES['attachments']['name'])) {
            
            $file_names = array_map('sanitize_file_name', $_FILES['attachments']['name']);
            
            foreach ($file_names as $key => $value) {
                if (isset(
                    $_FILES['attachments']['error'][$key],
                    $_FILES['attachments']['type'][$key],
                    $_FILES['attachments']['tmp_name'][$key],
                    $_FILES['attachments']['size'][$key]
                )) {
                    if ($_FILES['attachments']['error'][$key] === UPLOAD_ERR_OK) {
                        $attachments[] = array(
                            'name' => $value, // Already sanitized via array_map
                            'type' => sanitize_mime_type($_FILES['attachments']['type'][$key]),
                            // No sanitization needed for tmp_name
                            'tmp_name' => $_FILES['attachments']['tmp_name'][$key],  // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
                            'error' => (int) $_FILES['attachments']['error'][$key],
                            'size' => (int) $_FILES['attachments']['size'][$key]
                        );
                    }
                }
            }
        }

        $data['attachments'] = $attachments;
        $data['user_id'] = $current_user->ID;
        $data['source'] = 'web';

        $reply_id = NexlifyDesk_Tickets::add_reply($data);

        if (is_wp_error($reply_id)) {
            wp_send_json_error($reply_id->get_error_message());
        } else {
            ob_start();
            $ticket = NexlifyDesk_Tickets::get_ticket($data['ticket_id']);
            $reply = null; 
            
            $replies = NexlifyDesk_Tickets::get_replies($data['ticket_id']);
            foreach ($replies as $r) {
                if ($r->id == $reply_id) {
                    $reply = $r;
                    break;
                }
            }
            
            if ($reply) {
                $reply_user = get_userdata($reply->user_id);
                $reply->attachments = NexlifyDesk_Tickets::get_attachments($data['ticket_id'], $reply->id);
                
                include NEXLIFYDESK_PLUGIN_DIR . 'templates/frontend/partials/single-reply.php';
            }
            $reply_html = ob_get_clean();

            $response = array(
                'message' => __('Reply added successfully!', 'nexlifydesk'),
                'reply_html' => $reply_html
            );

            wp_send_json_success($response);
        }
    }
    
    public static function add_note() {
        check_ajax_referer('nexlifydesk-ajax-nonce', 'nonce');

        if (!is_user_logged_in()) {
            wp_send_json_error(__('You must be logged in to add a note.', 'nexlifydesk'));
        }

        if (!current_user_can('nexlifydesk_manage_tickets')) {
            wp_send_json_error(__('You do not have permission to add internal notes.', 'nexlifydesk'));
        }

        if (empty($_POST['ticket_id']) || empty($_POST['note'])) {
            wp_send_json_error(__('Missing required fields.', 'nexlifydesk'));
        }

        $ticket_id = absint($_POST['ticket_id']);
        $note = wp_kses_post(wp_unslash($_POST['note']));
        $current_user = wp_get_current_user();
        $ticket = NexlifyDesk_Tickets::get_ticket($ticket_id);
        if (!$ticket) {
            wp_send_json_error(__('Ticket not found.', 'nexlifydesk'));
        }

        $is_agent = in_array('nexlifydesk_agent', $current_user->roles) && !in_array('administrator', $current_user->roles);
        if ($is_agent) {
            $can_view_all = NexlifyDesk_Users::agent_can('nexlifydesk_view_all_tickets', $current_user->ID);
            if (!$can_view_all && (int)$ticket->assigned_to !== (int)$current_user->ID) {
                wp_send_json_error(__('You can only add notes to tickets assigned to you.', 'nexlifydesk'));
            }
        }

        $data = array(
            'ticket_id' => $ticket_id,
            'message' => $note,
            'user_id' => $current_user->ID,
            'is_admin_reply' => 1, 
            'is_internal_note' => true, 
            'attachments' => array() 
        );

        $reply_id = NexlifyDesk_Tickets::add_reply($data);

        if (is_wp_error($reply_id)) {
            wp_send_json_error($reply_id->get_error_message());
        } else {
            ob_start();
            $replies = NexlifyDesk_Tickets::get_replies($ticket_id);
            $note_obj = null;
            foreach ($replies as $r) {
                if ($r->id == $reply_id) {
                    $note_obj = $r;
                    break;
                }
            }
            if ($note_obj) {
                $note_obj->attachments = NexlifyDesk_Tickets::get_attachments($ticket_id, $note_obj->id);
                include NEXLIFYDESK_PLUGIN_DIR . 'templates/admin/partials/single-reply.php';
            }
            $note_html = ob_get_clean();

            wp_send_json_success(array(
                'message' => __('Internal note added successfully!', 'nexlifydesk'),
                'note_id' => $reply_id,
                'note_html' => $note_html
            ));
        }
    }
    
    public static function update_status() {
        check_ajax_referer('nexlifydesk-ajax-nonce', 'nonce');

        if (!is_user_logged_in()) {
            wp_send_json_error(__('You must be logged in to update ticket status.', 'nexlifydesk'));
        }

        if (!isset($_POST['ticket_id']) || !isset($_POST['status'])) {
            wp_send_json_error(__('Missing required fields.', 'nexlifydesk'));
        }

        $ticket_id = absint($_POST['ticket_id']);
        $status = sanitize_text_field(wp_unslash($_POST['status']));
        $current_user = wp_get_current_user();
        $ticket = NexlifyDesk_Tickets::get_ticket($ticket_id);

        if ($ticket && $ticket->status === 'closed' && !current_user_can('manage_options')) {
            wp_send_json_error(__('Only administrators can change the status of closed tickets.', 'nexlifydesk'));
        }

        $can_update_status = current_user_can('manage_options') || ($ticket && $ticket->assigned_to == $current_user->ID);

        if (!$can_update_status) {
            wp_send_json_error(__('You do not have permission to update this ticket status.', 'nexlifydesk'));
        }

        $allowed_statuses = ['open', 'pending', 'in_progress', 'resolved', 'closed'];
        if (!in_array($status, $allowed_statuses, true)) {
            wp_send_json_error(__('Invalid ticket status.', 'nexlifydesk'));
        }

        $success = NexlifyDesk_Tickets::update_ticket_status($ticket_id, $status);

        if ($success) {
            wp_send_json_success(__('Ticket status updated!', 'nexlifydesk'));
        } else {
            wp_send_json_error(__('Could not update ticket status.', 'nexlifydesk'));
        }
    }
    
    public static function upload_attachment() {
        check_ajax_referer('nexlifydesk-ajax-nonce', 'nonce');
        
        if (!is_user_logged_in()) {
            wp_send_json_error(__('You must be logged in to upload files.', 'nexlifydesk'));
        }
        
        if (empty($_FILES['file']) || !is_array($_FILES['file'])) {
            wp_send_json_error(__('No file uploaded or invalid file format.', 'nexlifydesk'));
        }
    
        $file = array(
            'name'     => isset($_FILES['file']['name']) ? sanitize_file_name($_FILES['file']['name']) : '',
            'type'     => isset($_FILES['file']['type']) ? sanitize_mime_type($_FILES['file']['type']) : '',
            'tmp_name' => isset($_FILES['file']['tmp_name']) ? $_FILES['file']['tmp_name'] : '', // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
            'error'    => isset($_FILES['file']['error']) ? (int)$_FILES['file']['error'] : UPLOAD_ERR_NO_FILE,
            'size'     => isset($_FILES['file']['size']) ? (int)$_FILES['file']['size'] : 0
        );
    
        if ($file['error'] !== UPLOAD_ERR_OK) {
            wp_send_json_error(__('File upload error occurred.', 'nexlifydesk'));
        }
    
        if (!is_uploaded_file($file['tmp_name'])) {
            wp_send_json_error(__('Invalid file upload.', 'nexlifydesk'));
        }
    
        $file_ext = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
        $allowed_extensions = array('jpg', 'jpeg', 'png', 'pdf');
        
        if (!in_array($file_ext, $allowed_extensions, true)) {
            wp_send_json_error(__('File type not allowed. Only JPG, JPEG, PNG, and PDF files are permitted.', 'nexlifydesk'));
        }
    
        $finfo = finfo_open(FILEINFO_MIME_TYPE);
        $actual_mime = finfo_file($finfo, $file['tmp_name']);
        finfo_close($finfo);
        
        $allowed_mimes = array(
            'image/jpeg' => array('jpg', 'jpeg'),
            'image/png'  => array('png'),
            'application/pdf' => array('pdf')
        );
        
        $mime_valid = false;
        foreach ($allowed_mimes as $mime => $extensions) {
            if ($actual_mime === $mime && in_array($file_ext, $extensions, true)) {
                $mime_valid = true;
                break;
            }
        }
        
        if (!$mime_valid) {
            wp_send_json_error(__('File content does not match file extension.', 'nexlifydesk'));
        }
    
        $settings = get_option('nexlifydesk_settings', array());
        $max_size = (isset($settings['max_file_size']) ? (int)$settings['max_file_size'] : 2) * 1024 * 1024;
    
        if ($file['size'] > $max_size) {
            wp_send_json_error(
                sprintf(
                    /* translators: %s: Maximum file size in megabytes */
                    __('File size exceeds maximum limit of %dMB.', 'nexlifydesk'),
                    isset($settings['max_file_size']) ? (int)$settings['max_file_size'] : 2
                )
            );
        }
    
        $upload_overrides = array(
            'test_form' => false,
            'test_type' => true,
            'mimes' => array(
                'jpg|jpeg' => 'image/jpeg',
                'png'      => 'image/png',
                'pdf'      => 'application/pdf'
            )
        );
    
        $uploaded_file = wp_handle_upload($file, $upload_overrides);
    
        if ($uploaded_file && !isset($uploaded_file['error'])) {
            // File was successfully uploaded
            wp_send_json_success(array(
                'name' => basename($uploaded_file['file']),
                'url'  => $uploaded_file['url'],
                'type' => $file['type'],
                'size' => $file['size']
            ));
        } else {
            $error_message = isset($uploaded_file['error']) ? $uploaded_file['error'] : __('Could not upload file.', 'nexlifydesk');
            wp_send_json_error($error_message);
        }
    }
    
    public static function admin_get_tickets() {
        check_ajax_referer('nexlifydesk-ajax-nonce', 'nonce');
        
        if (!current_user_can('nexlifydesk_manage_tickets')) {
            wp_send_json_error(__('You do not have permission to view tickets.', 'nexlifydesk'));
        }
        
        try {
        
            $current_user = wp_get_current_user();
            $status = isset($_POST['status']) && $_POST['status'] !== 'all' ? sanitize_text_field(wp_unslash($_POST['status'])) : '';
            $search = isset($_POST['search']) ? sanitize_text_field(wp_unslash($_POST['search'])) : '';
            
            $cache_key = 'nexlifydesk_admin_tickets_' . md5($status . '_' . $search . '_' . $current_user->ID);
            $tickets = wp_cache_get($cache_key);
            
            if (false === $tickets) {
                global $wpdb;
                
                $base_query = "SELECT t.*, c.name as category_name 
                            FROM {$wpdb->prefix}nexlifydesk_tickets t 
                            LEFT JOIN {$wpdb->prefix}nexlifydesk_categories c ON t.category_id = c.id 
                            WHERE 1=1";
                
                $where_conditions = array();
                $params = array();
                
                if (!empty($status)) {
                    $where_conditions[] = "t.status = %s";
                    $params[] = $status;
                }
                
                if (!empty($search)) {
                    $search_clean = ltrim($search, '#');
                    $keywords = preg_split('/\s+/', $search_clean, -1, PREG_SPLIT_NO_EMPTY);
                    foreach ($keywords as $kw) {
                        $like = '%' . $wpdb->esc_like($kw) . '%';
                        $where_conditions[] = "(t.ticket_id LIKE %s OR t.subject LIKE %s OR t.message LIKE %s OR EXISTS (SELECT 1 FROM {$wpdb->prefix}nexlifydesk_replies r WHERE r.ticket_id = t.id AND r.message LIKE %s))";
                        $params[] = $like;
                        $params[] = $like;
                        $params[] = $like;
                        $params[] = $like;
                    }
                }
                
                if (!current_user_can('manage_options')) {
                    if (in_array('nexlifydesk_agent', $current_user->roles)) {
                        $can_view_all = false;
                        
                        $assigned_position_slug = get_user_meta($current_user->ID, 'nexlifydesk_agent_position', true);
                        
                        if (!empty($assigned_position_slug)) {
                            $positions = get_option('nexlifydesk_agent_positions', array());
                            
                            if (isset($positions[$assigned_position_slug]) && 
                                isset($positions[$assigned_position_slug]['capabilities']) &&
                                is_array($positions[$assigned_position_slug]['capabilities'])) {
                                $can_view_all = in_array('nexlifydesk_view_all_tickets', $positions[$assigned_position_slug]['capabilities']);
                            }
                        }
                        
                        if (!$can_view_all) {
                            $where_conditions[] = "t.assigned_to = %d";
                            $params[] = $current_user->ID;
                        }
                    } else {
                        $where_conditions[] = "t.user_id = %d";
                        $params[] = $current_user->ID;
                    }
                }

                $query = $base_query;
                if (!empty($where_conditions)) {
                    $query .= " AND " . implode(" AND ", $where_conditions);
                }
                $query .= " ORDER BY t.created_at DESC";
                
                if (!empty($params)) {
                    // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery -- Query is built safely with escaped table names and validated parameters
                    $tickets = $wpdb->get_results($wpdb->prepare($query, $params));
                } else {
                    // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery -- Query is built safely with escaped table names
                    $tickets = $wpdb->get_results($query);
                }
                
                if ($tickets === null) {
                    throw new Exception('Database query failed: ' . $wpdb->last_error);
                }
                
                if (is_array($tickets)) {
                    foreach ($tickets as $ticket) {
                        if (isset($ticket->category_id)) {
                            $ticket->category = NexlifyDesk_Tickets::get_category($ticket->category_id);
                        }
                        if (isset($ticket->created_at)) {
                            $ticket->created_at = gmdate(get_option('date_format') . ' ' . get_option('time_format'), strtotime($ticket->created_at));
                        }
                        if (!empty($ticket->assigned_to)) {
                            $agent = get_userdata($ticket->assigned_to);
                            $ticket->assigned_to_display_name = $agent ? $agent->display_name : __('Unassigned', 'nexlifydesk');
                        } else {
                            $ticket->assigned_to_display_name = __('Unassigned', 'nexlifydesk');
                        }
                    }
                }
                
                wp_cache_set($cache_key, $tickets, '', 120);
            }

            wp_send_json_success($tickets);
            
        } catch (Exception $e) {
            wp_send_json_error(__('Error loading tickets. Please try again or contact support.', 'nexlifydesk'));
        }
    }
    
    public static function admin_assign_ticket() {
        check_ajax_referer('nexlifydesk-ajax-nonce', 'nonce');
        
        if (
            !current_user_can('manage_options') &&
            !NexlifyDesk_Users::agent_can('nexlifydesk_assign_tickets', get_current_user_id())
        ) {
            wp_send_json_error(__('You do not have permission to assign tickets.', 'nexlifydesk'));
        }
        
        if (!isset($_POST['ticket_id']) || !isset($_POST['user_id'])) {
            wp_send_json_error(__('Missing required parameters.', 'nexlifydesk'));
        }
        
        $ticket_id = absint($_POST['ticket_id']);
        $user_id = absint($_POST['user_id']);
        
        if (!$ticket_id || !is_numeric($user_id)) {
            wp_send_json_error(__('Invalid ticket ID or user ID.', 'nexlifydesk'));
        }
        
        $ticket_before = NexlifyDesk_Tickets::get_ticket($ticket_id);
        
        global $wpdb;
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Table name is safe and controlled
        $result = $wpdb->update(
            NexlifyDesk_Database::get_table('tickets'),
            array(
                'assigned_to' => $user_id,
                'updated_at' => current_time('mysql')
            ),
            array('id' => $ticket_id),
            array('%d', '%s'),
            array('%d')
        );
        
        if ($result) {
            wp_cache_delete('nexlifydesk_ticket_' . intval($ticket_id));
            
            $current_user = wp_get_current_user();
            
            $statuses = array('all', 'open', 'pending', 'resolved', 'closed');
            foreach ($statuses as $status) {
                wp_cache_delete('nexlifydesk_admin_tickets_' . md5($status . '__' . $current_user->ID));
            }
            
            if ($ticket_before && !empty($ticket_before->assigned_to)) {
                foreach ($statuses as $status) {
                    wp_cache_delete('nexlifydesk_assigned_tickets_' . intval($ticket_before->assigned_to) . '_' . $status);
                }
            }
            
            foreach ($statuses as $status) {
                wp_cache_delete('nexlifydesk_assigned_tickets_' . intval($user_id) . '_' . $status);
            }
            
            if ($ticket_before && !empty($ticket_before->user_id)) {
                foreach ($statuses as $status) {
                    wp_cache_delete('nexlifydesk_user_tickets_' . intval($ticket_before->user_id) . '_' . $status);
                }
            }
            
            wp_send_json_success(__('Ticket assigned successfully!', 'nexlifydesk'));
        } else {
            wp_send_json_error(__('Could not assign ticket.', 'nexlifydesk'));
        }
    }

    public static function add_category() {
        if (ob_get_level()) {
            ob_end_clean();
        }
        ob_start();
        
        if (!headers_sent()) {
            header('Content-Type: application/json; charset=utf-8');
        }
        
        if (!isset($_POST['nexlifydesk_category_nonce']) || 
            !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nexlifydesk_category_nonce'])), 'nexlifydesk_save_category')) {
            ob_end_clean(); 
            wp_send_json_error(__('Security check failed. Please refresh the page and try again.', 'nexlifydesk'));
            exit;
        }

        if (!current_user_can('manage_options')) {
            ob_end_clean(); 
            wp_send_json_error(__('You do not have permission to add categories.', 'nexlifydesk'));
            exit;
        }

        global $wpdb;

        if (empty($_POST['category_name'])) {
            ob_end_clean(); 
            wp_send_json_error(__('Category name is required.', 'nexlifydesk'));
            exit;
        }

        $category_name = sanitize_text_field(wp_unslash($_POST['category_name']));
        $category_description = isset($_POST['category_description']) ? 
            sanitize_textarea_field(wp_unslash($_POST['category_description'])) : '';
        $slug = sanitize_title($category_name);

        if (empty($slug)) {
            ob_end_clean(); 
            wp_send_json_error(__('Invalid category name - unable to generate slug.', 'nexlifydesk'));
            exit;
        }

        $table_name = NexlifyDesk_Database::get_table('categories');

        $cache_key = 'nexlifydesk_category_slug_check_' . md5($slug);
        wp_cache_delete($cache_key);

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Table name is safe and controlled, no caching needed for real-time category existence check to prevent race conditions
        $existing = $wpdb->get_row(
            $wpdb->prepare(
                "SELECT id, is_active FROM {$table_name} WHERE slug = %s",
                $slug
            )
        );

        if ($existing) {
            if ($existing->is_active == 1) {
                ob_end_clean(); 
                wp_send_json_error(__('A category with this name already exists.', 'nexlifydesk'));
                exit;
            } else {
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Table name is safe and controlled
                $result = $wpdb->update(
                    NexlifyDesk_Database::get_table('categories'),
                    array(
                        'name' => $category_name,
                        'description' => $category_description,
                        'is_active' => 1
                    ),
                    array('id' => $existing->id),
                    array('%s', '%s', '%d'),
                    array('%d')
                );

                if ($result === false) {
                    ob_end_clean(); 
                    wp_send_json_error(
                        sprintf(
                            /* translators: %s: Database error message */
                            __('Failed to reactivate category: %s', 'nexlifydesk'),
                            $wpdb->last_error ?: 'Unknown database error'
                        )
                    );
                    exit;
                }

                NexlifyDesk_Admin::clear_all_category_caches($existing->id);

                ob_end_clean();
                wp_send_json_success(array(
                    'message' => __('Category reactivated successfully!', 'nexlifydesk'),
                    'category_id' => $existing->id
                ));
                exit;
            }
        }

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Table name is safe and controlled
        $result = $wpdb->insert(
            NexlifyDesk_Database::get_table('categories'),
            array(
                'name' => $category_name,
                'slug' => $slug,
                'description' => $category_description,
                'is_active' => 1
            ),
            array('%s', '%s', '%s', '%d')
        );

        if ($result === false) {
            ob_end_clean();
            wp_send_json_error(
                sprintf(
                    /* translators: %s: Database error message */
                    __('Failed to add category: %s', 'nexlifydesk'),
                    $wpdb->last_error ?: 'Unknown database error'
                )
            );
            exit;
        }

        // Clear caches
        NexlifyDesk_Admin::clear_all_category_caches($wpdb->insert_id);

        ob_end_clean();
        wp_send_json_success(array(
            'message' => __('Category added successfully!', 'nexlifydesk')
        ));
        exit;
    }

    public static function delete_category() {
        check_ajax_referer('nexlifydesk-ajax-nonce', 'nonce');
    
        if (!current_user_can('manage_options')) {
            wp_send_json_error(__('You do not have permission to delete categories.', 'nexlifydesk'));
        }
    
        if (!isset($_POST['category_id'])) {
            wp_send_json_error(__('Category ID is required.', 'nexlifydesk'));
        }
    
        global $wpdb;
        $category_id = absint($_POST['category_id']);
        
        if (!$category_id) {
            wp_send_json_error(__('Invalid category ID.', 'nexlifydesk'));
        }

        $ticket_count = NexlifyDesk_Admin::get_category_ticket_count($category_id);
        
        if ($ticket_count > 0) {
            $available_categories = NexlifyDesk_Admin::get_categories_for_reassignment($category_id);
            
            if (empty($available_categories)) {
                wp_send_json_error(
                    sprintf(
                        /* translators: %d: Number of tickets */
                        __('Cannot delete category. It contains %d tickets and there are no other categories available for reassignment. Please create another category first.', 'nexlifydesk'),
                        $ticket_count
                    )
                );
            }
            
            $reassign_to = isset($_POST['reassign_to']) ? absint($_POST['reassign_to']) : 0;
            
            if (!$reassign_to) {
                wp_send_json_error(array(
                    'type' => 'reassignment_required',
                    'message' => sprintf(
                        /* translators: %d: Number of tickets */
                        __('This category contains %d tickets. Please select a category to reassign them to:', 'nexlifydesk'),
                        $ticket_count
                    ),
                    'ticket_count' => $ticket_count,
                    'available_categories' => $available_categories
                ));
            }
            
            $valid_reassignment = false;
            foreach ($available_categories as $cat) {
                if ($cat->id == $reassign_to) {
                    $valid_reassignment = true;
                    break;
                }
            }
            
            if (!$valid_reassignment) {
                wp_send_json_error(__('Invalid reassignment category selected.', 'nexlifydesk'));
            }
            
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Table name is safe and controlled, no caching needed for real-time category existence check to prevent race conditions
            $reassign_result = $wpdb->update(
                $wpdb->prefix . 'nexlifydesk_tickets',
                array('category_id' => $reassign_to),
                array('category_id' => $category_id),
                array('%d'),
                array('%d')
            );
            
            if ($reassign_result === false) {
                wp_send_json_error(__('Failed to reassign tickets. Category deletion cancelled.', 'nexlifydesk'));
            }
        }
        
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Table name is safe and controlled, no caching needed for real-time category existence check to prevent race conditions
        $result = $wpdb->update(
            NexlifyDesk_Database::get_table('categories'),
            array('is_active' => 0),
            array('id' => $category_id),
            array('%d'),
            array('%d')
        );
    
        if ($result) {
            NexlifyDesk_Admin::clear_all_category_caches($category_id);
            
            $success_message = __('Category deleted successfully!', 'nexlifydesk');
            if ($ticket_count > 0) {
                $success_message = sprintf(
                    /* translators: 1: Number of tickets */
                    __('Category deleted successfully! %1$d tickets were reassigned to the selected category.', 'nexlifydesk'),
                    $ticket_count
                );
            }
            
            wp_send_json_success($success_message);
        } else {
            wp_send_json_error(__('Could not delete category.', 'nexlifydesk'));
        }
    }

    public static function reassign_orphaned_tickets_ajax() {
        check_ajax_referer('nexlifydesk-ajax-nonce', 'nonce');
        if (!current_user_can('manage_options')) {
            wp_send_json_error(__('You do not have permission.', 'nexlifydesk'));
        }
        $count = NexlifyDesk_Tickets::handle_orphaned_tickets();
        if ($count > 0) {
            /* translators: %d: Number of orphaned tickets reassigned */
            wp_send_json_success(sprintf(__('Orphaned tickets reassigned: %d', 'nexlifydesk'), $count));
        } else {
            wp_send_json_success(__('No orphaned tickets found to reassign.', 'nexlifydesk'));
        }
    }

    /**
     * Test AWS WorkMail connection
     */
    public static function test_aws_connection() {
        
        if (!current_user_can('nexlifydesk_manage_tickets') && !current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('You do not have permission.', 'nexlifydesk')));
        }

        $imap_available = extension_loaded('imap') && function_exists('imap_open');
        if (!$imap_available) {
            wp_send_json_error(array('message' => __('IMAP extension is not properly installed or configured on this server. Please contact your hosting provider to enable the PHP IMAP extension with all required functions.', 'nexlifydesk')));
        }

        if (
            !isset($_POST['nonce']) ||
            !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'nexlifydesk_aws_test')
        ) {
            wp_send_json_error(array('message' => __('Invalid nonce.', 'nexlifydesk')));
        }

        $region = sanitize_text_field(wp_unslash($_POST['region'] ?? ''));
        $organization_id = sanitize_text_field(wp_unslash($_POST['organization_id'] ?? ''));
        $email = sanitize_email(wp_unslash($_POST['email'] ?? ''));
        $password = sanitize_text_field(wp_unslash($_POST['password'] ?? ''));
        $access_key_id = sanitize_text_field(wp_unslash($_POST['access_key_id'] ?? ''));
        $secret_access_key = sanitize_text_field(wp_unslash($_POST['secret_access_key'] ?? ''));

        if ($password === '••••••••••••••••') {
            $current_settings = get_option('nexlifydesk_imap_settings', []);
            if (!empty($current_settings['aws_password'])) {
                $password = nexlifydesk_decrypt($current_settings['aws_password']);
            }
        }

        $is_ssl_enabled = (
            is_ssl() || 
            (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on') ||
            (isset($_SERVER['HTTP_X_FORWARDED_PROTO']) && $_SERVER['HTTP_X_FORWARDED_PROTO'] === 'https') ||
            (isset($_SERVER['HTTP_X_FORWARDED_SSL']) && $_SERVER['HTTP_X_FORWARDED_SSL'] === 'on') ||
            (isset($_SERVER['REQUEST_SCHEME']) && $_SERVER['REQUEST_SCHEME'] === 'https') ||
            (wp_parse_url(home_url(), PHP_URL_SCHEME) === 'https')
        );
        
        if (defined('NEXLIFYDESK_FORCE_SSL_ENABLED') && NEXLIFYDESK_FORCE_SSL_ENABLED) {
            $is_ssl_enabled = true;
        }
        
        if (!$is_ssl_enabled) {
            wp_send_json_error(array('message' => __('SSL certificate is required for AWS WorkMail integration.', 'nexlifydesk')));
        }

        if (empty($region) || empty($organization_id) || empty($email) || empty($password)) {
            wp_send_json_error(array('message' => __('Please fill in all required fields: AWS Region, Organization ID, Email Address, and Email Password.', 'nexlifydesk')));
        }

        try {
            $messages = [];
            
            $imap_host = "imap.mail.{$region}.awsapps.com";
            $imap_port = 993;
            
            if (function_exists('imap_errors')) {
                imap_errors();
            }
            if (function_exists('imap_alerts')) {
                imap_alerts();
            }
            
            $connection = @imap_open(
                "{{$imap_host}:{$imap_port}/imap/ssl}INBOX",
                $email,
                $password,
                OP_READONLY | OP_SILENT
            );

            if (!$connection) {
                $error = 'Unknown connection error';
                
                if (function_exists('imap_last_error')) {
                    $last_error = imap_last_error();
                    if ($last_error) {
                        $error = $last_error;
                    }
                }
                
                $error_message = sprintf(
                    /* translators: %1$s: Error message from IMAP connection */
                    __('AWS WorkMail IMAP connection failed: %1$s', 'nexlifydesk'),
                    $error
                );
                
                if (strpos($error, 'connect') !== false || strpos($error, 'timeout') !== false) {
                    $error_message .= ' ' . __('This usually indicates a network connectivity issue or firewall blocking. Please verify your server can connect to AWS WorkMail servers.', 'nexlifydesk');
                } elseif (strpos($error, 'authenticate') !== false || strpos($error, 'login') !== false) {
                    /* translators: 1: Organization ID, 2: Email, 3: AWS Region */
                    $error_message .= ' ' . sprintf(__('Please verify your Organization ID (%1$s), Email (%2$s), and Password are correct for region %3$s.', 'nexlifydesk'), $organization_id, $email, $region);
                } else {
                    /* translators: 1: Organization ID, 2: Email, 3: AWS Region */
                    $error_message .= ' ' . sprintf(__('Please verify your AWS WorkMail configuration: Organization ID (%1$s), Email (%2$s), Region (%3$s).', 'nexlifydesk'), $organization_id, $email, $region);
                }
                
                wp_send_json_error(array('message' => $error_message));
            }

            imap_close($connection);
            /* translators: 1: Email address, 2: AWS region */
            $messages[] = sprintf(__('AWS WorkMail IMAP connection successful for %1$s in region %2$s', 'nexlifydesk'), $email, $region);
            
            if (!empty($access_key_id) && !empty($secret_access_key)) {
                $original_settings = get_option('nexlifydesk_imap_settings', []);
                $test_settings = array_merge($original_settings, [
                    'aws_region' => $region,
                    'aws_access_key_id' => $access_key_id,
                    'aws_secret_access_key' => $secret_access_key
                ]);
                update_option('nexlifydesk_imap_settings', $test_settings);
                
                $aws_handler_path = plugin_dir_path(dirname(__FILE__)) . 'email-source/providers/aws-ses/aws-handler.php';
                if (file_exists($aws_handler_path)) {
                    require_once $aws_handler_path;
                    
                    $auth = new NexlifyDesk_AWS_Auth();
                    $ses_result = $auth->validate_ses_credentials();
                    
                    if (is_wp_error($ses_result)) {
                        $messages[] = 'AWS SES credentials validation failed: ' . $ses_result->get_error_message();
                    } else {
                        $messages[] = 'AWS SES credentials validated successfully';
                    }
                }
                
                update_option('nexlifydesk_imap_settings', $original_settings);
            } else {
                $messages[] = 'AWS API credentials (Access Key ID, Secret Access Key) not provided - SES features will not be available';
            }

            $final_message = implode('. ', $messages);
            
            $note = ' ' . __('Note: A successful connection test means IMAP authentication works. If email fetching still has issues, check your email provider settings, firewall, or contact support.', 'nexlifydesk');
            
            wp_send_json_success(array('message' => $final_message . $note));

        } catch (Exception $e) {
            /* translators: %s: Error message returned from the connection test */
            wp_send_json_error(array('message' => sprintf(__('Connection test failed: %s', 'nexlifydesk'), $e->getMessage())));
        }
    }

    /*
    * Test custom IMAP connection
    */
    public static function test_custom_connection() {
        check_ajax_referer('nexlifydesk-ajax-nonce', 'nonce');

        if (!extension_loaded('imap')) {
            wp_send_json_error(array('message' => __('IMAP extension is not installed on this server. Please contact your hosting provider to enable the PHP IMAP extension.', 'nexlifydesk')));
        }

        $host = isset($_POST['host']) ? sanitize_text_field(wp_unslash($_POST['host'])) : '';
        $port = isset($_POST['port']) ? intval($_POST['port']) : 993;
        $username = isset($_POST['username']) ? sanitize_text_field(wp_unslash($_POST['username'])) : '';
        $password = isset($_POST['password']) ? sanitize_text_field(wp_unslash($_POST['password'])) : '';
        $encryption = isset($_POST['encryption']) ? sanitize_text_field(wp_unslash($_POST['encryption'])) : 'ssl';
        $protocol = isset($_POST['protocol']) ? sanitize_text_field(wp_unslash($_POST['protocol'])) : 'imap';

        if ($password === '••••••••••••••••') {
            $settings = get_option('nexlifydesk_imap_settings', array());
            $password = nexlifydesk_get_safe_password($settings['password'] ?? '');
        }

        if (empty($host) || empty($port) || empty($username) || empty($password)) {
            wp_send_json_error('Missing required connection details.');
        }

        $mailbox = "{" . $host . ":" . $port . "/" . $protocol . "/" . $encryption . "}INBOX";
        $inbox = @imap_open($mailbox, $username, $password, OP_READONLY);

        if ($inbox) {
            imap_close($inbox);
            wp_send_json_success(['message' => 'Connection successful!']);
        } else {
            $error = imap_last_error();
            wp_send_json_error(['message' => 'Connection failed!']);
        }
    }

    /**
     * Manual email fetch for testing
     */
    public static function manual_fetch_emails() {

        if (!current_user_can('nexlifydesk_manage_tickets') && !current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('You do not have permission.', 'nexlifydesk')));
        }

        $provider = isset($_POST['provider']) ? sanitize_text_field(wp_unslash($_POST['provider'])) : '';
        $nonce = isset($_POST['nonce']) ? sanitize_text_field(wp_unslash($_POST['nonce'])) : '';

        if ($provider === 'custom') {
            if (!$nonce || !wp_verify_nonce($nonce, 'nexlifydesk-ajax-nonce')) {
                wp_send_json_error(array('message' => __('Invalid nonce.', 'nexlifydesk')));
            }
            
            $result = nexlifydesk_fetch_custom_emails();
            
            if (isset($result['error'])) {
                wp_send_json_error(array('message' => $result['error']));
            } else {
                wp_send_json_success(array('message' => $result['message'] ?? 'Custom email fetch test successful!'));
            }
        }

        if (!$nonce || !wp_verify_nonce($nonce, 'nexlifydesk_aws_test')) {
            wp_send_json_error(array('message' => __('Invalid nonce.', 'nexlifydesk')));
        }

        $region = sanitize_text_field(wp_unslash($_POST['region'] ?? ''));
        $organization_id = sanitize_text_field(wp_unslash($_POST['organization_id'] ?? ''));
        $email = sanitize_email(wp_unslash($_POST['email'] ?? ''));
        $password = sanitize_text_field(wp_unslash($_POST['password'] ?? ''));
        $access_key_id = sanitize_text_field(wp_unslash($_POST['access_key_id'] ?? ''));
        $secret_access_key = sanitize_text_field(wp_unslash($_POST['secret_access_key'] ?? ''));

        if ($password === '••••••••••••••••') {
            $current_settings = get_option('nexlifydesk_imap_settings', []);
            if (!empty($current_settings['aws_password'])) {
                $password = nexlifydesk_decrypt($current_settings['aws_password']);
            }
        }

        if (empty($region) || empty($organization_id) || empty($email) || empty($password)) {
            wp_send_json_error(array('message' => __('Please fill in all required AWS fields first.', 'nexlifydesk')));
        }

        try {
            $current_settings = get_option('nexlifydesk_imap_settings', []);
            $test_settings = array_merge($current_settings, [
                'enabled' => true,
                'provider' => 'aws',
                'aws_region' => $region,
                'aws_organization_id' => $organization_id,
                'aws_email' => $email,
                'aws_password' => $password
            ]);
            
            if (!empty($access_key_id) && !empty($secret_access_key)) {
                $test_settings['aws_access_key_id'] = $access_key_id;
                $test_settings['aws_secret_access_key'] = $secret_access_key;
            }
            
            update_option('nexlifydesk_imap_settings', $test_settings);
            
            $email_pipe_path = plugin_dir_path(dirname(__FILE__)) . 'email-source/nexlifydesk-email-pipe.php';
            if (file_exists($email_pipe_path)) {
                require_once $email_pipe_path;
            }
            
            $aws_handler_path = plugin_dir_path(dirname(__FILE__)) . 'email-source/providers/aws-ses/aws-handler.php';
            if (file_exists($aws_handler_path)) {
                require_once $aws_handler_path;
            }
            
            if (function_exists('nexlifydesk_fetch_emails')) {
                nexlifydesk_fetch_emails();
                
                if (function_exists('nexlifydesk_fetch_aws_emails')) {
                    $result = nexlifydesk_fetch_aws_emails();
                    
                    if (is_array($result) && isset($result['error'])) {
                        update_option('nexlifydesk_imap_settings', $current_settings);
                        wp_send_json_error(array('message' => $result['error']));
                        return;
                    }
                    
                    if (is_array($result) && isset($result['success']) && $result['success']) {
                        update_option('nexlifydesk_imap_settings', $current_settings);
                        wp_send_json_success(array('message' => $result['message']));
                        return;
                    }
                }
            } else {
                throw new Exception('Email fetch function not found');
            }
            
            update_option('nexlifydesk_imap_settings', $current_settings);
            
            global $wpdb;
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table name is safe and not user input
            $table_name = $wpdb->prefix . 'nexlifydesk_tickets';
            $recent_tickets_cache_key = 'nexlifydesk_recent_email_tickets_' . md5(gmdate('Y-m-d H:i:s', strtotime('-2 minutes')));
            $recent_tickets = wp_cache_get($recent_tickets_cache_key);

            if (false === $recent_tickets) {
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Table name is safe and not user input
                $recent_tickets = $wpdb->get_results(
                    $wpdb->prepare(
                        "SELECT ticket_id, subject, source, created_at 
                        FROM {$table_name} 
                        WHERE created_at >= %s 
                        AND source = 'email'
                        ORDER BY created_at DESC
                        LIMIT 10",
                        gmdate('Y-m-d H:i:s', strtotime('-2 minutes'))
                    )
                );
                wp_cache_set($recent_tickets_cache_key, $recent_tickets, '', 120);
            }
            
            $message = "Email fetch completed successfully!\n";
            
            if (!empty($recent_tickets)) {
                $message .= "Found " . count($recent_tickets) . " recent email ticket(s):\n";
                foreach ($recent_tickets as $ticket) {
                    $message .= "- #{$ticket->ticket_id}: {$ticket->subject} ({$ticket->created_at})\n";
                }
            } else {
                $message .= "No new email tickets found. This could mean:\n";
                $message .= "- No new emails in the mailbox\n";
                $message .= "- Emails are being filtered out (spam, duplicates)\n";
                $message .= "- All emails have been processed already\n";
            }
            
            wp_send_json_success(array('message' => $message));
            
        } catch (Exception $e) {
            if (isset($current_settings)) {
                update_option('nexlifydesk_imap_settings', $current_settings);
            }
            
            /* translators: %s: Error message returned from the email fetch */
            wp_send_json_error(array('message' => sprintf(__('Email fetch failed: %s', 'nexlifydesk'), $e->getMessage())));
        }
    }

    /**
     * Handle bulk actions for tickets
     */
    public static function handle_bulk_action() {
        // Check nonce
        check_ajax_referer('nexlifydesk-ajax-nonce', 'nonce');
        
        // Check permissions
        if (!current_user_can('nexlifydesk_manage_tickets')) {
            wp_send_json_error(__('You do not have permission to perform this action.', 'nexlifydesk'));
        }
        
        $bulk_action = sanitize_text_field(wp_unslash($_POST['bulk_action'] ?? ''));
        $ticket_ids = array_map('intval', (array) ($_POST['ticket_ids'] ?? []));
        
        if (empty($bulk_action) || empty($ticket_ids)) {
            wp_send_json_error(__('Invalid bulk action or no tickets selected.', 'nexlifydesk'));
        }
        
        $updated_count = 0;
        $errors = [];
        
        foreach ($ticket_ids as $ticket_id) {
            $ticket = NexlifyDesk_Tickets::get_ticket($ticket_id);
            if (!$ticket) {
                /* translators: %d: Ticket ID */
                $errors[] = sprintf(__('Ticket #%d not found.', 'nexlifydesk'), $ticket_id);
                continue;
            }
            
            $current_user = wp_get_current_user();
            $is_agent = in_array('nexlifydesk_agent', $current_user->roles) && !in_array('administrator', $current_user->roles);
            if ($is_agent && (int)$ticket->assigned_to !== (int)$current_user->ID) {
                /* translators: %d: Ticket ID */
                $errors[] = sprintf(__('You cannot perform actions on ticket #%d as it is not assigned to you.', 'nexlifydesk'), $ticket_id);
                continue;
            }
            
            try {
                switch ($bulk_action) {
                    case 'assign':
                        $agent_id = intval($_POST['agent_id'] ?? 0);
                        $result = NexlifyDesk_Tickets::assign_ticket($ticket_id, $agent_id);
                        if ($result) {
                            $updated_count++;
                        } else {
                            /* translators: %d: Ticket ID */
                            $errors[] = sprintf(__('Failed to assign ticket #%d.', 'nexlifydesk'), $ticket_id);
                        }
                        break;
                        
                    case 'status':
                        $status = sanitize_text_field(wp_unslash($_POST['status'] ?? ''));
                        $allowed_statuses = ['open', 'in_progress', 'pending', 'resolved', 'closed'];
                        if (!in_array($status, $allowed_statuses)) {
                            /* translators: 1: Status, 2: Ticket ID */
                            $errors[] = sprintf(__('Invalid status "%1$s" for ticket #%2$d.', 'nexlifydesk'), $status, $ticket_id);
                            continue 2;
                        }
                        
                        $result = NexlifyDesk_Tickets::update_ticket_status($ticket_id, $status);
                        if ($result) {
                            $updated_count++;
                        } else {
                            /* translators: %d: Ticket ID */
                            $errors[] = sprintf(__('Failed to update status for ticket #%d.', 'nexlifydesk'), $ticket_id);
                        }
                        break;
                        
                    case 'priority':
                        $priority = sanitize_text_field(wp_unslash($_POST['priority'] ?? ''));
                        $allowed_priorities = ['low', 'medium', 'high', 'urgent'];
                        if (!in_array($priority, $allowed_priorities)) {
                            /* translators: 1: Priority, 2: Ticket ID */
                            $errors[] = sprintf(__('Invalid priority "%1$s" for ticket #%2$d.', 'nexlifydesk'), $priority, $ticket_id);
                            continue 2;
                        }
                        
                        $result = NexlifyDesk_Tickets::update_ticket_priority($ticket_id, $priority);
                        if ($result) {
                            $updated_count++;
                        } else {
                            /* translators: %d: Ticket ID */
                            $errors[] = sprintf(__('Failed to update priority for ticket #%d.', 'nexlifydesk'), $ticket_id);
                        }
                        break;
                        
                    case 'delete':
                        if (!current_user_can('manage_options')) {
                            /* translators: %d: Ticket ID */
                            $errors[] = sprintf(__('You do not have permission to delete ticket #%d.', 'nexlifydesk'), $ticket_id);
                            continue 2;
                        }
                        
                        $result = NexlifyDesk_Tickets::delete_ticket($ticket_id);
                        if ($result) {
                            $updated_count++;
                        } else {
                            /* translators: %d: Ticket ID */
                            $errors[] = sprintf(__('Failed to delete ticket #%d.', 'nexlifydesk'), $ticket_id);
                        }
                        break;
                        
                    default:
                        /* translators: %s: Bulk action name */
                        $errors[] = sprintf(__('Unknown bulk action "%s".', 'nexlifydesk'), $bulk_action);
                        break;
                }
            } catch (Exception $e) {
                /* translators: 1: Ticket ID, 2: Error message */
                $errors[] = sprintf(__('Error processing ticket #%1$d: %2$s', 'nexlifydesk'), $ticket_id, $e->getMessage());
            }
        }
        
        $messages = [];
        if ($updated_count > 0) {
            if ($bulk_action === 'delete') {
                /* translators: %d: Number of tickets deleted */
                $messages[] = sprintf(_n('%d ticket deleted permanently.', '%d tickets deleted permanently.', $updated_count, 'nexlifydesk'), $updated_count);
            } else {
                /* translators: %d: Number of tickets updated */
                $messages[] = sprintf(_n('%d ticket updated successfully.', '%d tickets updated successfully.', $updated_count, 'nexlifydesk'), $updated_count);
            }
        }
        
        if (!empty($errors)) {
            $messages[] = __('Errors occurred:', 'nexlifydesk') . ' ' . implode(', ', $errors);
        }
        
        if ($updated_count > 0) {
            wp_send_json_success(implode(' ', $messages));
        } else {
            wp_send_json_error(implode(' ', $messages));
        }
    }

    /**
     * Refresh ticket list - returns updated ticket data
     */
    public static function refresh_ticket_list() {
        if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'nexlifydesk-ajax-nonce')) {
            wp_send_json_error(array(
                'message' => __('Session expired. Please reload the page.', 'nexlifydesk'),
                'code' => 'nonce_expired'
            ));
        }
        
        if (!current_user_can('nexlifydesk_manage_tickets')) {
            wp_send_json_error(array('message' => __('You do not have permission to view tickets.', 'nexlifydesk')));
        }
        
        try {
            $last_refresh = isset($_POST['last_refresh']) ? intval($_POST['last_refresh']) : 0;
            
            wp_cache_flush_group('nexlifydesk_tickets_grid');
            
            $tickets = NexlifyDesk_Tickets::get_tickets_for_grid(['per_page' => 100]);
            
            if (!is_array($tickets)) {
                $tickets = array();
            }
            
            $formatted_tickets = array();
            $current_user_id = get_current_user_id();
            
            foreach ($tickets as $ticket) {
                if (!is_object($ticket) || !isset($ticket->id)) {
                    continue;
                }
                
                $user = get_userdata($ticket->user_id);
                $assigned_agent = $ticket->assigned_to ? get_userdata($ticket->assigned_to) : null;
                
                if (!$user) {
                    $customer_details = function_exists('nexlifydesk_extract_customer_details') ? 
                        nexlifydesk_extract_customer_details($ticket->message) : 
                        ['name' => '', 'email' => '', 'message' => $ticket->message];
                    $customer_name = $customer_details['name'] ?: 'Guest';
                    $customer_email = $customer_details['email'] ?: 'N/A';
                } else {
                    $customer_name = $user->display_name;
                    $customer_email = $user->user_email;
                }
                
                $is_unread = 0;
                if (isset($ticket->is_unread)) {
                    $is_unread = intval($ticket->is_unread);
                } else {
                    global $wpdb;
                    $cache_key = 'nexlifydesk_ticket_read_' . intval($ticket->id) . '_' . intval($current_user_id);
                    $read_record = wp_cache_get($cache_key, '', 60);
                    if (false === $read_record) {
                        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Custom table requires direct query
                        $read_record = $wpdb->get_row(
                            $wpdb->prepare(
                                "SELECT read_at FROM {$wpdb->prefix}nexlifydesk_ticket_reads 
                                 WHERE ticket_id = %d AND user_id = %d",
                                $ticket->id,
                                $current_user_id
                            )
                        );
                        wp_cache_set($cache_key, $read_record, '', 60);
                    }
                    
                    if (!$read_record) {
                        $is_unread = 1;
                    } else {
                        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Custom table requires direct query
                        $newer_replies = $wpdb->get_var(
                            $wpdb->prepare(
                                "SELECT COUNT(*) FROM {$wpdb->prefix}nexlifydesk_replies 
                                 WHERE ticket_id = %d AND user_id != %d 
                                 AND created_at > %s AND is_internal_note = 0",
                                $ticket->id,
                                $current_user_id,
                                $read_record->read_at
                            )
                        );
                        $is_unread = $newer_replies > 0 ? 1 : 0;
                    }
                }
                
                $formatted_tickets[] = array(
                    'id' => $ticket->id,
                    'ticket_id' => $ticket->ticket_id,
                    'subject' => $ticket->subject,
                    'message' => $ticket->message,
                    'status' => $ticket->status,
                    'priority' => $ticket->priority,
                    'user_id' => $ticket->user_id,
                    'user_name' => $customer_name,
                    'user_email' => $customer_email,
                    'assigned_to' => $ticket->assigned_to,
                    'assigned_to_display_name' => $assigned_agent ? $assigned_agent->display_name : '',
                    'created_at' => $ticket->created_at,
                    'updated_at' => $ticket->updated_at,
                    'last_reply_time' => isset($ticket->last_reply_time) ? $ticket->last_reply_time : $ticket->created_at,
                    'is_unread' => $is_unread
                );
            }
            
            $response_data = array(
                'tickets' => $formatted_tickets,
                'new_nonce' => wp_create_nonce('nexlifydesk-ajax-nonce')
            );
            
            wp_send_json_success($response_data);
            
        } catch (Exception $e) {
            wp_send_json_error(array(
                'message' => __('Error refreshing ticket list.', 'nexlifydesk'),
                'error' => $e->getMessage()
            ));
        } catch (Error $e) {
            wp_send_json_error(array(
                'message' => __('Fatal error refreshing ticket list.', 'nexlifydesk'),
                'error' => $e->getMessage()
            ));
        }
    }

    /**
     * Mark ticket as read
     */
    public static function mark_ticket_read() {
        // Check nonce
        check_ajax_referer('nexlifydesk-ajax-nonce', 'nonce');
        
        // Check permissions
        if (!current_user_can('nexlifydesk_manage_tickets')) {
            wp_send_json_error(array('message' => __('You do not have permission to mark tickets as read.', 'nexlifydesk')));
        }
        
        if (!isset($_POST['ticket_id'])) {
            wp_send_json_error(array('message' => __('Ticket ID is required.', 'nexlifydesk')));
        }
        
        $ticket_id = intval($_POST['ticket_id']);
        
        try {
            $current_user_id = get_current_user_id();
            
            global $wpdb;
            $reads_table = $wpdb->prefix . 'nexlifydesk_ticket_reads';
            
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Custom table requires direct REPLACE query
            $result = $wpdb->replace(
                $reads_table,
                array(
                    'ticket_id' => $ticket_id,
                    'user_id' => $current_user_id,
                    'read_at' => current_time('mysql')
                ),
                array('%d', '%d', '%s')
            );
            // Clear cache for this ticket/user read status
            $cache_key = 'nexlifydesk_ticket_read_' . intval($ticket_id) . '_' . intval($current_user_id);
            wp_cache_delete($cache_key);
            
            if ($result === false) {
                throw new Exception('Failed to update read status in database');
            }
            
            wp_cache_flush_group('nexlifydesk_tickets_grid');
            
            wp_send_json_success(array('message' => __('Ticket marked as read.', 'nexlifydesk')));
            
        } catch (Exception $e) {
            wp_send_json_error(array('message' => __('Error marking ticket as read.', 'nexlifydesk')));
        }
    }

    /**
     * Test Google connection
     */
    public static function test_google_connection() {
        check_ajax_referer('nexlifydesk-ajax-nonce', 'nonce');

        if (!current_user_can('nexlifydesk_manage_tickets') && !current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('You do not have permission.', 'nexlifydesk')));
        }

        $client_id = isset($_POST['client_id']) ? sanitize_text_field(wp_unslash($_POST['client_id'])) : '';
        $client_secret = isset($_POST['client_secret']) ? sanitize_text_field(wp_unslash($_POST['client_secret'])) : '';

        if ($client_secret === '••••••••••••••••') {
            $settings = get_option('nexlifydesk_imap_settings', array());
            $client_secret = !empty($settings['google_client_secret']) ? nexlifydesk_decrypt($settings['google_client_secret']) : '';
        }

        if (empty($client_id) || empty($client_secret)) {
            wp_send_json_error(array('message' => __('Please fill in both Google Client ID and Client Secret.', 'nexlifydesk')));
        }

        try {
            $email_pipe_path = plugin_dir_path(dirname(__FILE__)) . 'email-source/nexlifydesk-email-pipe.php';
            if (file_exists($email_pipe_path)) {
                require_once $email_pipe_path;
            }

            if (function_exists('nexlifydesk_get_google_access_token')) {
                $access_token = nexlifydesk_get_google_access_token();
                if ($access_token) {
                    $response = wp_remote_get('https://www.googleapis.com/oauth2/v2/userinfo', [
                        'headers' => ['Authorization' => 'Bearer ' . $access_token]
                    ]);

                    if (!is_wp_error($response) && wp_remote_retrieve_response_code($response) === 200) {
                        $user_info = json_decode(wp_remote_retrieve_body($response), true);
                        $email = $user_info['email'] ?? 'Unknown';
                        wp_send_json_success(array('message' => "Google connection successful! Authorized as: {$email}"));
                    } else {
                        wp_send_json_error(array('message' => __('Google connection failed. Please re-authorize.', 'nexlifydesk')));
                    }
                } else {
                    wp_send_json_error(array('message' => __('Google not authorized. Please complete the OAuth authorization first.', 'nexlifydesk')));
                }
            } else {
                wp_send_json_error(array('message' => __('Google handler functions not available.', 'nexlifydesk')));
            }
        } catch (Exception $e) {
            /* translators: %s: Error message returned from the Google connection test */
            wp_send_json_error(array('message' => sprintf(__('Google connection test failed: %s', 'nexlifydesk'), $e->getMessage())));
        }
    }

    /**
     * Manual Google email fetch for testing
     */
    public static function manual_fetch_google_emails() {
        check_ajax_referer('nexlifydesk-ajax-nonce', 'nonce');

        if (!current_user_can('nexlifydesk_manage_tickets') && !current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('You do not have permission.', 'nexlifydesk')));
        }

        try {
            $email_pipe_path = plugin_dir_path(dirname(__FILE__)) . 'email-source/nexlifydesk-email-pipe.php';
            if (file_exists($email_pipe_path)) {
                require_once $email_pipe_path;
            }

            if (function_exists('nexlifydesk_fetch_google_emails')) {
                $result = nexlifydesk_fetch_google_emails();
                
                if (is_array($result) && isset($result['error'])) {
                    wp_send_json_error(array('message' => $result['error']));
                    return;
                }
                
                if (is_array($result) && isset($result['success']) && $result['success']) {
                    wp_send_json_success(array('message' => $result['message']));
                    return;
                }
            } else {
                throw new Exception('Google email fetch function not found');
            }
            
            global $wpdb;
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table name is safe and not user input
            $table_name = $wpdb->prefix . 'nexlifydesk_tickets';
            $recent_tickets_cache_key = 'nexlifydesk_recent_google_tickets_' . md5(gmdate('Y-m-d H:i:s', strtotime('-2 minutes')));
            $recent_tickets = wp_cache_get($recent_tickets_cache_key);

            if (false === $recent_tickets) {
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Table name is safe and not user input
                $recent_tickets = $wpdb->get_results(
                    $wpdb->prepare(
                        "SELECT ticket_id, subject, source, created_at 
                        FROM {$table_name} 
                        WHERE created_at >= %s 
                        AND source = 'email'
                        ORDER BY created_at DESC
                        LIMIT 10",
                        gmdate('Y-m-d H:i:s', strtotime('-2 minutes'))
                    )
                );
                wp_cache_set($recent_tickets_cache_key, $recent_tickets, '', 120);
            }
            
            $message = "Google email fetch completed successfully!\n";
            
            if (!empty($recent_tickets)) {
                $message .= "Found " . count($recent_tickets) . " recent email ticket(s):\n";
                foreach ($recent_tickets as $ticket) {
                    $message .= "- #{$ticket->ticket_id}: {$ticket->subject} ({$ticket->created_at})\n";
                }
            } else {
                $message .= "No new email tickets found. This could mean:\n";
                $message .= "- No new emails in Gmail\n";
                $message .= "- Emails are being filtered out (spam, duplicates)\n";
                $message .= "- All emails have been processed already\n";
            }
            
            wp_send_json_success(array('message' => $message));
            
        } catch (Exception $e) {
            /* translators: %s: Error message returned from the Google email fetch */
            wp_send_json_error(array('message' => sprintf(__('Google email fetch failed: %s', 'nexlifydesk'), $e->getMessage())));
        }
    }

    /**
     * Get the maximum allowed POST size from server configuration
     * 
     * @return int Maximum POST size in bytes
     */
    private static function get_max_post_size() {
        $post_max_size = ini_get('post_max_size');
        $upload_max_filesize = ini_get('upload_max_filesize');
        $post_max_bytes = self::convert_to_bytes($post_max_size);
        $upload_max_bytes = self::convert_to_bytes($upload_max_filesize);
        
        return min($post_max_bytes, $upload_max_bytes);
    }
    
    /**
     * Convert PHP ini size format to bytes
     * 
     * @param string $size Size string like "8M", "1G", "512K"
     * @return int Size in bytes
     */
    private static function convert_to_bytes($size) {
        $size = trim($size);
        $last = strtolower($size[strlen($size) - 1]);
        $size = (int) $size;
        
        switch ($last) {
            case 'g':
                $size *= 1024;
                // Fall through
            case 'm':
                $size *= 1024;
                // Fall through
            case 'k':
                $size *= 1024;
                break;
        }
        
        return $size;
    }

    /**
     * AWS IMAP Diagnostics - provides detailed information about IMAP setup, required in production.
     */
    public static function aws_diagnostics() {
        
        if (!current_user_can('nexlifydesk_manage_tickets') && !current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('You do not have permission.', 'nexlifydesk')));
        }

        if (
            !isset($_POST['nonce']) ||
            !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'nexlifydesk_aws_test')
        ) {
            wp_send_json_error(array('message' => __('Invalid nonce.', 'nexlifydesk')));
        }

        $diagnostics = [];
        $settings = get_option('nexlifydesk_imap_settings', []);
        $aws_configured = false;
        
        if (!empty($settings['aws_region']) && !empty($settings['aws_organization_id']) && 
            !empty($settings['aws_email']) && !empty($settings['aws_password'])) {
            $aws_configured = true;
        }
        
        if (!$aws_configured) {
            $diagnostics['aws_configuration'] = [
                'status' => 'WARNING',
                'message' => 'No AWS WorkMail credentials configured. Please configure AWS settings first before running diagnostics.'
            ];
            wp_send_json_success(array('diagnostics' => $diagnostics));
            return;
        }
        
        $diagnostics['imap_extension'] = [
            'status' => extension_loaded('imap') ? 'OK' : 'FAILED',
            'message' => extension_loaded('imap') ? 'IMAP extension is loaded' : 'IMAP extension is NOT loaded'
        ];
        
        $required_functions = ['imap_open', 'imap_search', 'imap_fetchheader', 'imap_body', 'imap_close', 'imap_last_error'];
        $missing_functions = [];
        
        foreach ($required_functions as $function) {
            if (!function_exists($function)) {
                $missing_functions[] = $function;
            }
        }
        
        $diagnostics['imap_functions'] = [
            'status' => empty($missing_functions) ? 'OK' : 'FAILED',
            'message' => empty($missing_functions) ? 'All required IMAP functions are available' : 'Missing functions: ' . implode(', ', $missing_functions)
        ];
        
        $ssl_transports = stream_get_transports();
        $has_ssl = in_array('ssl', $ssl_transports) && in_array('tls', $ssl_transports);
        
        $diagnostics['ssl_support'] = [
            'status' => $has_ssl ? 'OK' : 'FAILED',
            'message' => $has_ssl ? 'SSL/TLS support is available' : 'SSL/TLS support is NOT available'
        ];
        
        $is_ssl_enabled = (
            is_ssl() || 
            (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on') ||
            (isset($_SERVER['HTTP_X_FORWARDED_PROTO']) && $_SERVER['HTTP_X_FORWARDED_PROTO'] === 'https') ||
            (isset($_SERVER['HTTP_X_FORWARDED_SSL']) && $_SERVER['HTTP_X_FORWARDED_SSL'] === 'on') ||
            (isset($_SERVER['REQUEST_SCHEME']) && $_SERVER['REQUEST_SCHEME'] === 'https') ||
            (wp_parse_url(home_url(), PHP_URL_SCHEME) === 'https')
        );
        
        if (defined('NEXLIFYDESK_FORCE_SSL_ENABLED') && NEXLIFYDESK_FORCE_SSL_ENABLED) {
            $is_ssl_enabled = true;
        }
        
        $diagnostics['site_ssl'] = [
            'status' => $is_ssl_enabled ? 'OK' : 'WARNING',
            'message' => $is_ssl_enabled ? 'Site is running on HTTPS' : 'Site is NOT running on HTTPS (required for AWS WorkMail)'
        ];
        
        $aws_hosts_to_check = [
            'imap.mail.us-east-1.awsapps.com',
            'imap.mail.us-west-2.awsapps.com',
            'imap.mail.eu-west-1.awsapps.com'
        ];
        
        $connectivity_results = [];
        foreach ($aws_hosts_to_check as $host) {
            $ip = gethostbyname($host);
            $connectivity_results[] = ($ip !== $host) ? "$host: OK" : "$host: DNS resolution failed";
        }
        
        $diagnostics['aws_connectivity'] = [
            'status' => 'INFO',
            'message' => 'DNS Resolution Test: ' . implode(', ', $connectivity_results)
        ];
        
        $current_host = "imap.mail.{$settings['aws_region']}.awsapps.com";
        $diagnostics['current_config'] = [
            'status' => 'INFO',
            'message' => "Current AWS WorkMail host: {$current_host}, Organization: {$settings['aws_organization_id']}, Username: {$settings['aws_email']}"
        ];
        
        $aws_password = nexlifydesk_decrypt($settings['aws_password']);
        if (!empty($aws_password)) {
            try {
                $connection = @imap_open(
                    "{{$current_host}:993/imap/ssl}INBOX",
                    $settings['aws_email'],
                    $aws_password,
                    OP_READONLY | OP_SILENT
                );
                
                if ($connection) {
                    imap_close($connection);
                    $diagnostics['aws_connection_test'] = [
                        'status' => 'OK',
                        'message' => 'AWS WorkMail IMAP connection test successful with saved credentials'
                    ];
                } else {
                    $error = imap_last_error() ?: 'Connection failed';
                    $diagnostics['aws_connection_test'] = [
                        'status' => 'FAILED',
                        'message' => "AWS WorkMail IMAP connection test failed: {$error}"
                    ];
                }
            } catch (Exception $e) {
                $diagnostics['aws_connection_test'] = [
                    'status' => 'FAILED',
                    'message' => "AWS WorkMail connection test error: {$e->getMessage()}"
                ];
            }
        } else {
            $diagnostics['aws_connection_test'] = [
                'status' => 'WARNING',
                'message' => 'Cannot test AWS connection - password decryption failed'
            ];
        }
        
        $php_version = PHP_VERSION;
        $openssl_available = extension_loaded('openssl');
        
        $diagnostics['php_info'] = [
            'status' => $openssl_available ? 'OK' : 'WARNING',
            'message' => "PHP Version: {$php_version}, OpenSSL: " . ($openssl_available ? 'Available' : 'NOT Available')
        ];
        
        wp_send_json_success(array('diagnostics' => $diagnostics));
    }
}